/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.facade;

import java.sql.SQLException;

import jp.valtech.bts.connection.VoteDBConnection;
import jp.valtech.bts.dao.BtsDBException;
import jp.valtech.bts.dao.VoteDAO;
import jp.valtech.bts.data.Vote;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.util.Logging;

/**
 * {@link jp.valtech.bts.dao.VoteDAO}用のFacadeクラスです。
 * 
 * @author		<A href="mailto:s_imai@valtech.jp">S.Imai</A>
 * @version	Ver.0.8
 */
public class VoteDAOFacade implements Logging {

	/**
	 * 何もしない。
	 */
	public VoteDAOFacade() {
		;
	}

	/**
	 * 指定されたフィンガープリントに属する投票データ全件を取得します。
	 * 
	 * @param			fingerPrint			フィンガープリント
	 * @return			投票データ全件
	 * @throws 		BtsDBException
	 */
	public Vote[] get(String fingerPrint) throws BtsDBException {

		// DBコネクション取得
		VoteDBConnection dbcon = new VoteDBConnection();

		try {
			VoteDAO dao = new VoteDAO(dbcon);
			return dao.get(fingerPrint);

		} finally {
			dbcon.close();
			dbcon = null;

		}
	}

	
	/**
	 * 指定された投票データの投票数を1増やし、新しい投票データ全件を戻します。 指定された投票データが無い場合は新規投票データを挿入します。
	 * 
	 * @param			vote			投票データ
	 * @param			dbcon			DBコネクション
	 * @throws			SQLException
	 */
	public Vote[] cntPlus(Vote vote) throws BtsDBException, SQLException {

		// DBコネクション取得
		VoteDBConnection dbcon = new VoteDBConnection();

		try {
			VoteDAO dao = new VoteDAO(dbcon);
			int updateCnt = dao.cntPlus(vote);

			// 指定された投票データが無い場合
			if (updateCnt == 0) {

				// 新規投票データを挿入します
				dao.insert(vote);

			}

			dbcon.commit();

			// 新しい投票データ全件を戻す
			return dao.get(vote.getFingerPrint());

		} finally {
			dbcon.close();
			dbcon = null;
		}
	}
	

	/**
	 * 指定されたフィンガープリントに属する課題票の投票総数を返します。
	 * 
	 * @param			fingerPrint			フィンガープリント
	 * @return			課題票の投票総数
	 * @throws 		BtsDBException
	 */
	public int getSum(String fingerPrint) throws BtsDBException {

		// DBコネクション取得
		VoteDBConnection dbcon = new VoteDBConnection();

		try {
			VoteDAO dao = new VoteDAO(dbcon);
			return dao.getSum(fingerPrint);

		} finally {
			dbcon.close();
			dbcon = null;
		}
	}
	
	
	/**
	 * 指定の投票情報を登録します。
	 * 登録時には既に登録されている同一課題の登録情報を一旦クリアします。
	 * 
	 * @param		votes			課題票情報
	 * @throws		BtsDBException
	 */
	public void cleanInsert(Vote[] votes) throws BtsDBException {

		if(votes ==null || votes.length ==0) {
			return;
		}
		
		// DBコネクション取得
		VoteDBConnection dbcon = new VoteDBConnection();

		try {
	
			VoteDAO dao = new VoteDAO(dbcon);
			
			// 一旦削除
			dao.deleteByFingerPrint(votes[0].getFingerPrint());
			
			// 登録
			for (int i = 0; i < votes.length; i++) {
				dao.insert(votes[i]);
			}

			dbcon.commit();
		} catch (Exception e) {
			dbcon.rollback();

			// エラー情報の出力
			String msg = Messages.getString("VoteDAOFacade.0"); //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
			
		} finally {
			dbcon.close();
			dbcon = null;
		}
	}

}
