/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.facade;

import java.util.Date;
import java.util.List;

import jp.valtech.bts.connection.IssueDBConnection;
import jp.valtech.bts.dao.BtsDBException;
import jp.valtech.bts.dao.CommentHistoryDAO;
import jp.valtech.bts.dao.IssueDAO;
import jp.valtech.bts.dao.IssueHistoryDAO;
import jp.valtech.bts.data.Attachment;
import jp.valtech.bts.data.CommentHistory;
import jp.valtech.bts.data.Issue;
import jp.valtech.bts.data.IssueHistory;
import jp.valtech.bts.data.IssueType;
import jp.valtech.bts.data.MessagePacket;
import jp.valtech.bts.data.MessageType;
import jp.valtech.bts.ui.BtsException;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.ui.IBtsListener;
import jp.valtech.bts.util.BTSUtility;
import jp.valtech.bts.util.Logging;

/**
 * 課題票受信時に使うDAOの処理を集めたFacadeクラスです。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class ReceiveReleaseIssueFacade implements Logging {

	/** 課題票受信：受信した課題票をINSERT */
	private static final int RECEIVE_NEW_ISSUE = 0 ;
	/** 課題票受信：受信した課題票でUPDATE */
	private static final int RECEIVE_UPDATE_ISSUE = 1 ;
	/** 課題票受信：受信した課題票は競合 */
	private static final int RECEIVE_CONFLICT_ISSUE = 2 ;
	/** 課題票受信モード */
	private int receiveMode = RECEIVE_NEW_ISSUE;
	
	/** 受信した課題票 */
	private Issue receiveIssue;
	
	/** ローカルにある同一課題票のバージョン番号 */
	private Integer localVersion;
	
	/**
	 * 何もしない。
	 */
	public ReceiveReleaseIssueFacade() {
		;
	}

	
	/**
	 * 受信した課題票をローカルDBに保存します。課題票の種類により以下のように処理を分けます。
	 * <ul>
	 *   <li>受信した課題票と同一の課題票がローカルに無い。---受信した課題票をINSERTします。
	 *   <li>受信した課題票と同一の課題票がローカルに既にある。---受信した課題票でUPDATEします。
	 *   <li>受信した課題票と同一の課題票がローカルのものよりバージョンが古い。---受信した課題票を競合課題票として処理します。
	 * </ul>
	 * @param		receiveIssue		受信した課題票
	 * @param		histories			受信した課題票の履歴
	 * @param		comment				受信した課題票のコメント
	 * @throws		BtsException
	 */
	public void saveReceivedIssue(Issue receiveIssue) throws BtsException {
		
		IssueHistory[] histories = receiveIssue.getIssueHistories();
		CommentHistory[] comment = receiveIssue.getCommentHistories();

		// インスタンスに保持する
		this.receiveIssue = receiveIssue;
		
		//	DBコネクション取得
		IssueDBConnection dbcon = new IssueDBConnection();
		try {
			IssueDAO issueDAO = new IssueDAO( dbcon );
			Issue localIssue = issueDAO.getByFingerPrint(receiveIssue.getFingerPrint(), IssueType.RELEASE_VALUE);
			
			if(localIssue == null) {
				//### 新規登録。receiveIssueを登録する ###
				
				// 課題票登録
				addIssue(receiveIssue, dbcon);
				
				// 課題票履歴登録
				addIssueHistory(histories, dbcon);
				
				// 添付ファイル登録
				AttachmentFacade.addFromEntity(receiveIssue, dbcon);
				
				// コメント登録
				addCommentHistory(comment, dbcon);
				
				// 新規登録であることを設定
				receiveMode = RECEIVE_NEW_ISSUE;
				
			} else if( receiveIssue.getVersion().intValue() > localIssue.getVersion().intValue()  ) {
				//### 正常な更新。バージョン比較の結果、正常に受け付けることにする。 ###
				
				// 課題票更新 
				updateIssue(localIssue, receiveIssue, dbcon);
				
				// 課題票履歴を一旦クリア
				deleteIssueHistory(receiveIssue.getFingerPrint(), dbcon);
				// 課題票履歴を登録
				addIssueHistory(histories, dbcon);
				
				// 添付ファイルを一旦クリア
				AttachmentFacade.deleteAll(receiveIssue, IssueType.RELEASE_VALUE, dbcon);
				// 添付ファイル登録
				AttachmentFacade.addFromEntity(receiveIssue, dbcon);

				// コメントを一旦クリア
				deleteCommentHistory(comment, dbcon);
				// コメント登録
				addCommentHistory(comment, dbcon);

				// 更新であることを設定
				receiveMode = RECEIVE_UPDATE_ISSUE;
				
			} else {
				//### 競合発生。バージョン比較の結果receiveIssueをconflictとして処理する ###

				// 既に同一課題票でCONFLICTがある場合は削除する 
				deleteConflictIssue(receiveIssue, dbcon);
				
				// 課題票種別に「競合」を設定
				setConflict(receiveIssue, histories, comment);

				// 競合として課題票を登録
				addConflictIssue(receiveIssue, dbcon);
				
				// 競合として課題票履歴を登録
				addConflictIssueHistory(histories, dbcon);

				// 競合として添付ファイル登録
				AttachmentFacade.addFromEntity(receiveIssue, dbcon);

				// コメント登録
				addCommentHistory(comment, dbcon);

				// 競合であることを設定
				receiveMode = RECEIVE_CONFLICT_ISSUE;
				localVersion = localIssue.getVersion();
			}
			
			dbcon.commit();
		} catch (Exception e) {
			dbcon.rollback();

			// エラー情報の出力
			String msg = Messages.getString("ReceiveReleaseIssueFacade.0") ; //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
			throw new BtsException(e);
			
		} finally {
			dbcon.close();
			dbcon = null;
		}
	}
	
	
	/**
	 * 既に同一課題票でCONFLICTがある場合は削除します。
	 * 
	 * @param		receiveIssue		追加する課題票
	 * @param		dbcon				DBコネクション
	 * @throws		Exception
	 */
	private void deleteConflictIssue(Issue receiveIssue, IssueDBConnection dbcon) throws Exception {
		
		String fingerPrint = receiveIssue.getFingerPrint();
		IssueDAO issueDAO = new IssueDAO( dbcon );

		// 既に同一課題票のCONFLICTがあるかどうか確認する 
		Issue conflictIssue = issueDAO.getByFingerPrint(fingerPrint, IssueType.CONFLICT_VALUE);
		if(conflictIssue == null) {
			return;
		}

		// 課題票を削除
		issueDAO.deleteByFingerPrint(fingerPrint, IssueType.CONFLICT_VALUE);
		
		// 添付を削除
		AttachmentFacade.deleteAll(fingerPrint, IssueType.CONFLICT_VALUE, dbcon);

		// 課題票履歴を削除
		IssueHistoryDAO historyDAO = new IssueHistoryDAO(dbcon);
		historyDAO.deleteByFingerPrint(fingerPrint, IssueType.CONFLICT_VALUE);
		
		// 課題票コメント履歴を削除
		CommentHistoryDAO commentHistoryDAO = new CommentHistoryDAO(dbcon);
		commentHistoryDAO.deleteByFingerPrint(fingerPrint, IssueType.CONFLICT_VALUE);

		// 現在プラグインが保持しているリスナクラスを取得します。
		List listeners = BtsPlugin.getInstance().getListeners();
		
		// 各リスナにビューを最新表示をするよう通知します。
		for (int idx = 0; idx < listeners.size(); idx++) {
			IBtsListener listnener = (IBtsListener)listeners.get(idx);
			listnener.deleteIssue(conflictIssue);
		}
	}
	
	
	/**
	 * 競合課題票を削除します。
	 * 
	 * @param		receiveIssue		追加する課題票
	 * @param		dbcon				DBコネクション
	 * @throws		BtsDBException
	 */
	private void addConflictIssue(Issue receiveIssue, IssueDBConnection dbcon) throws BtsDBException {

		IssueDAO dao = new IssueDAO( dbcon );
		dao.addIssue( receiveIssue );
		
		// 現在プラグインが保持しているリスナクラスを取得します。
		List listeners = BtsPlugin.getInstance().getListeners();
		
		// 各リスナにビューを最新表示をするよう通知します。
		for (int idx = 0; idx < listeners.size(); idx++) {
			IBtsListener listnener = (IBtsListener)listeners.get(idx);
			listnener.addIssue(receiveIssue);
		}
	}
	
	
	/**
	 * 課題票種別に{@link IssueType#CONFLICT 「競合」}を設定します。
	 * ・{@link Issue#setType(String) 課題票の課題票種別}<br>
	 * ・{@link Attachment#setType(String) 添付ファイルの課題票種別}<br>
	 * ・{@link IssueHistory#setType(String) 課題票履歴の課題票種別}
	 * 
	 * @param		receiveIssue		課題票本体
	 * @param		histories			課題票の履歴
	 * @param		comment				課題票のコメント
	 */
	private void setConflict(Issue receiveIssue, IssueHistory[] histories, CommentHistory[] comment) {
	
		// 課題票本体の種別を「CONFLICT」に設定
		receiveIssue.setType(IssueType.CONFLICT_VALUE);

		// 課題票の添付ファイルの種別を「CONFLICT」に設定
		Attachment[] attachments =  receiveIssue.getAttachments();
		if(attachments != null) {
			for (int i = 0; i < attachments.length; i++) {
				attachments[i].setType(IssueType.CONFLICT_VALUE);
			}
		}
		
		// 課題票の履歴の種別を「CONFLICT」に設定
		for (int i = 0; i < histories.length; i++) {
			histories[i].setType(IssueType.CONFLICT_VALUE);
		}
		
		// コメント履歴の種別を「CONFLICT」に設定
		if(comment != null) {
			for (int i = 0; i < comment.length; i++) {
				comment[i].setType(IssueType.CONFLICT_VALUE);
			}
		}
	}
	
	
	/**
	 * 課題票の追加処理を行います。以下の処理を行います。
	 * <ul>
	 *   <li>追加した課題票の情報をローカルDBに登録
	 *   <li>「追加した課題票」と「メッセージ」を各Viewに再描画
	 * </ul>
	 * 
	 * @param		receiveIssue		追加する課題票
	 * @param		dbcon				DBコネクション
	 */
	private void addIssue(Issue receiveIssue, IssueDBConnection dbcon) throws BtsDBException {

		// 課題票を登録
		IssueDAO dao = new IssueDAO( dbcon );
		dao.addIssue( receiveIssue );
			
		// 現在プラグインが保持しているリスナクラスを取得します。
		List listeners = BtsPlugin.getInstance().getListeners();
		
		// 各リスナにビューを最新表示をするよう通知します。
		for (int idx = 0; idx < listeners.size(); idx++) {
			IBtsListener listnener = (IBtsListener)listeners.get(idx);
			listnener.addIssue(receiveIssue);
		}
	}
	

	/**
	 * 課題票コメントを登録します。
	 * 
	 * @param		comment			登録するコメント
	 * @param		dbcon			DBコネクション
	 * @throws		BtsDBException
	 */
	private void addCommentHistory(CommentHistory[] comment, IssueDBConnection dbcon)  throws BtsDBException {

		if(comment==null || comment.length==0) {
			return;
		}
		
		// 履歴を登録
		CommentHistoryDAO dao = new CommentHistoryDAO( dbcon );
		for (int i = 0; i < comment.length; i++) {
			dao.addComment( comment[i] );
		}
	}


	/**
	 * 課題票コメントを削除します。
	 * 
	 * @param		comment			削除対象のコメント
	 * @param		dbcon			DBコネクション
	 * @throws		BtsDBException
	 */
	private void deleteCommentHistory(CommentHistory[] comment, IssueDBConnection dbcon)  throws BtsDBException {

		if(comment==null || comment.length==0) {
			return;
		}

		// 履歴を削除
		CommentHistoryDAO dao = new CommentHistoryDAO( dbcon );
		dao.deleteByFingerPrint(comment[0].getFingerPrint(), IssueType.RELEASE_VALUE);
	}

	
	
	/** 
	 * 課題票の履歴を登録します。
	 * 
	 * @param		histories			課題票の履歴
	 * @param		dbcon				DBコネクション
	 * @throws		BtsDBException
	 */
	private void addIssueHistory(IssueHistory[] histories, IssueDBConnection dbcon)  throws BtsDBException {

		// 履歴を登録
		IssueHistoryDAO dao = new IssueHistoryDAO( dbcon );
		for (int i = 0; i < histories.length; i++) {
			dao.addHistory( histories[i] );
		}
	}
		
	
	/**
	 * 課題票履歴を削除します。
	 * 
	 * @param		fingerprint			削除対象の課題票のFingerPrint
	 * @param		dbcon				DBコネクション
	 * @throws		BtsDBException
	 */
	private void deleteIssueHistory(String fingerprint, IssueDBConnection dbcon)  throws BtsDBException {

		// 履歴を登録
		IssueHistoryDAO dao = new IssueHistoryDAO( dbcon );
		dao.deleteByFingerPrint(fingerprint, IssueType.RELEASE_VALUE);
	}

	
	/** 
	 * 競合課題票の履歴を登録します。
	 * 
	 * @param		histories			競合課題票の履歴
	 * @param		dbcon				DBコネクション
	 * @throws		BtsDBException
	 */
	private void addConflictIssueHistory(IssueHistory[] histories, IssueDBConnection dbcon)  throws BtsDBException {
		IssueHistoryDAO dao = new IssueHistoryDAO( dbcon );

		// 既に存在する場合があるので一旦削除してから登録
		dao.deleteByFingerPrint(histories[0].getFingerPrint(), IssueType.CONFLICT_VALUE);

		for (int i = 0; i < histories.length; i++) {
			dao.addHistory( histories[i] );
		}
	}

	
	/**
	 * 課題票の更新処理を行います。
	 * <ul>
	 *   <li>受信した課題票情報でローカルDBを更新
	 *   <li>「更新した課題票」と「メッセージ」を各Viewに再描画
	 * </ul>
	 * 
	 * @param		issue			追加する課題票
	 * @param		type			課題票に設定する{@link jp.valtech.bts.data.IssueType 課題票種別}
	 */
	private int updateIssue(Issue localIssue, Issue receiveIssue, IssueDBConnection dbcon) throws BtsDBException {

		// ローカルDBに登録。失敗時はExeptionがthrowされるので、以降の処理は行われません。
		// 課題票を更新
		IssueDAO dao = new IssueDAO( dbcon );
		int updateCount = dao.modifyIssue( receiveIssue , IssueType.RELEASE_VALUE );
		
		// 現在プラグインが保持しているリスナクラスを取得します。
		List listeners = BtsPlugin.getInstance().getListeners();
		
		// 各リスナにビューを最新表示をするよう通知します。
		for (int idx = 0; idx < listeners.size(); idx++) {
			IBtsListener listnener = (IBtsListener)listeners.get(idx);
			listnener.updateIssue(localIssue, receiveIssue);
		}
		
		return updateCount;
	}

	
	
	/**
	 * 課題票受信時のメッセージを生成して保存します。
	 * 
	 * @param		fromUser			送信したユーザ
	 * @param		fromAddr			送信したユーザのアドレス
	 */
	public void saveReceiveMessage(String fromUser, String fromAddr) {
		MessagePacket message = new MessagePacket();

		// 送信ユーザ名設定
		message.setFromUser(fromUser);
		
		// 受信日時設定
		message.setSent(System.currentTimeMillis());
		
		// 課題票のフィンガープリント設定
		message.setFingerPrint(receiveIssue.getFingerPrint());
		
		// 競合かどうかで処理を分ける
		if(receiveMode == RECEIVE_CONFLICT_ISSUE) {
			// 競合のメッセージ種別設定
			message.setMessageType(MessageType.CONFLICT_VALUE);
			// 競合のメッセージ本文設定
			message.setMessage(createConflictMessage(fromUser, fromAddr));
			// 課題票の種別設定
			message.setIssueType(IssueType.CONFLICT_VALUE);

		} else {
			// メッセージ種別設定
			message.setMessageType(MessageType.ISSUE_VALUE);
			// メッセージ本文設定
			message.setMessage(createReceiveMessage(fromUser, fromAddr));
			// 課題票の種別設定
			message.setIssueType(IssueType.RELEASE_VALUE);
		}

        // 保存 ＆ ビュー表示更新
        MessagePacketFacade facade = new MessagePacketFacade();
        facade.addMessage(message);
	}


	/**
	 * 課題票受信メッセージを生成します。
	 * 
	 * @param		fromUser			課題票を送信した人
	 * @param		fromAddr			課題票を送信した人のIP
	 * @return		課題票受信メッセージ
	 */
	private String createReceiveMessage(String fromUser, String fromAddr) {
		StringBuffer msg = new StringBuffer();

		// １行目の課題票情報
		msg.append(Messages.getString("ReceiveReleaseIssueFacade.1")).append(receiveIssue.getDisplayIssueID()).append("]"); //$NON-NLS-1$
		msg.append(receiveIssue.getTitle());
		
		// 送信元情報
		msg.append(Messages.getString("ReceiveReleaseIssueFacade.2")); //$NON-NLS-1$
		msg.append("\n  (").append(fromAddr).append(")    ").append(fromUser);
		
		// 受信日
		msg.append(Messages.getString("ReceiveReleaseIssueFacade.3")); //$NON-NLS-1$
		msg.append( BTSUtility.formatDate(new Date()) );

		// バージョン
		msg.append(Messages.getString("ReceiveReleaseIssueFacade.4")); //$NON-NLS-1$
		msg.append("\n  Ver.").append( receiveIssue.getVersion());
		
		return msg.toString();
	}
	
	
	/**
	 * 課題票競合メッセージを生成します。
	 * 
	 * @param		fromUser			課題票を送信した人
	 * @param		fromAddr			課題票を送信した人のIP
	 * @return		競合メッセージ
	 */
	private String createConflictMessage(String fromUser, String fromAddr) {
		StringBuffer msg = new StringBuffer();

		// １行目の課題票情報
		msg.append(Messages.getString("ReceiveReleaseIssueFacade.5")).append(receiveIssue.getDisplayIssueID()).append("]"); //$NON-NLS-1$
		msg.append(receiveIssue.getTitle());
	
		// 競合メッセージ
		msg.append(Messages.getString("ReceiveReleaseIssueFacade.6")); //$NON-NLS-1$
		msg.append(Messages.getString("ReceiveReleaseIssueFacade.7")); //$NON-NLS-1$
		msg.append(Messages.getString("ReceiveReleaseIssueFacade.8")); //$NON-NLS-1$

		// 送信元情報
		msg.append(Messages.getString("ReceiveReleaseIssueFacade.9")); //$NON-NLS-1$
		msg.append("\n  (").append(fromAddr).append(")    ").append(fromUser);
		
		// 受信日
		msg.append(Messages.getString("ReceiveReleaseIssueFacade.10")); //$NON-NLS-1$
		msg.append( BTSUtility.formatDate(new Date()) );

		// バージョン
		msg.append(Messages.getString("ReceiveReleaseIssueFacade.11")); //$NON-NLS-1$
		msg.append(Messages.getString("ReceiveReleaseIssueFacade.12")).append( receiveIssue.getVersion()); //$NON-NLS-1$
		msg.append(Messages.getString("ReceiveReleaseIssueFacade.13")).append(localVersion); //$NON-NLS-1$

		return msg.toString();
	}


	/**
	 * インポートした課題票を登録します。
	 * インポートした課題票と同一の課題票がある場合は、それを一旦削除してからインポートの課題票を登録します。
	 * 
	 * @param		importIssue			インポートした課題票
	 * @throws		BtsException
	 */
	public void importReleaseIssue(Issue importIssue) throws BtsException {
		
		IssueHistory[] histories = importIssue.getIssueHistories();
		CommentHistory[] comment = importIssue.getCommentHistories();

		this.receiveIssue = importIssue;
		//	DBコネクション取得
		IssueDBConnection dbcon = new IssueDBConnection();
		try {
			// 現在プラグインが保持しているリスナクラスを取得します。
			List listeners = BtsPlugin.getInstance().getListeners();

			// 課題票を一旦削除
			IssueDAO issueDAO = new IssueDAO( dbcon );
			int deleteCnt = issueDAO.deleteByFingerPrint(importIssue.getFingerPrint(), IssueType.RELEASE_VALUE);
			if(deleteCnt != 0) {
				// 各リスナにビューを最新表示をするよう通知します。
				for (int idx = 0; idx < listeners.size(); idx++) {
					IBtsListener listnener = (IBtsListener)listeners.get(idx);
					listnener.deleteIssue(importIssue);
				}
			}

			// 課題票を登録
			issueDAO.addIssue(importIssue);
			// 各リスナにビューを最新表示をするよう通知します。
			for (int idx = 0; idx < listeners.size(); idx++) {
				IBtsListener listnener = (IBtsListener)listeners.get(idx);
				listnener.addIssue(importIssue);
			}
			
			// 課題票履歴を一旦クリア
			deleteIssueHistory(importIssue.getFingerPrint(), dbcon);
			// 課題票履歴を登録
			addIssueHistory(histories, dbcon);
			
			// 添付ファイルを一旦クリア
			AttachmentFacade.deleteAll(importIssue, IssueType.RELEASE_VALUE, dbcon);
			// 添付ファイル登録
			AttachmentFacade.addFromEntity(importIssue, dbcon);

			// コメントを一旦クリア
			deleteCommentHistory(comment, dbcon);
			// コメント登録
			addCommentHistory(comment, dbcon);

			
			dbcon.commit();

		} catch (Exception e) {
			dbcon.rollback();

			// エラー情報の出力
			String msg = Messages.getString("ReceiveReleaseIssueFacade.0") ; //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
			throw new BtsException(e);
			
		} finally {
			dbcon.close();
			dbcon = null;
		}
	}
	

	
	/**
	 * 課題票受信時のメッセージを生成して保存します。
	 * 
	 * @param		fromUser			送信したユーザ
	 * @param		fromAddr			送信したユーザのアドレス
	 */
	public void saveImportMessage() {
		MessagePacket message = new MessagePacket();

		// 送信ユーザ名設定
		IssueHistory[] histories = receiveIssue.getIssueHistories();
		String userName = histories[0].getUpdateUser();
		message.setFromUser(userName);
		
		// 受信日時設定
		message.setSent(System.currentTimeMillis());
		
		// 課題票のフィンガープリント設定
		message.setFingerPrint(receiveIssue.getFingerPrint());
		
		// メッセージ種別設定
		message.setMessageType(MessageType.ISSUE_VALUE);

		// メッセージ本文設定
		StringBuffer msg = new StringBuffer();
		msg.append(Messages.getString("ReceiveReleaseIssueFacade.15")).append(receiveIssue.getDisplayIssueID()).append("]"); //$NON-NLS-1$
		msg.append(receiveIssue.getTitle());
	
		// 送信元情報
		msg.append(Messages.getString("ReceiveReleaseIssueFacade.2")); //$NON-NLS-1$
		msg.append("\n  ").append(userName);

		msg.append(Messages.getString("ReceiveReleaseIssueFacade.14")); //$NON-NLS-1$

		message.setMessage(msg.toString());
		
		// 課題票の種別設定
		message.setIssueType(IssueType.RELEASE_VALUE);

        // 保存 ＆ ビュー表示更新
        MessagePacketFacade facade = new MessagePacketFacade();
        facade.addMessage(message);
	}

}
