/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.facade;

import java.util.Date;
import java.util.List;

import jp.valtech.bts.connection.IssueDBConnection;
import jp.valtech.bts.dao.AttachmentDAO;
import jp.valtech.bts.dao.CommentHistoryDAO;
import jp.valtech.bts.dao.IssueDAO;
import jp.valtech.bts.dao.IssueHistoryDAO;
import jp.valtech.bts.data.Issue;
import jp.valtech.bts.data.IssueHistory;
import jp.valtech.bts.data.IssueType;
import jp.valtech.bts.data.MessagePacket;
import jp.valtech.bts.data.MessageType;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.ui.IBtsListener;
import jp.valtech.bts.util.BTSUtility;
import jp.valtech.bts.util.Logging;

/**
 * 課題票受信時に使うDAOの処理を集めたFacadeクラスです。
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class ReceiveGarbageIssueFacade implements Logging {

	/** 更新したかどうかのフラグ */
	private boolean changeGarbage = false;
	
	/** 受信した課題票 */
	private Issue receiveIssue;

	
	/**
	 * 何もしない。
	 */
	public ReceiveGarbageIssueFacade() {
		;
	}

	
	/**
	 * 指定のFingerPrintを持つ課題票をごみ箱に入れます。
	 * 
	 * @param		receiveIssue		課題票
	 */
	public void changeGarbageIssue(Issue receiveIssue) {
		this.receiveIssue = receiveIssue;
		
		IssueDBConnection dbcon = new IssueDBConnection();
		try {
			
			IssueDAO issueDAO = new IssueDAO( dbcon );

			String fingerPrint = receiveIssue.getFingerPrint();
			
			// 既にGARBAGEバージョンの課題票がある場合は無視
			Issue garbageIssue = issueDAO.getByFingerPrint(fingerPrint, IssueType.GARBAGE_VALUE);
			if(garbageIssue != null) {
				this.changeGarbage = false;
				return;
			}

			// RELEASEバージョンの課題票がない場合は無視
			Issue releaseIssue = issueDAO.getByFingerPrint(fingerPrint, IssueType.RELEASE_VALUE);
			if(releaseIssue == null) {
				this.changeGarbage = false;
				return;
			}
			
			//### 受信した課題票をGARBAGEバージョンに変える ###

			// 課題票の種別をごみ箱に変更
			issueDAO.changeGarbageVersion(fingerPrint);
			
			// 添付の種別をごみ箱に変更
			AttachmentDAO attachmentDAO = new AttachmentDAO( dbcon );
			attachmentDAO.changeGarbageVersion(fingerPrint);

			// 課題票履歴の種別をごみ箱に変更
			IssueHistoryDAO historyDAO = new IssueHistoryDAO(dbcon);
			historyDAO.changeGarbageVersion(fingerPrint);

			// 課題票履歴にごみ箱履歴を追加
	        IssueHistory[] history = receiveIssue.getIssueHistories();
	        if(history != null && history.length != 0) {
	        	historyDAO.addHistory( history[0] );
	        }

	        // 課題票コメント履歴の種別をごみ箱に変更
			CommentHistoryDAO commentHistoryDAO = new CommentHistoryDAO(dbcon);
			commentHistoryDAO.changeGarbageVersion(fingerPrint);
			
			
			// ビュー表示を最新にする
			List listeners = BtsPlugin.getInstance().getListeners();
			for (int idx = 0; idx < listeners.size(); idx++) {
				IBtsListener listnener = (IBtsListener)listeners.get(idx);
				listnener.updateIssue(releaseIssue, receiveIssue);
			}

			this.changeGarbage = true;
			
			dbcon.commit();
		
		} catch (Exception e) {
			dbcon.rollback();

			// エラー情報の出力
			String msg = Messages.getString("ReceiveGarbageIssueFacade.0"); //$NON-NLS-1$
			logger.fatal(msg, e);
			BtsPlugin.getInstance().error(msg, e);
		} finally {
			dbcon.close();
			dbcon = null;
		}
	}


	/**
	 * 課題票受信時のメッセージを生成して保存します。
	 * 
	 */
	public void saveReceiveMessage() {
		
		IssueHistory[] history = receiveIssue.getIssueHistories();
        String fromUser = history[0].getUpdateUser();
        saveReceiveMessage(fromUser, null) ;
	}

	
	/**
	 * 課題票受信時のメッセージを生成して保存します。
	 * 
	 * @param		fromUser			送信したユーザ
	 * @param		fromAddr			送信したユーザのアドレス
	 */
	public void saveReceiveMessage(String fromUser, String fromAddr) {
		
		MessagePacket message = new MessagePacket();

		// 送信ユーザ名設定
		message.setFromUser(fromUser);
		
		// 受信日時設定
		message.setSent(new Date());
		
		// 課題票のフィンガープリント設定
		message.setFingerPrint(receiveIssue.getFingerPrint());
		
		// 課題票の種別設定
		message.setIssueType(IssueType.GARBAGE_VALUE);
		
		// メッセージ種別設定
		message.setMessageType(MessageType.ISSUE_VALUE);
		// メッセージ本文設定
		message.setMessage(createReceiveMessage(fromUser, fromAddr));

        // 保存 ＆ ビュー表示更新
        MessagePacketFacade facade = new MessagePacketFacade();
        facade.addMessage(message);
	}


	/**
	 * 課題票受信メッセージを生成します。
	 * 
	 * @param		fromUser			課題票を送信した人
	 * @param		fromAddr			課題票を送信した人のIP
	 * @return		課題票受信メッセージ
	 */
	private String createReceiveMessage(String fromUser, String fromAddr) {
		StringBuffer msg = new StringBuffer();

		// １行目の課題票情報
		msg.append(Messages.getString("ReceiveGarbageIssueFacade.1")).append(receiveIssue.getDisplayIssueID()).append("]"); //$NON-NLS-1$
		msg.append(receiveIssue.getTitle());
		
		// 送信元情報
		msg.append(Messages.getString("ReceiveGarbageIssueFacade.2")); //$NON-NLS-1$
        
		if(fromAddr ==null) {
			msg.append("    ").append(fromUser);
		} else {
			msg.append("\n  (").append(fromAddr).append(")    ").append(fromUser);
		}

		
		// 受信日
		msg.append(Messages.getString("ReceiveGarbageIssueFacade.3")); //$NON-NLS-1$
		msg.append( BTSUtility.formatDate(new Date()) );

		// バージョン
		msg.append(Messages.getString("ReceiveGarbageIssueFacade.4")); //$NON-NLS-1$
		msg.append(Messages.getString("ReceiveGarbageIssueFacade.5")); //$NON-NLS-1$

		// 更新が無かった場合
		if(!changeGarbage) {
			msg.append(Messages.getString("ReceiveGarbageIssueFacade.6")); //$NON-NLS-1$
		}

		return msg.toString();
	}
	
}
