/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.facade;

import java.util.ArrayList;

import jp.valtech.bts.connection.IssueDBConnection;
import jp.valtech.bts.dao.AttachmentDAO;
import jp.valtech.bts.data.Attachment;
import jp.valtech.bts.data.Issue;
import jp.valtech.bts.data.IssueType;
import jp.valtech.bts.util.AttachmentUtility;
import jp.valtech.bts.util.Logging;

/**
 * {@link jp.valtech.bts.dao.AttachmentDAO}用のFacadeクラスです。
 * 添付ファイル情報を元に以下の処理を連動して処理します。
 * <ul>
 *   <li>ファイルシステムへの書き込みまたは削除
 *   <li>添付情報をDBへ反映
 * </ul>
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class AttachmentFacade implements Logging {

	
	/**
	 * 更新時に追加指定された添付ファイルを追加します。<br>
	 * ・{@link AttachmentUtility#writeWorkspaceFromLocal(Attachment[])}を呼び出して、ファイルシステム上に指定の添付ファイルを作成します。<br>
	 * ・{@link AttachmentDAO#addAttachment(Attachment)}を呼び出して、DBに添付ファイル情報を追加します。
	 * 
	 * @param		deleteList			削除対象の添付ファイル情報
	 * @param		dbcon				DBコネクション
	 * @throws		Exception
	 */
	public static void add(Issue issue, IssueDBConnection dbcon) throws Exception {

		// 添付ファイルをワークスペース内に保存
		AttachmentUtility utility = new AttachmentUtility();
		Attachment[] addList =  utility.writeWorkspaceFromLocal(issue.getAttachments());
		
		// 保存情報が帰ってこない場合は新規登録が無いということなので、以降の処理をやらない
		if(addList == null || addList.length==0) {
			return;
		}
		
		// ワークスペース内に保存した情報をDBに登録
		AttachmentDAO attachmentDAO = new AttachmentDAO( dbcon );
		for (int i = 0; i < addList.length; i++) {
			logger.debug("AddAttachment:" + addList[i].getDisplayName());
			attachmentDAO.addAttachment(addList[i]);
		}
	}

	
	/**
	 * 更新時に削除指定された添付ファイルを削除します。<br>
	 * ・{@link AttachmentUtility#delete(Attachment[])}を呼び出して、ファイルシステム上から削除します。<br>
	 * ・{@link AttachmentDAO#delete(Attachment)}を呼び出して、DB上から添付ファイル情報を削除します。
	 * 
	 * @param		deleteList			削除対象の添付ファイル情報
	 * @param		dbcon				DBコネクション
	 * @throws		Exception
	 */
	public static void delete(Attachment[] deleteList, IssueDBConnection dbcon) throws Exception {

		if( deleteList == null || deleteList.length==0) {
			return;
		}
		
		// ワークスペース内の添付ファイルを削除
		AttachmentUtility utility = new AttachmentUtility();
		utility.delete(deleteList);
		
		// DBに登録されている添付ファイル情報を削除
		AttachmentDAO attachmentDAO = new AttachmentDAO( dbcon );
		for (int i = 0; i < deleteList.length; i++) {
			logger.debug("DeleteAttachment:" + deleteList[i].getDisplayName());
			attachmentDAO.delete(deleteList[i]);
		}
	}
	

	/**
	 * 課題票で持つ全ての登録済みの添付ファイルを削除します。<br>
	 * ・{@link AttachmentUtility#delete(Attachment[])}を呼び出して、ファイルシステム上から削除します。<br>
	 * ・{@link AttachmentDAO#delete(Attachment)}を呼び出して、DB上から添付ファイル情報を削除します。
	 * ・削除対象は、「第一引数の課題票が持つFingerPrint」と「第２引数の<code>type</code>」に一致する登録済みの添付情報です。
	 * 
	 * @param		issue			課題票
	 * @param		type			課題票種別
	 * @param		dbcon			DBコネクション
	 * @throws		Exception
	 */
	public static void deleteAll(Issue issue, String type, IssueDBConnection dbcon) throws Exception {
		deleteAll(issue.getFingerPrint(), type, dbcon);
	}


	/**
	 * 課題票で持つ全ての登録済みの添付ファイルを削除します。<br>
	 * ・{@link AttachmentUtility#delete(Attachment[])}を呼び出して、ファイルシステム上から削除します。<br>
	 * ・{@link AttachmentDAO#delete(Attachment)}を呼び出して、DB上から添付ファイル情報を削除します。
	 * ・削除対象は、「第一引数の課題票が持つFingerPrint」と「第２引数の<code>type</code>」に一致する登録済みの添付情報です。
	 * 
	 * @param		fingerPrint		FingerPrint
	 * @param		type			課題票種別
	 * @param		dbcon			DBコネクション
	 * @throws		Exception
	 */
	public static void deleteAll(String fingerPrint, String type, IssueDBConnection dbcon) throws Exception {
		
		AttachmentDAO attachmentDAO = new AttachmentDAO( dbcon );
		
		// 削除対照リストを作成
		ArrayList deleteList 
			= attachmentDAO.getByFingerPrint(fingerPrint, type);
		if( deleteList == null || deleteList.size()==0) {
			return;
		}
		Attachment[] deleteAttachments = (Attachment[])deleteList.toArray(new Attachment[0]);
		
		// ワークスペース内の添付ファイルを削除
		AttachmentUtility utility = new AttachmentUtility();
		utility.delete(deleteAttachments);
		
		// ログ用
		if(logger.isDebugEnabled()) {
			for (int i = 0; i < deleteAttachments.length; i++) {
				logger.debug("DeleteAllAttachment:" + deleteAttachments[i].getDisplayName());
			}
		}

		// DBに登録されている添付ファイル情報を削除
		attachmentDAO.deleteByFingerPrint(fingerPrint, type);
	}


	/**
	 * 課題票で持つ登録済みの添付ファイルのクローンを作成します。<br>
	 * ・{@link AttachmentUtility#clone(Attachment[])}を呼び出して、ファイルシステムにクローンの添付ファイルを登録します。<br>
	 * ・{@link AttachmentDAO#addAttachment(Attachment)}を呼び出して、DBに添付ファイル情報を登録します。
	 * 
	 * @param		issue			課題票
	 * @param		dbcon			DBコネクション
	 * @throws		Exception
	 */
	public static void clone(Issue issue, IssueDBConnection dbcon) throws Exception {

		// 添付ファイルをワークスペース内に保存
		AttachmentUtility utility = new AttachmentUtility();
		Attachment[] cloneList =  utility.clone(issue.getAttachments());
		
		// 保存情報が帰ってこない場合は新規登録が無いということなので、以降の処理をやらない
		if(cloneList == null || cloneList.length==0) {
			return;
		}
		
		// ワークスペース内に保存した情報をDBに登録
		AttachmentDAO attachmentDAO = new AttachmentDAO( dbcon );
		for (int i = 0; i < cloneList.length; i++) {
			logger.debug("CloneAttachment:" + cloneList[i].getDisplayName());
			attachmentDAO.addAttachment(cloneList[i]);
		}
	}
	

	/**
	 * 課題票情報が持つバイト配列の添付ファイル実体をワークスペース内に出力します。
	 * 出力したファイル情報をDBに登録します。
	 * 
	 * @param		issue			課題票情報
	 * @param		dbcon			データベースコネクション
	 * @throws		Exception
	 */
	public static void addFromEntity(Issue issue, IssueDBConnection dbcon) throws Exception {

		// バイト配列で持つ添付ファイルをワークスペース上にファイル出力する
		Attachment[] attachments= issue.getAttachments();
		if(attachments==null) {
			return;
		}
		
		AttachmentUtility utility = new AttachmentUtility();
		utility.writeWorkspaceFromEntity(attachments);
		
		// ワークスペース内に出力した情報をDBに登録
		AttachmentDAO attachmentDAO = new AttachmentDAO( dbcon );
		for (int i = 0; i < attachments.length; i++) {
			logger.debug("AddAttachment:" + attachments[i].getDisplayName());
			attachmentDAO.addAttachment(attachments[i]);
		}
	}
	

	/**
	 * DRAFT課題票が持つ添付ファイルの課題票種別を「DRAFT」から「RELEASE」に置き換えます。<br>
	 * <br>
	 * ・一旦、課題票種別'R'の添付情報を全部削除します。<br>
	 * ・課題票種別'D'の添付情報を全て'D'に置き換えます。<br>
	 * 
	 * @param		issue			課題票情報
	 * @param		dbcon			データベースコネクション
	 * @throws		Exception
	 */
	public static void replaceReleaseVersion(Issue issue, IssueDBConnection dbcon) throws Exception {

		// 課題票種別'R'の添付情報を一旦全部削除
		deleteAll(issue, IssueType.RELEASE_VALUE, dbcon);

		// 'D'→'R'に置き換える
		AttachmentDAO attachmentDAO = new AttachmentDAO( dbcon );
		attachmentDAO.replaceReleaseVersion(issue.getFingerPrint());
	}
	
}
