/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.dao;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;

import jp.valtech.bts.connection.MessageDBConnection;
import jp.valtech.bts.data.MessagePacket;

import org.apache.commons.dbutils.QueryRunner;
import org.apache.commons.dbutils.ResultSetHandler;
import org.apache.commons.dbutils.handlers.ArrayListHandler;
import org.apache.commons.dbutils.handlers.BeanListHandler;
import org.apache.commons.dbutils.handlers.ScalarHandler;

/**
 * <dl><dt><b>メッセージ用のDAO</b></dt>
 * <dd>
 * </dd>
 * <dt><b>使い方</b></dt>
 * <dd>
 * </dd>
 * </dl>
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class MessagePacketDAO extends DataBaseAccessor {

	/** テーブル作成SQL */
	private static final String CREATE_TABLE 
			= "CREATE TABLE MESSAGEPACKET" 
			+ "  ( MESSAGEID      INTEGER NOT NULL IDENTITY PRIMARY KEY" 
			+ "  , MESSAGETYPE    CHAR(1)" 
			+ "  , FROMUSER       VARCHAR(30)"
			+ "  , SENT           TIMESTAMP" 
			+ "  , MESSAGE        LONGVARCHAR" 
			+ "  , ATTACHFILENAME VARCHAR(60)" 
			+ "  , FINGERPRINT    VARCHAR(40)" 
			+ "  , ISSUETYPE      CHAR(1)" 
			+ "  , OPENED         BOOLEAN DEFAULT FALSE" 
			+ "  )";

	/** テーブル削除SQL */
	private static final String DROP_TABLE
			= "DROP TABLE MESSAGEPACKET CASCADE;";

	/** データ登録SQL */
	private static final String INSERT_SQL
			= "INSERT INTO MESSAGEPACKET"
			+ "  ( MESSAGETYPE"
			+ "  , FROMUSER"
			+ "  , SENT" 
			+ "  , MESSAGE" 
			+ "  , ATTACHFILENAME" 
			+ "  , FINGERPRINT"
			+ "  , ISSUETYPE"
			+  " , OPENED"
			+ "  ) VALUES (?, ?, ?, ?, ?, ?, ?, FALSE)";

	/** オープン済みに設定するSQL */
	private static final String SET_OPENED 
			= "UPDATE MESSAGEPACKET SET OPENED=TRUE WHERE MESSAGEID=?";

	/** データを全件取得するSQL */
	private static final String SELECT_ALL
			= "SELECT"
			+ "   MESSAGEID"
			+ " , MESSAGETYPE"
			+ " , FROMUSER"
			+ " , SENT"
			+ " , MESSAGE"
			+ " , ATTACHFILENAME"
			+ " , FINGERPRINT"
			+ " , ISSUETYPE"
			+ " , nvl(OPENED, false) OPENED"
			+ " FROM MESSAGEPACKET" ;
	
	/** 送信者一覧を取得するSQL */
	private static final String GET_FROMUSER
			= "SELECT DISTINCT FROMUSER FROM MESSAGEPACKET";

	/** データ削除SQL */
	private static final String DELETE_SQL
			= "DELETE FROM MESSAGEPACKET WHERE MESSAGEID=?";

	/** データ削除SQL */
	private static final String SELECT_MAX_ID
			= "SELECT MAX(MESSAGEID) FROM MESSAGEPACKET;";
	
	/**
	 * <DL><DT><B>コンストラクタ</B>
	 *  <DD></DD>
	 * </DL>
	 * @param connection
	 */
	public MessagePacketDAO( MessageDBConnection connection ) {
		super( connection );
	}

	/**
	 * テーブルを生成します。
	 * 
	 * @throws		BtsDBException
	 */
	public void createTable() throws BtsDBException {
		try {
			// テーブル生成用SQL実行
			new QueryRunner().update( this.con, CREATE_TABLE );
			
		} catch (Exception e) {
			throw new BtsDBException(e);
		}
	}

	
	/**
	 * テーブルを削除します。
	 * 
	 * @throws		BtsDBException
	 */
	public void dropTable() throws BtsDBException {
		try {
			// テーブル削除用SQL実行
			new QueryRunner().update( this.con, DROP_TABLE );	
		} catch (Exception e) {
			throw new BtsDBException(e);
		}
	}

	/**
	 * <DL><DT><B>データを登録します。</B></DL>
	 *  <DD></DD>
	 * </DL>
	 * @param messageList
	 * @throws BtsDBException
	 */
	public void addMessage( MessagePacket messagePacket ) throws BtsDBException {
		try {
			List param = new ArrayList();
			param.add( messagePacket.getMessageType() );
			param.add( messagePacket.getFromUser() );
			if( messagePacket.getSent() != null ){
				param.add( new Timestamp( messagePacket.getSent().getTime() ) );
			}else{
				param.add( null );
			}
			param.add( messagePacket.getMessage() );
			param.add( messagePacket.getAttachFileName());
			param.add( messagePacket.getFingerPrint() );
			param.add( messagePacket.getIssueType() );

			new QueryRunner().update( this.con, INSERT_SQL, param.toArray() );	
			
			
		} catch (Exception e) {
			throw new BtsDBException(e);
		}
	}

	/**
	 * <DL><DT><B>指定のメッセージIDのメッセージを削除します。</B></DT>
	 *  <DD></DD>
	 * </DL>
	 * @param messageList
	 * @throws BtsDBException
	 */
	public void deleteByMessageID( int messageID ) throws BtsDBException {
		try {
			new QueryRunner().update( this.con, DELETE_SQL, new Integer(messageID) );	
		} catch (Exception e) {
			throw new BtsDBException(e);
		}
	}

	/**
	 * <DL><DT><B>指定のメッセージIDのメッセージを「既読」に設定します。</B></DT>
	 *  <DD></DD>
	 * </DL>
	 * @param messageList
	 * @throws BtsDBException
	 */
	public void setOpend( int messageID ) throws BtsDBException {
		try {
			new QueryRunner().update( this.con, SET_OPENED, new Integer(messageID) );	
		} catch (Exception e) {
			throw new BtsDBException(e);
		}
	}

	
	/**
	 * <DL><DT>メッセージを全件取得します。<B></B></DT>
	 *  <DD></DD>
	 * </DL>
	 * @return		メッセージリスト
	 * @throws		BtsDBException
	 */
	public MessagePacket[] getAllMessage() throws BtsDBException {
		MessagePacket[] messages = new MessagePacket[]{};

		try {
			
			List result = getMessageList();
			messages = (MessagePacket[])result.toArray( messages );
			
		} catch (Exception e) {
			throw new BtsDBException(e);
		}
		
		return messages;
	}

	
	/**
	 * <DL><DT>メッセージを全件取得します。<B></B></DT>
	 *  <DD></DD>
	 * </DL>
	 * @return		メッセージリスト
	 * @throws		BtsDBException
	 */
	public List getMessageList() throws BtsDBException {

		try {
			ResultSetHandler rsh = new BeanListHandler(MessagePacket.class);
			
			List result = (List)new QueryRunner().query( this.con, SELECT_ALL, rsh );
			

			return result;
		} catch (Exception e) {
			throw new BtsDBException(e);
		}
		
	}
	
	/**
	 * 現在保持するメッセージIDの最大値を取得します。
	 * 
	 * @return 現在保持するメッセージIDの最大値
	 * @throws BtsDBException
	 */
	public int getMaxMessageID() throws BtsDBException{

		try {
			ResultSetHandler rsh = new ScalarHandler(1);
			
			Object result = new QueryRunner().query( this.con, SELECT_MAX_ID, rsh );
			if(result == null) {
				return 0;
			}
			return Integer.parseInt(result.toString());
			
		} catch (Exception e) {
			throw new BtsDBException(e);
		}
	}

	
	/**
	 * <DL><DT><B>メッセージ送信者一覧(重複なし)を取得します。</B></DL>
	 *  <DD></DD>
	 * </DL>
	 * @return					送信者一覧
	 * @throws BtsDBException
	 */
	public String[] getFromUser() throws BtsDBException{

		try {
			ResultSetHandler rsh = new ArrayListHandler();
			
			List list = (List) new QueryRunner().query( this.con, GET_FROMUSER, rsh );
			
			List result = new ArrayList();

			for (int i = 0; i < list.size(); i++) {

				if(((Object[])list.get(i))[0] != null && !"".equals(((Object[])list.get(i))[0])) {
					result.add(((Object[])list.get(i))[0]);
				}

			}

			return (String[]) result.toArray(new String[0]);
			
		} catch (Exception e) {
			throw new BtsDBException(e);
		}

	}


	
}
