/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.dao;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;

import jp.valtech.bts.connection.DBConnection;
import jp.valtech.bts.data.Issue;
import jp.valtech.bts.data.IssueHistory;

import org.apache.commons.dbutils.QueryRunner;
import org.apache.commons.dbutils.ResultSetHandler;
import org.apache.commons.dbutils.handlers.BeanHandler;
import org.apache.commons.dbutils.handlers.BeanListHandler;

/**
 * <dl><dt><b>課題票更新履歴用のDAO</b></dt>
 * <dd>
 * </dd>
 * <dt><b>使い方</b></dt>
 * <dd>
 * </dd>
 * </dl>
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public class IssueHistoryDAO extends DataBaseAccessor {

	/** テーブル作成SQL */
	private static final String CREATE_TABLE
			= "CREATE TABLE ISSUEHISTORY" 
			+ " ( FINGERPRINT     VARCHAR(30)"
			+ " , TYPE            CHAR(1) " 
			+ " , UPDATEATTRIBUTE LONGVARCHAR" 
			+ " , UPDATEBEFORE    LONGVARCHAR" 
			+ " , UPDATEAFTER     LONGVARCHAR" 
			+ " , UPDATEUSER      VARCHAR(30) " 
			+ " , UPDATEDATE      TIMESTAMP" 
			+ " ) ;"
			+ "CREATE INDEX IDX_HISTORY ON ISSUEHISTORY(FINGERPRINT) ;";

	/** テーブル削除SQL */
	private static final String DROP_TABLE
			= "DROP TABLE ISSUEHISTORY CASCADE;";

	/** データ登録SQL */
	private static final String INSERT_SQL
			= "INSERT INTO ISSUEHISTORY"
			+ " ( FINGERPRINT"
			+ " , TYPE"
			+ " , UPDATEATTRIBUTE"
			+ " , UPDATEBEFORE"
			+ " , UPDATEAFTER"
			+ " , UPDATEUSER"
			+ " , UPDATEDATE"
			+ " ) "
			+ " VALUES(?, ?, ?, ?, ?, ?, ?)";

	/** データを削除する */
	private static final String DELETE_SQL
			= "DELETE FROM ISSUEHISTORY WHERE FINGERPRINT=? AND TYPE=?";


	/** データを全件取得するSQL */
	private static final String SELECT_BY_FINGERPRINT
			= "SELECT" 
			+ "   FINGERPRINT"
			+ " , TYPE"
			+ " , UPDATEATTRIBUTE"
			+ " , UPDATEBEFORE"
			+ " , UPDATEAFTER"
			+ " , UPDATEUSER"
			+ " , UPDATEDATE"
			+ " FROM  ISSUEHISTORY"
			+ " WHERE FINGERPRINT=? AND TYPE=?"
			+ " ORDER BY UPDATEDATE DESC";


	/** ごみ箱に移動した時の履歴情報取得 */
	private static final String SELECT_MOVED_GARBAGE
			= "SELECT" 
			+ "   FINGERPRINT"
			+ " , TYPE"
			+ " , UPDATEATTRIBUTE"
			+ " , UPDATEBEFORE"
			+ " , UPDATEAFTER"
			+ " , UPDATEUSER"
			+ " , UPDATEDATE"
			+ " FROM  ISSUEHISTORY"
			+ " WHERE FINGERPRINT=? AND UPDATEATTRIBUTE=?";

	/** ごみ箱に移動するSQL */
	private static final String UPDATE_GARBAGE	
			= "UPDATE ISSUEHISTORY SET TYPE='G' WHERE FINGERPRINT=? AND TYPE='R'"; 

	/**
	 * <DL><DT><B>コンストラクタ</B>
	 *  <DD></DD>
	 * </DL>
	 * @param connection
	 */
	public IssueHistoryDAO( DBConnection connection ) {
		super( connection );
	}


	/**
	 * テーブルを生成します。
	 * 
	 * @throws		BtsDBException
	 */
	public void createTable() throws BtsDBException {
		try {
			// テーブル生成用SQL実行
			new QueryRunner().update( this.con, CREATE_TABLE );
			
		} catch (Exception e) {
			throw new BtsDBException(e);
		}
	}

	/**
	 * テーブルを削除します。
	 * 
	 * @throws		BtsDBException
	 */
	public void dropTable() throws BtsDBException {
		try {
			// テーブル削除用SQL実行
			new QueryRunner().update( this.con, DROP_TABLE );	
		} catch (Exception e) {
			throw new BtsDBException(e);
		}
	}

	/**
	 * <DL><DT><B>データを登録します。</B></DL>
	 *  <DD></DD>
	 * </DL>
	 * @param issueHistory
	 * @throws BtsDBException
	 */
	public void addHistory( IssueHistory issueHistory ) throws BtsDBException {
		try {
			List param = new ArrayList();
			param.add( issueHistory.getFingerPrint() );
			param.add( issueHistory.getType() );
			param.add( issueHistory.getUpdateAttribute() );
			param.add( issueHistory.getUpdateBefore() );
			param.add( issueHistory.getUpdateAfter() );
			param.add( issueHistory.getUpdateUser() );
			if( issueHistory.getUpdateDate() != null ){
				param.add( new Timestamp( issueHistory.getUpdateDate().getTime() ) );
			}else{
				param.add( null );
			}

			new QueryRunner().update( this.con, INSERT_SQL, param.toArray() );	
		} catch (Exception e) {
			throw new BtsDBException(e);
		}
	}

	
	/**
	 * <DL><DT><B></B></DL>
	 *  <DD></DD>
	 * </DL>
	 * @param issueHistory
	 * @throws BtsDBException
	 */
	public void deleteByFingerPrint(String fingerprint, String type) throws BtsDBException {
		try {
			List param = new ArrayList();
			param.add( fingerprint );
			param.add( type );
			new QueryRunner().update( this.con, DELETE_SQL, param.toArray() );	
		} catch (Exception e) {
			throw new BtsDBException(e);
		}
	}


	
	/**
	 * <DL>
	 *   <DT><B>指定された{@link Issue#getFingerPrint() FingerPrint}を持つ課題票の課題票履歴を全件取得します。</B></DT>
	 *   <DD></DD>
	 * </DL>
	 * @param			fingerprint			課題票の{@link Issue#getFingerPrint() FingerPrint}
	 * @param			type				課題票の{@link jp.valtech.bts.data.IssueType 種別}
	 * @return			指定された課題票システムIDに属する課題票履歴全件
	 * @throws			BtsDBException
	 */
	public IssueHistory[] getByFingerPrint(String fingerprint, String type) throws BtsDBException {
		IssueHistory[] issueHistories = new IssueHistory[]{};

		try {
			List param = new ArrayList();
			param.add( fingerprint );
			param.add( type );

			ResultSetHandler rsh = new BeanListHandler(IssueHistory.class);
			List result = (List)new QueryRunner().query( this.con, SELECT_BY_FINGERPRINT, param.toArray(), rsh );
			issueHistories = (IssueHistory[])result.toArray( issueHistories );
			
		} catch (Exception e) {
			throw new BtsDBException(e);
		}
		
		return issueHistories;
	}

	
	
	/**
	 * 課題票が「ごみ箱」に移動した時の履歴情報を取得します。
	 * 
	 * @param		fingerPrint		課題票の諮問
	 * @return		課題票が「ごみ箱」に移動した時の履歴情報
	 * @throws		BtsDBException
	 */
	public IssueHistory getMovedGarbageHistory(String fingerPrint) throws BtsDBException {
		
		try {
			List param = new ArrayList();
			param.add( fingerPrint );
			param.add( IssueHistory.ISSUE_MOVE_GARBAGE );

			// 検索
			ResultSetHandler rsh = new BeanHandler(IssueHistory.class);
			return (IssueHistory)new QueryRunner().query( this.con, SELECT_MOVED_GARBAGE, param.toArray(), rsh );
			
		} catch (Exception e) {
			throw new BtsDBException(e);
		}
	}

	/**
	 * <DL><DT><B>指定の課題票をごみ箱に移動します。</B></DT>
	 *  <DD></DD>
	 * </DL>
	 * @param		fingerPrint			削除対象の課題票のFingerPrint
	 * @throws 		BtsDBException
	 */
	public void changeGarbageVersion(String fingerPrint) throws BtsDBException{
		try {
			List param = new ArrayList();
			param.add(fingerPrint);

			new QueryRunner().update( this.con, UPDATE_GARBAGE, param.toArray() );	
		} catch (Exception e) {
			throw new BtsDBException(e);
		}
	}

}
