/***********************************************************************
 * Copyright(C) 2006 Valtech Co.,Ltd.
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 ***********************************************************************/
package jp.valtech.bts.connection;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.ResourceBundle;

import jp.valtech.bts.dao.BtsDBException;
import jp.valtech.bts.data.CurrentProject;
import jp.valtech.bts.ui.BtsPlugin;
import jp.valtech.bts.util.Logging;

/**
 * <dl><dt><b>データベース接続とトランザクション管理のための基底クラスです。</b></dt>
 * <dd>
 * </dd>
 * <dt><b>使い方</b></dt>
 * <dd>ソースのルートディレクトリにプロパティファイル<code>connection.properties</code>を用意します。
 * プロパティファイルに以下のキーと値を登録してください。
 * <ul>
 * 	<li><code>driver.class</code>:ドライバクラスの指定
 * 	<li><code>url</code>:接続先データベースのURL
 * 	<li><code>connection.prefix</code>:JDBC接続URLの先頭につける文字列やパラメータ
 * 	<li><code>connection.suffix</code>:JDBC接続URLの最後につける文字列やパラメータ
 * </ul>
 * 実際に利用するパラメータは、実装によって異なります。
 * </dd>
 * </dl>
 * 
 * @author		<A href="mailto:m_sugitou@valtech.jp">M.Sugito</A>
 * @version	Ver.0.8
 */
public abstract class DBConnection implements Logging{

	/** ファイルシステムのファイル区切り文字列 */
	private static final String SEPARATOR	= System.getProperty( "file.separator" );
	
	/** HSQLDBのプロパティファイル拡張子*/
	protected static final String DB_FILE_PROPERTIES = ".properties";
	
	/** HSQLDBのデータファイル拡張子*/
	protected static final String DB_FILE_SCRIPT = ".script";

	/** DBコネクション設定ファイル：connection.properties */
	private static final ResourceBundle resource	= ResourceBundle.getBundle( "connection" );

	/** 接続識別子のプレフィックス：『jdbc:hsqldb:file:』*/
	protected static final String CONNECTION_PREFIX = resource.getString("connection.prefix");

	/** 接続識別子のサフィックス  ：『;shutdown=true』 */
	protected static final String CONNECTION_SUFFIX = resource.getString("connection.suffix");
	
	/** ドライバクラス 　　　　　 ：『org.hsqldb.jdbcDriver』 */
	private static final String DRIVER_CLASS      = resource.getString("driver.class");

	/** データベースコネクション*/
	protected Connection con	= null;

	/** コネクション取得リトライ回数の最大数 */
	private int maxRetry=20;
	
	/** コネクション取得リトライ回数カウンター */
	private int retry = 0;
	
	/** 当該インスタンスで扱うDBファイルのPATH文字列*/
	protected String dbFilepath = "";
	
	static{
		try {
			Class.forName( DRIVER_CLASS );
		} catch ( ClassNotFoundException e ) {
			logger.fatal( e.getMessage() , e );
		}
	}


	/**
	 * コネクション生成。
	 * 処理種別を設定します。
	 * 
	 * @param		processType			処理種別
	 */
	public DBConnection(String processType) {
		this(null, processType, null);
	}
	
	/**
	 * コネクション生成。
	 * 処理種別とSSIDを設定します。
	 * 
	 * @param 		ssid				SSID
	 * @param		processType			処理種別
	 */
	public DBConnection(String ssid, String processType) {
		this(ssid, processType, null);
	}


	/**
	 * コネクションを生成します。
	 * @param		ssid			SSID
	 * @param		processType		処理種別
	 * @param		currentDir		ホームディレクトリ
	 */
	public DBConnection(String ssid, String processType, String currentDir) {
		try {
			// SSIDが無い場合は取得する。
			if(ssid ==null) {
				ssid = CurrentProject.getInsance().getSSID();
			}

			// カレントディレクトリを指定。「ワークスペースの.metadata/.plutins/jp.valtech.bts」を設定する。
			if(currentDir == null) {
				currentDir = BtsPlugin.getInstance().getStateLocation().toString();
			}

			// DB接続の識別子を生成
			StringBuffer dbFilepath = new StringBuffer();
			dbFilepath.append(currentDir).append(SEPARATOR).append(ssid).append(SEPARATOR).append(processType);
			
			this.dbFilepath = dbFilepath.toString(); 
			buildConnection();
		
		} catch ( BtsDBException e ) {
			throw new RuntimeException( e );
		}
	}

	
	/**
	 * HSQLDBへのコネクションを取得します。
	 * 
	 * @throws		BtsDBException
	 */
	protected void buildConnection() throws BtsDBException {

		// DB接続の識別子を生成
		StringBuffer dbURL = new StringBuffer();
		dbURL.append(CONNECTION_PREFIX);
		dbURL.append(this.dbFilepath);
		dbURL.append(CONNECTION_SUFFIX);

		logger.debug(dbURL);
		try {
			// コネクション生成
			this.con = DriverManager.getConnection(dbURL.toString());
			this.con.setAutoCommit( false );
			this.con.setReadOnly( false );
		
		}catch (SQLException e) {
			// コネクションを取得できなかったとき。
			// 再度取得処理を行う。取得処理の回数は変数maxRetryで決める
			// maxRetryの回数を超えた場合は諦めて例外をスローする
			retry++;
			if(retry > maxRetry) {
				String message = Messages.getString("DBConnection.6") ; //$NON-NLS-1$
				throw new BtsDBException(message, e);
			}
			
			logger.info(Messages.getString("DBConnection.7") + retry); //$NON-NLS-1$
			try {
				// １秒待ってからリトライ
				Thread.sleep(1000);
			}catch (Exception ex) {	}

			// コネクション取得リトライ
			buildConnection();
		}
	}


	/**
	 * トランザクションをコミットします。
	 * 
	 * @throws SQLException
	 */
	public void commit() throws SQLException{
		if( this.con != null ){
			this.con.commit();
		}
	}

	
	/**
	 * トランザクションをロールバックします。
	 *
	 */
	public void rollback(){
		if( this.con != null ){
			try {
				this.con.rollback();
			} catch ( SQLException e ) {
				logger.error( e.getMessage(), e );
			}
		}
	}

	
	/**
	 * データベースから切断します。
	 */
	public void close(){
		PreparedStatement ps = null;
		try {
			ps = this.con.prepareStatement( "shutdown;" );
			ps.execute();
		} catch ( SQLException e ) {
			logger.error( e.getMessage(), e );
		} finally {
			if( ps != null ){
				try {
					ps.close();
				} catch ( SQLException e ) {
					logger.error( e.getMessage(), e );
				}
				ps = null;
			}
		}
		
		if( this.con != null ){
			try {
				this.con.close();
			} catch ( SQLException e ) {
				logger.error( e.getMessage(), e );
			}
			this.con = null;
		}
	}
	
	/**
	 * 当インスタンスから{@link Connection}を取得します。
	 * 
	 * @return		データベース接続
	 */
	public Connection getConnection(){
		return this.con;
	}

	
	/**
	 * リトライ最大数を設定します。
	 * 
	 * @param		maxRetry		リトライ最大数
	 */
	protected void setMaxRetry(int maxRetry) {
		this.maxRetry = maxRetry;
	}
}
