/*
 * Copyright(C) 2006-2007 Valtech Corporation. All Rights Reserved.
 * 
 * $URL$
 * $Id$
 * 
 */
package jp.valtech.bts.command.server;

import java.util.List;

import jp.valtech.bts.data.NetworkConfig;
import jp.valtech.bts.facade.SyncHeaderFacade;
import jp.valtech.bts.facade.SyncServerFacade;
import jp.valtech.bts.network.Request;
import jp.valtech.bts.network.Response;
import jp.valtech.bts.network.command.TCPServerCommand;


/**
 * <dl>
 *   <dt><b>クライアントからの同期リクエストを受け付けます。</b></dt>
 *   <dd><ul>
 * 	   <li>リクエストから課題票ヘッダを取り出し、それを元にしてリクエスト元に返す課題票をローカルDBから見つけ出します。
 *     <li>リクエスト元が送信すべきデータを送ってきていない場合はエラーとしてレスポンスを返します。
 *     <li>エラーが発生した場合には、えらー情報をレスポンスメッセージに格納します。
 *   </ul></dd>
 * </dl>
 * 
 * @see			jp.valtech.bts.command.client.SyncIssue
 * @author		<A href="mailto:m_sugitou@valtech.jp">$Author$</A>
 * @version		$Rev$
 */
public class SyncIssueServer extends TCPServerCommand {

	/**
	 * なにもしない。
	 */
	public SyncIssueServer() {
		super();
	}

	/**
	 * {@link jp.valtech.bts.network.Request リクエストオブジェクト}から課題票ヘッダ情報を取得します。<br>
	 * 受け取った課題票情報を元にしてリクエスト元に返す課題票をローカルDBから見つけ出します。<br>
	 * {@link Response#OK}を設定して送信元に返信します。<br>
	 * 
	 * @param		request			リクエストオブジェクト
	 * @param		response		レスポンスオブジェクト
	 * @param		myconfig		ローカル端末のネットワーク情報
	 */
	public void execute(Request request, Response response, NetworkConfig myconfig){
		try {
			SyncHeaderFacade syncHeaderFacade = new SyncHeaderFacade();


			// ローカルの課題票ヘッダ情報
			List localIssueHeader = syncHeaderFacade.getIssueHeaderList();
			// ローカルのごみ箱ヘッダ情報
			List localGarbageHeader = syncHeaderFacade.getGarbageHeaderList();
	
			// ローカルにデータが無かった場合。処理結果OKを設定してデータはnullを返す。
			if(localIssueHeader == null && localGarbageHeader==null) {
				response.setCode(Response.OK);
				response.addParameter("issueList", null);
				response.addParameter("garbageList", null);
				reply(response);
				return;
			}
	
			// 受信した課題票ヘッダ取得
			List receiveIssueHeader = (List)request.getParameterObject( "issueHeader" );
			// 受信したごみ箱ヘッダ取得
			List receiveGarbageHeader = (List)request.getParameterObject( "garbageHeader" );
	

			// 送信する課題票を抽出する。
			// local課題票のバージョン == receive課題票のバージョン　-> 除外
			// local課題票のバージョン >  receive課題票のバージョン　-> 同期対象としてリクエスト元に返す
			// local課題票のバージョン <  receive課題票のバージョン　-> 除外
			//   （参考：IssueSyncHeader#equals() ）
			// ようするに、Localの方がバージョンが新しいもののみ同期データとしてリクエスト元に返す
			// バージョンが違っていてもreceiveの方が新しい場合は、リクエスト元に返さない。
			if(localIssueHeader != null && receiveIssueHeader != null) {
				localIssueHeader.removeAll(receiveIssueHeader);
			}
			
			// ごみ箱にする課題票を抽出する。
			// Listの中身はただのStringなのでString#equals(String) == trueとなるものが除外される
			if(localGarbageHeader != null && receiveGarbageHeader != null) {
				localGarbageHeader.removeAll(receiveGarbageHeader);
			}

			// リクエスト元に返す課題票情報を取得
			SyncServerFacade facade = new SyncServerFacade();
			facade.readSyncIssues(localIssueHeader, localGarbageHeader);

			// リクエスト元に返すデータをレスポンスオブジェクトに設定
			response.addParameter("syncIssues"  , facade.getSyncIssues());
			response.addParameter("syncGarbages", facade.getSyncGarbages());
			response.setCode(Response.OK);
		
		} catch (Exception e) {
			logger.fatal(e.getMessage(), e);
			response.setMessage(e.getMessage());
			response.setCode(Response.ERROR);
		}
		
		// レスポンスオブジェクトを返す
		reply(response);
	}

}
