/*
 * 
 * Licensed Materials - Property of IBM
 *
 * Open Platform Trust Services - An open source TCG PTS
 *
 * (C) Copyright International Business Machines Corp. 2007, 2011
 *
 */
package com.ibm.trl.tcg.pts.eventlog;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.ibm.trl.tcg.pts.engine.Event;
import com.ibm.trl.tcg.pts.engine.TPM;
import com.ibm.trl.tcg.pts.integrity.IntegrityReport;
import com.ibm.trl.tcg.pts.tools.Base64Tool;
import com.ibm.trl.tcg.pts.tools.HexTool;

/**
 * 
 * @author Seiji Munetoh (munetoh@users.sourceforge.jp)
 * 
 */
public class IML {

	/* Logger */
	private Log log = LogFactory.getLog(this.getClass());

	final static int MAX_SML_SIZE = 240000; // LIM/IMA
	private EventSet[] _e;
	private int _eventNum;
	private TPM _tpm;


	/**
	 * 
	 */
	public IML() {
		_e = null;
		_eventNum = 0;
		_tpm = new TPM();
	}

	public IML(TPM tpm) {
		_e = null;
		_eventNum = 0;
		_tpm = tpm;
	}

	/**
	 * @param file
	 * @return
	 * @throws Exception
	 */
	public void loadBIOSIML(String file) throws Exception {
		if (log.isTraceEnabled()) {
			log.trace("load BIOS IML file " + file);
		}

		byte[] sml = _loadBIOSIML(file);
		_e = createEventFromBIOSIML(sml);

	}

	private byte[] _loadBIOSIML(String file) throws Exception {
		File f = new File(file);
		if (f.isFile() == false) {
			throw new Exception("File not found " + file);
		}

		byte[] sml = null;

		try {
			// Step 1, check the length
			File f0=new File(file);
			int len=(int) f0.length();
			
			// Step 2, create byte[]
			FileInputStream fis = new FileInputStream(file);
			fis = new FileInputStream(file);
			sml = new byte[len];
			len = fis.read(sml);

			if (log.isDebugEnabled()) {
				log.debug(" IML file name = " + file);
				log.debug(" IML file size = " + len);
			}

		} catch (FileNotFoundException e) {
			log.error("Error, File not found. filename = " + file);
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
		return sml;
	}

	/**
	 * @param sml
	 * @throws Exception 
	 * 
	 */
	private EventSet[] createEventFromBIOSIML(byte[] sml) throws Exception {
		if (log.isTraceEnabled()) {
			log.trace("createEventFromBIOSIML ");
		}

		int ptr = 0;
		int pcrIndex;
		long type;
		byte digest[];
		byte eventdata[];

		EventSet[] e = new EventSet[24];
		for (int i = 0; i < 24; i++) {
			e[i] = new EventSet();
			e[i].setPcrIndex(i);
		}

		while (ptr < sml.length) {
			pcrIndex = (int)UINT32(sml, ptr);
			ptr += 4;
			type = UINT32(sml, ptr);
			ptr += 4;
			digest = SHA1_DIGEST(sml, ptr);
			ptr += 20;
			eventdata = EventData(sml, ptr);
			ptr += 4 + eventdata.length;

			if (pcrIndex < 24) {
				e[pcrIndex].addEvent(type, digest, eventdata);
			} else {
				log.error("malfunced Event at " + _eventNum + " ?");
			}

			if (log.isDebugEnabled()) {
				String str = "PCR[" + pcrIndex + "],Type[" + type + "],Length["
						+ eventdata.length + "],digest["
						+ HexTool.getHexString(digest) + "]";
				log.debug(str);
			}
			_tpm.extend(pcrIndex, digest); // 20071111 SM
			_eventNum++;
		}

		if (log.isTraceEnabled()) {
			log.trace("createEventFromBIOSIML - done");
		}
		return e;
	}

	/**
	 * @param smlFilename
	 * @throws Exception
	 */
	public void addEventFromBIOSIML(String smlFilename) throws Exception {
		if (smlFilename == null)
			return;

		byte[] sml = _loadBIOSIML(smlFilename);
		if (sml == null)
			return;

		int ptr = 0;

		int pcrIndex;
		long type;
		byte digest[];
		byte eventdata[];

		while (ptr < sml.length) {
			pcrIndex = (int)UINT32(sml, ptr);
			ptr += 4;
			type = UINT32(sml, ptr);
			ptr += 4;
			digest = SHA1_DIGEST(sml, ptr);
			ptr += 20;
			eventdata = EventData(sml, ptr);
			ptr += 4 + eventdata.length;

			_e[pcrIndex].addEvent(type, digest, eventdata);

			if (log.isDebugEnabled()) {
				String str = "PCR[" + pcrIndex + "],Type[" + type + "],Length["
						+ eventdata.length + "],digest["
						+ HexTool.getHexString(digest) + "]";
				log.debug(str);
			}
			_eventNum++;
		}
	}

	public void loadIntegrityReport(String filename) throws Exception {
		IntegrityReport ir = new IntegrityReport(filename, null);  // TODO set pubkey
		_e = ir.getEventSet();
		_eventNum = ir.getEventNum();
	}

	/**
	 * Event[pcrindex][eventindex]
	 * 
	 * @return
	 */
	public EventSet[] getEvents() {
		return _e;
	}

	/**
	 * Return Event list (whole pcr)
	 * 
	 * @return
	 */

	public Event[] getEventList() {
		Event[] event = new Event[_eventNum];

		int index = 0;
		for (int pcrindex = 0; pcrindex < 24; pcrindex++) {
			for (int i = 0; i < _e[pcrindex].size; i++) {
				event[index] = _e[pcrindex].getEvent(i);
				if (log.isTraceEnabled()) {
					log.trace(index + " " + i + " " + event[index].toString());
				}
				index++;
			}
		}

		return event;
	}

	/**
	 * Get events of pcr[pcrindex]
	 * 
	 * @param pcrindex
	 * @return
	 */
	public Event[] getEventList(int pcrindex) {
		Event[] event = new Event[_e[pcrindex].size];

		for (int i = 0; i < _e[pcrindex].size; i++) {
			event[i] = _e[pcrindex].getEvent(i);
		}
		return event;
	}

	/**
	 * @param message
	 * @return
	 * @throws Exception 
	 */
	public Event[] getEvents(byte[] message) throws Exception {
		int ptr = 0;
		int pcrIndex;
		long type;
		byte digest[];
		byte eventdata[];
		int num = 0;

		// 1st pass
		// TODO Verify PCRs at this time
		while (ptr < message.length) {
			pcrIndex = (int)UINT32(message, ptr);
			ptr += 4;
			type = UINT32(message, ptr);
			ptr += 4;
			digest = SHA1_DIGEST(message, ptr);
			ptr += 20;
			eventdata = EventData(message, ptr);
			ptr += 4 + eventdata.length;

			if (log.isTraceEnabled()) {
				log.trace("PCR    " + pcrIndex);
				log.trace("Type   " + type);
				log.trace("Length " + eventdata.length);
				log.trace(HexTool.getHexString(digest));
			}
			num++;
		}

		if (log.isDebugEnabled()) {
			log.debug("EventNum    " + num);
		}

		// 2nd pass
		Event e[] = new Event[num];
		ptr = 0;
		num = 0;
		while (ptr < message.length) {
			e[num] = new Event();
			ptr = e[num].set(message, ptr);
			num++;
		}

		return e;
	}

	/**
	 * Re order the events based on PCR index (same as TSS's getEvent Func)
	 * 
	 * @param message
	 * @return
	 * @throws Exception 
	 */
	public Event[] getEvents2(byte[] message) throws Exception {
		int ptr = 0;
		int pcrIndex;
		long type;
		byte digest[];
		byte eventdata[];
		int num = 0;

		int pcrNum[];
		pcrNum = new int[24];

		// 1st pass
		// TODO Verify PCRs at this time
		while (ptr < message.length) {
			pcrIndex = (int)UINT32(message, ptr);
			pcrNum[pcrIndex]++;
			ptr += 4;
			type = UINT32(message, ptr);
			ptr += 4;
			digest = SHA1_DIGEST(message, ptr);
			ptr += 20;
			eventdata = EventData(message, ptr);
			ptr += 4 + eventdata.length;

			if (log.isTraceEnabled()) {
				log.trace("PCR[" + pcrIndex);
				log.trace("],Type[0x" + Long.toString(type, 16));
				log.trace("],Length[" + eventdata.length + "],digest[");
				log.trace(HexTool.getHexString(digest));
			}
			num++;
		}

		int offset[];
		offset = new int[25];

		for (int i = 0; i < 24; i++) {
			offset[i + 1] = pcrNum[i] + offset[i];
			if (log.isTraceEnabled()) {
				log.trace("EventNum PCR[" + i + "]=   " + pcrNum[i]
						+ " offset=" + offset[i]);
			}
		}
		if (log.isTraceEnabled()) {
			log.trace("EventNum (Total)   " + num);
		}

		// 2nd pass
		// DEBUG
		// NG log.debug("SM DEBUG getEvents2 LIMIT NUM 100");
		// if (num>100) num=100;

		Event e[] = new Event[num];

		ptr = 0;
		while (ptr < message.length) {
			pcrIndex = (int)UINT32(message, ptr);
			ptr += 4;
			type = UINT32(message, ptr);
			ptr += 4;
			digest = SHA1_DIGEST(message, ptr);
			ptr += 20;
			eventdata = EventData(message, ptr);
			ptr += 4 + eventdata.length;

			num = offset[pcrIndex];
			e[num] = new Event(pcrIndex, type, digest, eventdata);
			offset[pcrIndex]++;
		}

		return e;
	}

	/**
	 * @param message
	 * @param ptr
	 * @return
	 * @throws Exception 
	 */
	private static byte[] EventData(byte[] message, int ptr) throws Exception {
		int len = (int)UINT32(message, ptr);

		if (len < 0 ) {
			throw new Exception("Bad event data length, len=" + len + ", nagative");
		}
		if (len > message.length - 4 ) {
			throw new Exception("Bad event data length, len=" + len + ", too large");
		}
		
		byte b[] = new byte[len];
		int i;
		for (i = 0; i < len; i++) {
			b[i] = message[ptr + 4 + i];
		}
		return b;
	}

	/**
	 * @param message
	 * @param ptr
	 * @return
	 */
	private static byte[] SHA1_DIGEST(byte[] message, int ptr) {
		byte b[] = new byte[20];
		int i;
		for (i = 0; i < 20; i++) {
			b[i] = message[ptr + i];
		}
		return b;
	}

	/**
	 * @param b
	 * @return
	 */
	public static long UINT32(byte b[], int ptr) {
		long i;

		i = b[ptr + 3] & 0xFF;
		i = i << 8;
		i += b[ptr + 2] & 0xFF;
		i = i << 8;
		i += b[ptr + 1] & 0xFF;
		i = i << 8;
		i += b[ptr] & 0xFF;

		return i;
	}

	/**
	 * @param e
	 * @return
	 */
	// public static byte[] getPCEventData(Event e) {
	// byte[] pcdata = EventData(e.eventData,4); // skip type
	// return pcdata;
	// }

	/**
	 * @param base64BiosSML
	 * @throws Exception 
	 */
	public void setBase64BiosSML(String base64BiosSML) throws Exception {
		byte[] sml = null;
		try {
			sml = Base64Tool.decode(base64BiosSML);
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		_e = createEventFromBIOSIML(sml);
	}

	public String toString() {
		String str = "";
		str += "Total " + _e.length + " events";
		return str;
	}

	public void print() {
		if (log.isDebugEnabled()) {
			log.debug("print IML ");
		}

		String pcrUsage[] = {
				"CRTM, POST BIOS, and Embedded Option ROMs", // 0
				"Host Platform Configuration", 
				"Option ROM Code",
				"Option ROM Configuration and Data", 
				"IPL",
				"IPL Configuration and Data", 
				"State Transition",
				"Host Platform Manufacturer Control", 
				"(Operating System)", // 8
				"", // 9
				"(IMA, LKM&ELF Components)", // 10
				"", // 11
				"", // 12
				"", // 13
				"", // 14
				"", // 15
				"Debug", // 16
				"", // 17
				"", // 18
				"", // 19
				"", // 20
				"", // 21
				"", // 22
				"Application Support", // 23
		};

		for (int pcrindex = 0; pcrindex < 24; pcrindex++) {
			if (_e[pcrindex].size > 0) {
				System.out.println("--- PCR[" + pcrindex + "] "
						+ pcrUsage[pcrindex] + " ---");
				Event[] elist = _e[pcrindex].getEventList();
				for (int i = 0; i < elist.length; i++) {
					System.out.println(elist[i].toString());
					if (log.isTraceEnabled()) {
						log.trace(elist[i].toString());
					}
				}
			}
		}

		_tpm.print();

		if (log.isDebugEnabled()) {
			log.debug("print IML - done");
		}

	}

	/**
	 * 
	 * Update/replace Event of pcrindex
	 * 
	 * @param pcrindex
	 * @param events
	 */
	public void updateEvent(int pcrindex, Event[] events) {
		_e[pcrindex].setEventList(events);

		// TODO Auto-generated method stub

	}


	/**
	 * Get Digest (base64 string) by Event Type
	 * 
	 * @param i
	 * @return
	 * @throws Exception
	 */

	public String getDigestByType(long type) throws Exception {
		String b64 = null;
		for (int pcrindex = 0; pcrindex < 24; pcrindex++) {
			if (_e[pcrindex].size > 0) {
				Event[] elist = _e[pcrindex].getEventList();
				for (int i = 0; i < elist.length; i++) {
					if (elist[i].getEventType() == type) {
						// HIT
						b64 = Base64Tool.encode(elist[i].getDigest());
						return b64;
					}
				}
			}
		}
		if (log.isWarnEnabled()) {
			log.warn("Event Type " + type + " not found...");
		}
		throw new Exception("Event Type " + type + " not found...");
	}

	public int size() {
		return _eventNum;
	}
}
