/*
 * 
 * Licensed Materials - Property of IBM
 *
 * Open Platform Trust Services - An open source TCG PTS
 *
 * (C) Copyright International Business Machines Corp. 2007, 2011
 *
 */
package com.ibm.trl.tcg.pts.engine;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.ibm.trl.tcg.pts.tools.Base64Tool;
import com.ibm.trl.tcg.pts.tools.DigestTool;
import com.ibm.trl.tcg.pts.tools.HexTool;

/**
 * @author Seiji Munetoh (munetoh@users.sourceforge.jp)
 * 
 */
public class TPM {

	/* Logger */
	private Log log = LogFactory.getLog(this.getClass());

	private byte[][] _pcr;

	private boolean[] _extended;

	public TPM() {
		_pcr = new byte[24][];
		_extended = new boolean[24];

		for (int i = 0; i < 16; i++) {
			_pcr[i] = new byte[20];
			_extended[i] = false;
			;
			for (int j = 0; j < 20; j++) {
				_pcr[i][j] = 0;
			}
		}
		for (int i = 16; i < 24; i++) {
			_pcr[i] = new byte[20];
			_extended[i] = false;
			;
			for (int j = 0; j < 20; j++) {
				_pcr[i][j] = (byte) 0xff;
			}
		}

	}

	/**
	 * @param _properties2
	 * @throws Exception
	 */
	public void init(PlatformProperties prop) throws Exception {
		for (int i = 0; i < 24; i++) {
			String name = "tpm.pcr." + i;
			//PlatformProperty p = prop.getPropertyByName(name);
			// if (log.isTraceEnabled()) {
			// log.trace(" " + name);
			// log.trace(" " + p.getName());
			// }
			String b64 = prop.getProperty(name);
			//_pcr[i] = Base64Tool.decode(p.getValue());
			_pcr[i] = Base64Tool.decode(b64);
		}
	}

	/**
	 * 
	 * Extend
	 * 
	 * @param pcrIndex
	 * @param digest
	 */
	public void extend(int pcrIndex, byte[] digest) {
		// log.error("tpm extend " + event.toString());
		if (digest == null) {
			log.error("Null Digest, Internal Error?");
		} else {
			// int pcrindex = (int) event.getPcrIndex();
			byte[] msg = new byte[40];
			for (int i = 0; i < 20; i++) {
				msg[i] = _pcr[pcrIndex][i];
				msg[i + 20] = digest[i];
			}
			_pcr[pcrIndex] = DigestTool.SHA1(msg);
			_extended[pcrIndex] = true;
		}
	}

	/**
	 * @param event
	 */
	public void extend(Event event) {
		// log.error("tpm extend " + event.toString());
		if (event.getDigest() == null) {
			// Push Event, ignore
		} else {
			int pcrindex = (int) event.getPcrIndex();
			byte[] msg = new byte[40];
			for (int i = 0; i < 20; i++) {
				msg[i] = _pcr[pcrindex][i];
				msg[i + 20] = event.getDigest()[i];
			}
			_pcr[pcrindex] = DigestTool.SHA1(msg);
			_extended[pcrindex] = true;
		}
	}

	/**
	 * @param _properties2
	 * @return
	 */
	public PlatformProperties update(PlatformProperties prop) {
		for (int i = 0; i < 24; i++) {
			String name = "tpm.pcr." + i;
			String value = Base64Tool.encode(_pcr[i]);
			prop.setProperty(name, value);
		}
		return prop;
	}

	/**
	 * 
	 * @param pcrindex
	 * @param pcrvalue
	 * @return 0 VALID, 1 INVALID, 2 UNVERIFYED,
	 */
	public int validate(int pcrindex, byte[] pcrvalue) {
		if (_extended[pcrindex] == false) {
			return 2;
		}
		for (int i = 0; i < 20; i++) {
			if (pcrvalue[i] != _pcr[pcrindex][i]) {
				return 1;
			}
		}
		return 0;
	}

	/**
	 * 
	 * Reset PCR (Power on Reset)
	 * 
	 * @param pcrnum
	 */
	public void resetPcr(int pcrnum) {
		if (pcrnum < 16) {
			for (int i = 0; i < 20; i++) {
				_pcr[pcrnum][i] = 0;
			}
		} else if (pcrnum < 24) {
			for (int i = 0; i < 20; i++) {
				_pcr[pcrnum][i] = (byte) 0xff;
			}
		}
	}

	/**
	 * Print PCRS
	 * 
	 * PCR-00: 6C 87 A9 A7 88 B5 26 D6 CE A9 7B A7 FE 97 01 D8 6E A9 F5 D5
	 * 
	 */
	public void print() {
		for (int i = 0; i < 24; i++) {
			String indexStr;
			if (i < 10) {
				indexStr = "0" + i;
			} else {
				indexStr = "" + i;
			}

			String digestStr = HexTool.getHEXString(_pcr[i], " ");
			System.out.println("PCR-" + indexStr + ": " + digestStr);
		}
	}
}
