/*
 * 
 * Licensed Materials - Property of IBM
 *
 * Open Platform Trust Services - An open source TCG PTS
 *
 * (C) Copyright International Business Machines Corp. 2007
 *
 */
package com.ibm.trl.tcg.pts.engine;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
//import java.util.Properties;
import java.util.Vector;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
// IBM JRE
//import org.apache.xml.serializer.OutputPropertiesFactory;
// SUN JRE
//import com.sun.org.apache.xml.internal.serializer.OutputPropertiesFactory;


import org.w3c.dom.DOMException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

import com.ibm.trl.tcg.pts.eventlog.EventSet;
import com.ibm.trl.tcg.pts.eventlog.RuntimeDigest;
//import com.ibm.trl.tcg.pts.integrity.aide.AIDE;
//import com.ibm.trl.tcg.pts.integrity.iidb.IIDB;
import com.ibm.trl.tcg.pts.tools.Base64Tool;
import com.ibm.trl.tcg.pts.tools.BlobTool;
import com.ibm.trl.tcg.pts.tools.HexTool;
//import com.sun.org.apache.xml.internal.serializer.OutputPropertiesFactory;

/**
 * 
 * UML2 FSM (abstruct level) -> DOM <- existing eventlog | V new FSM (manifest
 * level), Simple RIMM
 * 
 * For IBM Rational (XMI 2.0), NEC SDAM (XMI 2.1)
 * 
 * XMI2.0 Guard <transition xmi:id="EV_S_CRTM_VERSION_trans"
 * name="EV_S_CRTM_VERSION_trans" kind="external" source="Start"
 * target="EV_S_CRTM_VERSION"> <guard xmi:id="EV_S_CRTM_VERSION_trans_guard">
 * <specification xmi:type="uml:OpaqueExpression"
 * xmi:id="EV_S_CRTM_VERSION_trans_spec" body="pcrindex == 0, eventtype == 0x08,
 * digest == base64" /> </guard> </transition>
 * 
 * XMI2.1 Guard <transition xmi:id="IPL5_HOLD_loop" name="IPL5_HOLD_loop"
 * visibility="public" kind="local" target="IPL5_HOLD"
 * guard="_bFnG1AhJEdyNdsh2vL3IaQ" source="IPL5_HOLD"> <ownedRule
 * xmi:id="_bFnG1AhJEdyNdsh2vL3IaQ" name=""> <specification
 * xmi:type="uml:OpaqueExpression" xmi:id="_bFnG1QhJEdyNdsh2vL3IaQ" name="">
 * <body>pcrindex == 0x05</body> </specification> </ownedRule> </transition>
 * 
 * NEC SDAM (XMI 2.1) FinalState is not a psudo state <subvertex
 * xmi:type="uml:FinalState" xmi:id="Final" incoming="Final_trans"/>
 * 
 * @author Seiji Munetoh (munetoh@users.sourceforge.jp)
 * 
 */
public class FiniteStateMachine {

	/* TCG define three states */
	//private static final int VALID = 0;
	//private static final int INVALID = 1;
	//private static final int UNVERIFIED = 2;

	/* Logger */
	private Log log = LogFactory.getLog(this.getClass());

	// private EventGrub eventGrub;

	/* DOM for read */
	private Document _doc;
	private Element _root;
	private NodeList _regions;
	private Element _start;
	private Element _currentState;
	private Element _loopTrans;
	private Element _prevState;
	
	/* String for new FSM (UML2) */

	/* misc */
	private int _stayCount = 0;
	private String _name = "na";
	private String _nameIMA = "na";
	private boolean _expandLoop;
	private boolean _includeManifest;
	private boolean _wasLoop;
	private boolean _terminate;
	private Vector<Event> _restEvents;
	private PlatformProperties _properties;
	private int _transCount;
	private double _xmi = 2.0;
	private boolean _runtimeMode = false;
	private TPM _tpm;
	private int _biosActionCount;
	private EventLinuxTCB _imaEvent;
	private boolean _hold;
	private boolean _internalTrans;
	private boolean _verbose = false;
	private String _message;
	private int _countUnverifiedEvent;
	private String _reason = ""; // IMA
	private boolean _elseState;
	private boolean _ima_ng_invalid_event = false;
	private int _level = 0;

	/**
	 * Initialize FSM with Behavior Model UML2 State Diagram
	 * 
	 * @param modelFilename
	 * @throws Exception
	 */
	public FiniteStateMachine(String modelFilename) throws Exception {
		reset();
		loadBehaviorModelFile(modelFilename);
		_terminate = false;
		_transCount = 0;

		_tpm = new TPM();
		_biosActionCount = 0;
		_restEvents = new Vector<Event>();

	}

	/**
	 * @param rm
	 * @throws Exception
	 */
	public FiniteStateMachine(Element root) throws Exception {
		_root = root;
		setup();
		_terminate = false;
		_transCount = 0;

		_tpm = new TPM();
		_biosActionCount = 0;
		_restEvents = new Vector<Event>();
	}

	/**
	 * @param modelFilename
	 * @throws ParserConfigurationException
	 * @throws IOException
	 * @throws Exception
	 */
	public void loadBehaviorModelFile(String modelFilename) throws Exception {
		log.debug("Load Behevior Model " + modelFilename);

		/* Check file */
		if (modelFilename == null) {
			throw new Exception("Behavior Model Filename is null");
		}

		InputStream ips = this.getClass().getClassLoader().getResourceAsStream(
				modelFilename);
		if (ips == null) {
			throw new Exception("Behavior Model File not found "
					+ modelFilename);
		}

		/* DOM Setup for bahavior model */
		DocumentBuilderFactory dbfactory = DocumentBuilderFactory.newInstance();
		DocumentBuilder builder = dbfactory.newDocumentBuilder();
		_doc = builder.parse(ips);
		_root = _doc.getDocumentElement();
		setup();

	}

	/**
	 * 
	 */
	private void setup() {
		/* Check XMI version */
		String xmiVer = _root.getAttribute("xmi:version");
		_xmi = Double.valueOf(xmiVer);

		/* Check regions */
		_regions = _root.getElementsByTagName("region");
		for (int i = 0; i < _regions.getLength(); i++) {
			/* Check all subvertex(state) and find initial state */
			Element r = (Element) _regions.item(i);
			log.trace("region : " + r.getAttribute("name"));
			NodeList nl = r.getElementsByTagName("subvertex");
			for (int j = 0; j < nl.getLength(); j++) {
				Element sv = (Element) nl.item(j);
				String name = sv.getAttribute("name");
				log.trace("subvertex : " + name);
				if (name.equals("Start")) {
					_start = sv;
					_currentState = sv;
					_stayCount = 0;
				}
			}
		}
	}

	/**
	 * 
	 * validate Eventlog (IML)
	 * 
	 * @param eventList
	 * @param integrityProps 
	 * @param  
	 * @throws Exception
	 */
	public boolean validate(Event[] eventList, IntegrityProperties integrityProps) throws Exception {
		boolean rc = true;
		_countUnverifiedEvent = 0;

		/* check input and properties */
		if (eventList == null)
			throw new Exception("null event!!");
		if (_properties == null)
			throw new Exception(
					"_properties is null, please call reset() before valdate()");

		if (log.isDebugEnabled()) {
			log.debug(" Event # " + eventList.length);
			for (int i = 0; i < eventList.length; i++) {
				log.debug(eventList[i].toString());
			}
		}

		/* validate event[] */
		for (int i = 0; i < eventList.length; i++) {
			/* Validate Event and Event Sequance by Model/Manifest */
			if (validateEvent(eventList[i]) == false) {
				// no FSM for event? 
				log.error("validateEvent() was failed");
				log.error("  Event[" + i + "]:  " + eventList[i].toString());
				rc = false;
				integrityProps.setIntegrityProp(eventList[i], _level, "INVALID");
				throw new Exception("Wrong event. validation was failed. please see the logfile for the detail");
			} else {
				/* valid event integrity */
				if (_hold == false) {
					/* validate Event itself */
					boolean rce = eventList[i].validate();

					String str = eventList[i].getEventString();
					_message += eventList[i].getMessage();

					if (rce == false) {
						log.error("validate failed");
						log.error(" Event   : " + eventList[i].toString());
						log.error(" Message : " + _message);
						rc = false;
						integrityProps.setIntegrityProp(eventList[i], _level, "INVALID");
						throw new Exception("e[i].validate() was failed");
					} else {
						integrityProps.setIntegrityProp(eventList[i], _level, "VALID");
					}

					if (_verbose) {
						log.debug(str + " - " + _message);
					}
				} else {
					if (log.isDebugEnabled()) {
						log.debug("Hold event");
					}
					// TODO 
					integrityProps.setIntegrityProp(eventList[i], _level, "HOLD");
				}
			}

		}

		if (_terminate != true) {
			/* Push */
			log.debug("Push");
			Event et = new Event();
			validateEvent(et);
		}

		/* Finalize IMA Measurment */
		if (_imaEvent != null) {
			/* */
			_reason = _imaEvent.addSummary(_properties);
			if (_reason.equals("")) { // TODO FIX
				// no problems
				rc = true;
			} else {
				// some known problems
				log.error("INVALID :" + _reason);
				rc = false;

			}

			_imaEvent = null;
		}

		return rc;
	}

	/**
	 * 
	 * For VMM Status check
	 * 
	 * @return
	 * @throws Exception
	 */
	public boolean validate() throws Exception {
		//boolean rc = true;

		if (log.isDebugEnabled()) {
			log.debug("VMM Validate without Event ");
		}

		/* Create Dummy Event to drive FSM */
		Event[] events = new Event[2];
		byte[] buf = new byte[20];
		for (int i = 0; i < 20; i++)
			buf[i] = 0;

		events[0] = new Event(17, 1, buf, buf);
		events[1] = new Event(17, 1, buf, buf);

		/* Validate */
		return validate(events, null);
	}


	/**
	 * validate eventSet
	 * 
	 * 
	 * @param eventSet
	 * @return
	 * @throws Exception
	 */
	public boolean validate(EventSet[] eventSet) throws Exception {
		/* check the size */
		int len = 0;
		for (int i = 0; i < eventSet.length; i++) {
			len += eventSet[i].size;
		}

		/* aggregate */
		int k = 0;
		Event[] e = new Event[len];
		for (int i = 0; i < eventSet.length; i++) {
			for (int j = 0; j < eventSet[i].size; j++) {
				e[k] = eventSet[i].getEvent(j);
				log.debug(k + " " + e[k].toString());
				k++;
			}
		}

		return validate(e, null);

	}
	

	//private static final int EVENT_ERROR = -1;
	//private static final int EVENT_INVALID = 0;
	private static final int EVENT_INFOMATIVE = 1;
	private static final int EVENT_CODE = 2;
	private static final int EVENT_TERMINATED = 3;

	/**
	 * 
	 * Drive FSM with SML and create the Runtime Model
	 * 
	 * A loop with digest check is extracted to individual state e.g. CRTM
	 * 
	 * A loop with type or pcrindes check is keep the current model. e.g.
	 * Platform Specific Event, Boot Message
	 * 
	 * @param e
	 * @return return code -1 ERROR 0 invalid event 1 infomative event 2 code
	 *         measurement (manifest) 3 terminated
	 * 
	 * TODO define
	 */
	public int generateRuntimeModelByEventlog(Event event) throws Exception {
		int rc = EVENT_INFOMATIVE;// EVENT_ERROR;
		if (log.isDebugEnabled()) {
			log.debug("----------------------------------------------------------------------");
			log.debug("state: " + toString(_currentState));
			log.debug("event: " + event.toString());
		}

		/* Get outgoings of current state */
		// SDAM 
		//String[] outgoings = _currentState.getAttribute("outgoing").split(" "); // TODO MDT does not support outgoing attribute 
		// MDT
		Element[] outgoingTrans = getOutgoingTrans(_currentState);

		/* Final */
		if (_currentState.getAttribute("xmi:type").equals("uml:FinalState")) {
			return EVENT_TERMINATED;
		}

		if (_terminate) {
			_restEvents.add(event);
			return EVENT_TERMINATED;
		}

		if (outgoingTrans == null) {
			/* no outgoings, final? */
			String st = _currentState.getAttribute("xmi:type");
			if (st.equals("uml:FinalState")) {
				_terminate = true;
				_restEvents.add(event);
			} else {
				log.error("no outgoing at state "
						+ _currentState.getAttribute("name"));
			}
			throw new Exception("no outgoing at state from "
					+ _currentState.getAttribute("name"));
		} else {
			/* check outgoings */
			boolean hit = false;
			for (int i = 0; i < outgoingTrans.length; i++) {
				/* Get Next State */
				Element trans = outgoingTrans[i];

				String nextStateID = trans.getAttribute("target");
				Element nextState = getSubvertex(nextStateID);

				/* Check the Guards */
				String guardString = getGuardString(trans);

				if (guardString == null) {
					/* no guard to trans to the next state */
					if (log.isDebugEnabled())
						log.debug("no guard to : " + nextStateID + " "
								+ nextState.toString());

					_prevState = _currentState;
					_currentState = nextState;

					/* enter may has verifyProperty */
					updateDoActivity(_currentState, event);

					/* move to next state and run this method */
					if (log.isDebugEnabled())
						log
								.debug("note) Call generateRuntimeModelByEventlog with same event");

					/* try to use the same event for next state */
					return generateRuntimeModelByEventlog(event);
				} else {
					/* with guard, check the new guard */
					String newGuardString = validateAndUpdateGuard(event,
							guardString);

					if (newGuardString != null) {
						hit = true;
						/* guard was updated */
						if (_currentState != nextState) {
							/* move to the next state */
							_stayCount = 0;

							if (_expandLoop) {
								addNewTrans(_transCount, _currentState, trans,
										nextState, newGuardString);
								_transCount++;
								newGuardString = guardString;
							}

							/* if last state has a loop, delate the loop */
							if (_wasLoop) {
								deleteTrans(_loopTrans);
								/*
								 * Update GuardString, now this tans is runtime
								 * level
								 */
								updateGuardString(trans, newGuardString);
							} else {
								/*
								 * Update GuardString, now this tans is runtime
								 * level
								 */
								updateGuardString(trans, newGuardString);
							}

							_prevState = _currentState;
							_currentState = nextState;
							_wasLoop = false;
						} else {
							/* loop state */
							if (_expandLoop) {
								/* expand this loop */
								_wasLoop = true;
								_loopTrans = trans;
								_stayCount++;
								_prevState = addNewSubvertex2(_stayCount,
										_prevState, _currentState, trans,
										newGuardString);
							} else {
								/* keep this loop */
								_wasLoop = false;
							}
						}

						/* Set name */
						_name = _currentState.getAttribute("name") + "-"
								+ _stayCount;

						if (_includeManifest) {
							return EVENT_CODE; // TODO
						} else {
							return EVENT_INFOMATIVE; // TODO
						}
					} else { /* Update Gurd was failed? */
					} // guardString
				} // outgoings loop
			} // for loop

			/* Check no Trans condition */
			if (hit == false) {
				log.error("No Transfer #################");
				throw new Exception("no Trans?");
			}

		} // outgoings[0].equals("")

		return rc;// EVENT_INFOMATIVE;
	}

	/**
	 * 
	 * Validate Eventlog
	 * 
	 * @param event
	 * @throws Exception
	 */
	public boolean validateEvent(Event event) throws Exception {
		boolean rc = true;
		_hold = false;
		_message = ""; // reset

		if (log.isDebugEnabled()) {
			// log.trace("validateEvent start ");
			log.debug("--------------------------------------------------------------------------");
			log.debug("state: " + toString(_currentState) 
					+ " (xmi:type=" + _currentState.getAttribute("xmi:type")
					+ ", xmi:id=" + _currentState.getAttribute("xmi:id") + ")" );
			log.debug("event: " + event.toString());
			log.debug("--------------------------------------------------------------------------");
		}

		/* Get outgoings of current state */
		// SDAM
		//String[] outgoings = _currentState.getAttribute("outgoing").split(" "); // TODO MDT does not support outgoing attribute 
		// MDT
		Element[] outgoingTrans = getOutgoingTrans(_currentState);
		
		/* is terminated? */
		if (_terminate) {
			_message += "FSM terminated";
			if (log.isDebugEnabled()) {
				log.debug(_message);
			}
			_restEvents.add(event);
			_hold = true; // TODO
			return true; /* this is not a error */
		}

		//if (outgoings[0].equals("")) {
		if (outgoingTrans == null) {
			/* No output trans, is this Final state? */
			// TODO state:  (xmi:type=uml:State, xmi:id=Final)  ???? 
			// TODO UML is OK, BUT RM contains BAD xmi:type=uml:State 
			// TODO FIX RM gen
			String xt = _currentState.getAttribute("xmi:type");
			String xi = _currentState.getAttribute("xmi:id");
			if ((xt.equals("uml:FinalState") || (xi.equals("Final")))) {
				/* Final State */
				if (log.isDebugEnabled())
					log.debug("Final state");

				_restEvents.add(event);
				setProperty("integrity", "valid");
				_terminate = true;
				_hold = true; // TODO
				rc = true;
			} else {
				log.error("no outgoing at state ["
						+ _currentState.getAttribute("name") + "] xmi:type = " + xt + "xmi:id = " + xi);
				rc = false;
			}
		} else {
			/* has trans */
			_elseState = false;
			Element elseNextState = null;
			for (int i = 0; i < outgoingTrans.length; i++) {
				/*
				 * 2007-06-01 SM split has problem if two spaces are exist
				 * XXX__XXX => "XXX" "" "XXX"
				 * 
				 */
				if (outgoingTrans[i] == null) { // TODO never happen?
					// no outgoings...
					log.error("no outgoings...");
					rc = false;
				} else {
					/* Get Next State */
					Element trans = outgoingTrans[i];
					if (trans == null) {
						// No transition...
						log.error("getTransition of index = " + i + ", "
								+ "NULL" //+ outgoings[i] 
								            + " not found");
						//log.error(" outgoings.length=" + outgoings.length);
						//log.error(" outgoing["
						//		+ _currentState.getAttribute("outgoing") + "]");
					} else {
						String nextStateID = trans.getAttribute("target");
						Element nextState = getSubvertex(nextStateID);

						/* Check the Guards */
						String guardString = null;
						guardString = getGuardString(trans);

						if (guardString == null) {
							/* no guard trans to next state */
							if (log.isTraceEnabled()) {
								log.trace("validateGuard - SKIP ");
							}
							_prevState = _currentState;
							_currentState = nextState;

							checkDoActivity(nextState, event);

							/* Check this event by next transition */
							rc = validateEvent(event);
						} else if (guardString == "") {
							/* no guard trans to next state */
							if (log.isTraceEnabled()) {
								log.trace("validateGuard - SKIP ");
							}
							_prevState = _currentState;
							_currentState = nextState;

							checkDoActivity(nextState, event);

							/* Check this event by next transition */
							rc = validateEvent(event);
						} else {
							/* guard exist */
							if (validateGuard(event, guardString)) {
								/* Hit */
								if (log.isTraceEnabled()) {
									// log.trace("validateGuard  [" + guardString + "] - HIT ");
									log.trace("validateGuard : HIT => " + nextStateID);
								}

								_prevState = _currentState;
								_currentState = nextState;
								/* check Activity */
								if (checkDoActivity(nextState, event)) {
									/* event Hold State */
									if (log.isTraceEnabled()) {
										log.trace("event Hold State");
									}

									_restEvents.add(event);
									_hold = true; // TODO note tested yet
													// 07-6-13
									// SM
								} else {
									/* Consume this event */
									//if (log.isTraceEnabled()) {
									//	log.trace("Consume this event ");
									//}
									// TODO Linu-IMA-NG log - 0x00 -> 0xFF 
									// EXP PCR-10: 55 61 46 D8 2C 32 1B 21 7E CB CC 91 0D 5C 64 A5 46 65 7B 36
									// PCR-10: DD B4 A7 7F 66 4A 1D DE 07 64 B5 B4 D9 D9 73 CE 09 1A 86 03
									// PCR-10: 11 99 F4 1C 2C 18 0A 60 60 38 BA 01 85 2F 52 99 66 69 2E 8C
									// PCR-10: 55 61 46 D8 2C 32 1B 21 7E CB CC 91 0D 5C 64 A5 46 65 7B 36 OK
									// TODO when we should swap this? 1:Kernel,2:TSS,3:PTS?
									if (_ima_ng_invalid_event) {
										byte[] digest = 
										{(byte)0xff,(byte)0xff,(byte)0xff,(byte)0xff,(byte)0xff,(byte)0xff,(byte)0xff,(byte)0xff,
										 (byte)0xff,(byte)0xff,(byte)0xff,(byte)0xff,(byte)0xff,(byte)0xff,(byte)0xff,(byte)0xff,
										 (byte)0xff,(byte)0xff,(byte)0xff,(byte)0xff};		
										_tpm.extend((int) event.getPcrIndex(), digest);// 
									} else {
										_tpm.extend(event); /* Update PCRs */
									}
								}

								/* update PCR */
								if (_internalTrans) {
									/* Auto Trans */
									if (log.isTraceEnabled()) {
										log.trace("CAuto Trans ");
									}
									rc = validateEvent(event);
								}

								//if (log.isTraceEnabled()) {
								//	log.trace("validateEvent done rc " + rc
								//			+ " HIT ");
								//}
								return true;
								// rc = true;
							} else {
								/* miss */
								if (log.isTraceEnabled()) {
									log.trace("validateGuard : Miss X=> " + nextStateID);
								}
								
								if (_elseState) { // but else
									elseNextState = nextState;
								}
								rc = false;
							}
						}
					} // null check
				} // guard
			} // for trans

			/* no trand but else */
			if ((rc == false) && (_elseState)) {
				_prevState = _currentState;
				_currentState = elseNextState;

				checkDoActivity(elseNextState, event);

				/* Check this event by next transition */
				rc = validateEvent(event);
			}

		} // trans

		if (log.isTraceEnabled()) {
			log.trace("validateEvent done rc " + rc);
		}
		return rc;
	}

	/**
	 * 
	 * Validation for snapshot generation in Integrity Report
	 * 
	 * @param event
	 * @return return code -1 ERROR 0 measured event ==> snapshot 1 hold event
	 *         ==> used by another snapshot 3 terminated
	 * @throws Exception
	 */
	public int validateEvent2(Event event) throws Exception {
		boolean rc = validateEvent(event);

		if (rc) {
			if (_terminate)
				return 3;
			if (_hold)
				return 1;
			return 0; // OK
		} else {
			log.error("SM DEBUG validateEvent2 fail, rc = " + rc);
			return -1;
		}
	}

	/**
	 * 
	 * guardString 
	 *   name0 == value0, name1 == value1, name2 == value2 < < AND
	 * 
	 * 2007-05-04 Support, name != value for eventtype LogLevel TRACE: show
	 * details
	 * 
	 * @param event
	 * @param boolean
	 * @return
	 * @throws Exception
	 */
	private boolean validateGuard(Event event, String guardString)
			throws Exception {
		boolean rc = true;
		_expandLoop = false;
		_includeManifest = false;
		_internalTrans = false; // TODO 2006-06-17
		
		/* check */
		if (guardString == null) {
			return rc;
		}
		if (guardString == "") {
			return rc;
		}
		
		/* parse guardString */
		String[] guards = guardString.split(",");
		for (int i = 0; i < guards.length; i++) {
			String[] guards2 = guards[i].trim().split(" ");
			String name = guards2[0];


			// Single  N
			// TODO BIOS PCR1 else
			if (name.equals("else")) {
				if (_elseState) {
					log.error("model contains multiple else trans");
					throw new Exception("model contains multiple else trans");
				}
				_elseState = true;
				return false;
			} else {
				_elseState = false;
			}

			// Multiple,  N = V
			if (guards2.length == 1) {
				// [eventtype == 0x4, ]
				// log,debug("SMDEBUG guards[" + i +"] =[" + guards[i] + "] <= guardString = [" + guardString + "]");
				// SKIP
			} else {
				String eq = guards2[1];
				String value = guards2[2];
	
				if (name.equals("eventtype")) {
					long eventtype = Long.decode(value).longValue();
	
					if (validateLongEq(eventtype, eq, event.getEventType())) {
						if (log.isTraceEnabled()) {
							log.trace(" guard : [" + name + " == " + value + "] -  HIT");
						}
					} else {
						if (log.isTraceEnabled()) {
							log.trace(" guard : " + name + " == " + value + "("
									+ eventtype + ") Miss");
						}
						rc = false;
						break;
					}
				} // name == eventtype
				else if (name.equals("pcrindex")) {
					long pcrindex = Integer.decode(value).intValue();
	
					if (eq.equals("==")) {
						if (pcrindex == event.getPcrIndex()) {
							if (log.isTraceEnabled()) {
								log.trace(" guard : pcrindex == " + pcrindex
										+ " HIT");
							}
						} else {
							if (log.isTraceEnabled()) {
								log.trace(" guard : pcrindex must be "
										+ event.getPcrIndex() + " but " + pcrindex
										+ " Miss");
							}
							rc = false;
							break;
						}
					} else if (eq.equals("!=")) {
						if (pcrindex != event.getPcrIndex()) {
							if (log.isTraceEnabled()) {
								log.trace(" guard : pcrindex != " + pcrindex
										+ " HIT");
							}
						} else {
							if (log.isTraceEnabled()) {
								log.trace(" guard : pcrindex must not be "
										+ event.getPcrIndex() + " but " + pcrindex
										+ " MISS");
							}
							rc = false;
							break;
						}
					}
				} // name == pcrindex
				else if (name.equals("digest")) {
					/* Validate digest value */
					if (value.equals("base64")) {
						// System.err.println("BASE64");
						if (_runtimeMode) {
							if (log.isTraceEnabled()) {
								log.trace("in runtime mode, BASE64 is INVALID");
							}
							_message += " unverified?";
							_countUnverifiedEvent++;
							rc = false;
						} else {
							if (log.isTraceEnabled()) {
								log
										.trace("not in runtime mode (behavior mode), BASE64 is OK");
							}
						}
	
						_includeManifest = true;
						_expandLoop = true;
					} // value == base64
					else if (value.equals("ask")) {
						/* Digest is not in this Manifest, need to ask IIDB */
						if (log.isDebugEnabled()) {
							log.debug("not verifyed " + guardString
									+ " with IIDB (digest="
									+ Base64Tool.encode(event.getDigest())
									+ ")");
						}
						_message += " unverified";
					} // name == digest
					else if (value.equals(Base64Tool.encode(event.getDigest()))) {
						if (log.isTraceEnabled()) {
							log.trace("guard : [digest = " + value + "] - HIT (by manifest)");
						}
						_message += " digest is verifyed by Manifest,";
					} else {
						/* miss, check another transition */
						if (log.isTraceEnabled()) {
							log.trace("Value "
									+ Base64Tool.encode(event.getDigest()) + " != "
									+ value + " missing");
						}
						rc = false;
						_message += " unverified";
						break;
					}
				} else {
					/* Internal Trans */
					if (log.isTraceEnabled()) {
						log.trace("Internal Transition (not use Event)");
					}
					/* properties */
					_internalTrans = true;
	
					//PlatformProperty prop = _properties.getPropertyByName(name);
					String propValue = _properties.getProperty(name);
					if (propValue == null) {
						log.error("null return from getPropertyByName(" + name + ")");
					} else {
						String refValue = propValue;
		
						if (eq.equals("==")) {
							if (value.equals(refValue)) {
								if (log.isTraceEnabled()) {
									log.trace(name + " == " + value + " HIT");
								}
							} else {
								if (log.isTraceEnabled()) {
									log.trace(name + " == " + value + " != " + refValue
											+ "(ref) MISS");
								}
								rc = false;
								break;
							}
						} else if (eq.equals("!=")) {
							if (value.equals(refValue)) {
								if (log.isTraceEnabled()) {
									log.trace(name + " != " + value + " == " + refValue
											+ "(ref) MISS");
								}
								rc = false;
								break;
							} else {
								if (log.isTraceEnabled()) {
									log.trace(name + " != " + value + " HIT");
								}
							}
						} else {
							log.error("Internal Error? unknown equation " + eq);
							throw new Exception("TODO? unknown eq " + eq);
						}
					}
				}
			}  // N = V
		} // for
		
		return rc;
	}

	/**
	 * check doActivity and run functions
	 * 
	 * <doActivity name="setAssertion(name,value)" ...> 0 1 2 values digest set
	 * base64 digest
	 * 
	 * 2007-06-16 SM update/Cleanup
	 * 
	 * @param nextState
	 * @throws Exception
	 *             return true:event hold, false:go next
	 */
	private boolean checkDoActivity(Element nextState, Event event)
			throws Exception {

		String stateName = nextState.getAttribute("name");
		_nameIMA = null;

		/* Check doActivity */
		NodeList nl = nextState.getElementsByTagName("doActivity");
		if (nl.getLength() == 0) {
			if (log.isDebugEnabled()) {
				log.debug("checkDoActivity - no doActivity");
			}
			return false; // not contains eventHold()
		}

		if (nl.getLength() > 1) {
			throw new Exception("multiple doActivities");
		}

		Element e = (Element) nl.item(0);
		String name = e.getAttribute("name");
		String[] func = name.split("[(,)]");
		for (int j = 0; j < func.length; j++) {
			func[j] = func[j].trim();
		}
		
		if (name == "") {
			//if (log.isDebugEnabled()) {
			//	log.debug("checkDoActivity - name is missing => no action");
			//}
			return false;
		}
		
		
		if (log.isDebugEnabled()) {
			log.debug("checkDoActivity - name = [" + name + "]");
		}

		if (name.equals("eventHold()")) {
			// skip
			return true;
		}

		/* General Functions */
		else if (func[0].equals("setAssertion")) {
			if (func[2].equals("digest")) {
				// set digest
				setProperty(func[1], Base64Tool.encode(event.getDigest()));
				if (log.isDebugEnabled()) {
					log.debug("setAssertion name  : " + func[1]);
					log.debug("setAssertion value : "
							+ HexTool.getHexString(event.getDigest()));
				}
			} else if (func[2].equals("eventdata")) {
				// TODO String or Binary?
				// setProperty(func[1], Base64Tool.encode(event.getDigest()));
				setProperty(func[1], event.getEventDataStr());
				if (log.isDebugEnabled()) {
					log.debug("setAssertion name  : " + func[1]);
					log
							.debug("setAssertion value : "
									+ event.getEventDataStr());
				}
			} else {
				if (log.isDebugEnabled()) {
					log
							.debug("FiniteStatemachine/checkDoActivity setAssertion "
									+ func[1] + " = " + func[2]);
				}
				setProperty(func[1], func[2]);
				if (log.isDebugEnabled()) {
					log.debug("setAssertion name  : " + func[1]);
					log.debug("setAssertion value : " + func[2]);
				}
			}
			_message += " " + func[1] + ",";
		}

		/* BIOS Special Functions */

		/* Bootloader Special Functions */

		else if (name.equals("validateMBR()")) {
			String refMbrDigest = getProperty("ipl.mbr.digest");
			String refMbrMaskedDigest = getProperty("ipl.mbr.masked.digest");

			/* Validate BIOS Measurement */
			RuntimeDigest rd = new RuntimeDigest();
			byte[] mbr = event.getPCEventData();

			/* mode2, 446(IBM) */
			String mbrDigest = rd.getStage1Digest(mbr, 2); // TODO try mode 1
			// also... 07-7-4 SM
			if (mbrDigest.equals(refMbrDigest) == false) {
				/* The IPL code measured by BIOS and Grub dose not match */
				log.error("Measured MBR does not match");
				log
						.error("BIOS Measure : "
								+ HexTool.getHexString(Base64Tool
										.decode(refMbrDigest)));
				log.error("ReCalc(446)  : "
						+ HexTool.getHexString(Base64Tool.decode(mbrDigest)));
				// log.error("Measured MBR does not match");
				// + mbrDigest + " != " + refMbrDigest + "(ref)");
				throw new Exception("Measured MBR does not match " + mbrDigest
						+ " != " + refMbrDigest + "(ref)");
			}

			String mbrMaskedDigest = rd.getStage1MaskedDigest(mbr);
			if (mbrMaskedDigest.equals(refMbrMaskedDigest) == false) {
				/* For DEBUG */
				if (log.isDebugEnabled()) {
					log.debug("Masked MBR does not match "
							+ HexTool.getHexString(mbr));
				}
				throw new Exception("Masked MBR does not match "
						+ mbrMaskedDigest + " != " + refMbrMaskedDigest
						+ "(ref)");
			}

			// 0x90 => -112
			if ((mbr[0x04b] != -112) || (mbr[0x04c] != -112)) {
				throw new Exception("MBR(4bh-4ch are not a nop (90h) "
						+ HexTool.getHexString(mbr[0x04b]) + " " + mbr[0x04b]
						+ " " + HexTool.getHexString(mbr[0x04c]) + " "
						+ mbr[0x04c] + " ");
			}
			_message += " Good IPL(MBR) Image,";
		} else if (name.equals("validateEltoritoBootImage()")) {
			/* get proiperties */
			String refMbrDigest = getProperty("ipl.mbr.digest");
			String refDigest0 = getProperty("ipl.eltorito.digest");
			String refDigest1 = getProperty("ipl.eltorito.hp.digest");
			String refDigest2 = getProperty("ipl.eltorito.ibm.digest");

			/* Validate */
			//boolean state = false;
			if (refMbrDigest.equals(refDigest0)) {
				_message += " Good ElTorito Boot Image,";
				//state = true;
			} else if (refMbrDigest.equals(refDigest1)) {
				_message += " Good ElTorito Boot Image(HP),";
				//state = true;
			} else if (refMbrDigest.equals(refDigest2)) {
				// see ReferenceManifest()
				_message += " Good ElTorito Boot Image(IBM)),";
				//state = true;
			} else {
				_message += " Unknown ElTorito Boot Image,";
				// if (log.isDebugEnabled()) {
				log.error("BootImage did not match " + refMbrDigest);
				// }
				// throw new Exception("BootImage did not match " +
				// refMbrDigest);
			}

		}

		/* Linux IMA Special Functions */

		// else if (func[0].equals("checkKernelCommandline")) {
		// if (_imaEvent == null)
		// _imaEvent = new EventLinuxTCB();
		// _imaEvent.setKernelCommandline(_properties, func[1]);
		// }
		/* Validate ima=1 in kernel cmdline */
		else if (name.startsWith("validateKernelCmdline")) {
			String str = func[1] + "=" + func[2];

			// String cmdline =
			// _properties.getPropertyByName("linux.kernel.comandline").getValue();
			String cmdline = _properties
					//.getPropertyValueByName("linux.kernel.comandline");
					.getProperty("linux.kernel.comandline");

			if (cmdline == null) {
				log.warn(" verifyKernelCmdline NG " + name
						+ " Quote include PCR5? ");
				throw new Exception("verifyKernelCmdline failed " + name
						+ " Quote include PCR5? ");
			} else {
				if (cmdline.contains(str)) {
					if (log.isDebugEnabled()) {
						log.debug(" verifyKernelCmdline OK " + name);
					}
				} else {
					log.warn(" verifyKernelCmdline NG " + name + " ");
					throw new Exception("verifyKernelCmdline failed " + name);
				}
			}

		}

		/* Validate IMA Boot Aggregate (Original Linux-IMA)*/
		else if (name.equals("validateImaAggregate()")) {
			if (_imaEvent == null)
				_imaEvent = new EventLinuxTCB();
			_tpm.update(_properties);
			// 20071117 SM set status to properties
			// use NG
			if (_imaEvent.checkImaAggregateNG(_properties, stateName, event)) {
				// Good
				setProperty("ima.aggregate", "valid");
			} else {
				// NG
				setProperty("ima.aggregate", "invalid");
			}

			_message += " Good Boot Aggrigate,";
		}
		/* Validate IMA's measurement */
		else if (name.equals("validateImaMeasurement()")) {
			if (_imaEvent == null)
				_imaEvent = new EventLinuxTCB();
			// TODO CHANGE for 0.2.3
			/* set snapshot name too */
			//_nameIMA = _imaEvent.validateImaMeasurement(
			//			_properties, 
			//			stateName,
			//			event, _iidb);
			//_message += _imaEvent.message;

			// _nameIMA = event.getEventDataStr();
			// System.err.println("SM DEBUG FSM/ " + stateName + " " +
			// _nameIMA);
		}
		
		/* Validate IMA Boot Aggregate (New Linux-IMA)*/
		else if (name.equals("validateImaAggregateNG()")) {
			if (_imaEvent == null)
				_imaEvent = new EventLinuxTCB();
			
			//_tpm.resetPcr(10); // TODO get pcr index
			_tpm.update(_properties);

			if (_imaEvent.checkImaAggregateNG(_properties, stateName, event)) { // Good
				setProperty("ima.bootaggrigate", "valid");
				_message += " Good Boot Aggrigate,";
			} else { // NG
				setProperty("ima.bootaggrigate", "invalid");
				_message += " BAD Boot Aggrigate,";
			}
		}
		/* Validate IMA-NG's measurement */
		else if (name.equals("validateImaMeasurementNG()")) {
			if (_imaEvent == null)
				_imaEvent = new EventLinuxTCB();

			/* set snapshot name too */
			_nameIMA = _imaEvent.validateImaMeasurementNG(_properties, stateName,
					event); 
			_message += _imaEvent.message;
			_ima_ng_invalid_event = _imaEvent.ima_ng_invalid_event;
		}
		
		
		/* OLD Functions pls. update 2007-06-17 SM */

		else if (name.equals("addBIOSAction()")) {

			String pcr = Long.valueOf(event.getPcrIndex()).toString();
			//String pname = "bios.action." + "pcr" + pcr + "."
			//		+ _biosActionCount;
			String pname = "bios.pcr." + pcr + ".action." + _biosActionCount;
			String msg = event.getEventDataStr();

			//setProperty(pname + ".pcrindex", pcr);
			//setProperty(pname + ".message", msg);
			setProperty(pname + "", msg);
			
			if (log.isDebugEnabled()) {
				log.debug("addBIOSAction() num = " + _biosActionCount);
				log.debug("name  : " + pname);
				log.debug("value : " + msg);
			}

			_biosActionCount++;
		} else if (name.equals("addBIOSSpecificProperty()")) {
			// System.err.println(" checkEnter : " +
			// nextState.getAttribute("xmi:id"));
			// System.err.println(" checkEnter : " + name + " N/A");
			// addBIOSSpecificProperty(event);
			//PlatformProperty p = PCPlatformSpecificEvent.getProperty(event);
			//_properties.add(p);
			PCPlatformSpecificEvent.setProperties(_properties, event);
		//} else if (name.equals("setGrubProperty()")) {
		//	EventGrub eventGrub = new EventGrub();
		//	eventGrub.setGrubProperty(_properties, stateName, event);
		} else if (name.equals("resetTPM()")) { // TODO
			// if (log.isDebugEnabled()) {
			log.error(" resetTPM() : " + name + " N/A");
			// }
		} else if (name.startsWith("resetPCR")) { // TODO
			int pcrnum = Integer.valueOf(func[1]).intValue();
			_tpm.resetPcr(pcrnum);
			if (log.isDebugEnabled()) {
				log.debug(" reset PCR[" + pcrnum + "] " + name + " N/A");
			}
		} else if (name.startsWith("validateProperty")) { // TODO
			//String p = func[1];
			//String digest = func[2];
			if (_properties.verifyProperty(func[1], func[2])) {
				// HIT
				if (log.isDebugEnabled()) {
					log.debug(" validateProperty OK " + name);
				}
			} else {
				// MISS
				log.error(" validateProperty failed " + func[1] + " != "
						+ func[2]);
						//+ _properties.getPropertyByName(func[1]).getValue()
						//+ "");
			}

			// if (log.isDebugEnabled()) {
			// }
		}
		// else if (name.startsWith("setIMAProperty")) { // TODO
		// //if (log.isDebugEnabled()) {
		// log.warn(" setIMAProperty : " + name + " N/A");
		// //}
		// }
		else if (name.startsWith("verifyVMM")) {
			_properties.setProperty("vmm.drtm", "no(demo)");
			// TODO to validate VMM we need curent PCR value instead of eventlog
			// since VMM does not keep eventlog
			log.error("verifyVMM - NA ");
		}
		else if (name.startsWith("flashFSM")) {
			// TODO flashFSM
			log.debug("flashFSM - NA ");
		}
		else if (name.startsWith("setLinuxKernelCmdlineAssertion")) {			
			// TODO setLinuxKernelCmdlineAssertion
			// linux.kernel.cmdline.exec-shield=0
			String msg = event.getEventDataStr();
			String options[] = msg.split(" "); 
			//for (int i = 0; i < options.length; i++) {
			// SKIP ./vmlinuz-2.6.32-71.14.1.el6.x86_6
			for (int i = 1; i < options.length; i++) {
				String  option[] = options[i].split("=");
				if (option.length == 1) {
					String pname = "linux.kernel.cmdline." +  option[0];
					_properties.setProperty(pname, "");
				} else if (option.length == 2){
					String pname = "linux.kernel.cmdline." +  option[0];
					_properties.setProperty(pname, option[1]);
				} else {
					log.error("setLinuxKernelCmdlineAssertion - NA " + msg);
				}
			}
			//log.error("setLinuxKernelCmdlineAssertion - NA " + msg);
		} else if (name.startsWith("startCollector")) {
			// TODO add prop gen
			log.error("startCollector() is not supported");
		} else if (name.startsWith("addIntelTxtTbootProperty")) {
			// TODO add prop gen
			log.error("addIntelTxtTbootProperty() is not supported");
			long eventtype = event.getEventType();
			if (eventtype == 0x100) {
				// EV_TBOOT_SINIT_V6
				// intel.txt.tboot.sinit.hash.hex
				byte ed[] = event.getEventData();
				_properties.setProperty(
						"intel.txt.tboot.sinit.hash.hex",
						HexTool.getHexString(ed, 0, 20));
			} else if (eventtype == 0x101) {
				// EV_TBOOT_STM_V6
				// intel.txt.tboot.bios.acm.id.hex
				byte ed[] = event.getEventData();
				_properties.setProperty(
						"intel.txt.tboot.bios.acm.id.hex",
						HexTool.getHexString(ed, 0, 20));
			} else if (eventtype == 0x102) {
				// EV_TBOOT_POCTL
				// intel.txt.tboot.pol.control.hex
				// intel.txt.tboot.pol.hash.hex
				byte ed[] = event.getEventData();
				_properties.setProperty(
						"intel.txt.tboot.pol.control.hex",
						HexTool.getHexString(ed, 0, 4));
				_properties.setProperty(
						"intel.txt.tboot.pol.hash.hex",
						HexTool.getHexString(ed, 4, 20));
			} else if (eventtype == 0x103) {
				// EV_TBOOT_SINIT_V7
				// intel.txt.tboot.sinit.hash.hex
				byte ed[] = event.getEventData();
				_properties.setProperty(
						"intel.txt.tboot.sinit.hash.hex",
						HexTool.getHexString(ed, 0, 32));
			} else if (eventtype == 0x110) {
				// EV_TBOOT_MLE_HASH
				// intel.txt.tboot.mle.hash.hex

			} else if (eventtype == 0x111) {
				// EV_TBOOT_MODULE
				// intel.txt.tboot.pcr.N.module.command.hash.hex
				// intel.txt.tboot.pcr.N.module.file.hash.hex
				// intel.txt.tboot.pcr.N.module.command
				// intel.txt.tboot.pcr.N.module.filename
				long pcrindex = event.getPcrIndex();
				byte ed[] = event.getEventData();
				_properties.setProperty(
						"intel.txt.tboot.pcr." + pcrindex + ".module.command.hash.hex",
						HexTool.getHexString(ed, 0, 20));
				_properties.setProperty(
						"intel.txt.tboot.pcr." + pcrindex + ".module.file.hash.hex",
						HexTool.getHexString(ed, 20, 20));
				long size = BlobTool.UINT32(ed, 40);
				if (size <= ed.length - 44) {
					if (size > 0) {
						_properties.setProperty(
								"intel.txt.tboot.pcr." + pcrindex + ".module.command",
								BlobTool.getString(ed, 44, (int) size));
					}
					long size2 = BlobTool.UINT32(ed, (int) (44 + size));
					if (size2 <= (ed.length - 44 - size - 4)) {
						if (size2 > 0) {
							_properties.setProperty(
									"intel.txt.tboot.pcr." + pcrindex + ".module.filename",
									BlobTool.getString(
											ed, 
											(int)(44 + size + 4), 
											(int) size2));
						}
					}
				}
			}
			
		} else {
			// if (log.isDebugEnabled()) {
			log.error("checkDoActivity - " + "checkEnter, state: "
					+ nextState.getAttribute("xmi:id") + ", enter : [" + name
					+ "] << unknown ");
			// }
			// System.err.print("checkEnter, state: " +
			// nextState.getAttribute("xmi:id"));
			// System.err.println(", enter : " + name + " << unknown ");
			throw new Exception("checkEnter, state: "
					+ nextState.getAttribute("xmi:id") + ", enter : [" + name
					+ "] << unknown ");
		}

		if (log.isDebugEnabled()) {
			log.debug("checkDoActivity done");
		}

		return false; // not contains eventHold()
	}

	public void setProperty(String name, String value) {
		try {
			// System.err.println("SM DEBUG setProperty :" + name + " = " +
			// value);
			_properties.setProperty(name, value);
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	public String getProperty(String name) {
		//PlatformProperty out = null;
		try {
			//out = _properties.getPropertyByName(name);
			//return out.getValue();
			return _properties.getProperty(name);
		} catch (Exception e) {
			e.printStackTrace();
		}
		return null;
	}

	/**
	 * @param state
	 * @param attribute
	 */
	private void removeIncoming(Element state, String id) {
		updateSubvertexIncoming(state, id, "");

	}

	/**
	 * @param state
	 * @param attribute
	 */
	private void removeOutgoing(Element state, String id) {

		updateSubvertexOutgoing(state, id, "");

	}

	/**
	 * 
	 * new
	 * 
	 * @param count
	 * @param state
	 * @param nextState
	 * @param trans
	 * @param newGuardString
	 * @throws Exception
	 * @throws DOMException
	 */
	private Element addNewSubvertex2(int count, Element state1, Element state2,
			Element loopTrans, String newGuardString) throws DOMException,
			Exception {

		// System.err.println("DEBUG " +
		// state1.getAttribute("name") + "--" +
		// state2.getAttribute("name") );

		Element prevTrans = getTrans(state1, state2);

		// System.err.println("DEBUG " +
		// state1.getAttribute("name") + "---" +
		// prevTrans.getAttribute("name") + "-->" +
		// state2.getAttribute("name") + "==" +
		// loopTrans.getAttribute("name"));

		String newTransName = state2.getAttribute("name") + "_"
				+ Integer.valueOf(count).toString() + "_trans";
		String newStateName = state2.getAttribute("name") + "_"
				+ Integer.valueOf(count).toString();

		Element region = (Element) state2.getParentNode();

		/* Update prev. Trans target */
		updateTransTarget(prevTrans, state2.getAttribute("xmi:id"),
				newStateName);

		/* Create New State just before the newTrans */
		Element subvertex = _doc.createElement("subvertex");
		subvertex.setAttribute("xmi:type", state2.getAttribute("xmi:type"));
		subvertex.setAttribute("xmi:id", newStateName);
		subvertex.setAttribute("name", newStateName);
		subvertex.setAttribute("incoming", prevTrans.getAttribute("xmi:id")); // TODO MDT does not support incoming attribute 
		subvertex.setAttribute("outgoing", newTransName); // TODO MDT does not support outgoing attribute 

		/* Create New Trans just before the State2 */
		Element newTrans = _doc.createElement("transition");
		newTrans.setAttribute("xmi:id", newTransName);
		newTrans.setAttribute("name", newTransName);
		newTrans.setAttribute("source", newStateName);// state1.getAttribute("xmi:id"));
		newTrans.setAttribute("target", state2.getAttribute("xmi:id"));
		newTrans.setAttribute("kind", "external");

		if (_xmi == 2.0) {
			Element guard = _doc.createElement("guard");
			guard.setAttribute("xmi:id", newTransName + "_guard");

			Element specification = _doc.createElement("specification");
			specification.setAttribute("xmi:type", "uml:OpaqueExpression");
			specification.setAttribute("xmi:id", newTransName + "_spec");
			specification.setAttribute("body", getGuardString(loopTrans));
			guard.appendChild(specification);
			newTrans.appendChild(guard);
		} else if (_xmi == 2.1) {
			newTrans.setAttribute("guard", newTransName + "_guard");
			Element guard = _doc.createElement("ownedRule");
			guard.setAttribute("xmi:id", newTransName + "_guard");
			guard.setAttribute("name", "");

			Element specification = _doc.createElement("specification");
			specification.setAttribute("name", "");
			specification.setAttribute("xmi:type", "uml:OpaqueExpression");
			specification.setAttribute("xmi:id", newTransName + "_spec");

			Element body = _doc.createElement("body");
			body.appendChild(_doc.createTextNode(newGuardString));

			guard.appendChild(specification);
			newTrans.appendChild(guard);

		} else {
			throw new Exception();
		}

		/* Update State2 input */
		updateSubvertexIncoming(state2, prevTrans.getAttribute("xmi:id"),
				newTransName);

		/* Insert */
		region.insertBefore(subvertex, state2);
		region.insertBefore(newTrans, state2);

		/* set new GuardString to new Trans */
		updateGuardString(newTrans, newGuardString);

		return subvertex;
	}

	/**
	 * @param state
	 * @param nextState
	 * @param nextState2
	 * @param newGuardString
	 * @throws Exception
	 */
	private Element addNewTrans(int count, Element srcState, Element trans,
			Element destState, String newGuardString) throws Exception {

		String newTransID = trans.getAttribute("xmi:id") + "_"
				+ Integer.valueOf(count).toString();
		String newTransName = trans.getAttribute("name") + "_"
				+ Integer.valueOf(count).toString();

		Element region = (Element) trans.getParentNode();

		/* Create New Trans just before the State2 */
		Element newTrans = _doc.createElement("transition");
		newTrans.setAttribute("xmi:id", newTransID);
		newTrans.setAttribute("name", newTransName);
		newTrans.setAttribute("source", srcState.getAttribute("xmi:id"));
		newTrans.setAttribute("target", destState.getAttribute("xmi:id"));
		newTrans.setAttribute("kind", "external");

		if (_xmi == 2.0) {
			Element guard = _doc.createElement("guard");
			guard.setAttribute("xmi:id", newTransName + "_guard");

			Element specification = _doc.createElement("specification");
			specification.setAttribute("xmi:type", "uml:OpaqueExpression");
			specification.setAttribute("xmi:id", newTransName + "_spec");
			specification.setAttribute("body", newGuardString);
			guard.appendChild(specification);
			newTrans.appendChild(guard);

		} else if (_xmi == 2.1) {
			newTrans.setAttribute("guard", newTransName + "_guard");

			Element guard = _doc.createElement("ownedRule");
			guard.setAttribute("xmi:id", newTransName + "_guard");
			guard.setAttribute("name", "");

			Element specification = _doc.createElement("specification");
			specification.setAttribute("xmi:type", "uml:OpaqueExpression");
			specification.setAttribute("xmi:id", newTransName + "_spec");

			Element body = _doc.createElement("body");
			body.appendChild(_doc.createTextNode(newGuardString));
			specification.appendChild(body);
			guard.appendChild(specification);
			newTrans.appendChild(guard);
		} else {
			throw new Exception("wrong XMI version : " + _xmi);
		}

		/* Update Src State input */
		addSubvertexOutgoing(srcState, newTransID);

		/* Update Dest State input */
		addSubvertexIncoming(destState, newTransID);

		/* Insert */
		region.insertBefore(newTrans, trans);

		return newTrans;
	}

	/**
	 * @param destState
	 * @param newTransID
	 */
	private void addSubvertexIncoming(Element state, String newTransID) {
		String ins = newTransID + " " + state.getAttribute("incoming"); // TODO MDT does not support incoming attribute 
		state.setAttribute("incoming", ins);
	}

	/**
	 * @param srcState
	 * @param newTransID
	 */
	private void addSubvertexOutgoing(Element state, String newTransID) {
		String outs = newTransID + " " + state.getAttribute("outgoing"); // TODO MDT does not support outgoing attribute 
		// System.err.println("Outs " + outs);
		state.setAttribute("outgoing", outs);
	}

	/**
	 * @param state
	 * @param id
	 * @param string
	 */
	private void updateSubvertexIncoming(Element state, String transName,
			String newTransName) {
		String outs = state.getAttribute("incoming"); // TODO MDT does not support incoming attribute 

		Pattern pattern = Pattern.compile(transName);
		Matcher matcher = pattern.matcher(outs);

		state.setAttribute("incoming", matcher.replaceFirst(newTransName)); // TODO MDT does not support incoming attribute 
		// TODO Auto-generated method stub

	}

	/**
	 * @param state1
	 * @param attribute
	 * @param newTransName
	 */
	private void updateSubvertexOutgoing(Element state, String transName,
			String newTransName) {
		String outs = state.getAttribute("outgoing"); // TODO MDT does not support outgoing attribute 

		Pattern pattern = Pattern.compile(transName);
		Matcher matcher = pattern.matcher(outs);

		state.setAttribute("outgoing", matcher.replaceFirst(newTransName) // TODO MDT does not support outgoing attribute 
				.trim());

	}

	/**
	 * @param prevTrans
	 * @param attribute
	 * @param newStateName
	 */
	private void updateTransTarget(Element trans, String stateName,
			String newStateName) {
		String target = trans.getAttribute("target");
		Pattern pattern = Pattern.compile(stateName);
		Matcher matcher = pattern.matcher(target);
		trans.setAttribute("target", matcher.replaceFirst(newStateName));
	}

	/**
	 * Update Guard String in transition Element
	 * 
	 * @param trans
	 * @param newGuardString
	 * @throws Exception
	 */
	private void updateGuardString(Element trans, String newGuardString)
			throws Exception {

		if (_xmi == 2.0) {
			NodeList nl = trans.getElementsByTagName("specification");
			for (int j = 0; j < nl.getLength(); j++) {
				Element e = (Element) nl.item(j);
				e.setAttribute("body", newGuardString);
			}
		} else if (_xmi == 2.1) {
			NodeList nl = trans.getElementsByTagName("specification");
			if (nl.getLength() == 1) {
				Element e = (Element) nl.item(0);

				/* append new Body */
				Element body = _doc.createElement("body");
				body.appendChild(_doc.createTextNode(newGuardString));

				/* body is exist? */
				NodeList nl2 = trans.getElementsByTagName("body");
				if (nl2.getLength() == 0) {
					/* no body element, just append */
					e.appendChild(body);
				} else if (nl2.getLength() == 1) {
					/* body exist, replace */
					e.replaceChild(body, nl2.item(0));
				} else {
					throw new Exception("to many body element");
				}

			} else if (nl.getLength() == 0) {
				// no spec?
				throw new Exception("no specification element");
			} else {
				throw new Exception("to many specification element");
			}

		}
	}

	/**
	 * Behavior mode -> Runtime Mode Update digest value of functions in
	 * doActivity
	 * 
	 * verifyProperty(name,value) value hexdigest digest valid
	 * 
	 * @param state
	 * @param event
	 * @throws Exception
	 */
	private void updateDoActivity(Element state, Event event) throws Exception {
		if (log.isDebugEnabled()) {
			log.debug("updateEnter state : " + state.toString());
			log.debug("updateEnter event : " + event.toString());
		}

		NodeList nl = state.getElementsByTagName("doActivity");

		for (int i = 0; i < nl.getLength(); i++) {
			Element e = (Element) nl.item(i);
			String job = e.getAttribute("name");
			if (log.isDebugEnabled())
				log.debug("updateEnter job : " + job);
			String[] je = job.split("[(,)]");
			for (int j = 0; j < je.length; j++) {
				if (log.isDebugEnabled())
					log.debug("sp " + je[j]);
				je[j] = je[j].trim();
			}
			if (je[0].equals("verifyProperty")) {
				if (je[2].equals("hexdigest")) {
					try {
						String hexdigest = _properties.getProperty(je[1]);
						String newJob = "verifyProperty(" + je[1] + ","
								+ hexdigest + ")";
						if (log.isDebugEnabled())
							log.debug("New " + newJob);

						e.setAttribute("name", newJob);
					} catch (Exception e1) {
						log.error("Prop not found " + je[1]);
						e1.printStackTrace();
					}
				} else if (je[2].equals("digest")) {
					try {
						String digest = _properties.getProperty(je[1]);
						String newJob = "verifyProperty(" + je[1] + ","
								+ digest + ")";
						if (log.isDebugEnabled())
							log.debug("New " + newJob);

						e.setAttribute("name", newJob);
					} catch (Exception e1) {
						log.error("Prop not found " + je[1]);
						e1.printStackTrace();
					}
				} else if (je[2].equals("valid")) {
					if (_properties.verifyProperty(je[1], je[2])) {
						log.debug("valid : " + je[1] + " " + je[2]);
					} else {
						throw new Exception("verifyProperty was failed" + je[1]
								+ " = " + _properties.getProperty(je[1])
								+ " must be " + je[2]);

					}
				} else {
					throw new Exception("Unknown verifyProperty " + je[2]);
				}

			}
		}
	}

	/**
	 * Get transition element between state1 and state2
	 * 
	 * @param state1
	 * @param state2
	 * @return
	 */
	private Element getTrans(Element state1, Element state2) {
		// MDT scan all trans.
		for (int i = 0; i < _regions.getLength(); i++) {
			Element region = (Element) _regions.item(i);
			NodeList nl = region.getElementsByTagName("transition");

			for (int j = 0; j < nl.getLength(); j++) {
				Element trans = (Element) nl.item(j);
				if (trans.getAttribute("source").equals(state1.getAttribute("xmi:id"))) {
					if (trans.getAttribute("target").equals(state2.getAttribute("xmi:id"))) {
						return trans;
					}
				}	
			}

		}
		log.error("getTrans : " + state1 + " -> " + state2 + " not found!");
		return null;
	}

	
	/**
	 * @param region
	 * @param nextStateID
	 * @return
	 * @throws Exception
	 */
	private Element getSubvertex(String id) throws Exception {
		for (int i = 0; i < _regions.getLength(); i++) {
			Element region = (Element) _regions.item(i);
			NodeList nl = region.getElementsByTagName("subvertex");

			for (int j = 0; j < nl.getLength(); j++) {
				Element e = (Element) nl.item(j);
				if (e.getAttribute("xmi:id").equals(id)) {
					return e;
				}
			}
		}
		throw new Exception("getSubvertex : " + id + " not found!");
	}

	private Element getSubvertexByName(String name) throws Exception {
		for (int i = 0; i < _regions.getLength(); i++) {
			Element region = (Element) _regions.item(i);
			NodeList nl = region.getElementsByTagName("subvertex");

			for (int j = 0; j < nl.getLength(); j++) {
				Element e = (Element) nl.item(j);
				if (e.getAttribute("name").equals(name)) {
					return e;
				}
			}
		}
		throw new Exception("getSubvertex : " + name + " not found!");
	}

	/**
	 * @param trans
	 * @return Guard String or null if not exist
	 * @throws Exception
	 */
	public String getGuardString(Element trans) throws Exception {

		if (_xmi == 2.0) {
			NodeList nl = trans.getElementsByTagName("specification");
			if (nl.getLength() != 1) {
				log.error("SMDEBUG xmi2.0 multiple body " + nl.getLength());
				return null;
			}
			Element e = (Element) nl.item(0);
			return e.getAttribute("body"); // TODO check if trahs has
		} else if (_xmi == 2.1) {
			NodeList nl = trans.getElementsByTagName("body");
			if (nl.getLength() != 1) {
				log.error("SMDEBUG xmi2.1 multiple body " + nl.getLength());
				return null;
			}
			Element e = (Element) nl.item(0);
			return e.getTextContent();
		}
		/* no gurad String */
		return null;
	}

	/**
	 * @param trans
	 * @throws Exception
	 */
	private void deleteTrans(Element trans) throws Exception {
		/* remove outgoing entry */
		Element state = getSubvertex(trans.getAttribute("source"));
		removeOutgoing(state, trans.getAttribute("xmi:id"));

		/* remove incomming entry */
		removeIncoming(state, trans.getAttribute("xmi:id"));

		/* delete trans */
		Element region = (Element) trans.getParentNode();
		region.removeChild(trans);

		// TODO Auto-generated method stub

	}

	/**
	 * 
	 * @param eventtype
	 * @param eq
	 * @param l
	 * @return
	 * @throws Exception
	 */
	boolean validateLongEq(long a, String eq, long b) throws Exception {
		// 2011-04-19 Munetoh
		// IR contains unsigned int data in int format(b). but Manifest value (a) is long > 0 
		if (b < 0) {
			b = b + 0x100000000L;			
		}
		
		if (eq.trim().equals("==")) {
			if (a == b)
				return true;
		} else if (eq.trim().equals("!=")) {
			// log.debug("SM DEBUG != in validateLongEq
			// #######################");
			if (a != b)
				return true;
		} else {
			throw new Exception("N/A");
		}
		
		// TODO 
		// log.error("validateLongEq fail " + a + " != " + b );
		
		return false;
	}

	/**
	 * 
	 * guardString name0 == value0, name1 == value1, name2 == value2 < < AND
	 * 
	 * 
	 * @param event
	 * @param new
	 *            guardString
	 * @return
	 * @throws Exception
	 */
	private String validateAndUpdateGuard(Event event, String guardString)
			throws Exception {
		boolean rc = true;
		_expandLoop = false; // if digest == base64, insert new state
		_includeManifest = false;
		String newGuardString = "";

		if (guardString == null) {
			return null;
		}
		if (guardString.equals("")) {
			log.error("Wrong Guard String : [" + guardString + "]");
			throw new Exception("Wrong Guard String : [" + guardString + "]");
		}

		/* parse guardString */
		String[] guards = guardString.split(",");
		for (int i = 0; i < guards.length; i++) {

			String[] guards2 = guards[i].trim().split(" ");
			if (guards2.length != 3) {
				// TODO BIOS PCR1 use else but re-think about this
				if (guards[i].equals("else")) {
					// keep this
					return "else";
				}
				log.error("Wrong Guard String : [" + guards[i] + "]");
				log.error("Wrong Guard String : [" + guardString + "]");
				throw new Exception("Wrong Guard String : [" + guards[i]);
			}
			String name = guards2[0];
			String eq = guards2[1];
			String value = guards2[2];
			boolean put = false;

			if (name.equals("eventtype")) {
				long eventtype = Long.decode(value).longValue();// v0.1.3 Lenovo use 0x8000000X, .intValue();
				if (eq.equals("==")) {
					if (eventtype == event.getEventType()) { // HIT
						newGuardString += guards[i].trim();
						put = true;
					} else { // MISS
						rc = false;
						break;
					}
				} else if (eq.equals("!=")) {
					if (eventtype != event.getEventType()) { // HIT
						newGuardString += guards[i].trim();
						put = true;
					} else { // MISS
						rc = false;
						break;
					}
				}

			} else if (name.equals("pcrindex")) {
				long pcrindex = Integer.decode(value).intValue();
				if (eq.equals("==")) {
					if (pcrindex == event.getPcrIndex()) { // HIT
						newGuardString += guards[i].trim();
						put = true;
					} else { // MISS
						rc = false;
						break;
					}
				} else if (eq.equals("!=")) {
					if (pcrindex != event.getPcrIndex()) { // HIT
						newGuardString += guards[i].trim();
						put = true;
					} else { // MISS
						rc = false;
						break;
					}
				} else {
					throw new Exception(
							"Sorry not supported. please update the code for "
									+ eq);
				}
			} else if (name.equals("digest")) { 
				// digest must be valid
				// model
				//   "digest == base64"
				//   "digest == ask"
				// manifest
				//   "digest == vhvewKp0tNyweZQ+cFKAlsyphfg="
				//
				if (eq.equals("==") == false) { // bad expression
					throw new Exception(
							"Sorry not supported. please update the code");
				}
				// System.err.println("digest " + value);
				if (value.equals("base64")) {
					/* Behavior -> Runtime */
					// System.err.println(" guard : " + name + " == "
					// + Base64Tool.encode(event.digest)
					// + " (get from SML)");
					_includeManifest = true;
					_expandLoop = true;
					newGuardString += "digest == "
							+ Base64Tool.encode(event.getDigest());
					put = true;
				} else if (value.equals("ask")) {
					/* Behavior but ASK IIDB */
					if (log.isDebugEnabled())
						log.debug(" guard : " + name + " == ask");
					_includeManifest = true;
					_expandLoop = false;

					newGuardString += "digest == ask";
					put = true;
				} else if (value.equals(Base64Tool.encode(event.getDigest()))) {
					/* Runtime */
					log.debug("DEBUG digest : " + value + " == "
							+ Base64Tool.encode(event.getDigest()));

					newGuardString += guards[i].trim();
					put = true;
				} else { // Miss
					log.error("Miss - name : " + name);
					log.error("DEBUG digest : " + value + " != "
							+ Base64Tool.encode(event.getDigest()));
					rc = false;
					break;
				}
			} else { // NA
				log.error("DEBUG [" + guards[i] + "] [" + name + "][" + value
						+ "]");
				rc = false;
			}
			if (put) {
				newGuardString += ",";
			}
		}

		if (rc == true) {
			/* return new GuardString */
			newGuardString.trim();
			int length = newGuardString.length();
			String last = newGuardString.substring(length - 1, length);
			if (last.equals(",")) {
				newGuardString = newGuardString.substring(0, newGuardString
						.length() - 1);
			}

			return newGuardString;
		} else {
			return null;
		}
	}

	/**
	 * name: validateEvent2 fill the name of event/snapshot object
	 * 
	 * 
	 * @return
	 */
	public String getName() {
		/* IMA Measured File -> use filename as name */
		if (_nameIMA != null)
			return _nameIMA;

		/* else -> use state name */
		return _name;
	}

	/**
	 * @param newFsmFilename
	 */
	public void toFile(String newFsmFilename) {
		// log.debug("DEBUG toFile " + newFsmFilename);
		try {
			TransformerFactory transFactory = TransformerFactory.newInstance();
			Transformer transformer = transFactory.newTransformer();

			transformer.setOutputProperty(
					javax.xml.transform.OutputKeys.ENCODING, "UTF-8");
			transformer.setOutputProperty(
					javax.xml.transform.OutputKeys.INDENT, "yes");
			transformer.setOutputProperty(
					javax.xml.transform.OutputKeys.METHOD, "xml");
			// 
			// transformer.setOutputProperty(org.apache.xml.serializer.OutputPropertiesFactory.S_KEY_INDENT_AMOUNT,
			// "2");
			//transformer.setOutputProperty(
			//		OutputPropertiesFactory.S_KEY_INDENT_AMOUNT, "4");

			// DOMSource source = new DOMSource(_runtime_doc);
			DOMSource source = new DOMSource(_doc);
			File newXML = new File(newFsmFilename);
			FileOutputStream os = new FileOutputStream(newXML);
			StreamResult result = new StreamResult(os);
			transformer.transform(source, result);

			log.info("Write Model to : " + newFsmFilename);

		} catch (TransformerException e) {
			e.printStackTrace();
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}

	}

	public String toString() {
		//int i = 0;
		String buf = "";
		Element current = _start;

		try {
			buf += transToString(current, "", 0);
		} catch (Exception e) {
			e.printStackTrace();
		}

		return buf;
	}

	/**
	 * @param current
	 * @return
	 * @throws Exception
	 */
	private String transToString(Element current, String indent, int count)
			throws Exception {
		if (count > 200)
			return ")))\n";
		if (current == null)
			return indent + "------------------------\n";
		String buf = "";
		Element[] ts = getOutgoingTrans(current);

		for (int i = 0; i < ts.length; i++) {
			// String S1 = current.getAttribute("name");
			String S1 = current.getAttribute("xmi:id");
			String tr = ts[i].getAttribute("xmi:id");
			String S2 = ts[i].getAttribute("target");
			// log.debug(S1 + "--" + tr + "-->" + S2 );
			buf += indent + S1 + "--[" + tr + "]-->" + S2;

			if (S1.equals(S2)) {
				buf += "(loop)\n";
			} else if (S2.equals("Final")) {
				buf += "(FIN)\n";
			} else {
				count++;
				indent += " ";
				buf += "\n";
				buf += transToString(getSubvertex(S2), indent, count);
			}
		}

		return buf;
	}

	/**
	 * Get trans list from state
	 * @param current
	 * @return
	 */
	private Element[] getOutgoingTrans(Element state) {
		// SDAM
		/*
		String outs = state.getAttribute("outgoing"); // TODO MDT does not support outgoing attribute 
		String[] out = outs.trim().split(" ");
		Element[] ts = new Element[out.length];
		for (int i = 0; i < out.length; i++) {
			ts[i] = getTransition(out[i]);
		}
		return ts;
		*/
		// MDT 
		int count =0;
		Element[] tmp = new Element[20]; // TODO MAX 20???
		// scan all trans 
		for (int i = 0; i < _regions.getLength(); i++) {
			Element region = (Element) _regions.item(i);
			NodeList nl = region.getElementsByTagName("transition");

			for (int j = 0; j < nl.getLength(); j++) {
				Element trans = (Element) nl.item(j);
				if (trans.getAttribute("source").equals(state.getAttribute("xmi:id"))) {
					// Hit
					tmp[count]=trans;
					count++;
				}	
			}

		}

		if (count == 0) {			
			return null;
		}
		else {
			// Array
			Element[] ts = new Element[count];
			for (int i = 0; i < count; i++) {
				ts[i] = tmp[i];
			}
			return ts;
		}
	}

	/**
	 * @return
	 */
	public NodeList getSubvertexs() {

		return _root.getElementsByTagName("subvertex");
	}

	/**
	 * @return
	 */
	public NodeList getTransitions() {
		return _root.getElementsByTagName("transition");
	}

	/**
	 * 
	 */
	public void reset() {
		_terminate = false;
		_currentState = _start;
		_properties = new PlatformProperties();
		_properties.init();
	}

	/**
	 * @param props
	 */
	public void reset(PlatformProperties props) {
		_terminate = false;
		_currentState = _start;
		_properties = props;

		try {
			_tpm.init(props);
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}

	}

	public void setTveProperties(PlatformProperties props) {
		_properties = null;
		_properties = props;
	}

	/**
	 * @return
	 */
	public PlatformProperties getTveProperties() {
		_tpm.update(_properties);
		return _properties;
	}

	/**
	 * 
	 * Get Events
	 * 
	 * @return
	 */
	public Event[] getEvents() {
		Event[] e = new Event[_restEvents.size()];
		for (int i = 0; i < _restEvents.size(); i++) {
			e[i] = (Event) _restEvents.get(i);
		}
		return e;
	}

	/**
	 * 
	 */
	// public void verbose() {
	// _verbose = true;
	// }
	/**
	 * @param state
	 * @return
	 */
	private String toString(Element state) {
		return state.getAttribute("name");
	}

	/**
	 * replace gurad by given digest value
	 * 
	 * @param src
	 *            source state name (can be null)
	 * @param dst
	 *            target state name
	 * @param digest
	 *            Reference digest value in Base64
	 * @throws Exception
	 */
	public void setReferenceDigest(String src, String dst, String digest)
			throws Exception {

		setReferenceDigest(src, dst, "digest", digest);
	}

	/**
	 * replace gurad by given digest value
	 * 
	 * @param src
	 *            source state name (can be null)
	 * @param dst
	 *            target state name
	 * @param propName
	 *            property name
	 * @param digest
	 *            Reference digest value in Base64
	 * @throws Exception
	 */
	public void setReferenceDigest(String src, String dst, String propName,
			String digest) throws Exception {
		/* Find State(subvertex) */
		Element srcElement = getSubvertexByName(src);
		Element dstElement = getSubvertexByName(dst);
		Element tr = getTrans(srcElement, dstElement);

		String guardString = getGuardString(tr);

		if (log.isDebugEnabled())
			log.debug("setReferenceDigest " + src + "->" + dst + " "
					+ guardString);

		/* Create new guardString */
		String newGuardString = "";
		String[] guards = guardString.split(",");
		for (int i = 0; i < guards.length; i++) {
			String[] guards2 = guards[i].trim().split(" ");
			if (guards2.length != 3) {
				throw new Exception("Wrong Guard String : [" + guards[i]);
			}
			String name = guards2[0];
			//String eq = guards2[1];
			//String value = guards2[2];
			//boolean put = false;

			if (name.equals(propName)) {
				newGuardString += propName + " == " + digest + ",";
			} else {
				newGuardString += guards[i] + ",";
			}
		}

		// TODO remove last "," -- not work
		newGuardString.trim();
		int length = newGuardString.length();
		String last = newGuardString.substring(length - 1, length);
		if (log.isDebugEnabled()) {
			log.debug("SM DEBUG " + length + " [" + last + "] ["
					+ newGuardString + "] #################################");
		}
		if (last.equals(",")) {
			newGuardString = newGuardString.substring(0, newGuardString
					.length() - 1);
			if (log.isDebugEnabled()) {
				log.debug("SM DEBUG remive , : " + newGuardString);
			}
		}

		updateGuardString(tr, newGuardString);

	}

	/**
	 * 
	 * Replace Assertions
	 * 
	 * before <subvertex name="SetMBRReferenceDigest"...> <doActivity
	 * name="setAssertion(ipl.mbr.digest.masked,digest)" ...>
	 * 
	 * after <subvertex name="SetMBRReferenceDigest"...> <doActivity
	 * name="setAssertion(ipl.mbr.digest.masked,4GeSPk42AWZs4OkR+jkOBpNhaHg=)"
	 * ...>
	 * 
	 * 
	 * @param string
	 * @param stage1Digest
	 * @throws Exception
	 */
	public void setReferenceAssertionDigest(String stateName,
			String stage1Digest) throws Exception {
		/* Lookup stateName */
		Element state = getSubvertexByName(stateName);
		NodeList nl = state.getElementsByTagName("doActivity");
		if (nl.getLength() != 1)
			throw new Exception("State " + stateName
					+ " does not contains doActivity");

		Element doActivity = (Element) nl.item(0);
		String name = doActivity.getAttribute("name");

		String[] func = name.split("[(,)]");
		for (int j = 0; j < func.length; j++)
			func[j] = func[j].trim();

		if (func[0].equals("setAssertion")) {
			String newName = "setAssertion(" + func[1] + "," + stage1Digest
					+ ")";
			doActivity.setAttribute("name", newName);
		} else {
			throw new Exception("State " + stateName
					+ " does not contains setAssertion");
		}
	}

	public void setVerifyPropertyDigest(String stateName, String digest)
			throws Exception {
		/* Lookup stateName */
		Element state = getSubvertexByName(stateName);
		NodeList nl = state.getElementsByTagName("doActivity");
		if (nl.getLength() != 1)
			throw new Exception("State " + stateName
					+ " does not contains doActivity");

		Element doActivity = (Element) nl.item(0);
		String name = doActivity.getAttribute("name");

		String[] func = name.split("[(,)]");
		for (int j = 0; j < func.length; j++)
			func[j] = func[j].trim();

		if (func[0].equals("validateProperty")) {
			String newName = "validateProperty(" + func[1] + "," + digest + ")";
			doActivity.setAttribute("name", newName);
		} else {
			throw new Exception("State " + stateName
					+ " does not contains verifyProperty");
		}
		
		if (log.isDebugEnabled()) {
			log.debug("setVerifyPropertyDigest, state = " + stateName + 
					", digest = " + digest);
		}

	}

	/**
	 * 
	 * Set Runtime Mode. Ignore "digest == base64" guard
	 * 
	 * 
	 */
	public void setRuntimeMode() {
		_runtimeMode = true;
	}

	public double getXMIVersion() {
		return _xmi;
	}

	public Element getRoot() {
		return _root;
	}

	/**
	 * 
	 * @param pcrindex
	 * @param pcr
	 * @return 0 VALID 1 INVALID 2 UNVERIFYED
	 */
	public int validatePcrValue(int pcrindex, byte[] pcrvalue) {
		return _tpm.validate(pcrindex, pcrvalue);
	}

	/**
	 * Display Verification Info
	 * 
	 */
	public void setVerbose() {
		_verbose = true;
	}

	public String getReason() {
		return _reason;
	}

	public void setLevel(int level) {
		_level = level;
	}
}
