/*
 * 
 * Licensed Materials - Property of IBM
 *
 * Open Platform Trust Services - An open source TCG PTS
 *
 * (C) Copyright International Business Machines Corp. 2007
 *
 */
package com.ibm.trl.tcg.pts.engine;

import java.util.Enumeration;
import java.util.Hashtable;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

//import com.ibm.trl.tcg.pts.integrity.aide.AIDE;
//import com.ibm.trl.tcg.pts.integrity.iidb.IIDB;
import com.ibm.trl.tcg.pts.tools.Base64Tool;
import com.ibm.trl.tcg.pts.tools.DigestTool;
import com.ibm.trl.tcg.pts.tools.HexTool;

/**
 * 
 * EventLog by Linux IMA
 * 
 * @author Seiji Munetoh (munetoh@users.sourceforge.jp)
 * 
 */
public class EventLinuxTCB {
	/* Logger */
	private Log log = LogFactory.getLog(this.getClass());


	// defined by Linux Kernel, security/integrity/ima/ima.h
	private static final int IMA_EVENT_NAME_LEN_MAX = 255;

	private int _imaMeasurmentCount;
	private int _summaryImaValidNum;
	private int _summaryImaInvalidNum;
	private int _summaryImaUnverifiedNum;
	private int _imaInvalidMeasurmentCount;
	
	public String message;

	Hashtable<String, String> resultArray;

	boolean ima_ng_invalid_event;


	public EventLinuxTCB() {
		_imaMeasurmentCount = 0;
		_imaInvalidMeasurmentCount = 0;
		_summaryImaValidNum = 0;
		_summaryImaInvalidNum = 0;
		_summaryImaUnverifiedNum = 0;
		resultArray = new Hashtable<String, String>();
	}

	
	/**
	 * 
	 * Validate Linux-IMA Boot Aggrigate
	 * 
	 * http://lwn.net/Articles/139878/ OLD
	 * SHA1(PCR[0]...PCR[7])
	 * 
	 * 2007-11-09 Munetoh implement... :-)
	 * 
	 * @param prop
	 * @param stateName
	 * @param event
	 * @throws Exception
	 */
	public boolean checkImaAggregate(PlatformProperties prop,
			String stateName, Event event) throws Exception {

		boolean rc = true;

		byte[] buf = new byte[8 * 20];
		for (int i = 0; i < 8; i++) {
			String base64 = prop.getProperty("tpm.pcr." + i);
			byte[] b = Base64Tool.decode(base64);

			if (b.length != 20) {
				log.error("Wrong digest" + b.length + " base64 " + base64);
			} else {
				for (int j = 0; j < 20; j++) {
					buf[20 * i + j] = b[j];
				}
			}
		}

		byte[] digest = DigestTool.SHA1(buf);
		byte[] given = event.getDigest();

		/* compare */
		for (int i = 0; i < 20; i++) {
			if (digest[i] != given[i]) {
				rc = false;
			}
		}
		if (rc == false) {
			log.warn("IMA Aggrigate validation was Failed");
			log.warn("  digest(res) :" + HexTool.getHexString(event.getDigest()));
			log.warn("  digest(exp) :" + HexTool.getHexString(digest));
			log.warn("    =SAH1(buf):" + HexTool.getHexString(buf));
			
		}
		return rc;
	}
	
	/**
	 * 
	 * Validate Linux-IMA Boot Aggregate (Kernel 2.6.31)
	 *
	 *  security/integrity/ima/ima_crypto.c ima_calc_boot_aggregate()
	 *  security/integrity/ima/ima_init.c   ima_add_boot_aggregate()
	 *  
	 *
	 * 2010-02-24 new Linux-IMA template
	 * 
	 * @param _properties
	 * @param stateName
	 * @param event
	 * @throws Exception
	 */
	public boolean checkImaAggregateNG(PlatformProperties _properties,
			String stateName, Event event) throws Exception {

		boolean rc = true;

		/* get boot aggregate value from given PCR[0-7]*/
		byte[] buf = new byte[8 * 20];
		for (int i = 0; i < 8; i++) {
			String base64 = _properties.getProperty("tpm.pcr." + i);
			byte[] b = Base64Tool.decode(base64);

			if (b.length != 20) {
				log.error("Wrong digest" + b.length + " base64 " + base64);
			} else {
				for (int j = 0; j < 20; j++) {
					buf[20 * i + j] = b[j];
				}
			}
		}

		byte[] digest = DigestTool.SHA1(buf);
		
		
		/* get template */
		byte[] eventData = event.getEventData();
		byte[] nameArray = new byte[eventData.length - 20];
		byte[] imaDigest = new byte[20];
		for (int i=0;i<20;i++)               imaDigest[i]=eventData[i];
		for (int i=0;i<nameArray.length;i++) nameArray[i]=eventData[20 + i];

		// Check template_digest
		for (int i = 0; i < 20; i++) {
			if (digest[i] != imaDigest[i]) {
				rc = false;
			}
		}
		if (rc == false) {
			log.warn("IMA Aggrigate validation was Failed");
			log.warn("  digest(res) :" + HexTool.getHexString(imaDigest) + " from IML");
			log.warn("  digest(exp) :" + HexTool.getHexString(digest) + " from Calc");
			log.warn("     SAH1(buf):" + HexTool.getHexString(buf) + " Aggregated PCRs");
			log.warn("        event :" + event.toStructString());
			return rc;
		}
		
		/* Padding */ 
		byte[] buf2 = new byte[digest.length + IMA_EVENT_NAME_LEN_MAX + 1];
		for (int i=0;i<buf2.length;i++) buf2[i]= 0;
		for (int i=0;i<digest.length;i++) buf2[i]= digest[i];
		for (int i=0;i<nameArray.length;i++) buf2[digest.length+i]= nameArray[i];		
		byte[] digest2 = DigestTool.SHA1(buf2);
		
		byte[] given = event.getDigest();

		/* compare */
		for (int i = 0; i < 20; i++) {
			if (digest2[i] != given[i]) {
				rc = false;
			}
		}
		if (rc == false) {
			log.warn("IMA Aggrigate validation was Failed");
			log.warn("  digest(res) :" + HexTool.getHexString(event.getDigest()) + " from IML");
			log.warn("  digest(exp) :" + HexTool.getHexString(digest2) + " from Calc");
			log.warn("    =SHA1(    :" + HexTool.getHexString(buf2));
			log.warn("     SAH1(buf):" + HexTool.getHexString(buf));
			log.warn("        event :" + event.toStructString());
			}
		return rc;
	}

	/**
		 * 
		 * Validate IMA Measurement
		 * 
		 * IMA measurement
		 * 
		 * @param prop
		 * @param stateName
		 * @param event
		 * @param iidb
		 * @return
		 * @throws Exception
		 */
		public String validateImaMeasurement(PlatformProperties prop,
				String stateName, Event event) throws Exception {
	
			if (log.isDebugEnabled()) {
				log.debug("setIMAProperty start");
			}
	
			String name = new String(event.getEventData());

			message = "";
	
			_imaMeasurmentCount++;
			return name;
		}


	/**
	 * 
	 * 
	 * @param prop
	 * @param stateName
	 * @param event
	 * @param aide
	 * 
	 * @return
	 * @throws Exception
	 */
	public String validateImaMeasurementNG(PlatformProperties prop,
			String stateName, Event event) throws Exception {

		if (log.isDebugEnabled()) {
			log.debug("setIMAProperty start");
		}

		String pname = "os.measure." + _imaMeasurmentCount;
		
		// IML digest
		byte[] digest = event.getDigest();
		String base64Digest = Base64Tool.encode(digest);
		String hexDigest = HexTool.getHexString(digest);
		
		// IMA template
		byte[] eventData = event.getEventData();
		byte[] nameArray = new byte[eventData.length - 20];
		byte[] imaDigest = new byte[20];
		for (int i=0;i<20;i++) imaDigest[i]=eventData[i];
		for (int i=0;i<nameArray.length;i++) nameArray[i]=eventData[20 + i];
		
		String name = new String(nameArray);
		String base64ImaDigest =  Base64Tool.encode(imaDigest);
		String hexImaDigest = HexTool.getHexString(imaDigest);
		String base64NullDigest = "AAAAAAAAAAAAAAAAAAAAAAAAAAA="; // ZERO

		/* add to prop */
		prop.setProperty(pname + ".name", name);
		prop.setProperty(pname + ".digest", base64Digest);
		prop.setProperty(pname + ".template.digest", base64ImaDigest);
		// for DEBUG
		prop.setProperty(pname + ".digest.hex", hexDigest);
		prop.setProperty(pname + ".template.digest.hex", hexImaDigest);

		message = "";
		
		ima_ng_invalid_event = false;
		if (base64ImaDigest.equals(base64NullDigest)) { /* TODO  INVALID log */
			log.warn( HexTool.getHexString(imaDigest) + " " + name + " - INVALIDATED by IMA");
			// TODO Find previous/original measurement
			prop.setProperty(pname + ".name", name);
			prop.setProperty(pname + ".integrity", "invalid");
			prop.setProperty(pname + ".path", "unknown");
			_summaryImaInvalidNum++;
			_imaInvalidMeasurmentCount++;
			message += " INVALID ";
			ima_ng_invalid_event = true;
		}

		_imaMeasurmentCount++;

		return name;
	}


	/**
		 * TODO Under construction, fix TSS too
		 * IMA-NG
		 * EventData = digest[20] + Name[len]
		 * 
		 * @param prop
		 * @param stateName
		 * @param event
		 * @return
		 * @throws Exception
		 */
		public String validateImaMeasurementNG_IIDB(PlatformProperties prop,
				String stateName, Event event) throws Exception {
	
			if (log.isDebugEnabled()) {
				log.debug("setIMAProperty start");
			}
			
			byte[] eventData = event.getEventData();
			byte[] nameArray = new byte[eventData.length - 20];
			byte[] imaDigest = new byte[20];
			for (int i=0;i<20;i++) imaDigest[i]=eventData[i];
			for (int i=0;i<nameArray.length;i++) nameArray[i]=eventData[20 + i];
			
			String name = new String(nameArray);
	
			/* Digest is not in this Manifest, need to ask IIDB */
			message = "";
			_imaMeasurmentCount++;
	
			return name;
		}


	/**
	 * 
	 * See linuxtcb.statesiagram
	 * 
	 * this function generate following assertions linux.kernel.lsm :
	 * exist/notextst linux.kernel.lsm.ima: enabled/disabled/unknown
	 * linux.kernel.lsm.selinux: enabled/disabled/unknown
	 * linux.kernel.lsm.apparmor: enabled/disabled/unknown
	 * 
	 * 
	 * @param _properties
	 * @param string
	 * @throws Exception
	 */
	public void setKernelCommandline(PlatformProperties prop, String cmdlineProp)
			throws Exception {
		String cmdline = prop.getProperty(cmdlineProp);
		String[] cla = cmdline.split(" ");
		int ima = -1;
		int selinux = -1;
	
		/* set defaults */
		prop.setProperty("linux.kernel.lsm", "unknown");
		prop.setProperty("linux.kernel.lsm.ima", "unknown");
		prop.setProperty("linux.kernel.lsm.selinux", "unknown");
		prop.setProperty("linux.kernel.lsm.apparmor", "unknown");
	
		for (int i = 0; i < cla.length; i++) {
			log.debug("" + cla[i]);
			if (cla[i].equals("ima=1"))
				ima = 1;
			if (cla[i].equals("ima=0"))
				ima = 0;
			if (cla[i].equals("selinux=1"))
				selinux = 1;
			if (cla[i].equals("selinux=0"))
				selinux = 0;
		}
	
		if (ima == 1) {
			prop.setProperty("linux.kernel.lsm", "exist");
			prop.setProperty("linux.kernel.lsm.ima", "enabled");
		} else if (ima == 0) {
			prop.setProperty("linux.kernel.lsm", "exist");
			prop.setProperty("linux.kernel.lsm.ima", "disabled");
		} else if (selinux == 1) {
			prop.setProperty("linux.kernel.lsm", "exist");
			prop.setProperty("linux.kernel.lsm.selinux", "enabled");
		} else if (selinux == 0) {
			prop.setProperty("linux.kernel.lsm", "exist");
			prop.setProperty("linux.kernel.lsm.selinux", "disabled");
		} else {
			prop.setProperty("linux.kernel.lsm", "unknown");
		}
	
	}


	/**
	 * 
	 * @param prop
	 * @return
	 * @throws Exception
	 */
	public String addSummary(PlatformProperties prop) throws Exception {
		String remediation = "";
		int count = 0;

		Enumeration<String> e = resultArray.elements();
		while (e.hasMoreElements()) {
			String str = e.nextElement().toString();
			remediation += str + ",";
			count++;
		}
		
		if (count > 0) {
			remediation += count + " vulnelable packages are found,";
		} else {
			remediation += "";
		}
	
		prop.setProperty("ima.summary.total", "" + _imaMeasurmentCount);
		prop.setProperty("ima.summary.valid", "" + _summaryImaValidNum);
		prop.setProperty("ima.summary.invalid", "" + _summaryImaInvalidNum);
		prop.setProperty("ima.summary.unverified", ""
				+ _summaryImaUnverifiedNum);
	
		return remediation;
	}

}
