/*
 * This file is part of the OpenPTS project.
 *
 * The Initial Developer of the Original Code is International
 * Business Machines Corporation. Portions created by IBM
 * Corporation are Copyright (C) 2010 International Business
 * Machines Corporation. All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the Common Public License as published by
 * IBM Corporation; either version 1 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * Common Public License for more details.
 *
 * You should have received a copy of the Common Public License
 * along with this program; if not, a copy can be viewed at
 * http://www.opensource.org/licenses/cpl1.0.php.
 */

/**
 * \file src/ptscd.c
 * \brief TCG IF-M Collector Daemon
 * @author Seiji Munetoh <munetoh@users.sourceforge.jp>
 * @date 2010-05-12
 * cleanup 2011-01-20,21 SM
 *
 * TODO(munetoh) sepalate collector daemon and colletor
 * TODO(munetoh) multi threads
 * TODO(munetoh) MLS
 *   LC_ALL=C ./src/ptscd -h
 *   LC_MESSAGES="ja_JP.UTF-8" ./src/ptscd -h
 *
 *   LC_ALL=C /usr/sbin/ptscd -h
 *   LC_MESSAGES="ja_JP.UTF-8" /usr/sbin/p/ptscd -h
 * msginit --locale=ja --input=openpts.pot
 # /usr/local/share/locale/ja/LC_MESSAGES/openpts.mo
 */

#if 0
#ifdef ENABLE_NLS
#include <locale.h>
#include <libintl.h>
#define _(x) gettext(x)
#define N_(x) gettext_noop(x)
#define gettext_noop(x) (x)
#else
#define _(x) x
#endif
#endif  // 0

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <netdb.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>  // inet_ntoa
#include <unistd.h>

#include <signal.h>

#include <sys/stat.h>
#include <fcntl.h>

#include <openpts.h>

int verbose = 0; /**< DEBUG */

static struct sockaddr_in server_addr;
static struct sockaddr_in client_addr;
static struct hostent *server_ip;

static int sock;
static int client_sock;
static socklen_t client_size;

static int terminate = 0;

// TODO add lock for HUP, IF-M

OPENPTS_CONFIG *conf = NULL;
int prop_num = 0;
OPENPTS_PROPERTY *start = NULL;
OPENPTS_PROPERTY *end = NULL;


#ifdef ENABLE_NLS
#ifdef HAVE_CATGETS
    /* catgets */
    nl_catd catd;
#endif
#endif

// test on Linux
// 2011-02-24 SM make check => Pass
// #undef HAVE_DAEMON

#ifndef HAVE_DAEMON
// http://linux.die.net/man/3/daemon
int my_daemon(int nochdir, int noclose) {
    int pid;
    int sid;
    int fd;

    DEBUG("my_daemon() - start\n");

    /* fork */
    pid = fork();
    if (pid < 0) {
        ERROR("fork error\n");
        exit(1);
    }
    if (pid > 0) {
        /* parent */
        exit(0);
    }

    DEBUG("my_daemon() - pid = %d\n", pid);

    /* new process group */
    sid = setsid();
    if (sid < 0) {
        ERROR("setsid error\n");
        exit(1);
    }

    DEBUG("my_daemon() - sid = %d\n", sid);

    if (nochdir == 0) {
        /* daemon() changes the current working directory to the root ("/"). */
        chdir("/");
    }

    if (noclose == 0) {
        /* daemon() will redirect standard input, standard output and standard 
           error to /dev/null. */
        // TODO we should not need to close descriptors open by parent
        // for (fd = getdtablesize(); fd >= 0;--fd) close(fd);
        /* stdin */
        fd = open("/dev/null", O_RDWR);
        if (fd < 0) {
            ERROR("open error\n");
            exit(1);
        }
        /* stdout */
        dup(fd);
        /* stderr */
        dup(fd);

        /* 750 */
        umask(027);
    }

    DEBUG("my_daemon() - end\n");

    return 0;
}
#endif  // !HAVE_DAEMON


/**
 * Signal
 */
void sigcatch(int sig) {
#ifdef CONFIG_AUTO_RM_UPDATE
    int rc;

    INFO("ptscd receive signal %d (1:HUP)\n", sig);
    /* update RMs */

    /* reset new RM */
    conf->newrm_uuid = NULL;

    rc = update(conf, prop_num, start, end, 0);
    if (rc != PTS_SUCCESS) {
        ERROR("Update fail\n");
    } else {
        DEBUG("Update OK\n");
    }
    INFO("ptscd update RM (New RM UUID=%s)\n", conf->newrm_uuid->str);
#else
    INFO("no ARU, but ptscd got signal %d\n", sig);
#endif
}


/**
 * collector daemon
 *
 * TODO support single connection.
 * TODO for multiple conenction, multiple ctxs are requires. 
 * TODO disable remote conenction
 */ 
int collector(
            OPENPTS_CONFIG *conf,  // global conf
            int forground,
            int debug,
            const char* dirname) {
    int rc;
    OPENPTS_CONTEXT *ctx = NULL;
    PTS_IF_M_Attribute *read_tlv = NULL;

    /* Init RMs */
    rc = getRmSetDir(conf);
    if (rc != PTS_SUCCESS) {
        ERROR("collector() - getRmSetDir() was failed\n");
        return PTS_INTERNAL_ERROR;
    }

    rc = getNewRmSetDir(conf);
    if (rc != PTS_SUCCESS) {
        /* don't care */
        DEBUG("collector() - getNewRmSetDir() was failed - never mind\n");
    }

    // TODO verify IML before start

    DEBUG("collector, port           : %d\n", conf->port);

    /* Gen. Daemon UUID */
    conf->daemon_uuid = newUuid();
    conf->str_daemon_uuid = getStringOfUuid(conf->daemon_uuid);
    conf->time_daemon_uuid = getDateTimeOfUuid(conf->daemon_uuid);

    /* syslog message */
    INFO("start collector daemon (port=%d, System UUID=%s, RM UUID = %s, Daemon UUID=%s)\n",
        conf->port, conf->uuid->str, conf->rm_uuid->str, conf->str_daemon_uuid);

    /* sock */
    sock = socket(AF_INET, SOCK_STREAM, 0);
    if (sock < 0) {
        perror("sv:socket");
        goto err;
    }

    server_ip = gethostbyname("localhost");
    if (server_ip == NULL) {
        perror("server:gethostbyname");
        goto err;
    }

    memset((char *) &server_addr, 0, sizeof(server_addr));
    server_addr.sin_family = AF_INET;
    server_addr.sin_port = htons(conf->port);

    /* local access only */
    DEBUG("local access only\n");
    server_addr.sin_addr.s_addr = htonl(INADDR_LOOPBACK);

    /* bind */
    rc = bind(sock, (struct sockaddr *) &server_addr, sizeof(server_addr));
    if (rc < 0) {
        perror("server:bind");
        goto err;
    }

    // DEBUG("bind\n");


    rc = listen(sock, 5);
    if (rc == -1) {
        ERROR("server:listen");
        goto err;
    }

    /* daemon */
    if (forground == 0) {
#ifdef HAVE_DAEMON
        DEBUG("Use daemon()\n");
        if (daemon(0, 0) == -1) {
            ERROR("daemon");
            return -1;
        }
#else
        DEBUG("Use my_daemon()\n");
        if (my_daemon(0, 0) == -1) {
            ERROR("daemon");
            return -1;
        }
#endif
    }

    /* forground or daemon below */

    /* Ignore SIGPIPE */
    {
        struct sigaction sa;
        sa.sa_handler = SIG_IGN;
        sa.sa_flags = 0;
        if (sigemptyset(&sa.sa_mask) == -1 || sigaction(SIGPIPE, &sa, 0) == -1) {
          ERROR("failed to ignore SIGPIPE; sigaction");
          exit(EXIT_FAILURE);
        }
    }

    /* Extend OpenPTS start event */
    extendEvCollectorStart(conf);


    /* Collector <-> Verifier - handshake loop */
    do {
        client_sock = accept(sock, (struct sockaddr *) &client_addr,
                                &client_size);

        // TODO check connection to be single
        if (client_sock < 0) {
            if (errno != EINTR)
                perror("accept");
            else
                continue;
        } else {  // GET
            // TODO access log, syslog?
            INFO("open  IF-M PTS connection (address %s, port:%d)\n",
                inet_ntoa(client_addr.sin_addr), client_addr.sin_port);

            ctx = newPtsContext(conf);

            // TODO new ctx for the new connection

            /* handshake loop */
            for (;;) {
                /* V->C request */

                read_tlv = readPtsTlvFromSock(client_sock);  // ifm.c, malloc tlv
                if (read_tlv == NULL) {
                    INFO("close IF-M PTS connection (address %s, port:%d)\n",
                        inet_ntoa(client_addr.sin_addr), client_addr.sin_port);
                    /* free current context */
                    freePtsContext(ctx);
                    break;
                }

                /* check bad TLV */
                if (read_tlv->type == 0)
                    break;

                if (read_tlv->length > 0 && read_tlv->value == NULL)
                    break;

                /* C->V responces */
                switch (read_tlv->type) {
                case OPENPTS_CAPABILITIES:
                    // TODO define CAPABILITIES structure
                    /* check the UUID */
                    if (read_tlv->length != sizeof(OPENPTS_IF_M_Capability)) {  // TODO use defined name
                        ERROR("Bad PTS_CAPABILITIES, len = %d != 32\n", read_tlv->length);
                    } else {
                        OPENPTS_IF_M_Capability *cap;
                        cap = (OPENPTS_IF_M_Capability *) read_tlv->value;
                        /* get version */
                        // TODO
                        /* get verifier's UUID */
                        ctx->uuid = malloc(sizeof(PTS_UUID));
                        memcpy(ctx->uuid, &cap->platform_uuid, 16);
                        ctx->str_uuid = getStringOfUuid(ctx->uuid);

                        /* syslog */
                        INFO("verifier (UUID=%s)\n", ctx->str_uuid);

                        /* send PTS_CAPABILITIES msg. to verifier (=UUID) */
                        rc = writePtsTlvToSock(ctx, client_sock, OPENPTS_CAPABILITIES);
                    }
                    break;

                case DH_NONCE_PARAMETERS_REQUEST:
                    /* check */
                    if (read_tlv->length != 4) {
                        ERROR("Bad DH_NONCE_PARAMETERS_REQUEST, len = %d != 4\n", read_tlv->length);
                    } else {
                        /* req -> res */
                        ctx->nonce->req->reserved      = read_tlv->value[0];
                        ctx->nonce->req->min_nonce_len = read_tlv->value[1];
                        ctx->nonce->req->dh_group_set  = (read_tlv->value[2]<<8) | read_tlv->value[3];

                        rc = getDhResponce(ctx->nonce);

                        /* send responce */
                        rc = writePtsTlvToSock(
                                ctx, client_sock, DH_NONCE_PARAMETORS_RESPONSE);
                    }
                    break;
                case DH_NONCE_FINISH:
                    /* check */
                    if (read_tlv->length != 152) {  // TODO  how to calc this size?
                        ERROR("Bad DH_NONCE_FINISH, len = %d != 152\n", read_tlv->length);
                    } else {
                        /* finish  */
                        ctx->nonce->fin->reserved            = read_tlv->value[0];
                        ctx->nonce->fin->nonce_length        = read_tlv->value[1];
                        ctx->nonce->fin->selected_hash_alg   = (read_tlv->value[2]<<8) | read_tlv->value[3];

                        /* public */
                        ctx->nonce->fin->dh_initiator_public = malloc(ctx->nonce->pubkey_length);
                        memcpy(
                            ctx->nonce->fin->dh_initiator_public,
                            &read_tlv->value[4],
                            ctx->nonce->pubkey_length);

                        /* nonce */
                        ctx->nonce->fin->dh_initiator_nonce = malloc(ctx->nonce->fin->nonce_length);
                        memcpy(
                            ctx->nonce->fin->dh_initiator_nonce,
                            &read_tlv->value[4 + ctx->nonce->pubkey_length],
                            ctx->nonce->fin->nonce_length);

                        rc = calcDhFin(ctx->nonce);

                        /* no responce */
                    }
                    break;
                case REQUEST_RIMM_SET:
                    /* check */
                    if (read_tlv->length != 0) {
                        ERROR("Bad REQUEST__RIMM_SET, len = %d != 0\n", read_tlv->length);
                    } else {
                        rc = writePtsTlvToSock(
                                ctx, client_sock, RIMM_SET);
                    }
                    break;
                case REQUEST_NEW_RIMM_SET:
                    /* check */
                    if (read_tlv->length != 0) {
                        ERROR("Bad REQUEST_NEW_RIMM_SET, len = %d != 0\n", read_tlv->length);
                    } else {
                        rc = writePtsTlvToSock(
                                ctx, client_sock, NEW_RIMM_SET);
                    }
                    break;
                case REQUEST_INTEGRITY_REPORT:
                    /* check */
                    if (read_tlv->length != 0) {
                        ERROR("Bad REQUEST_INTEGRITY_REPORT, len = %d != 0\n", read_tlv->length);
                    } else {
                        rc = writePtsTlvToSock(ctx, client_sock, INTEGRITY_REPORT);
                    }
                    break;
                case VERIFICATION_RESULT:
                    /* no responce */
                    DEBUG_IFM("finish\n");
                    break;
#ifdef CONFIG_AIDE
                case REQUEST_AIDE_DATABASE:
                    /* check */
                    if (read_tlv->length != 0) {
                        ERROR("Bad REQUEST_AIDE_DATABASE, len = %d != 0\n", read_tlv->length);
                    } else {
                        rc = writePtsTlvToSock(ctx, client_sock, AIDE_DATABASE);
                    }
                    break;
#endif
                case REQUEST_TPM_PUBKEY:
                    /* check */
                    if (read_tlv->length != 0) {
                        ERROR("Bad REQUEST_TPM_PUBKEY, len = %d != 0\n", read_tlv->length);
                    } else {
                        rc = writePtsTlvToSock(ctx, client_sock, TPM_PUBKEY);  // ifm.c
                    }
                    break;
                case NONCE:
                    /* check */
                    if (read_tlv->length != 20) {
                        ERROR("Bad NONCE, len = %d != 20\n", read_tlv->length);
                    } else {
                        /* set nonce */
                        ctx->nonce->nonce_length = 20;
                        if (ctx->nonce->nonce != NULL) {
                            free(ctx->nonce->nonce);
                        }
                        ctx->nonce->nonce = malloc(20);
                        memcpy(ctx->nonce->nonce, read_tlv->value, 20);
                        DEBUG_IFM("nonce[%d] : \n", ctx->nonce->nonce_length);
                    }
                    break;

                default:
                    // not supported type
                    ERROR("PTS IF-M type 0x%08x is not supported\n", read_tlv->type);
                    // TODO(munetoh) send error
                    // return (-1);
                    // exit(0);
                    break;
                }  // switch case

                /* free TLV */
                if (read_tlv != NULL) {
                    if (read_tlv->value != NULL) {
                        free(read_tlv->value);
                    }
                    free(read_tlv);
                }
            }  // GET loop
            /* out */
            /* free TLV for break out */
            if (read_tlv != NULL) {
                if (read_tlv->value != NULL) {
                    free(read_tlv->value);
                }
                free(read_tlv);
            }

            // printf("Connection closed. ???\n");
        }  // if
    } while (terminate == 0);  // TODO(munetoh) signal

  err:
    return (-1);
}




/**
 * Usage
 */
void usage(void) {
    fprintf(stderr, NLS(1, 1, "OpenPTS Collector Daemon\n\n"));
    fprintf(stderr, NLS(1, 2, "Usage: ptscd [options] [command]\n\n"));
    fprintf(stderr, NLS(1, 3, "Commands: (forgrand)\n"));
    fprintf(stderr, NLS(1, 4, "  -i                    Initialize PTS collector\n"));
    fprintf(stderr, NLS(1, 5, "  -u                    Update the RM\n"));
#ifdef CONFIG_AUTO_RM_UPDATE
    fprintf(stderr, NLS(1, 6, "  -U                    use HUP signal if the ptscd is running to update the RM\n"));
#endif
    fprintf(stderr, NLS(1, 7, "  -D                    Display the configulation\n"));
    // TODO add selftest
    fprintf(stderr, "\n");
    fprintf(stderr, NLS(1, 8, "Miscellaneous:\n"));
    fprintf(stderr, NLS(1, 9, "  -h                    Show this help message\n"));
    fprintf(stderr, NLS(1, 10, "  -v                    Verbose mode. Multiple -v options increase the verbosity.\n"));
    fprintf(stderr, "\n");
    fprintf(stderr, NLS(1, 11, "Options:\n"));
    fprintf(stderr, NLS(1, 12, "  -p port               Set port number. default is %d\n"), PTSCD_PORT);
    fprintf(stderr, NLS(1, 13, "  -c configfile         Set configulation file. defalt is %s\n"), PTSCD_CONFIG_FILE);
    fprintf(stderr, NLS(1, 14, "  -f                    foreground, run in the foreground."));
    fprintf(stderr, NLS(1, 15, "                        Logging goes to stderr " "instead of syslog.\n"));
    fprintf(stderr, NLS(1, 16, "  -P name=value         Set properties.\n"));
    fprintf(stderr, NLS(1, 17, "  -R                    Remove RMs\n"));
    fprintf(stderr, NLS(1, 18, "  -z                    Use the SRK secret to all zeros (20 bytes of zeros)"));
    // fprintf(stderr, "  -d dirname            Debug\n");

    fprintf(stderr, "\n");
}

#define COMMAND_DAEMON   0
#define COMMAND_INIT     1
#define COMMAND_STATUS   2
#define COMMAND_SELFTEST 3
#define COMMAND_UPDATE   4

#ifdef CONFIG_AUTO_RM_UPDATE
#define COMMAND_AUTO_UPDATE 5
#endif


/**
 * name=value
 */
OPENPTS_PROPERTY *getPropertyFromArg(char *arg) {
    char *name;
    char *value;
    char * eq;
    int len;
    OPENPTS_PROPERTY *prop;

    if ((eq = strstr(arg, "=")) != NULL) {
        /* remove CR */
        len = strlen(arg);
        *eq = 0;
        name = arg;
        value = eq + 1;

        prop = newProperty(name, value);
        return prop;
    } else {
        fprintf(stderr, "bad property %s\n", arg);
        return NULL;
    }
}

/**
 * main of "openpts" command 
 */
int main(int argc, char *argv[]) {
    int forground = 0;
    int debug = 0;
    int command = COMMAND_DAEMON;
    int rc = -1;
    int c;
    int port = 0;  // PTSCD_PORT;
    char *config_filename = NULL;
    int remove = 0;
    // int srk_well_known = 0;

    /* properties by cmdline  */
    OPENPTS_PROPERTY *prop;

#ifdef ENABLE_NLS
#ifdef HAVE_CATGETS
    /* catgets */
    // nl_catd catd;
    catd = catopen("ptscd", 0);
#else
    /* gettext */
    setlocale(LC_ALL, "");
    bindtextdomain(PACKAGE, LOCALEDIR);
    textdomain(PACKAGE);
#endif
#endif


    // DEBUG OFF
    verbose = 0;

    if (SIG_ERR == signal(SIGHUP, sigcatch)) {
        ERROR("failed to set the signal handler");
        return -1;
    }



    conf = newPtsConfig();
    if (conf == NULL) {
        ERROR("internal error\n");  // TODO(munetoh)
        return -1;
    }

    /* command option */
    while ((c = getopt(argc, argv, "ifc:uUDtvp:P:Rzh")) != EOF) {
        switch (c) {
        case 'i':
            command = COMMAND_INIT;
            break;
        case 'u':
            command = COMMAND_UPDATE;
            break;
        case 'U':
#ifdef CONFIG_AUTO_RM_UPDATE
            command = COMMAND_AUTO_UPDATE;
#endif
            break;
        case 'D':
            command = COMMAND_STATUS;
            break;
        case 't':
            command = COMMAND_SELFTEST;
            break;
        case 'f':
            forground = 1;
            break;
        case 'c':
            config_filename = optarg;
            break;
        case 'v':
            debug++;
            break;
        case 'R':
            remove = 1;
            break;
        case 'z':
            conf->srk_password_mode = 1;
            break;
        case 'p':
            port = atoi(optarg);
            break;
        case 'P':
            prop = getPropertyFromArg(optarg);
            if (start == NULL) {
                start = prop;
                end = prop;
                prop->next = NULL;
            } else {
                end->next = prop;
                end = prop;
                prop->next = NULL;
            }
            prop_num++;
            break;
        case 'h':
            /* help */
        default:
            usage();
            return -1;
            break;
        }
    }
    argc -= optind;
    argv += optind;

    /* DEBUG level, 1,2,3 */
    if (debug > 2) {
        verbose = DEBUG_FLAG | DEBUG_FSM_FLAG | DEBUG_IFM_FLAG;
    } else if (debug > 1) {
        verbose = DEBUG_FLAG | DEBUG_IFM_FLAG;
    } else if (debug > 0) {
        verbose = DEBUG_FLAG;
    }

    DEBUG("DEBUG mode\n");

    if ((command == COMMAND_DAEMON) && (forground == 0)) {
        /* daemon, use syslog */
        DEBUG("use syslog \n");
        setenv("PTSCD_DAEMON", "1", 1);
    }

    if (port != 0) {
        /* use given port */
        conf->port = port;
    }

    /* load config */
    if (config_filename == NULL) {
        DEBUG("config file               : %s\n", PTSCD_CONFIG_FILE);
        rc = readPtsConfig(conf, PTSCD_CONFIG_FILE);
        if (rc != PTS_SUCCESS) {
            ERROR("read config file, '%s' was failed - abort\n", PTSCD_CONFIG_FILE);
            goto free;
        }
    } else {
        DEBUG("config file               : %s\n", config_filename);
        rc = readPtsConfig(conf, config_filename);
        if (rc != PTS_SUCCESS) {
            ERROR("read config file, '%s' was failed - abort\n", config_filename);
            goto free;
        }
    }

    /* initialize the  colelctor */
    if (command == COMMAND_INIT) {
        DEBUG("Initialize Reference Manifest\n");
        rc = init(conf, prop_num, start, end);
        goto free;
    }

    /* RM UUID */
    rc = readOpenptsUuidFile(conf->rm_uuid);
    if (rc != PTS_SUCCESS) {
        ERROR("read RM UUID file %s was failed, initialize ptscd first\n", conf->rm_uuid->filename);
        goto free;
    } else {
        DEBUG("conf->str_rm_uuid         : %s\n", conf->rm_uuid->str);
    }

    /* NEWRM UUID */
    rc = readOpenptsUuidFile(conf->newrm_uuid);
    if (rc != PTS_SUCCESS) {
        DEBUG("conf->str_newrm_uuid      : missing (file:%s)\n", conf->newrm_uuid->filename);
        // goto free;
    } else {
        DEBUG("conf->str_newrm_uuid      : %s (for next boot)\n", conf->newrm_uuid->str);
    }

    /* load RSA PUB key */
    // TODO single key => multiple keys?
#ifdef CONFIG_NO_TSS
        TODO("CONFIG_NO_TSS, no TPM_PUBKEY\n");
        conf->pubkey_length = 0;
        conf->pubkey = NULL;
#else
        /* get PUBKEY */
        rc = getTssPubKey(
                conf->uuid->uuid,
                TSS_PS_TYPE_SYSTEM,
                conf->srk_password_mode,
                conf->tpm_resetdalock,
                NULL,
                &conf->pubkey_length,
                &conf->pubkey);
        if (rc != TSS_SUCCESS) {
            ERROR("getTssPubKey() fail rc=0x%x srk password mode=%d\n",
                rc, conf->srk_password_mode);
        }
#endif

    /* run */
    switch (command) {
#ifdef CONFIG_AUTO_RM_UPDATE
        case COMMAND_AUTO_UPDATE:
            /* update by command, but HUP is better */
            DEBUG("Update Reference Manifest\n");
            /* update RMs */
            rc = update(conf, prop_num, start, end, remove);
            if (rc != PTS_SUCCESS) {
                printf("update was fail\n");
            }
            break;
#endif
        case COMMAND_STATUS:
            rc = printCollectorStatus(conf);
            break;
        case COMMAND_SELFTEST:
            rc = selftest(conf, prop_num, start, end);
            if (rc == OPENPTS_SELFTEST_SUCCESS) {
                printf("selftest - OK\n");
            } else if (rc == OPENPTS_SELFTEST_RENEWED) {
                printf("selftest - Renewed\n");
            } else if (rc == OPENPTS_SELFTEST_FALLBACK) {
                printf("selftest -> fallback - TBD\n");
            } else if (rc == OPENPTS_SELFTEST_FAILED) {
                printf("selftest -> fail\n");
            } else {
                printf("TBD\n");
            }
            break;
        case COMMAND_UPDATE:
            /* del RM_UUID */
            conf->rm_uuid->status = OPENPTS_UUID_FILENAME_ONLY;
            if (conf->rm_uuid->uuid != NULL) freeUuid(conf->rm_uuid->uuid);
            if (conf->rm_uuid->str != NULL) free(conf->rm_uuid->str);
            if (conf->rm_uuid->time != NULL) free(conf->rm_uuid->time);
            conf->rm_uuid->uuid = NULL;
            conf->rm_uuid->str = NULL;
            conf->rm_uuid->time = NULL;

            /* gen new RM_UUID and RM */
            rc = newrm(conf, prop_num, start, end);
            if (rc != PTS_SUCCESS) {
                ERROR("newrm() fail\n");
                goto free;
            }

            /* self test */
            rc = selftest(conf, prop_num, start, end);
            if (rc == OPENPTS_SELFTEST_SUCCESS) {
                INFO("manifest generation - success\n");
            } else if (rc == OPENPTS_SELFTEST_RENEWED) {
                TODO("TBD\n");
            } else {
                TODO("TBD\n");
            }
            break;
        case COMMAND_DAEMON:
            /* self check */
            // TODO selftest
            if (conf->selftest == 1) {
            rc = selftest(conf, prop_num, start, end);  // collector.c
                if (rc == OPENPTS_SELFTEST_SUCCESS) {
                    DEBUG("selftest - OK\n");
                    INFO("selftest - success\n");
                } else if (rc == OPENPTS_SELFTEST_RENEWED) {
                    DEBUG("selftest - Renewed\n");
                    INFO("Changed to new manifests (as result of system update)\n");
                } else if (rc == OPENPTS_SELFTEST_FALLBACK) {
                    TODO("selftest -> fallback - TBD\n");
                } else if (rc == OPENPTS_SELFTEST_FAILED) {
                    if (conf->autoupdate == 1) {
                        TODO("selftest -> fail -> gen new RM\n");
                        /* del RM_UUID */
                        conf->rm_uuid->status = OPENPTS_UUID_FILENAME_ONLY;
                        if (conf->rm_uuid->uuid != NULL) freeUuid(conf->rm_uuid->uuid);
                        if (conf->rm_uuid->str != NULL) free(conf->rm_uuid->str);
                        if (conf->rm_uuid->time != NULL) free(conf->rm_uuid->time);
                        conf->rm_uuid->uuid = NULL;
                        conf->rm_uuid->str = NULL;
                        conf->rm_uuid->time = NULL;

                        /* gen new RM_UUID and RM */
                        rc = newrm(conf, prop_num, start, end);
                        if (rc != PTS_SUCCESS) {
                            ERROR("newrm() fail\n");
                            goto free;
                        }
                        rc = selftest(conf, prop_num, start, end);
                        if (rc == OPENPTS_SELFTEST_SUCCESS) {
                            DEBUG("selftest - OK\n");
                            INFO("selftest was faild, new manifests has been generated\n");
                        } else if (rc == OPENPTS_SELFTEST_RENEWED) {
                            DEBUG("selftest - Renewed\n");
                        } else {
                            TODO("TBD\n");
                        }
                    } else {
                        INFO("selftest was faild, but keep existing manifests, run ptscd -i again\n");
                    }
                } else {
                    TODO("TBD\n");
                }
            } else {
                DEBUG("selftest - SKIP\n");
                INFO("selftest - skipped\n");
            }

            /* run colelctor daemon */
            rc = collector(conf, forground, debug, NULL);
            break;
        default:
            ERROR("bad command\n");
            break;
    }

 free:
    freePtsConfig(conf);

    return rc;
}
