/*
 * This file is part of the OpenPTS project.
 *
 * The Initial Developer of the Original Code is International
 * Business Machines Corporation. Portions created by IBM
 * Corporation are Copyright (C) 2010 International Business
 * Machines Corporation. All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the Common Public License as published by
 * IBM Corporation; either version 1 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * Common Public License for more details.
 *
 * You should have received a copy of the Common Public License
 * along with this program; if not, a copy can be viewed at
 * http://www.opensource.org/licenses/cpl1.0.php.
 */

/**
 * \file src/openpts.c
 * \brief main of openpts command
 * @author Seiji Munetoh <munetoh@users.sourceforge.jp>
 * @date 2010-07-25
 * cleanup 2011-01-21 SM
 *
 * This is verifier and utility to maintain the collector/verifier
 *
 *
 * Local host (for TEST)
 *
 * e.g.
 * $ ../src/openpts -i -f -p 5557 localhost
 * target            : localhost
 * configulation     : /home/munetoh/.openpts/07139e72-1ef7-11e0-9833-001f160c9c28/target.conf
 * manifest[0]       : /home/munetoh/.openpts/07139e72-1ef7-11e0-9833-001f160c9c28//074c9722-1ef7-11e0-9833-001f160c9c28/rm0.xml
 * manifest[1]       : /home/munetoh/.openpts/07139e72-1ef7-11e0-9833-001f160c9c28//074c9722-1ef7-11e0-9833-001f160c9c28/rm1.xml
 * validation policy : /home/munetoh/.openpts/07139e72-1ef7-11e0-9833-001f160c9c28/policy.conf
 * 
 * $ ../src/openpts localhost
 * target        : localhost
 * port          : 5557 (localhost)
 * policy file   : /home/munetoh/.openpts/07139e72-1ef7-11e0-9833-001f160c9c28/policy.conf
 * property file : /home/munetoh/.openpts/07139e72-1ef7-11e0-9833-001f160c9c28/vr.property
 * integrity     : valid
 *
 *
 * Remote host w/ SSH port forword
 *
 *  Verifier |            SSH tunneling         |    Collector
 *
 *  openpts-->localhost:5557
 *                       ssh -------> sshd:22
 *                                       sshd-->localhost:5556
 *                                                      =ptscd
 * 
 * e.g.
 * $ ./src/openpts -i -s -S 5567 -l munetoh -p 5557 localhost
 * munetoh@localhost's password: 
 * target            : localhost
 * configulation     : /home/munetoh/.openpts/07139e72-1ef7-11e0-9833-001f160c9c28/target.conf
 * manifest[0]       : /home/munetoh/.openpts/07139e72-1ef7-11e0-9833-001f160c9c28//074c9722-1ef7-11e0-9833-001f160c9c28/rm0.xml
 * manifest[1]       : /home/munetoh/.openpts/07139e72-1ef7-11e0-9833-001f160c9c28//074c9722-1ef7-11e0-9833-001f160c9c28/rm1.xml
 * validation policy : /home/munetoh/.openpts/07139e72-1ef7-11e0-9833-001f160c9c28/policy.conf
 *
 * ./src/openpts -localhost
 * target        : localhost
 * port          : 5557 (remotehost)
 * port(ssh)     : 5567 (localhost)
 * username(ssh) : munetoh
 * policy file   : /home/munetoh/.openpts/07139e72-1ef7-11e0-9833-001f160c9c28/policy.conf
 * property file : /home/munetoh/.openpts/07139e72-1ef7-11e0-9833-001f160c9c28/vr.property
 * integrity     : valid
 *
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <netdb.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <fcntl.h>

#include <unistd.h>
#include <sys/wait.h>
#include <dirent.h>
#include <limits.h>

#define USE_SCANDIR

#include <openpts.h>

int verbose = 0; /**< DEBUG */

#define LINE "--------------------------------------------------------------------"

/**
 * Usage
 */
void usage(void) {
    fprintf(stderr, "OpenPTS command\n\n");
    fprintf(stderr, "Usage: openpts [options] [command] target\n\n");
    fprintf(stderr, "Commands:\n");
    fprintf(stderr, "  -i                    Initialize PTS verifier with target(collector)\n");
    fprintf(stderr, "  -u                    Update PTS verifier with target(collector)\n");
    fprintf(stderr, "  -D                    Display the configulation (ALL)\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "Miscellaneous:\n");
    fprintf(stderr, "  -h                    Show this help message\n");
    fprintf(stderr, "  -v                    Verbose mode. Multiple -v options increase the verbosity.\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "Options:\n");
    fprintf(stderr, "  -p port               Set port number. default is %d\n", PTSCD_PORT);
    fprintf(stderr, "  -c configfile         Set configulation file. defalt is %s\n", PTSCD_CONFIG_FILE);
    fprintf(stderr, "  -f                    Force init, delete existing target(collector) info\n");
    fprintf(stderr, "  -s                    use SSH tunnel (port forward)\n");
    fprintf(stderr, "  -S port               setup SSH tunnel, set local port\n");
    fprintf(stderr, "  -l username           ssh username\n");
    fprintf(stderr, "\n");
}

#define INIT    0
#define VERIFY  1
#define UPDATE  2
#define DISPLAY 3

/**
 * print target info
 */
int printTargetInfo(OPENPTS_CONFIG *conf, char *conf_dir, char *uuid, int cnt) {
    char * target_conf_dir = NULL;
    char * target_conf_filename = NULL;
    int rc;

    /* target dir, conf */
    target_conf_dir = getFullpathName(conf_dir, uuid);
    target_conf_filename = getFullpathName(target_conf_dir, "target.conf");

    /* read target.conf */
    rc = readTargetConf(conf, target_conf_filename);
    if (rc != PTS_SUCCESS) {
        ERROR("printTargetInfo() - readTargetConf failed\n");
        ERROR("  uuid = %s\n", uuid);
        ERROR("  conf_dir = %s\n", conf_dir);
        return PTS_INTERNAL_ERROR;
    }

    printf("%s\n", LINE);
    printf("target[%d] uuid            : %s\n", cnt, uuid);
    printf("target[%d] config          : %s\n", cnt, target_conf_filename);
    printf("target[%d] hostname        : %s\n", cnt, conf->hostname);
    printf("target[%d] port            : %d\n", cnt, conf->port);
    if (conf->ssh_mode == OPENPTS_SSH_MODE_ON) {
        printf("target[%d] SSH             : on\n", cnt);
        printf("target[%d] SSH remote user : %s\n", cnt, conf->ssh_username);
        printf("target[%d] SSH local port  : %d\n", cnt, conf->ssh_port);
    } else {
        printf("target[%d] SSH             : off\n", cnt);
    }

    return PTS_SUCCESS;
}


#ifdef USE_SCANDIR   // scandir or opendir
/**
 * selectDir
 *
 * select target dir
 */
// TODO use selectUuidDir in uuid.c
static int selectDir(const struct dirent *entry) {
    int len;

    /* skip . .. dirs */
    if (0 == strcmp(".", entry->d_name)) return 0;
    if (0 == strcmp("..", entry->d_name)) return 0;

    /* UUID */
    /* skip bad dir name - by length */
    len = strlen(entry->d_name);
    // TODO ("UUID dirname len = %d, %s\n",len, entry->d_name);
    if (len != 36) return 0;

    /* Dir HIT */
    if (entry->d_type == DT_DIR) return 1;

    return 0;
}

#endif

/**
 * main of "openpts" command 
 *
 *
 * IntegrationTest: check_ifm
 *
 */
int main(int argc, char *argv[]) {
    int command = VERIFY;
    int rc = 0;
    int c;
    int port = PTSCD_PORT;

    OPENPTS_CONFIG *conf = NULL;  // conf for openpts
    OPENPTS_CONTEXT *ctx = NULL;
    char * execpath = NULL;
    char * config_filename = NULL;
    char * target_hostname = NULL;
    int initialized = 0;  // 0 -> 1 -> 2
    char * target_conf_dir = NULL;
    char * target_conf_filename = NULL;
    int force = 0;
    int ssh = 0;
    int ssh_config = 0;
    int ssh_port = 0;
    char *ssh_user = NULL;
    int i;
    OPENPTS_TARGET *target_collector = NULL;
    OPENPTS_CONFIG *target_conf = NULL;    // conf for target
    int new_target = 0;  // indicate new target
    int debug = 0;

    verbose = 0;
    execpath = argv[0];

    /* args */
    while ((c = getopt(argc, argv, "ivuDp:c:dfsS:l:h")) != EOF) {
        switch (c) {
        case 'i':
            command = INIT;
            break;
        case 'v':
            debug++;
            break;
        case 'u':
            command = UPDATE;
            break;
        case 'D':
            command = DISPLAY;
            break;
        case 'p':
            port = atoi(optarg);
            break;
        case 'c':
            config_filename = optarg;
            break;
        case 'd':
            verbose = DEBUG_FLAG | DEBUG_FSM_FLAG;
            break;
        case 'f':
            force = 1;
            break;
        case 's':
            ssh = 1;
            ssh_config = 1;
            break;
        case 'S':
            ssh = 1;
            ssh_config = 1;
            ssh_port = atoi(optarg);
            break;
        case 'l':
            ssh_user = optarg;
            break;
        case 'h':
            /* fall through */
        default:
            usage();
            return -1;
            break;
        }
    }
    argc -= optind;
    argv += optind;

    target_hostname = argv[0];

    /* set the DEBUG level, 1,2,3 */
    if (debug > 2) {
        verbose = DEBUG_FLAG | DEBUG_FSM_FLAG | DEBUG_IFM_FLAG;
    } else if (debug > 1) {
        verbose = DEBUG_FLAG | DEBUG_IFM_FLAG;
    } else if (debug > 0) {
        verbose = DEBUG_FLAG;
    }

    /* new config */
    conf = newPtsConfig();
    if (conf == NULL) {
        printf("ERROR\n");  // TODO(munetoh)
        return -1;
    }

    /* check/create config file - HOME./openpts/openpts.conf */
    if (config_filename == NULL) {
        /* use system default config file */
        char dirbuf[BUF_SIZE];
        char confbuf[BUF_SIZE];
        char uuidbuf[BUF_SIZE];

        snprintf(dirbuf, BUF_SIZE, "%s/.openpts", getenv("HOME"));
        snprintf(confbuf, BUF_SIZE, "%s/.openpts/openpts.conf", getenv("HOME"));
        snprintf(uuidbuf, BUF_SIZE, "%s/.openpts/uuid", getenv("HOME"));

        /* check dir */
        if (checkDir(dirbuf) != PTS_SUCCESS) {
            char ans;
            printf("%s is missing. create [Y/n]:", dirbuf);
            rc = scanf("%[YyNn]", &ans);

            /* new UUID */
            conf->uuid = newOpenptsUuid();
            conf->uuid->filename = smalloc(uuidbuf);
            conf->uuid->status = OPENPTS_UUID_FILENAME_ONLY;
            genOpenptsUuid(conf->uuid);

            if ((ans == 'Y') || (ans == 'y')) {
                rc = mkdir(dirbuf, S_IRUSR | S_IWUSR | S_IXUSR);
                rc = writeOpenptsUuidFile(conf->uuid, 1);
                rc = writeOpenptsConf(conf, confbuf);

            } else {
                printf("Bad answer %c, exit\n", ans);
                rc = -1;
                goto error;
            }
        }

        /* check conf  */

        DEBUG("read conf file          : %s\n", confbuf);
        rc = readOpenptsConf(conf, confbuf);

        if (rc != 0) {
            ERROR("readOpenptsConf() failed\n");
            rc = -1;
            goto error;
        } else {
            initialized++;  // 0->1
        }
    } else {
        /* use given config file */
        DEBUG("read conf file       : %s\n", config_filename);
        rc = readOpenptsConf(conf, config_filename);
        if (rc != 0) {
            ERROR("config file [%s] - missing\n", config_filename);
            rc = -1;
            goto error;
        } else {
            initialized++;  // 0->1
        }
    }

    /* check/create target conf dir */
    if (target_hostname != NULL) {
        /* look up the conf of target(hostname) */
        rc = getTargetList(conf, conf->config_dir);
        if (rc != PTS_SUCCESS) {
            TODO("main() - getTargetList rc =%d\n", rc);
        }
        if (verbose & DEBUG_FLAG) {
            DEBUG("target list\n");
            printTargetList(conf, "");  // uuid.c
        }

        /* set the target hostname:port and search */
        if (conf->hostname != NULL) {
            TODO("realloc conf->hostname\n");
            free(conf->hostname);
        }
        conf->hostname = smalloc(target_hostname);
        conf->port = port;
        target_collector =  getTargetCollector(conf);
        if (target_collector == NULL) {
            /* missing, new target => malloc temp target */
#if 0
            target_collector = (OPENPTS_TARGET *) malloc(sizeof(OPENPTS_TARGET));
            memset(target_collector, 0, sizeof(OPENPTS_TARGET));
            target_collector->state = 999;  // TODO temp
            target_collector->target_conf = newPtsConfig();  // TODO free
            new_target = 1;
            target_conf = target_collector->target_conf;
#else
            new_target = 1;
            target_conf = newPtsConfig();
#endif
        } else {
            /* HIT exist */
            target_conf_dir  = getTargetConfDir(conf);  // HOME/.openpts/UUID
            target_conf_filename = smalloc(target_collector->target_conf_filename);
            target_conf = (OPENPTS_CONFIG*)target_collector->target_conf;
        }

    } else {
        if (command == DISPLAY) {
#ifdef USE_SCANDIR  // scandir
            int cnt = 0;
            int dir_num;
            struct dirent **dir_list;
            char * base_conf_dir;

            DEBUG("USE_SCANDIR\n");

            printf("Show openpts config\n");
            printf("%s\n", LINE);
            printf("config file               : %s\n", conf->config_file);
            printf("uuid                      : %s\n", conf->uuid->str);

            /* move to config dir */
            if ((chdir(conf->config_dir)) != 0) {
                fprintf(stderr, "Accessing config directory %s\n", conf->config_dir);
                rc = -1;
                goto error;
            }

            /* scan dirs */
            dir_num = scandir(".", &dir_list, &selectDir, NULL);
            if ( dir_num == -1 ) {
                fprintf(stderr, "no target data\n");
                rc = -1;
                goto error;
            }

            base_conf_dir = smalloc(conf->config_dir);

            // TODO read conf modify the conf structure
            for (cnt = 0; cnt < dir_num; cnt++) {
                DEBUG("Config %d at %s %s\n", cnt, base_conf_dir, dir_list[cnt]->d_name);
                printTargetInfo(conf, base_conf_dir, dir_list[cnt]->d_name, cnt);
                free(dir_list[cnt]);
             }
            free(dir_list);
            free(base_conf_dir);

            printf("%s\n", LINE);

#else  // opndir
            DIR *dir;
            struct dirent *de;
            int cnt = 0;
            char * base_conf_dir;

            ERROR("DDDD\n");

            printf("Show openpts config\n");
            printf("%s\n", LINE);
            printf("config file               : %s\n", conf->config_file);
            // printf("uuid                      : %s\n", conf->str_uuid);

            if ((dir = opendir(conf->config_dir)) == NULL) {
                ERROR("TODO\n");
                rc = -1;
                goto error;
            }

            base_conf_dir = smalloc(conf->config_dir);

            // TODO use readdir_r
            for (de = readdir(dir); de != NULL; de = readdir(dir)) {
                if (de->d_type == DT_DIR) {
                    if (!strncmp(de->d_name, "..\0", 3)) {
                        // skip parents dir
                    } else if (!strncmp(de->d_name, ".\0", 2)) {
                        // skip this dir
                    } else {
                        printTargetInfo(conf, base_conf_dir, de->d_name, cnt);
                        cnt++;
                    }
                }
            }
            closedir(dir);
            free(base_conf_dir);
            printf("%s\n", LINE);
#endif
            rc = 0;
            goto free;
        } else {
            usage();
            rc = -1;
            goto error;
        }
    }
    if (command == DISPLAY) {
        printTargetInfo(conf, conf->config_dir, target_hostname, 0);
        printf("%s\n", LINE);
        rc = 0;
        goto free;
    }

    /* verify -> read target conf */
    if (command == VERIFY) {
        /* check the target dir  */
        if (checkDir(target_conf_dir) == PTS_SUCCESS) {
            initialized++;  // 1->2

            // rc = readTargetConf(conf, target_conf_filename);
            // if ((rc != PTS_SUCCESS) && (force == 0)) {
            //     ERROR("%s dir exist, but target.conf is missing. please initialize again with -f option\n",
            //         target_conf_dir);
            // }
        }
    }

    /* check the port # */
    if (port != target_conf->port) {
        // TODO to be tested
        if (target_conf->port == 0) {
            /* no conf? */
            target_conf->port = port;
        } else {
            DEBUG("port=%d != conf->port=%d (used)\n", port, target_conf->port);
            port = target_conf->port;
        }
    }

    // TODO reset old settings if exist (previous config)
    if (command == INIT) {
        // TODO ("reset conf side\n");
        // target_collector->target_conf->ssh_mode = 0;
        target_conf->ssh_mode = 0;
        if (target_conf->hostname != NULL) {
            // DEBUG("realloc target_conf->hostname\n"); TODO realloc happen
            free(target_conf->hostname);
        }
        target_conf->hostname = smalloc(target_hostname);
        target_conf->port = port;
    }

    /* SSH */
    if ((ssh == 1) || (target_conf->ssh_mode == OPENPTS_SSH_MODE_ON)) {
        pid_t pid;
        int status;
        char forward[BUF_SIZE];
        char ssh_target[BUF_SIZE];

        /* SSH mode */
        target_conf->ssh_mode = OPENPTS_SSH_MODE_ON;  // enable SSH

        /* SSH username */
        if (ssh_user != NULL) {
            /* given by cmd option */
            if (target_conf->ssh_username != NULL) {
                if (strcmp(target_conf->ssh_username, ssh_user)) {
                    /* username was changed? */
                    TODO("name change -> update SSH forword\n");
                }
            } else {
                /* no username in conf, use given username */
                target_conf->ssh_username = smalloc(ssh_user);
            }
        }

        /* SSH local port */
        if (ssh_port != 0) {
            /* given by cmd option */
            if (target_conf->ssh_port != 0) {
                if (ssh_port != target_conf->ssh_port) {
                    printf("SSH localport was changed from %d to %d\n", conf->ssh_port, ssh_port);
                    target_conf->ssh_port = ssh_port;
                }
            } else {
                /* no username in conf, use given username */
                target_conf->ssh_port = ssh_port;
            }
        }

        /* setup local port forward */
        // ssh -2 -N -f -L 5557:localhost:5556 foo@localhost

        snprintf(forward, BUF_SIZE, "%d:localhost:%d", target_conf->ssh_port, port);
        snprintf(ssh_target, BUF_SIZE, "%s@%s", target_conf->ssh_username, target_hostname);
        /* config namespace : target.ssh.* */

        /* ask user name if not defined yet, save to target.conf,  target.ssh.username */

        /* define local port, target.ssh.local.port */
        DEBUG("target      : %s\n", target_hostname);
        DEBUG("ssh forward : %s\n", forward);
        DEBUG("ssh target  : %s\n", ssh_target);

        if (ssh_config == 1) {
            /* ssh -L */
            pid = fork();
            if (pid < 0) {
                ERROR("\n");
                rc = -1;
                goto error;
            }

            if (pid == 0) {
                /* child */
                DEBUG("exec      : /usr/bin/ssh -2 -N -f -L %s %s\n", forward, ssh_target);
                execl("/usr/bin/ssh", "ssh", "-2", "-N", "-f", "-L", forward, ssh_target,  NULL);
                /* execl failed */
                exit(-1);
            } else {
                /* parent */
                waitpid(pid, &status, 0);
                // TODO ("status = %d\n", status);
                if (WIFEXITED(status)) {
                    /* 1 : OK */
                    // TODO ("Exit status %d\n", WEXITSTATUS(status));
                } else if (WIFSIGNALED(status)) {
                    ERROR("Signal status %d\n", WIFSIGNALED(status));
                    rc = -1;
                    goto error;
                } else {
                    ERROR("Bad exit");
                    rc = -1;
                    goto error;
                }
            }
        }

        DEBUG("SSH tunnel mode, port=%d\n", target_conf->ssh_port);

        /* OK, connect to the local port */
        target_hostname = "localhost";
        port = target_conf->ssh_port;
    }

    /* new context */
    ctx = newPtsContext(conf);
    if (ctx == NULL) {
        printf("ERROR\n");  // TODO(munetoh)
        rc = -1;
        goto error;
    }
    ctx->target_conf = target_conf;

    /* command */
    if (command == INIT) {
        /* get UUID, RMs, AIDE DB */
        DEBUG("enroll with %s  (SSH uses localost)\n", target_hostname);
        DEBUG("conf->config_dir %s\n", conf->config_dir);
        rc =  enroll(ctx, target_hostname, port, conf->config_dir, force);  // verifier.c
        if (rc != 0) {
            fprintf(stderr, "enroll was failed, rc = %d\n", rc);
            printReason(ctx);
            goto error;
        }

        /* gen  initial policy  and AIDE-ignorelist */
        // DEBUG("Generate initial validation policy for %s\n", target);
        // rc = readTargetConf(ctx->conf, ctx->target_conf_filename);  // TODO BAD use new conf for the target
        // if (rc != PTS_SUCCESS) {
        //     ERROR(" readTargetConf failed\n");
        //     // TODO so?
        // }

        DEBUG("conf->config_dir %s\n", conf->config_dir);
        rc =  verifier(ctx, target_hostname, port, conf->config_dir, 1);  // init
        if (rc != OPENPTS_RESULT_VALID) {
            fprintf(stderr, "initial verification was failed, rc = %d\n", rc);
            printReason(ctx);
            goto error;
        }

        /* message */
        printf("Target            : %s\n", target_hostname);
        printf("Collector UUID    : %s\n", ctx->target_conf->uuid->str);

        if (ctx->target_conf != NULL) {
            if (ctx->target_conf->rm_uuid != NULL) {
                printf("Manifest UUID     : %s\n", ctx->target_conf->rm_uuid->str);
                for (i = 0; i< ctx->conf->rm_num; i ++) {
                    printf("manifest[%d]       : %s\n", i, ctx->target_conf->rm_filename[i]);
                }
            }
            printf("configulation     : %s\n", ctx->target_conf->config_file);
            printf("validation policy : %s\n", ctx->target_conf->policy_filename);
        } else {
            // TODO never happen?
            printf("configulation     : new target\n");
        }




        // printf("validation policy : %s\n", ctx->conf->policy_filename);
        // printf("aide database     : %s\n", ctx->conf->aide_database_filename);
        // printf("aide ignore list  : %s\n", ctx->conf->aide_ignorelist_filename);
    } else if (command == VERIFY) {
        /* verify */
        if (initialized < 2) {
            fprintf(stderr, "ERROR: target %s is not initialized yet. please enroll with %s first\n\n",
                target_hostname, target_hostname);
            usage();
            rc = -1;
            goto error;
        }

        // TODO(munetoh) control by policy? or conf?
        ctx->conf->ima_validation_unknown = 1;

        /* vefify*/
        rc =  verifier(ctx, target_hostname, port, conf->config_dir, 0);  // normal

        /* messages */
        // printf("target        : %s\n", argv[0]);
        printf("Target            : %s\n", argv[0]);
        if (target_conf != NULL) {
            printf("Collector UUID    : %s ", target_conf->uuid->str);
            // TODO set this when load the uuid
            if (target_conf->uuid->time == NULL) {
                target_conf->uuid->time = getDateTimeOfUuid(target_conf->uuid->uuid);
            }
            printf("(date: %04d-%02d-%02d-%02d:%02d:%02d)\n",
                target_conf->uuid->time->year + 1900,
                target_conf->uuid->time->mon + 1,
                target_conf->uuid->time->mday,
                target_conf->uuid->time->hour,
                target_conf->uuid->time->min,
                target_conf->uuid->time->sec);
            printf("Manifest UUID     : %s ", target_conf->rm_uuid->str);
            printf("(date: %04d-%02d-%02d-%02d:%02d:%02d)\n",
                target_conf->rm_uuid->time->year + 1900,
                target_conf->rm_uuid->time->mon + 1,
                target_conf->rm_uuid->time->mday,
                target_conf->rm_uuid->time->hour,
                target_conf->rm_uuid->time->min,
                target_conf->rm_uuid->time->sec);

            if (target_conf->ssh_mode == 1) {
                printf("port              : %d (remotehost)\n", target_conf->port);
                printf("port(ssh)         : %d (localhost)\n", target_conf->ssh_port);
                printf("username(ssh)     : %s\n", target_conf->ssh_username);
            } else {
                printf("port              : %d (localhost)\n", target_conf->port);
            }
            printf("policy file       : %s\n", target_conf->policy_filename);
            printf("property file     : %s\n", target_conf->prop_filename);  // TODO ptoperty or prop
        } else {
            // ERROR("\n");
        }

        if (rc == OPENPTS_RESULT_VALID) {
            printf("integrity         : valid\n");
        } else if (rc == OPENPTS_RESULT_INVALID) {
            printf("integrity         : invalid\n");
            printReason(ctx);
        } else if (rc == OPENPTS_RESULT_UNKNOWN) {
            printf("integrity         : unknown\n");
            printReason(ctx);
        } else if (rc == PTS_VERIFY_FAILED) {
            printf("integrity         : invalid ()\n");
            printReason(ctx);
        } else {
            printf("integrity         : unknown (INTERNAL ERROR) rc=%d\n", rc);
            printReason(ctx);
        }


#ifdef CONFIG_AUTO_RM_UPDATE
        if (conf->newrm_exist > 0) {
            char uuid_str[40]; // 37?
            PTS_DateTime *uuid_time;
            char ans;

            if (verbose) {
                DEBUG("NEWRM_UUID\n");
                printHex("NEWRM UUID (remote)  : ", (BYTE*)conf->aru_newrm_uuid, 16, "\n");
                if (target_conf->newrm_uuid->uuid != NULL) {
                    printHex("NEWRM UUID (local)   : ", (BYTE*)target_conf->newrm_uuid->uuid, 16, "\n");
                } else {
                    printf("NEWRM UUID (local)   : missing\n");
                }
            }
            /* Check local newrm */
            if (target_conf->newrm_uuid->uuid != NULL) {
                if (memcmp(
                        (BYTE*)conf->aru_newrm_uuid,
                        (BYTE*)target_conf->newrm_uuid->uuid, 16) == 0) {
                    /* HIT */
                    printf("---------------------------------------------------------\n");
                    printf("New Manifest UUID : %s ", target_conf->newrm_uuid->str);
                    printf("(date: %04d-%02d-%02d-%02d:%02d:%02d)\n",
                        target_conf->newrm_uuid->time->year + 1900,
                        target_conf->newrm_uuid->time->mon + 1,
                        target_conf->newrm_uuid->time->mday,
                        target_conf->newrm_uuid->time->hour,
                        target_conf->newrm_uuid->time->min,
                        target_conf->newrm_uuid->time->sec);
                    goto free;
                } else {
                    /* local is old? */
                    printf("New reference manifest has been received, but update exist\n");
                }
            }

            /* msg */
            printf("---------------------------------------------------------\n");
            encodeBase64((unsigned char *)uuid_str, (unsigned char *)conf->aru_newrm_uuid, 16);
            uuid_time = getDateTimeOfUuid(conf->aru_newrm_uuid);
            printf("New Manifest UUID : %s ", uuid_str);
            printf("(date: %04d-%02d-%02d-%02d:%02d:%02d)\n",
                uuid_time->year + 1900,
                uuid_time->mon + 1,
                uuid_time->mday,
                uuid_time->hour,
                uuid_time->min,
                uuid_time->sec);

            printf("New reference manifest exist. update? [Y/n]\n");
            rc = scanf("%[YyNn]", &ans);

            DEBUG("conf->config_dir %s\n", conf->config_dir);

            if ((ans == 'Y') || (ans == 'y')) {
                rc = updateNewRm(ctx, target_hostname, port, conf->config_dir);  // aru.c
                if (rc == PTS_SUCCESS) {
                    printf("Save new reference manifest\n");
                    // TODO UUID
                } else {
                }
            } else if ((ans == 'N') || (ans == 'n')) {
                printf("keep current manifest\n");
            } else {
                printf("Bad answer %c, exit\n", ans);
                rc = -1;
                goto error;
            }

            // TODO validate new RM
            // TODO e.g. gen new RM by verifier and compare both
        } else if (rc == PTS_RULE_NOT_FOUND) {
            // char ans;
            printf("New reference manifest exist. if this is expected change, update the manifest by openpts -i -f \n");
        } else {
            DEBUG("no newrm\n");
        }
#else
        if (rc == PTS_RULE_NOT_FOUND) {
            // char ans;
            printf("New reference manifest exist. if this is expected change, update the manifest by openpts -i -f \n");
        }
#endif
    } else if (command == UPDATE) {
       TODO("TBD\n");
    } else {
        usage();
    }

    rc = 0;

 error:
 free:
    if (target_conf_dir != NULL) free(target_conf_dir);
    if (target_conf_filename != NULL) free(target_conf_filename);

    // TODO just use the target?
    // if ((new_target == 1) && (target_collector  != NULL)) {
    //    if (target_collector->str_uuid != NULL) free(target_collector->str_uuid);
    //    if (target_collector->uuid != NULL) free(target_collector->uuid);
    //    if (target_collector->time != NULL) free(target_collector->time);
    //    if (target_collector->dir != NULL) free(target_collector->dir);
    //    if (target_collector->target_conf_filename != NULL) free(target_collector->target_conf_filename);
    //    if (target_collector->target_conf != NULL) freePtsConfig((OPENPTS_CONFIG *)target_collector->target_conf);
    // }

    // if (ctx->target_conf != NULL) {
    //    freePtsConfig(ctx->target_conf);
    //    ctx->target_conf = NULL;
    // }
    // if (ctx->target_conf->config_file != NULL) {
    //    // TODO somehow valgrind report the leak
    //    // TODO("ctx->target_conf->config_file != NULL\n");
    //    free(ctx->target_conf->config_file);
    //    ctx->target_conf->config_file = NULL;
    // }

    if ((new_target == 1) && (target_conf != NULL)) {
        /* free new target conf */
        freePtsConfig(target_conf);
    }

    if (ctx != NULL) {
        ctx->target_conf = NULL;
        freePtsContext(ctx);
    }

    freePtsConfig(conf);

    return rc;
}
