/*
 * 
 * Licensed Materials - Property of IBM
 *
 * Open Platform Trust Services - An open source TCG PTS
 *
 * (C) Copyright International Business Machines Corp. 2007
 *
 */
package com.ibm.trl.tcg.pts.vulnerability.tool;

import java.util.regex.Pattern;

/**
 * RPM version compare.
 * 
 * @author Seiji Munetoh, Megumi Nakamura
 */
public class PackageVersionTool {

	public static final int EQUAL = 0;

	public static final int GREATER = 1;

	public static final int LESS = -1;

	public static final int CHECK_AGAIN = 100;

	protected static final int DEFAULT_VALUE = 10;

	/**
	 * Compare two version numbers, A(vera), B(verb). 1 means A is larget than
	 * B. -1 means B is larger than A. 0 means A is the same as B.
	 * 
	 * @param vera
	 *            The version number and the release number
	 * @param verb
	 *            The version number and the release number
	 * @return 1(A&gt;B), -1(A&lt;B), 0(A==B)
	 */
	public static int compareTwoVersion(String vera, String verb) {
		if (vera.equals(verb)) {
			return EQUAL;
		}

		// parse versions
		String[] tmp = getVersionRelease(vera);
		String ver1 = tmp[0];
		String rel1 = tmp[1];
		tmp = getVersionRelease(verb);
		String ver2 = tmp[0];
		String rel2 = tmp[1];

		// check
		int rtn = compareFigureStrings(patternCheck("[-._\\p{Punct}]+", ver1),
				patternCheck("[-._]+", ver2));
		if (rtn == EQUAL) {
			rtn = compareFigureStrings(patternCheck("[-._\\p{Punct}]+", rel1),
					patternCheck("[-._]+", rel2));
		}
		return rtn;
	}

	/**
	 * Get the version number and release number.
	 * 
	 * @param verrel
	 *            version-release
	 * @return String[0]=Version, String[1]=Release
	 */
	protected static String[] getVersionRelease(String verrel) {
		String versionRelease = verrel;
		if (verrel.indexOf(":") > 0) {
			String[] removeColon = patternCheck(":", verrel);
			if (removeColon.length >= 2) {
				versionRelease = removeColon[1];
			} else if (removeColon.length == 1) {
				versionRelease = removeColon[0];
			}
		}

		String[] rtn = new String[2];
		String[] parts = patternCheck("-", versionRelease);

		/* Get Version */
		if (parts.length > 0) {
			rtn[0] = parts[0];
		} else {
			rtn[0] = null;
		}

		/* Get Release */
		rtn[1] = null;
		if (parts.length > 1) {
			rtn[1] = versionRelease.substring(rtn[0].length() + 1);
		}
		return rtn;
	}

	/**
	 * compare each character in two strings as version numbers. 1 means
	 * strings1 is larget than strings2. -1 means strings2 is larger than
	 * strings1. 0 means strings1 is the same as strings2.
	 * 
	 * @param strings1
	 *            The array of version number split by the dot
	 * @param strings2
	 *            The array of version number split by the dot
	 * @return 1(strings1&gt;strings2), -1(strings1&lt;strings2),
	 *         0(strings1==strings2)
	 */
	protected static int compareFigureStrings(String[] strings1,
			String[] strings2) {
		Pattern patternAlpha = Pattern.compile("[a-zA-Z:\\p{Punct}]+");
		
		int len1 = strings1.length;
		int len2 = strings2.length;
		int maxlen;
		if (len1 > len2) {
			maxlen = len1;
		} else {
			maxlen = len2;
		}

		int i1, i2;
		String[] tmp1a = null;
		String[] tmp1b = null;
		String[] tmp2a = null;
		String[] tmp2b = null;
		for (int count = 0; count < maxlen; count++) {
			tmp1a = null;
			tmp1b = null;
			tmp2a = null;
			tmp2b = null;
			if ((count >= len1) && (len2 > len1)) {
				i1 = -1;
			} else {
				if (strings1[count] != null) {
					strings1[count] = removeDistribution(strings1[count]);
					tmp1b = getFigures(strings1[count]);
					if (patternAlpha.matcher(strings1[count]).find()) { // contains
																		// distribution-name
						strings1[count] = removeDistribution(strings1[count]);

						if (patternAlpha.matcher(strings1[count]).find()) { // contains
																			// a,b,c,...
							tmp1a = getAlphabets(strings1[count]);
							i1 = -1;
						} else {
							i1 = Integer.parseInt(patternAlpha
									.split(strings1[count])[0]);
						}
					} else {
						i1 = Integer.parseInt(patternAlpha
								.split(strings1[count])[0]);
					}
				} else {
					i1 = -1;
				}
			}

			if ((count >= len2) && (len1 > len2)) {
				i2 = -1;
			} else {
				if (strings2[count] != null) {
					strings2[count] = removeDistribution(strings2[count]);
					tmp2b = getFigures(strings2[count]);
					if (patternAlpha.matcher(strings2[count]).find()) {
						strings2[count] = removeDistribution(strings2[count]);
						if (patternAlpha.matcher(strings2[count]).find()) {
							tmp2a = getAlphabets(strings2[count]);
							i2 = -1;
						} else {
							i2 = Integer.parseInt(patternAlpha
									.split(strings2[count])[0]);
						}

					} else {
						i2 = Integer.parseInt(patternAlpha
								.split(strings2[count])[0]);
					}
				} else {
					i2 = -1;
				}
			}

			// compare - alphabet & figure
			int rtn = DEFAULT_VALUE;
			int rtn2 = DEFAULT_VALUE;
			if (tmp1a != null && tmp2a != null) { // alphabet
				rtn = checkAlphabets(tmp1a, tmp2a);
				if (rtn == CHECK_AGAIN) {
					return CHECK_AGAIN;
				}
			}
			if (tmp1b != null || tmp2b != null) { // figure
				rtn2 = checkFigures(tmp1b, tmp2b);
				if (rtn2 == CHECK_AGAIN) {
					return CHECK_AGAIN;
				}
			}

			if (rtn == EQUAL) {
				if (rtn2 == LESS) {
					return LESS;
				} else if (rtn2 == GREATER) {
					return GREATER;
				}
			} else if (rtn == DEFAULT_VALUE && rtn2 != DEFAULT_VALUE
					&& (i1 == -1 || i2 == -1)) {
				// compare figure (not by alphabet)
				return rtn2;

			} else if (rtn2 == EQUAL) {
				if (rtn == LESS) {
					return LESS;
				} else if (rtn == GREATER) {
					return GREATER;
				}
			} else if (rtn != DEFAULT_VALUE && rtn2 != DEFAULT_VALUE
					&& (i1 != -1 || i2 != -1)) {
				return CHECK_AGAIN;
			} else {
				// compare - figure
				if (i2 > i1) {
					return LESS;
				} else if (i2 < i1) {
					return GREATER;
				} else if (i1 == -1 && i2 == -1) {
					if (rtn == rtn2) {
						return rtn; // same as rtn2
					} else {
						return rtn2;
					}
				} else {
					return CHECK_AGAIN;
				}
			}

		}
		return EQUAL;
	}

	/**
	 * Compare two string[] of figure.
	 * 
	 * @param var1
	 *            The string array of figure
	 * @param var2
	 *            The string array of figure
	 * @return RPMTool.CHECH_AGAIN, RPMTool.LESS(var1&lt;var2),
	 *         RPMTool.GREATER(var1&gt;var2), RPMTool.EQUAL
	 */
	private static int checkFigures(String[] var1, String[] var2) {
		int minlen;
		if (var1 == null && var2 != null) {
			return LESS;
		} else if (var1 != null && var2 == null) {
			return GREATER;
		} else if (var1 == null && var2 == null) {
			return CHECK_AGAIN;
		} else {
			if (var1.length > var2.length) {
				minlen = var2.length;
			} else {
				minlen = var1.length;
			}
		}
		for (int a = 0; a < minlen; a++) {
			if (!var1[a].equalsIgnoreCase(var2[a])) {
				int i1 = Integer.parseInt(var1[a]);
				int i2 = Integer.parseInt(var2[a]);
				if (i1 < i2) {
					return LESS;
				} else if (i1 > i2) {
					return GREATER;
				}
			}
		}
		return EQUAL;
	}

	/**
	 * Compare two string[] of alphabet.
	 * 
	 * @param var1
	 *            The string array of alphabet
	 * @param var2
	 *            The string array of alphabet
	 * @return RPMTool.CHECH_AGAIN, RPMTool.LESS(var1&lt;var2),
	 *         RPMTool.GREATER(var1&gt;var2), RPMTool.EQUAL
	 */
	private static int checkAlphabets(String[] var1, String[] var2) {
		if (var1 == null || var2 == null) {
			return CHECK_AGAIN;
		}
		int minlen;
		if (var1.length > var2.length) {
			minlen = var2.length;
		} else {
			minlen = var1.length;
		}

		// compare
		for (int a = 0; a < minlen; a++) {
			if (!var1[a].equalsIgnoreCase(var2[a])) {
				char[] ch1 = var1[a].toCharArray();
				char[] ch2 = var2[a].toCharArray();
				if (ch1.length != ch2.length) {
					return CHECK_AGAIN;
				} else {
					for (int b = 0; b < ch1.length; b++) {
						if (ch1[b] < ch2[b]) {
							return LESS;
						} else if (ch1[b] > ch2[b]) {
							return GREATER;
						}
					}
				}
			}
		}
		return EQUAL;
	}

	/**
	 * Split the string by pattern. ex. patternCheck(".-", "x.y.z-wv") ==>
	 * String[]{x,y,z,wv}
	 * 
	 * @param pattern
	 *            The string of Regular expression
	 * @param target
	 *            The string to be split
	 * @return The split string array
	 */
	protected static String[] patternCheck(String pattern, String target) {
		Pattern ptn = Pattern.compile(pattern);
		if (target == null) {
			String[] tmp = new String[1];
			tmp[0] = null;
			return tmp;
		} else {
			return ptn.split(target);
		}
	}

	/**
	 * Split the version number into the array of figures. for example.
	 * 0:3.9p1-8.RHEL4.17 -> 0, 3, 9, 1, 8, 4, 17
	 * 
	 * @param ver
	 *            The version number
	 * @return The array of the version number split by the dot and the alphabet
	 */
	private static String[] getFigures(String ver) {
		Pattern pattern = Pattern.compile("[\\p{Punct}:_.a-zA-Z\\-]+");
		String figure = pattern.matcher(ver).replaceAll(" ").trim();
		if (figure != null) {
			if (figure.length() == 0) {
				return null;
			}
			return patternCheck("[ ]+", figure);
		} else {
			return null;
		}
	}

	/**
	 * Split the version number into the array of alphabet characters. for
	 * example 0:3.9p1-8.RHEL4.17 -> p, RHEL.
	 * 
	 * @param ver
	 *            The version number
	 * @return The array of the version number split by the dot and the figures
	 */
	private static String[] getAlphabets(String ver) {
		Pattern pattern = Pattern.compile("[:_.0-9\\-]+");
		String alphabet = pattern.matcher(ver).replaceAll(" ").trim();
		return patternCheck("[ ]+", alphabet); // alphabet.split(" ");
	}

	/**
	 * Get the distribution name from the version number.
	 * 
	 * @param ver
	 *            The version number
	 * @return The distribution name
	 */
	public static String checkDistribution(String ver) {
		Pattern pattern = Pattern.compile("[:_.0-9\\-]+");
		String alphabet = pattern.matcher(ver).replaceAll(" ").trim();
		String[] packAlphabet = alphabet.split(" ");
		String dist = null;
		for (int i = 0; i < packAlphabet.length; i++) {
			if (isDistro(packAlphabet[i])) {
				dist = packAlphabet[i];
			}
		}
		return dist;
	}

	/**
	 * Judge whether the param string is distribution name.
	 * 
	 * @param string
	 *            String
	 * @return true(string is a distribution)/false(string is not a
	 *         distribution)
	 */
	private static boolean isDistro(String string) {
		return (string.equalsIgnoreCase("fc")
				|| string.equalsIgnoreCase("rhel")
				|| string.equalsIgnoreCase("centos")
				|| string.equalsIgnoreCase("rh") || string
				.equalsIgnoreCase("el"))
				|| string.equalsIgnoreCase("etch") // debian
				|| string.equalsIgnoreCase("sarge")
				|| string.equalsIgnoreCase("sid");
	}

	/**
	 * Remove the distribution name from the version string.
	 * 
	 * @param ver
	 *            The version string
	 * @return The version string replaced the distribution name to ":"
	 */
	private static String removeDistribution(String ver) {
		String dist = checkDistributionVersion(ver);
		if (dist != null) {
			return ver.replaceAll(dist, ":");
		} else {
			return ver;
		}
	}

	/**
	 * Check the distribution name in the version number(param).
	 * 
	 * @param ver
	 *            The version number
	 * @return The distribution name
	 */
	public static String checkDistributionVersion(String ver) {
		String dist = checkDistribution(ver);
		if (dist != null) {
			int begin = ver.indexOf(dist);
			int end = begin + dist.length();
			if (end < ver.length()) {
				end += 1;
			}
			String distVer = ver.substring(begin, end);
			if (isDistroVersion(distVer)) {
				return distVer;
			}
		}
		return dist;
	}

	/**
	 * Judge whether the param string is distribution name and version.
	 * 
	 * @param string
	 *            String
	 * @return true(string is a distribution)/false(string is not a
	 *         distribution)
	 */
	private static boolean isDistroVersion(String string) {
		return (string.equalsIgnoreCase("fc5")
				|| string.equalsIgnoreCase("fc6")
				|| string.equalsIgnoreCase("rhel3")
				|| string.equalsIgnoreCase("rhel4")
				|| string.equalsIgnoreCase("rhel5")
				|| string.equalsIgnoreCase("centos4")
				|| string.equalsIgnoreCase("rh8")
				|| string.equalsIgnoreCase("rh9")
				|| string.equalsIgnoreCase("el3")
				|| string.equalsIgnoreCase("el4") 
				|| string.equalsIgnoreCase("el5")
				|| string.equalsIgnoreCase("etch") // debian
				|| string.equalsIgnoreCase("sarge")
				|| string.equalsIgnoreCase("sid"));

	}

}
