/*
 * 
 * Licensed Materials - Property of IBM
 *
 * Open Platform Trust Services - An open source TCG PTS
 *
 * (C) Copyright International Business Machines Corp. 2007
 *
 */
package com.ibm.trl.tcg.pts.integrity;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.util.Properties;
import java.util.UUID;
import java.util.Vector;
import java.util.zip.GZIPInputStream;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import com.ibm.trl.tcg.pts.engine.Event;
import com.ibm.trl.tcg.pts.engine.FiniteStateMachine;
import com.ibm.trl.tcg.pts.engine.PlatformProperties;
import com.ibm.trl.tcg.pts.eventlog.EventSet;
import com.ibm.trl.tcg.pts.eventlog.IML;
import com.ibm.trl.tcg.pts.eventlog.PCRS;
import com.ibm.trl.tcg.pts.tools.Base64Tool;
import com.ibm.trl.tcg.pts.tools.HexTool;
import com.ibm.trl.tcg.pts.tools.PcrComposite;
import com.ibm.trl.tcg.pts.tools.SignatureTool;
import com.ibm.trl.tcg.pts.tools.XmlTool;

/**
 * Object to handle TCG Integrity Report
 * 
 * Reference: TCG Infrastructure Working Group Integrity Report Schema
 * Specification Specification Version 1.0 Revision 1.0 17 November 2006 FINAL
 * https://www.trustedcomputinggroup.org/specs/IWG/IntegrityReport_Schema_Specification_v1.0.pdf
 * 
 * 
 * IntegrityReport file -> IML IML -> IntegrityReport file IML + Model ->
 * IntegrityReport file
 * 
 * 
 * @author Seiji Munetoh
 * 
 */
public class IntegrityReport {

	/* Logger */
	private Log log = LogFactory.getLog(this.getClass());

	// private boolean _verbose = false;
	private String _filename;

	// TODO _schemaDir is relative path from target XML file
	// private String _schemaDir = "../pvs/tcg/schemas";
	// private String _schemaDir = "../../pvs/tcg/schemas/";
	// private String _schemaDir = "../../schemas/";

	// private String _schemaDir = "C:/Documents and Settings/munetoh/My
	// Documents/Eclipse Workplace/3.2-meti/pvs/tcg/schemas";
	// private String[] _schemas = {
	// _schemaDir + "Reference_Integrity_Measurement_Manifest_v13_ibm.xsd",
	// _schemaDir + "SimpleObject_v8_ibm.xsd",
	// _schemaDir + "Core_Integrity_Manifest_v16_ibm.xsd",
	// _schemaDir + "Integrity_Report_Manifest_v19_ibm.xsd"
	// };
	private TCGSchemas _tcgSchemas;

	private String _namespace = "http://www.trustedcomputinggroup.org/XML/SCHEMA/1_0/integrity_report#";

	private Document _doc;

	private Element _ir;

	private boolean addDigestMethod = true;

	private int _eventNum = 0;

	private PcrComposite _pcrComposite;

	private String _base64SignatureValue;

	private byte[] _pubkey;

	private byte[][] _startHashPcrs = null;

	private byte[][] _pcrs = null;

	private int _pcrNum = 24;

	// private int dbIndex;

	/**
	 * Create new integrity report object
	 * 
	 */
	public IntegrityReport() {
		// this.dbIndex = dbIndex;

		// TODO _tcgSchemas = new TCGSchemas();
		init();
	}

	/**
	 * Create integrity report object from file (XML)
	 * 
	 * @param filename
	 * @throws Exception
	 */
	public IntegrityReport(String filename) throws Exception {
		if (log.isTraceEnabled()) {
			log.trace("IntegrityReport");
			log.trace(" filename : " + filename);
		}

		_filename = filename;
		this.load(filename);
		this.validateQuoteData();
	}

	/**
	 * 
	 * 
	 * @param irFilename
	 * @param option,
	 *            1:ziped file
	 * @throws Exception
	 */
	public IntegrityReport(String filename, int option) throws Exception {
		if (log.isTraceEnabled()) {
			log.trace("IntegrityReport");
			log.trace(" filename : " + filename);
		}

		_filename = filename;
		if (option == 1) {
			this.loadGzip(filename);
		} else {
			this.load(filename);
		}
		this.validateQuoteData();
	}

	/**
	 * Create new Integrity Report DOM Tree
	 */
	private void init() {
		// http://java.sun.com/j2se/1.5.0/ja/docs/ja/api/java/util/UUID.html
		UUID uuid = UUID.randomUUID();
		String uuidStr = uuid.toString();
		try {
			DocumentBuilderFactory factory = DocumentBuilderFactory
					.newInstance();
			DocumentBuilder builder = factory.newDocumentBuilder();
			DOMImplementation domImpl = builder.getDOMImplementation();
			_doc = domImpl.createDocument(_namespace, "Report", null);

			_ir = _doc.getDocumentElement();
			_ir.setAttribute("xmlns:xsi",
					"http://www.w3.org/2001/XMLSchema-instance");
			_ir
					.setAttribute("xmlns:core",
							"http://www.trustedcomputinggroup.org/XML/SCHEMA/1_0_1/core_integrity#");
			_ir
					.setAttribute("xmlns:stuff",
							"http://www.trustedcomputinggroup.org/XML/SCHEMA/1_0/simple_object#");
			// _ir.setAttribute("RevLevel", "0");
			_ir.setAttribute("ID", "_" + uuidStr);
			_ir.setAttribute("UUID", uuidStr);

		} catch (ParserConfigurationException e) {
			e.printStackTrace();
		}

	}

	public void addQuoteData(PcrComposite pcrComposite,
	// int SizeOfSelect,
			// String base64PcrSelect,
			// int pcrNums,
			// int[] pcrIndex,
			// String[] base64PcrValue,

			// String base64DigestValue,
			// String base64ExternalData,
			String base64SignatureValue, byte[] pubKey) {
		Element qd = _doc.createElement("QuoteData");
		qd.setAttribute("ID", "XXXX"); // xs:ID
		_ir.appendChild(qd);

		Element q = _doc.createElement("Quote");
		qd.appendChild(q);

		Element pc = _doc.createElement("PcrComposite");
		q.appendChild(pc);

		Element ps = _doc.createElement("PcrSelection");
		ps.setAttribute("SizeOfSelect", Integer.valueOf(pcrComposite
				.getSizeOfSelect()).toString());// xs:unsignedShort
		ps.setAttribute("PcrSelect", Base64Tool.encode(pcrComposite
				.getPcrSelect()));// xs:base64Binary
		pc.appendChild(ps);

		Element vs = _doc.createElement("ValueSize"); // xs:unsignedLong
		int selectedPcrNums = pcrComposite.getSelectedPcrNums();
		vs.appendChild(_doc.createTextNode(new Integer(20 * selectedPcrNums)
				.toString()));
		pc.appendChild(vs);

		log.debug("selectedPcrNums = " + selectedPcrNums);
		log.debug(pcrComposite.toString());

		for (int i = 0; i < selectedPcrNums; i++) {
			Element pv = _doc.createElement("PcrValue");
			pv.setAttribute("PcrNumber", new Integer(pcrComposite
					.getPcrIndex(i)).toString());
			pv.appendChild(_doc.createTextNode(Base64Tool.encode(pcrComposite
					.getPcrValue(i))));
			pc.appendChild(pv);
		}

		Element qi = _doc.createElement("QuoteInfo");
		q.appendChild(qi);

		qi.setAttribute("VersionMajor", "1");// xs:unsignedByte
		qi.setAttribute("VersionMinor", "1");// xs:unsignedByte
		qi.setAttribute("VersionRevMajor", "0");// xs:unsignedByte
		qi.setAttribute("VersionRevMinor", "0");// xs:unsignedByte
		qi.setAttribute("Fixed", "QUOT");// xs:normalizedString
		qi.setAttribute("DigestValue", Base64Tool.encode(pcrComposite
				.getDigestValue()));// ds:DigestValueType
		qi.setAttribute("ExternalData", Base64Tool.encode(pcrComposite
				.getExternalData())); // xs:base64Binary

		// Element ti = _doc.createElement("TpmInfo"); // option

		Element ts = _doc.createElement("TpmSignature");
		qd.appendChild(ts);

		Element sm = _doc.createElement("SignatureMethod"); // ds:SignatureMethodType
		sm.setAttribute("Algorithm",
				"http://www.w3.org/2000/09/xmldsig#rsa-sha1");
		ts.appendChild(sm);

		Element sv = _doc.createElement("SignatureValue"); // ds:SignatureValueType
		sv.appendChild(_doc.createTextNode(base64SignatureValue));
		// sv.appendChild(_doc.createTextNode(base64DigestValue)); // TODO DEBUG
		ts.appendChild(sv);

		Element ki = _doc.createElement("KeyInfo"); // ds:KeyInfoType
		ts.appendChild(ki);

		Element kv = _doc.createElement("KeyValue"); // ds:KeyInfoType
		kv.appendChild(_doc.createTextNode(Base64Tool.encode(pubKey)));
		ki.appendChild(kv);

	}

	/**
	 * 
	 * @param iml
	 * @param fsmList
	 * @throws Exception
	 */
	public void addSnapshotFromIML2(IML iml, FiniteStateMachine[] fsmList)
			throws Exception {

		/* Init Properties */
		PlatformProperties properties = new PlatformProperties();
		properties.init();

		/* UUID */
		// TODO Wrong
		UUID uuid = UUID.randomUUID();
		String uuidStr = uuid.toString();

		/* PCR */
		for (int pcrindex = 0; pcrindex < _pcrNum; pcrindex++) {
			/* Event */
			Event[] events = iml.getEventList(pcrindex);
			if (log.isDebugEnabled()) {
				log
						.debug("---------------------------------- EVENTS START for FSM "
								+ pcrindex);
				for (int i = 0; i < events.length; i++)
					log.debug(events[i].toString());
				log.debug("---------------------------------- EVENTS END");
			}

			/* Check IML and models */
			boolean snapshotState = false;
			FiniteStateMachine fsm = null;

			if (events == null) {
				// no events
				snapshotState = false;
			} else if (events.length > 0) {
				/* Get FSM */
				if (fsmList.length <= pcrindex) {
					snapshotState = false;
				} else {
					fsm = fsmList[pcrindex];
					if (fsm == null) {
						snapshotState = false;
					} else {
						snapshotState = true;
					}
				}
			}

			/* Generate XML */
			if (snapshotState) {
				// events
				fsm.reset(properties);
				Element stuffSimpleSnapshotObject = createStuffSimpleSnapshotObject();// new
																						// Element[24];

				int rc = 0;
				for (int i = 0; i < events.length; i++) {
					if (log.isDebugEnabled()) {
						log.debug(events[i].toString());
						log.debug("----------------------------------------"
								+ i);
					}

					/* Validate Event */
					rc = fsm.validateEvent2(events[i]);
					if (rc == 0) {
						/* add to snapshot */
						String name = fsm.getName();
						String id = "_" + uuidStr + "_sha1_" + i;
						stuffSimpleSnapshotObject
								.appendChild(createStuffObjects(events[i], id,
										name));
						if (log.isDebugEnabled())
							log.debug("ADD : " + name);
					} else if (rc == 1) {
						// hold event
						if (log.isDebugEnabled())
							log.debug("HOLD");
					} else if (rc == 3) {
						// terminate
						if (log.isDebugEnabled())
							log.debug("terminate");
					} else {
						// FSM transfer failed
						throw new Exception("FSM Trans was Failed at "
								+ events[i].toString() + ", FSM : "
								+ fsm.getName());// +
					}
				} // for event

				/* Snapshot final */
				/* Get metadata */
				// Element cid = getCoreComponentIDfromRM(rmList[j],j,i);
				/* add to IR */

				_ir.appendChild(createShapshotCollection(
						stuffSimpleSnapshotObject, pcrindex));

				if (log.isDebugEnabled()) {
					log.debug("shapshot of pcr " + pcrindex + " of  "
							+ fsm.getName() + " added");
				}

				/* Update for next FSM */
				// System.err.println("SM DEBUG : " + j + " "+
				// rmList[j].getName() + " -- done ");
				events = fsm.getEvents();
				if (events.length > 0) {
					log.debug("Events are remains for PCR " + pcrindex);
				}
				properties = fsm.getTveProperties();
				fsm = null;
			} // snapshotState
			else {
				/* snapshot w/o event */
				System.err.println("W/O E " + pcrindex);
				_ir.appendChild(createShapshotCollection(null, pcrindex));
			}
		} // for pcrindex
	}

	/**
	 * 
	 * @param iml
	 * @param rmList
	 * @param pcrComposite
	 * @throws Exception
	 */
	public void addSnapshotFromIML3(IML iml, ReferenceManifest[] rmList,
			PcrComposite pcrComposite) throws Exception {

		/* Gen IR without QuoteInfo/PcrComposite */
		if (pcrComposite == null) {
			pcrComposite = new PcrComposite();
			pcrComposite.setPcrNumber(_pcrNum);
			for (int i = 0; i < _pcrNum; i++) {
				// System.err.println("PCR " + i);
				pcrComposite.selectPcr(i, _pcrs[i]); // TODO DUMMY
			}
		}

		// Event[] events = iml.getEventList();
		PlatformProperties properties = new PlatformProperties();
		properties.init();

		UUID uuid = UUID.randomUUID(); // TODO Wrong
		String uuidStr = uuid.toString(); // TODO

		/* Reference Manifest[] */
		for (int rmindex = 0; rmindex < rmList.length; rmindex++) {
			if (log.isDebugEnabled()) {
				log.debug("============================================= RM["
						+ rmindex + "]");
				log.debug("RM name : " + rmList[rmindex].getName());
			}
			/* FSM */
			FiniteStateMachine[] fsmList = rmList[rmindex].getValidationModel();

			if (log.isDebugEnabled()) {
				for (int i = 0; i < fsmList.length; i++) {
					if (fsmList[i] != null) {
						log.debug("FSM[" + i + "] : " + fsmList[i].getName());
					} else {
						log.debug("FSM[" + i + "] : null");
					}
				}
			}
			/* PCRs */
			for (int pcrindex = 0; pcrindex < _pcrNum; pcrindex++) {
				/* Event */
				Event[] events = iml.getEventList(pcrindex);

				if (log.isDebugEnabled()) {
					log
							.debug("---------------------------------- EVENTS START for FSM "
									+ pcrindex);
					for (int i = 0; i < events.length; i++) {
						log.debug(events[i].toString());
					}
					log.debug("---------------------------------- EVENTS END");
				}

				/* Check IML, models, PcrComposite */
				boolean snapshotState = false;
				boolean addSnapshotWOE = false; // with out event

				FiniteStateMachine fsm = null;

				/* Check used PCR by PCR Composite */
				if (pcrComposite.selected(pcrindex)) {
					// log.debug("Quoted PCR " + pcrindex);
					/* PCR was Quoted */
					if (events == null) {
						// No Event
						log.warn("Quoted but no Events for PCR " + pcrindex);
						snapshotState = false;
						addSnapshotWOE = true;
					} else if (events.length > 0) {
						/* Get FSM */
						if (fsmList.length <= pcrindex) {
							log.warn("Quoted but no FSM for PCR " + pcrindex
									+ ", FSM is not enough. " + 
									fsmList.length + " <= " + pcrindex);
							snapshotState = false;
							addSnapshotWOE = true;
						} else {
							fsm = fsmList[pcrindex];
							if (fsm == null) {
								log.warn("Quoted but no FSM for PCR " + pcrindex);
								snapshotState = false;
								addSnapshotWOE = true;
							} else {
								// Event and FSM
								snapshotState = true;
							}
						}
					} else { // length ==0
						log.warn("Quoted but no Events for PCR " + pcrindex + ", len=0");
						snapshotState = false;
						addSnapshotWOE = true;
					}
				} else {
					// Not selected Skip
					//log.warn("Not Quoted PCR " + pcrindex);
					if (log.isDebugEnabled()) {
						log.debug("Not Quoted for PCR " + pcrindex);
					}
				}

				/* Create Snapshot without Event */
				if (addSnapshotWOE) {
					// System.err.println("W/O E " + pcrindex);
					_ir.appendChild(createShapshotCollection(null, pcrindex));
				}

				/* Create Snapshot with Event */
				if (snapshotState) {
					// System.err.println("W E " + pcrindex);
					fsm.reset(properties);
					Element stuffSimpleSnapshotObject = createStuffSimpleSnapshotObject();// new
																							// Element[24];

					int rc = 0;
					for (int i = 0; i < events.length; i++) {
						if (log.isDebugEnabled()) {
							log.debug(events[i].toString());
							log
									.debug("----------------------------------------"
											+ i);
						}
						rc = fsm.validateEvent2(events[i]);
						if (rc == 0) { /* Measured */
							/* add to snapshot */
							String name = fsm.getName();
							// int pcrIndex = (int) events[i].getPcrIndex();

							/* ok add this event to snapshot */
							String id = "_" + uuidStr + "_sha1_" + i;
							stuffSimpleSnapshotObject
									.appendChild(createStuffObjects(events[i],
											id, name));
							if (log.isDebugEnabled()) {
								log.debug("ADD : " + name);
							}
						} else if (rc == 1) { /* Hold */
							// hold event
							if (log.isDebugEnabled())
								log.debug("HOLD");
						} else if (rc == 3) {
							// terminate
							if (log.isDebugEnabled())
								log.debug("terminate");
							// break;
						} else {

							log.error("FSM Trans was Failed at "
									+ events[i].toString() + ", FSM : "
									+ fsm.getName());
							log.error(events[i].toString());
							log.error(events[i].toStringBase64());
							throw new Exception("FSM Trans was Failed at "
									+ events[i].toString() + ", FSM : "
									+ fsm.getName());// +
							// ", RM File: " + rmList[j]);
						}
					} // for event

					/* Snapshot final */
					/* Get metadata */
					// Element cid = getCoreComponentIDfromRM(rmList[j],j,i);
					/* add to IR */

					Element cid = getCoreComponentIDfromRM(rmList[rmindex],
							rmindex, pcrindex);
					/* add to IR */
					_ir.appendChild(createShapshotCollection(
							stuffSimpleSnapshotObject, cid, pcrindex));
					if (log.isDebugEnabled())
						log.debug("shapshot of pcr " + pcrindex + " of  "
								+ fsm.getName() + " added");

					// _ir.appendChild(createShapshotCollection(stuffSimpleSnapshotObject,pcrindex));
					// if (_verbose) log.debug("shapshot of pcr " + pcrindex +
					// " of " + fsm.getName() + " added" );

					/* Update for next FSM */
					// System.err.println("SM DEBUG : " + j + " "+
					// rmList[j].getName() + " -- done ");
					events = fsm.getEvents();
					if (events.length > 0) {
						log.debug("Events are remains for PCR " + pcrindex);
						for (int i = 0; i < events.length; i++) {
							log.debug(events[i].toString());
						}
					}
					iml.updateEvent(pcrindex, events);
					properties = fsm.getTveProperties();
					fsm = null;
				} // snapshotState
			} // for pcrindex

		}

		// TODO Auto-generated method stub

	}

	/**
	 * 1) load integrity report (XML file) 2) create DOM tree
	 * 
	 * @param filename
	 */
	private void load(String filename) {
		try {
			DocumentBuilderFactory dbfactory = DocumentBuilderFactory
					.newInstance();
			DocumentBuilder builder = dbfactory.newDocumentBuilder();
			_doc = builder.parse(new File(filename));
			_ir = _doc.getDocumentElement();
		} catch (Exception e) {
			log.error("loadFile " + filename + " failed");
			e.printStackTrace();
		}
	}

	private void loadGzip(String filename) {
		try {
			FileInputStream fis = new FileInputStream(filename);
			BufferedInputStream bis = new BufferedInputStream(fis);
			GZIPInputStream gis = new GZIPInputStream(bis);

			DocumentBuilderFactory dbfactory = DocumentBuilderFactory
					.newInstance();
			DocumentBuilder builder = dbfactory.newDocumentBuilder();
			// _doc = builder.parse(new File(filename));
			_doc = builder.parse(gis);
			_ir = _doc.getDocumentElement();
		} catch (Exception e) {
			log.error("Load Giped File " + filename + " failed");
			e.printStackTrace();
		}
	}

	/**
	 * save to file
	 * 
	 * @param filename
	 * @throws Exception
	 */
	public void save(String filename) throws Exception {
		//XmlTool.saveToFile(filename, _doc, true);
		XmlTool.saveToFile(filename, _doc, false);
		// XmlTool.saveToFile(filename,_doc,false); // w/ Quote
		if (log.isDebugEnabled())
			log.debug("Write IR to : " + filename);

	}

	/**
	 * 1) load BIOS binary IML file 2) put them to SnapshotCollection per PCR
	 * index
	 * 
	 * note) PcrIndex, EventType and EvetData are added in stuff:Objects Element
	 * 
	 * SnapshotCollection core:ComponentID core:VendorID core:VendorGUID
	 * core:DigestMethod core:Values stuff:SimpleSnapshotObject stuff:Objects
	 * stuff:Hash pcrindex eventtype eventdata
	 * 
	 * @param filename
	 * @throws Exception
	 */
	public void loadBIOSIML(String filename) throws Exception {
		// BiosEventLog bel = new BiosEventLog(filename);
		IML iml = new IML();
		iml.loadBIOSIML(filename);

		byte[] zero = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
				0, 0 };

		// Event [] events = bel.getEvents();
		EventSet[] events = iml.getEvents();

		for (int pcrindex = 0; pcrindex < events.length; pcrindex++) {
			if (events[pcrindex].size > 0) {
				/* add SnapshotCollection */
				UUID uuid = UUID.randomUUID();
				String uuidStr = uuid.toString();

				Element sc = _doc.createElement("SnapshotCollection");
				sc.setAttribute("Id", "_" + uuidStr);
				sc.setAttribute("UUID", uuidStr);
				sc.setAttribute("RevLevel", "0");

				Element cCID = _doc.createElement("core:ComponentID");
				Element cVID = _doc.createElement("core:VendorID");
				Element cGID = _doc.createElement("core:VendorGUID");

				cCID.setAttribute("Id", "_" + uuidStr + "_cid");
				cCID.setAttribute("SimpleName", "unknown");
				cVID.setAttribute("Name", "unknown");
				cGID.appendChild(_doc.createTextNode("unknown"));
				cVID.appendChild(cGID);
				cCID.appendChild(cVID);
				sc.appendChild(cCID);

				if (addDigestMethod) {
					Element cDM = _doc.createElement("core:DigestMethod");
					cDM.setAttribute("Id", "sha1");
					cDM.setAttribute("Algorithm", "unknown");
					sc.appendChild(cDM);
					addDigestMethod = false;
				}
				Element cV = _doc.createElement("core:Values");

				Element sS = _doc.createElement("stuff:SimpleSnapshotObject");

				Event[] elist = events[pcrindex].getEventList();
				for (int i = 0; i < elist.length; i++) {
					Element sO = _doc.createElement("stuff:Objects");
					sO.setAttribute("Name", "unknown");

					Element sH = _doc.createElement("stuff:Hash");
					sH.setAttribute("Id", "_" + uuidStr + "_sha1_" + i);
					sH.setAttribute("AlgRef", "sha1");
					sH.appendChild(_doc.createTextNode(Base64Tool
							.encode(elist[i].getDigest())));

					Element sPI = _doc.createElement("pcrindex"); // TODO not
																	// TCG
																	// Standard
					String piStr = "" + elist[i].getPcrIndex();
					sPI.appendChild(_doc.createTextNode(piStr));

					Element sET = _doc.createElement("eventtype"); // TODO not
																	// TCG
																	// Standard
					String etStr = "" + elist[i].getEventType();
					sET.appendChild(_doc.createTextNode(etStr));

					Element sED = _doc.createElement("eventdata"); // TODO not
																	// TCG
																	// Standard
					sED.appendChild(_doc.createTextNode(Base64Tool
							.encode(elist[i].getEventData())));

					sO.appendChild(sH);
					sO.appendChild(sPI);
					sO.appendChild(sET);
					sO.appendChild(sED);
					sS.appendChild(sO);

				}
				cV.appendChild(sS);
				sc.appendChild(cV);

				Element ph = _doc.createElement("PcrHash");
				ph.setAttribute("IsResetable", "false");
				ph.setAttribute("StartHash", Base64Tool.encode(zero));
				ph.setAttribute("Id", "_" + uuidStr + "_pcrhash");
				ph.setAttribute("AlgRef", "sha1");
				ph.setAttribute("Number", "" + pcrindex); // TODO PCR Index
				if (_pcrs != null) {
					// System.err.println("PCR " + pcrindex);
					ph.appendChild(_doc.createTextNode(Base64Tool
							.encode(_pcrs[pcrindex])));
				} else {
					// System.err.println("PCR " + pcrindex + "BULL");
					ph
							.appendChild(_doc.createTextNode(Base64Tool
									.encode(zero)));
				}
				sc.appendChild(ph);
				_ir.appendChild(sc);

			} else {
				/* no event - skip */
			}
		}

	}

	/**
	 * 
	 * get/copy core:ComponentID from Manifest to Report
	 * 
	 * @param manifest
	 * @param pcrIndex
	 * @param fsmIndex
	 * @return
	 */
	private Element getCoreComponentIDfromRM(ReferenceManifest manifest,
			int fsmIndex, int pcrIndex) {
		// 20071210 Munetoh GCJ ERROR
		// java.lang.NullPointerException
		//   at gnu.xml.dom.DomNode.dispatchEvent(libgcj.so.7rh)
		//   at gnu.xml.dom.DomAttr.mutating(libgcj.so.7rh)
		//   at gnu.xml.dom.DomAttr.setNodeValue(libgcj.so.7rh)
		//   at gnu.xml.dom.DomElement.setAttribute(libgcj.so.7rh)
		//   at com.ibm.trl.tcg.pts.integrity.IntegrityReport.getCoreComponentIDfromRM(openpts5.jar.so)

		if (log.isDebugEnabled()) {
			log.debug("getCoreComponentIDfromRM ");
			log.debug("  manifest : " + manifest);
			log.debug("  fsmIndex : " + fsmIndex);
			log.debug("  pcrIndex : " + pcrIndex);
		}
		
		Element cciSrc = manifest.getCoreComponentID();
		
		log.debug("getCoreComponentIDfromRM A");
		
		Element cciDst = (Element) _doc.importNode(cciSrc, true);
		
		String name = "_cid_" + fsmIndex + "_" + pcrIndex;
		log.debug("getCoreComponentIDfromRM B name : " + name);
		
		
		if (cciDst == null) {
			log.error("getCoreComponentIDfromRM  cciDst == null");
		}
		else {
			try {
				cciDst.setAttribute("Id", name);
			} catch (NullPointerException e) {
				// TODO GCJ ERROR,  Manual Copy 
				//e.printStackTrace();		
				String tagName = cciSrc.getNodeName();
				cciDst = _doc.createElement(tagName);
				cciDst.setAttribute("Id", name);
				
				log.warn("Somehow importNode did not work with GCJ. We copy the element instead of importNode."
						+", tagName : " + tagName);
			}
		}
		
		if (log.isDebugEnabled()) {
			log.debug("getCoreComponentIDfromRM done");
			log.debug("  cciDst : " + cciDst);
		}
		
		return cciDst;
	}

	private Node createStuffObjects(Event event, String id, String name) {
		Element sO = _doc.createElement("stuff:Objects");
		sO.setAttribute("Name", name);

		Element sH = _doc.createElement("stuff:Hash");
		// sH.setAttribute("Id", "_" + uuidStr + "_sha1_" + i);
		sH.setAttribute("Id", id);
		sH.setAttribute("AlgRef", "sha1");
		sH.appendChild(_doc
				.createTextNode(Base64Tool.encode(event.getDigest())));

		Element sPI = _doc.createElement("pcrindex"); // TODO not TCG Standard
		String piStr = "" + event.getPcrIndex();
		sPI.appendChild(_doc.createTextNode(piStr));

		Element sET = _doc.createElement("eventtype"); // TODO not TCG Standard
		String etStr = "" + event.getEventType();
		sET.appendChild(_doc.createTextNode(etStr));

		Element sED = _doc.createElement("eventdata"); // TODO not TCG Standard
		sED.appendChild(_doc.createTextNode(Base64Tool.encode(event
				.getEventData())));

		sO.appendChild(sH);
		sO.appendChild(sPI);
		sO.appendChild(sET);
		sO.appendChild(sED);
		return sO;
	}

	private Element createStuffSimpleSnapshotObject() {
		Element sS = _doc.createElement("stuff:SimpleSnapshotObject");
		return sS;
	}

	/**
	 * Create Dumy CID
	 * 
	 * 
	 * @param sS
	 * @param pcrindex
	 * @return
	 */
	private Node createShapshotCollection(Element sS, int pcrindex) {
		if (log.isTraceEnabled()) {
			log.trace("createShapshotCollection, pcrindex = " + pcrindex);
		}

		Element _cid;
		_cid = _doc.createElement("core:ComponentID");
		_cid.setAttribute("Id", "_ComponentID");
		_cid.setAttribute("SimpleName", "unknown");
		_cid.setAttribute("VersionString", "unknown");
		_cid.setAttribute("VersionBuild", "unknown");
		_cid.setAttribute("ModelSystemClass", "unknown");

		Element vid = _doc.createElement("core:VendorID");
		vid.setAttribute("Name", "unknown");

		// TODO How to get the SmiVendorId?
		Element svid = _doc.createElement("core:SmiVendorId");
		svid.appendChild(_doc.createTextNode("0"));

		// TODO How to get the TcgVendorId?
		Element tvid = _doc.createElement("core:TcgVendorId");
		tvid.appendChild(_doc.createTextNode("0"));

		vid.appendChild(svid);
		vid.appendChild(tvid);
		_cid.appendChild(vid);
		return createShapshotCollection(sS, _cid, pcrindex);
	}

	/**
	 * 
	 * <SnapshotCollection Id="_6b91c378-f7bf-4d75-b22b-52995b96a032"
	 * RevLevel="0" UUID="6b91c378-f7bf-4d75-b22b-52995b96a032">
	 * 
	 * @param sS
	 * @param cCID
	 *            core:ComponentID
	 * @param pcrindex
	 * @return
	 */
	private Element createShapshotCollection(Element sS, Element cCID,
			int pcrindex) {
		if (log.isTraceEnabled()) {
			log.trace("createShapshotCollection, pcrindex = " + pcrindex);
		}

		byte[] zero = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
				0, 0 }; // TODO wrong

		/* add SnapshotCollection */
		UUID uuid = UUID.randomUUID();
		String uuidStr = uuid.toString();

		Element sc = _doc.createElement("SnapshotCollection");
		sc.setAttribute("Id", "_" + uuidStr);
		sc.setAttribute("UUID", uuidStr);
		sc.setAttribute("RevLevel", "0");

		// Element cCID = _doc.createElement("core:ComponentID");
		// Element cVID = _doc.createElement("core:VendorID");
		// Element cGID = _doc.createElement("core:VendorGUID");

		// cCID.setAttribute("Id","_" + uuidStr + "_cid");
		// cCID.setAttribute("SimpleName","unknown");
		// cVID.setAttribute("Name","unknown");
		// cGID.appendChild(_doc.createTextNode("unknown"));
		// cVID.appendChild(cGID);
		// cCID.appendChild(cVID);
		sc.appendChild(cCID);

		if (addDigestMethod) {
			Element cDM = _doc.createElement("core:DigestMethod");
			cDM.setAttribute("Id", "sha1");
			cDM.setAttribute("Algorithm", "unknown");
			sc.appendChild(cDM);
			addDigestMethod = false;
		}
		Element cV = _doc.createElement("core:Values");

		// Element sS = _doc.createElement("stuff:SimpleSnapshotObject");

		if (sS != null) {
			cV.appendChild(sS);
		}
		sc.appendChild(cV);

		Element ph = _doc.createElement("PcrHash");
		ph.setAttribute("IsResetable", "false");
		ph.setAttribute("StartHash", Base64Tool
				.encode(_startHashPcrs[pcrindex]));
		ph.setAttribute("Id", "_" + uuidStr + "_pcrhash");
		ph.setAttribute("AlgRef", "sha1");
		ph.setAttribute("Number", "" + pcrindex); // TODO PCR Index

		// IR Generation
		if (_pcrs != null) {
			ph.appendChild(_doc.createTextNode(Base64Tool
					.encode(_pcrs[pcrindex])));
		} else {
			ph.appendChild(_doc.createTextNode(Base64Tool.encode(zero)));
		}

		// ph.appendChild(_doc.createTextNode(Base64Tool.encode(zero)));
		sc.appendChild(ph);

		return sc;
	}

	/**
	 * Get list of snapsot name SnapshotCollection core:ComponentID SimpleName
	 * -------- use this
	 * 
	 * 
	 * TODO list order must be a boot order .... IR must contains the such order
	 * info... check the schema
	 * 
	 * @return
	 */
	public String[] getNameList() {
		// log.debug("getNameList");
		Vector names = new Vector();

		NodeList nlSC = _ir.getElementsByTagName("SnapshotCollection");
		for (int i = 0; i < nlSC.getLength(); i++) {
			Element elSC = (Element) nlSC.item(i);
			NodeList nlCCI = elSC.getElementsByTagName("core:ComponentID");
			Element elCCI = (Element) nlCCI.item(0); // TODO must be 0
			String name = elCCI.getAttribute("SimpleName");

			// log.debug( i + " " + name);

			if (names.size() == 0) {
				log.debug("ADD (1st)");
				/* single SimpleName */
				names.add(name);
			} else {
				/* check the same name? */
				boolean hit = false;
				for (int j = 0; j < names.size(); j++) {
					String nameRef = (String) names.get(j);
					if (nameRef.equals(name)) {
						// log.debug(j + "HIT " + nameRef);
						hit = true;
					} else {
						// log.debug(j + "ADD " + nameRef);
						// Miss
						// names.add(name);
					}
				}
				if (hit == false)
					names.add(name);
			}
		}

		/* Vector -> String[] */
		String[] out = new String[names.size()];
		for (int i = 0; i < names.size(); i++) {
			out[i] = (String) names.get(i);
		}

		// log.debug("getNameList -- done");
		return out;
	}

	/**
	 * get events in this report, single snapshot for each PCR
	 * 
	 * @return
	 * @throws Exception
	 */
	public EventSet[] getEventSet() throws Exception {
		EventSet[] events = new EventSet[24]; // TODO 24?
		for (int i = 0; i < 24; i++) {
			events[i] = new EventSet();
			events[i].setPcrIndex(i);
		}

		/* SnapshotCollection per PCR index */
		NodeList nlSC = _ir.getElementsByTagName("SnapshotCollection");
		for (int i = 0; i < nlSC.getLength(); i++) {
			Element elSC = (Element) nlSC.item(i);

			/* Check PCR index of this SnapshotCollection */
			NodeList nlPH = elSC.getElementsByTagName("PcrHash");
			Element elPH = (Element) nlPH.item(0); // TODO must be 0
			int pcrindex = new Integer(elPH.getAttribute("Number"));

			/* fill events */
			NodeList nlSO = elSC.getElementsByTagName("stuff:Objects");
			for (int j = 0; j < nlSO.getLength(); j++) {
				Element elSO = (Element) nlSO.item(j);
				/* Digest */
				NodeList nlSH = elSO.getElementsByTagName("stuff:Hash");
				Element elSH = (Element) nlSH.item(0);
				byte[] digest = Base64Tool.decode(elSH.getTextContent());
				/* Type */
				NodeList nlET = elSO.getElementsByTagName("eventtype");
				Element elET = (Element) nlET.item(0);
				int type = new Integer(elET.getTextContent());
				/* EventData */
				NodeList nlED = elSO.getElementsByTagName("eventdata");
				Element elED = (Element) nlED.item(0);
				byte[] eventdata = Base64Tool.decode(elED.getTextContent());
				events[pcrindex].addEvent(type, digest, eventdata);
				_eventNum++;
			}
		}
		return events;
	}

	/**
	 * get events in this report which has given name (SimpleName attribute)
	 * 
	 * @return
	 * @throws Exception
	 */
	public EventSet[] getEventSetByName(String name) throws Exception {
		EventSet[] events = new EventSet[24]; // TODO 24?
		for (int i = 0; i < 24; i++) {
			events[i] = new EventSet();
			events[i].setPcrIndex(i);
		}

		/* SnapshotCollection per PCR index */
		NodeList nlSC = _ir.getElementsByTagName("SnapshotCollection");
		for (int i = 0; i < nlSC.getLength(); i++) {
			Element elSC = (Element) nlSC.item(i);

			/* Check Name */
			NodeList nlCCI = elSC.getElementsByTagName("core:ComponentID");
			Element elCCI = (Element) nlCCI.item(0); // TODO must be 0
			String simpleName = elCCI.getAttribute("SimpleName");
			if (simpleName.equals(name)) {
				/* Check PCR index of this SnapshotCollection */
				NodeList nlPH = elSC.getElementsByTagName("PcrHash");
				Element elPH = (Element) nlPH.item(0); // TODO must be 0
				int pcrindex = new Integer(elPH.getAttribute("Number"));

				/* fill events */
				NodeList nlSO = elSC.getElementsByTagName("stuff:Objects");
				for (int j = 0; j < nlSO.getLength(); j++) {
					Element elSO = (Element) nlSO.item(j);
					/* Digest */
					NodeList nlSH = elSO.getElementsByTagName("stuff:Hash");
					Element elSH = (Element) nlSH.item(0);
					byte[] digest = Base64Tool.decode(elSH.getTextContent());
					/* Type */
					NodeList nlET = elSO.getElementsByTagName("eventtype");
					Element elET = (Element) nlET.item(0);
					int type = new Integer(elET.getTextContent());
					/* EventData */
					NodeList nlED = elSO.getElementsByTagName("eventdata");
					Element elED = (Element) nlED.item(0);
					byte[] eventdata = Base64Tool.decode(elED.getTextContent());
					events[pcrindex].addEvent(type, digest, eventdata);
					_eventNum++;
				}
			} // if name
		} // for snapshot
		return events;
	}

	/**
	 * get event number in this report
	 * 
	 * @return
	 */
	public int getEventNum() {
		return _eventNum;
	}

	public PcrComposite getPcrComposite() {
		// TODO if (_pcrComposite == null)
		return _pcrComposite;
	}

	// public void setSchemaPath(String path) throws Exception {
	// _tcgSchemas = new TCGSchemas(path);
	// //_tcgSchemas.setSchemaDir(path);
	// }

	// public void verbose() {
	// _verbose = true;
	//	
	// }

	public String getBase64String() throws Exception {
		return XmlTool.getBase64String(_doc);
	}

	/**
	 * 
	 * IR with PCRs
	 * 
	 * <PcrHash AlgRef="sha1" Id="_0c0de11f-b658-4d54-a2f4-10fbc2493124_pcrhash"
	 * IsResetable="false" Number="0" StartHash="AAAAAAAAAAAAAAAAAAAAAAAAAAA=">
	 * AAAAAAAAAAAAAAAAAAAAAAAAAAA=</PcrHash> <<< Here
	 * 
	 * 
	 * @param imlFiles
	 * @param pcrsFilename
	 * @param rmFiles
	 * @param quotePropFilename
	 * @throws Exception
	 */
	public void generateIntegrityReportFromIML(String[] imlFiles,
			String pcrsFilename, String[] rmFiles, String quotePropFilename)
			throws Exception {

		/* pcrs file -> _pcr[24][20] */
		
		
		try {
			PCRS pcrs = new PCRS();
			pcrs.loadfile(pcrsFilename);
			_pcrNum = pcrs.size();
			_pcrs = pcrs.getByteArray();
			
			/*
			_pcrs = new byte[24][20];
			FileReader in = new FileReader(pcrsFilename);
			BufferedReader br = new BufferedReader(in);
			String line;
			int pcrIndex = 0;
			while ((line = br.readLine()) != null) {
				String[] sp = line.split(" ");
				for (int i = 0; i < 20; i++) {
					int si = HexTool.getByte(sp[1 + i]);
					_pcrs[pcrIndex][i] = (byte) si;
				}
				pcrIndex++;
			}
			br.close();
			in.close();

			_pcrNum = pcrIndex;
			*/
			
			// for (int i=0;i<_pcrNum;i++) {
			// System.out.println(HexTool.getHEXString(_pcrs[i], ""));
			// }
		} catch (Exception e) {
			// file not found?
			System.err.println(e);
			e.printStackTrace();
		}

		/* Set Start Hash */
		_startHashPcrs = new byte[24][20];
		for (int pcrIndex = 0; pcrIndex < _pcrNum; pcrIndex++) {
			if ((pcrIndex < 16) || (pcrIndex == 23)) {
				for (int i = 0; i < 20; i++) {
					_startHashPcrs[pcrIndex][i] = 0;
				}
			} else {
				for (int i = 0; i < 20; i++) {
					_startHashPcrs[pcrIndex][i] = (byte) 0xff;
				}
			}
		}

		// throw new Exception("TBD");

		generateIntegrityReportFromIML(imlFiles, rmFiles, quotePropFilename);
	}

	/**
	 * 
	 * Create Integrity Report by using Behavior FSM model (PTS Client Mode)
	 * 
	 * 
	 * @param imlFiles
	 * @param rmFiles
	 * @return
	 * @throws Exception
	 */
	private void generateIntegrityReportFromIML(String[] imlFiles,
			String[] rmFiles, String quotePropFilename) throws Exception {

		/* create IML object */
		IML iml = new IML();
		iml.loadBIOSIML(imlFiles[0]);
		for (int i = 1; i < imlFiles.length; i++) {
			iml.addEventFromBIOSIML(imlFiles[i]);
		}

		/* create Reference Manifest objects */
		ReferenceManifest[] rmList = new ReferenceManifest[rmFiles.length];
		for (int i = 0; i < rmFiles.length; i++) {
			rmList[i] = new ReferenceManifest();
			rmList[i].loadFile(rmFiles[i]);
		}

		// TODO we assume index = pcrindex, not work for multiple fsm for the
		// pcrindex
		// FiniteStateMachine[] fsmList = new
		// FiniteStateMachine[fsmFiles.length];
		// for (int i=0;i<fsmFiles.length;i++) {
		// fsmList[i]= new FiniteStateMachine(fsmFiles[i]);
		// }

		if (quotePropFilename != null) {
			/* Select PCRs to Quote */
			tpmQuote(quotePropFilename);
			/* Get Quote Message */
			this.addQuoteData(_pcrComposite, _base64SignatureValue, _pubkey);
			this.addSnapshotFromIML3(iml, rmList, _pcrComposite);

		} else {
			this.addSnapshotFromIML3(iml, rmList, null);
		}

		// TODO RM or Policy defines the set of PCR to be measured.
		// at this time we contains all policies

		// if (_verbose) ir.verbose();

		// return ir;
	}

	/**
	 * tpm_quote - work with tpm_quote command
	 * 
	 * @param quoteOutputFilename
	 * @throws Exception
	 */
	public void tpmQuote(String quoteOutputFilename) throws Exception {
		Properties prop = new Properties();
		FileInputStream fis = new FileInputStream(quoteOutputFilename);
		prop.load(fis);
		String strPcrNums = prop.getProperty("pcrnums");

		/* PCR Composite */
		_pcrComposite = new PcrComposite();

		/* PCR Nums */

		int pcrNums = 16; // TODO
		if (strPcrNums != null) {
			pcrNums = Integer.valueOf(strPcrNums).intValue();
		}
		int sizeOfSelect = pcrNums / 8;
		_pcrComposite.setPcrNumber(pcrNums);

		for (int i = 0; i < pcrNums; i++) {
			String strPcrValue = prop.getProperty("pcr." + i);
			if (strPcrValue != null) {
				byte[] pcr = HexTool.getBytes(strPcrValue);
				_pcrComposite.selectPcr(i, pcr);
			}
		}

		/* RSA Pub Key */
		// TODO
		// TCPA_PUBKEY
		// TCPA_KEY_PARAMS
		// TCPA_ALGORITHM_ID
		// TCPA_ENC_SCHEME p198
		// TCPA_SIG_SCHEME
		// UINT32 size
		// BYTE parms
		// UINT32 keyLength
		// UINT32 numPrime
		// UINT32 exposize
		// BYTE[] expo
		// TCPA_STORE_PUBKEY
		// UINT32 length
		// BYTE[] key
		// 00 00 00 01 RSA
		// 00 01 TCPA_ES_NONE
		// 00 02 TCPA_SS_RSASSAPKCS1v15_SHA1
		// 00 00 00 0c
		// 00 00 08 00
		// 00 00 00 02
		// 00 00 00 00 exposize=0 => expo = 0x00010001
		// 00 00 01 00
		// 9d121fb371724ef0f4ee8b796f51c7198f1128045d9d8eed1966ad485b1edad6104f84cbc6fdada3de1e4c2d8a8cddc7caf27865dba71f91c905b756021addafa6f702665569f589cc69542cb8340100da4ab3228f854a7b908d0d6c3176f875d9188dd073816c09a94fd6ddef2dcb6fe43272d4b29cd0267276ab763b4e232a8d1d0c2dd60c650d023aaa31532f6ed9c2f723000ff70e00b585ea828908d19d42a0df0b68f481c9a8cf4907f55eddcfe52b8ea07a00c42ce3b44c60a72738279d08246a72c5ab7957c7633267c84b28cd7841b579c922e655096f76a312c95a398c8841b7487e9d181bbb59e30e9f2cf46854d9d34b9bae3eb77c7607536617
		String strPubkey = prop.getProperty("pubkey");
		byte[] pubKeyTmp = HexTool.getBytes(strPubkey);
		int offset = 25; // TODO 2048 bit RSA Key ONLY
		byte[] pubKey = new byte[pubKeyTmp.length - offset + 1];
		pubKey[0] = 0; // MSB=0
		for (int i = 0; i < pubKeyTmp.length - offset; i++) {
			pubKey[1 + i] = pubKeyTmp[offset + i];
		}
		// HexTool.printHex(pubKey); System.out.println(" ");
		_pubkey = pubKey;

		// byte[] pubExp = {0x00,0x01,0x00,0x01};

		// BigInteger biM = new BigInteger(pubKey);
		// BigInteger biE = new BigInteger(pubExp);
		// RSAPublicKeySpec rpks = new RSAPublicKeySpec(biM,biE);
		// KeyFactory kf = KeyFactory.getInstance("RSA");

		// PublicKey key = (PublicKey) kf.generatePublic(rpks);

		/* Verify Signature */

		String strQuoteInfo = prop.getProperty("quoteinfo");
		byte[] quoteInfo = HexTool.getBytes(strQuoteInfo);

		// quoteInfo[10] = (byte) 0xFF;

		String strSignature = prop.getProperty("signature");
		byte[] signature = HexTool.getBytes(strSignature);
		_base64SignatureValue = Base64Tool.encode(signature);

		// TCPA_QUOTE_INFO = TCPA_VERSION + "QUOTE" + TCPA_COMPOSITE_HASH +
		// TCPA_NONCE

		String strNonce = prop.getProperty("nonce");
		byte[] nonce = HexTool.getBytes(strNonce);
		_pcrComposite.setExtraData(nonce);

		fis.close();
	}

	/**
	 * TODO same func in VR
	 * 
	 * @return
	 * @throws Exception
	 */
	public int validateQuoteData() throws Exception {
		/* Validate Signature */

		NodeList nlKV = _ir.getElementsByTagName("KeyValue");

		/* IR without Quote */
		if (nlKV.item(0) == null) {
			if (log.isDebugEnabled()) {
				log.debug("No KeyValue in IR");
			}
			return 2;
		}

		byte[] pubKey = Base64Tool.decode(nlKV.item(0).getTextContent());

		byte[] message = new byte[48];
		NodeList nlQI = _ir.getElementsByTagName("QuoteInfo");
		Element qi = (Element) nlQI.item(0);

		message[0] = 1; // TODO get from IR
		message[1] = 1;// TODO
		message[2] = 0;// TODO
		message[3] = 0;// TODO
		message[4] = 'Q';// TODO
		message[5] = 'U';// TODO
		message[6] = 'O';// TODO
		message[7] = 'T';// TODO
		byte[] dv = Base64Tool.decode(qi.getAttribute("DigestValue"));
		for (int i = 0; i < 20; i++)
			message[8 + i] = dv[i];
		byte[] ed = Base64Tool.decode(qi.getAttribute("ExternalData"));
		for (int i = 0; i < 20; i++)
			message[28 + i] = ed[i];
		// TODO Valdate ED if nonce from server

		NodeList nlSV = _ir.getElementsByTagName("SignatureValue");
		byte[] signature = Base64Tool.decode(nlSV.item(0).getTextContent());

		if (SignatureTool.validate(pubKey, message, signature) == false) {
			log.error("Wrong SIgnature but CONTINUE");
			// return 1; // INVALID
		}

		/* Validate PCR Composite <-> message */
		/* PcrSelection */

		NodeList nlPS = _ir.getElementsByTagName("PcrSelection");
		Element ps = (Element) nlPS.item(0);
		int sizeOfSelect = new Integer(ps.getAttribute("SizeOfSelect"));

		_pcrComposite = new PcrComposite();
		_pcrComposite.setPcrNumber(sizeOfSelect * 8); // TODO

		NodeList nlPV = _ir.getElementsByTagName("PcrValue");
		for (int i = 0; i < nlPV.getLength(); i++) {
			Element pv = (Element) nlPV.item(i);
			int pcrindex = new Integer(pv.getAttribute("PcrNumber"));
			_pcrComposite.selectPcr(pcrindex, Base64Tool.decode(pv
					.getTextContent()));
		}

		_pcrComposite.setExtraData(ed);

		if (_pcrComposite.validate(message) == false) {
			return 1; // INVALID
		}

		return 0; // VALID
		// return 2; // UNVERIFYED
	}

	static int MAX_FILE_COUNT = 2;

	private static void usage() {
		System.out.println("Usage: report [OPTIONS]");
		System.out.println(" OPTIONS");
		System.out.println(" --create --prop propfile --out irfile");
		System.out.println("     create IntegrityReport from IML");

	}

	public static void main(String[] args) {
		int opType = 0; // 1:create
		// int opMode = 0; // 0:platform 1:runtime
		// int fileCount = 0;
		// String[] inputFilename = new String[MAX_FILE_COUNT];
		String outputFilename = null;
		String propFilename = null;
		String pcrsFilename = null;

		for (int i = 0; i < args.length; ++i) {
			if ("--create".equals(args[i])) {
				opType = 1;
			} else if ("--prop".equals(args[i])) {
				propFilename = args[++i];
			} else if ("--out".equals(args[i])) {
				outputFilename = args[++i];
			} else {
				usage();
			}
		}

		try {
			switch (opType) {
			case 1: // create IR from IML
				/* Check Properties */
				String propDir = "";
				File file = new File(propFilename);
				if (file.exists() == false) {
					throw new Exception("IR gen properties file is not found "
							+ propFilename);
				} else {
					propDir = file.getParent();
				}
				/* Load Properties */
				Properties prop = new Properties();
				prop.load(new FileInputStream(propFilename));

				String[] imlFiles = {
						propDir + "/" + prop.getProperty("platform.iml"),// biosImlFilename,
						propDir + "/" + prop.getProperty("runtime.iml")// runtimeImlFilename};
				};
				String[] rmFiles = {
						propDir + "/" + prop.getProperty("platform.manifest"),// platformRmFilename,
						propDir + "/" + prop.getProperty("runtime.manifest"),// runtimeRmFilename
				};
				pcrsFilename = propDir + "/" + prop.getProperty("pcrs");

				// IntegrityReport ir = pts.createIRbyBFSM(
				// imlFiles,
				// biosBehaviorModelFilenames,
				// quotePropFilename);
				IntegrityReport ir1 = new IntegrityReport();
				ir1.generateIntegrityReportFromIML(imlFiles, pcrsFilename,
						rmFiles, null);

				System.out.println(ir1.toString());
				ir1.save(outputFilename);
				break;
			default:
				usage();
				break;

			}
		} catch (Exception e) {
			System.err.println("Internal Error");
			e.printStackTrace();
		}

		// TODO Auto-generated method stub

	}

	public NodeList getElementsByTagName(String string) {
		return _ir.getElementsByTagName(string);
	}

	public boolean validateXML() throws Exception {
		TCGSchemas schema = new TCGSchemas();
		return schema.validate(_filename);
	}

	public String getGipedBase64String() throws Exception {
		return XmlTool.getGipedBase64String(_doc);
	}

}
