/*
 * 
 * Licensed Materials - Property of IBM
 *
 * Open Platform Trust Services - An open source TCG PTS
 *
 * (C) Copyright International Business Machines Corp. 2007
 *
 */
package com.ibm.trl.tcg.pts.eventlog;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.security.NoSuchAlgorithmException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.ibm.trl.tcg.pts.tools.Base64Tool;
import com.ibm.trl.tcg.pts.tools.DigestTool;
import com.ibm.trl.tcg.pts.tools.HexTool;

/**
 * 
 * @author Seiji Munetoh
 * 
 */
public class RuntimeDigest {

	/* Logger */
	private Log log = LogFactory.getLog(this.getClass());

	private String _path = null;

	/**
	 * 
	 * byte[] mode
	 * 
	 * 
	 */
	public RuntimeDigest() {
		_path = null;
	}

	/**
	 * 
	 * File Image Mode
	 * 
	 * @param grubInstallPath
	 */
	public RuntimeDigest(String grubInstallPath) throws Exception {
		File f = new File(grubInstallPath);

		if (f.exists() == false) {
			throw new Exception("Grub Install path is not found. "
					+ grubInstallPath);
		} else if (f.isDirectory()) {
			_path = grubInstallPath;
		} else {
			_path = null;
		}
	}

	public String getDigest(String filename) throws Exception {
		String filename2;
		if (_path != null) {
			filename2 = _path + "/" + filename;
		} else {
			filename2 = filename;
		}

		File f = new File(filename2);
		if (f.isFile()) {
			// try {
			FileInputStream fis = new FileInputStream(filename2);
			InputStream in = new BufferedInputStream(fis);
			byte[] buf = new byte[512];
			int len;
			DigestTool sha1 = new DigestTool();

			sha1.init();
			while ((len = in.read(buf)) != -1) {
				sha1.update(len, buf);
			}

			byte[] byteout = sha1.digest();
			if (log.isDebugEnabled())
				log.debug("sha1 digest  " + HexTool.getHexString(byteout));
			String base64out = Base64Tool.encode(byteout);
			if (log.isDebugEnabled())
				log.debug("getDigest " + filename2 + " = " + base64out);
			fis.close();

			return base64out;
		} else {
			throw new Exception("File not found? " + filename2);
		}

		// return null;
	}

	/**
	 * 
	 * @param filename
	 * @param mode
	 *            0: 512 full 1: 440 2: 446
	 * @return
	 */
	public String getStage1Digest(String filename, int mode) throws Exception {
		if (_path != null) {
			String filename2 = _path + "/" + filename;
			File f = new File(filename2);
			if (f.isFile()) {
				InputStream in = new BufferedInputStream(new FileInputStream(
						filename2));
				byte[] buf = new byte[512];
				int len;
				len = in.read(buf);

				return getStage1Digest(buf, mode);

			} else {
				throw new Exception("Grub stage1, File not found?");
			}
		}

		return null;
	}

	/**
	 * 
	 * 
	 * 
	 * @param buf
	 * @param mode
	 *            0: 512 full 1: 440 2: 446
	 * @return
	 */
	public String getStage1Digest(byte[] buf, int mode) {
		try {
			DigestTool sha1 = new DigestTool();
			sha1.init();

			if (mode == 1) {
				sha1.update(440, buf);
			} else if (mode == 2) {
				sha1.update(446, buf);

			} else {
				sha1.update(512, buf);
			}
			byte[] byteout = sha1.digest();
			String base64out = Base64Tool.encode(byteout);

			if (log.isDebugEnabled()) {
				log.debug("masked MBR " + HexTool.getHexString(byteout)
						+ " len " + 446);
			}

			return base64out;
		} catch (NoSuchAlgorithmException e) {
			e.printStackTrace();
		}

		return null;
	}

	/**
	 * 
	 * Get Stage1 masked digest
	 * 
	 * Ref /pvs/src/com/ibm/trl/pvs/engine/ValidationModule.java
	 * validateMBR(String masked, Event e)
	 * 
	 * 
	 * @param string
	 * @throws Exception
	 */
	public String getStage1MaskedDigest(String filename) throws Exception {
		if (_path != null) {
			String filename2 = _path + "/" + filename;
			File f = new File(filename2);
			if (f.isFile()) {
				InputStream in = new BufferedInputStream(new FileInputStream(
						filename2));
				byte[] buf = new byte[512];
				int len;
				len = in.read(buf);

				return getStage1MaskedDigest(buf);

			} else {
				throw new Exception("Grub stage1, File not found?");
			}
		}

		return null;
	}

	/**
	 * 
	 * @param buf
	 * @return
	 */

	/*
	 * 
	 * BUF 512
	 * eb489000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000302ff000080010000000008faeb07f6c2807502b280ea597c000031c08ed88ed0bc0020fba0407c3cff740288c252be097de8cc00668b4c10be057cc644ff01668b1e447cc7041000c74402010066895c08c7440600706631c08944046689440cb442cd137205bb0070eb08be2a7de88f00eb4f60b800bbcd1a6685c0751566a37d7db801bbbf457dbe008e6631c96631d2cd1a618e06487c601eb900018edb31f631fffcf3a51f61ff26427cbe0f7de84e00eb0ebe147de84600eb06be1e7de83e00be237de83800ebfe47525542200047656f6d0048617264204469736b005265616400204572726f72004348534e41005443474552522000bb0100b40ecd10ac3c0075f4c31c00000000000700f00100000400000000000000617d000020000000040000000d000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000055aa
	 * MASK 480
	 * ffffff0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
	 * MASKED 446
	 * eb48900000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000faeb07f6c2807502b280ea597c000031c08ed88ed0bc0020fba0407c3cff740288c252be097de8cc00668b4c10be057cc644ff01668b1e447cc7041000c74402010066895c08c7440600706631c08944046689440cb442cd137205bb0070eb08be2a7de88f00eb4f60b800bbcd1a6685c0751566a37d7db801bbbf457dbe008e6631c96631d2cd1a618e06487c601eb900018edb31f631fffcf3a51f61ff26427cbe0f7de84e00eb0ebe147de84600eb06be1e7de83e00be237de83800ebfe47525542200047656f6d0048617264204469736b005265616400204572726f72004348534e41005443474552522000bb0100b40ecd10ac3c0075f4c31c00000000000700f00100000400000000000000617d000020000000040000000d00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
	 * 
	 * BUF 446
	 * eb4890d066bc007c00008ec08ed889e666bf0006000066b900010000f3a5ea2306000080fa807c0580fa877e02b2808816490766bfbe07000031f666b9040302ff000020010000000002fa9090f6c2807502b280ea597c000031c08ed88ed0bc0020fba0407c3cff740288c252be097de8cc00668b4c10be057cc644ff01668b1e447cc7041000c74402010066895c08c7440600706631c08944046689440cb442cd137205bb0070eb08be2a7de88f00eb4f60b800bbcd1a6685c0751566a37d7db801bbbf457dbe008e6631c96631d2cd1a618e06487c601eb900018edb31f631fffcf3a51f61ff26427cbe0f7de84e00eb0ebe147de84600eb06be1e7de83e00be237de83800ebfe47525542200047656f6d0048617264204469736b005265616400204572726f72004348534e41005443474552522000bb0100b40ecd10ac3c0075f4c31c00000000000700f00100000400000000000000617d000020000000040000000d000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000009ab206000000
	 * MASK 480
	 * ffffff0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
	 * MASKED 446
	 * eb48900000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000fa9090f6c2807502b280ea597c000031c08ed88ed0bc0020fba0407c3cff740288c252be097de8cc00668b4c10be057cc644ff01668b1e447cc7041000c74402010066895c08c7440600706631c08944046689440cb442cd137205bb0070eb08be2a7de88f00eb4f60b800bbcd1a6685c0751566a37d7db801bbbf457dbe008e6631c96631d2cd1a618e06487c601eb900018edb31f631fffcf3a51f61ff26427cbe0f7de84e00eb0ebe147de84600eb06be1e7de83e00be237de83800ebfe47525542200047656f6d0048617264204469736b005265616400204572726f72004348534e41005443474552522000bb0100b40ecd10ac3c0075f4c31c00000000000700f00100000400000000000000617d000020000000040000000d00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
	 * 
	 * 
	 */
	public String getStage1MaskedDigest(byte[] buf) {

		byte[] mask = {
				// =====================================================================================
				// This mask table is only work for GNU GRUB bootloader
				// =====================================================================================
				// 000-002 CODE (Jump)
				// notes) grub fills nop(90h) to 003
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				// =====================================================================================
				// 003-00A OEM ID/Windows name and version. (8 bytes)
				// http://www.microsoft.com/technet/archive/win98/reskit/part2/wrkc10.mspx?mfr=true
				// Table 10.6
				// http://www.microsoft.com/technet/prodtechnol/Windows2000Pro/reskit/part7/proch32.mspx?mfr=true
				// Table 32.6 Boot Sector Sections on a FAT16 Volume
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				// =====================================================================================
				// 00B-03D BPB
				// BIOS Parameter Block (58 bytes)
				// http://en.wikipedia.org/wiki/BIOS_parameter_block
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				// 010
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				// 020
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				// 030
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				// 038-03D
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				// =====================================================================================
				// 03E-04C Grub Parameter Block (12 bytes)
				(byte) 0x00,
				(byte) 0x00, // stage1 version
				(byte) 0x00, // boot drive
				(byte) 0x00, // force lba (=0)
				(byte) 0x00,
				(byte) 0x00, // stage2 address
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00, // stage2 sectors (0x8000)
				(byte) 0x00,
				(byte) 0x00, // stage2 segment 0x800
				// =====================================================================================
				// 04A-04C CODE
				(byte) 0xff, // FAh CLI => FAh cli
				// EBh 07h jmp 1f => 90h 90h nop
				// see the code at stage2/builtin.c line 2275
				// TODO sigh....
				// 04b, 04c
				(byte) 0x00,
				(byte) 0x00,
				// 04D-04F CODE
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				// 050-058 CODE
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				// 060
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				// 070
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				// 080
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				// 090
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				// 0A0
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				// 0B0
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				// 0C0
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				// 0E0
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				// 0F0
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				// 100
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				// 110
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				// 120
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				// 130
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				// 140
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				// 150
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				// 160
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				// 170
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				// 180 CODE
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				// 190 CODE
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				// 1A0 CODE
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				// 1B0 CODE
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				(byte) 0xff,
				// =====================================================================================
				// 1B8-1BD NT Magic
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				// =====================================================================================
				// 1BE-1BF Partition Table
				(byte) 0x00,
				(byte) 0x00,
				// 1C0-1CF Partition Table
				(byte) 0x00, (byte) 0x00, (byte) 0x00, (byte) 0x00,
				(byte) 0x00, (byte) 0x00, (byte) 0x00, (byte) 0x00,
				(byte) 0x00, (byte) 0x00, (byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				(byte) 0x00,
				// 1E0-1EF Partition Table
				(byte) 0x00, (byte) 0x00, (byte) 0x00, (byte) 0x00,
				(byte) 0x00, (byte) 0x00, (byte) 0x00, (byte) 0x00,
				(byte) 0x00, (byte) 0x00, (byte) 0x00, (byte) 0x00,
				(byte) 0x00, (byte) 0x00, (byte) 0x00,
				(byte) 0x00,
				// 1F0-1F8 Partition Table
				(byte) 0x00, (byte) 0x00, (byte) 0x00, (byte) 0x00,
				(byte) 0x00, (byte) 0x00, (byte) 0x00, (byte) 0x00,
				// 1F8-1FC Partition Table
				(byte) 0x00, (byte) 0x00, (byte) 0x00, (byte) 0x00,
				(byte) 0x00, (byte) 0x00,
				// =====================================================================================
				// 1FD-1FF signature aa55
				(byte) 0x00, (byte) 0x00, };

		// if (_path != null) {
		// String filename2 = _path + "/" + filename;
		// File f = new File(filename2);
		// if (f.isFile()) {
		try {
			// InputStream in = new BufferedInputStream(
			// new FileInputStream(filename2));
			// byte[] buf = new byte[512];
			// int len;
			DigestTool sha1 = new DigestTool();

			// len = in.read(buf);
			sha1.init();

			byte[] maskedMBR = new byte[446];
			for (int i = 0; i < 446; i++) {
				maskedMBR[i] = (byte) (mask[i] & buf[i]);
			}

			if (log.isTraceEnabled()) {
				log.trace("BUF " + buf.length + " "
						+ HexTool.getHexString((buf)));
				log.trace("MASK " + mask.length + " "
						+ HexTool.getHexString((mask)));
				log.trace("\nMASKED " + maskedMBR.length + " "
						+ HexTool.getHexString((maskedMBR)));
			}

			sha1.update(446, maskedMBR);
			byte[] byteout = sha1.digest();
			String base64out = Base64Tool.encode(byteout);

			if (log.isDebugEnabled()) {
				log.debug("masked MBR  " + HexTool.getHexString(byteout)
						+ " len " + 446);
			}

			return base64out;

			// } catch (FileNotFoundException e) { // FileInputStream
			// e.printStackTrace();
			// } catch (IOException e) { // in.read
			// e.printStackTrace();
		} catch (NoSuchAlgorithmException e) {
			e.printStackTrace();
		}
		// }
		// }

		return null;
	}

	/**
	 * 
	 * first 496 bytes of file
	 * 
	 * @param filename
	 * @return
	 * @throws Exception
	 */
	public String getStage15Digest(String filename) throws Exception {
		if (_path != null) {
			String filename2 = _path + "/" + filename;
			File f = new File(filename2);
			if (f.isFile()) {
				// try {
				FileInputStream fis = new FileInputStream(filename2);
				InputStream in = new BufferedInputStream(fis);
				byte[] buf = new byte[512];
				int len = 0;
				DigestTool sha1 = new DigestTool();

				sha1.init();
				len = in.read(buf);
				// TODO check len
				sha1.update(496, buf);
				byte[] byteout = sha1.digest();
				String base64out = Base64Tool.encode(byteout);

				if (log.isDebugEnabled()) {
					log.debug("1st 496 bytes " + HexTool.getHexString(byteout)
							+ " len " + 496);
					log.debug("getDigest " + filename2 + " = " + base64out);
				}

				fis.close();

				return base64out;

				// } catch (FileNotFoundException e) { // FileInputStream
				// e.printStackTrace();
				// } catch (IOException e) { // in.read
				// e.printStackTrace();
				// } catch (NoSuchAlgorithmException e) {
				// // TODO Auto-generated catch block
				// e.printStackTrace();
				// }
			} else {
				throw new Exception("Grub stage 1.5, File not found?");
			}
		}
		return null;
	}

	/**
	 * 
	 * skip first sector skip first 70x bytes in second sector add padding
	 * 
	 * @param string
	 * @return
	 * @throws Exception
	 */
	public String getStage15fsDigest(String filename) throws Exception {
		if (_path != null) {
			String filename2 = _path + "/" + filename;
			File f = new File(filename2);
			if (f.isFile()) {
				// try {
				InputStream in = new BufferedInputStream(new FileInputStream(
						filename2));
				byte[] buf = new byte[512];
				int len;
				int len2 = 0;
				DigestTool sha1 = new DigestTool();

				sha1.init();
				/* 1st sector - Skip */
				len = in.read(buf);
				/* 2nd sector */
				len = in.read(buf);
				sha1.update(buf, 0x70, 512 - 0x70);

				while ((len = in.read(buf)) != -1) {
					sha1.update(buf, 0, len);
					len2 = len;
				}
				/* padding */
				if (len2 != 512) {
					// log.debug("SM DEBUG padding len= " + len2);
					for (int i = 0; i < 512; i++)
						buf[i] = 0;
					sha1.update(buf, len2, 512 - len2);

				}

				byte[] byteout = sha1.digest();
				String base64out = Base64Tool.encode(byteout);
				// log.debug("SM DEBUG getDigest " + filename2 + "
				// = " + base64out);
				if (log.isDebugEnabled()) {
					log.debug("rest bytes " + HexTool.getHexString(byteout)
							+ " len " + len);
					log.debug("getDigest " + filename2 + " = " + base64out);
				}

				return base64out;

				// } catch (FileNotFoundException e) { // FileInputStream
				// e.printStackTrace();
				// } catch (IOException e) { // in.read
				// e.printStackTrace();
				// } catch (NoSuchAlgorithmException e) {
				// // TODO Auto-generated catch block
				// e.printStackTrace();
				// }
			} else {
				throw new Exception("Grub stage 1.5 fs, File not found?");
			}
		}
		return null;
	}

	public String getDigest(String path, String filename) throws Exception {
		_path = path;

		return getDigest(filename);
	}

	/**
	 * 
	 * @param knoppixPath
	 * @param filename
	 * @param mode
	 *            0 : TCG PC Client 1.2 spec 1st 512 bytes of Boot image -
	 *            Panasonic 1 : 1st 2048 bytes of Boot image - HP 2 : Lenovo
	 * @return
	 * @throws Exception
	 */
	public String getEltoritoDigest(String filename, int mode) throws Exception {

		String filename2 = _path + "/" + filename;
		File f = new File(filename2);
		if (f.isFile() == false) {
			throw new Exception("file not found : " + filename2);
		}

		/* setup data stream */
		FileInputStream fis = new FileInputStream(filename2);
		InputStream in = new BufferedInputStream(fis);
		int len;
		int len2 = 0;
		DigestTool sha1 = new DigestTool();

		/* calc digest */
		switch (mode) {
		case 0: {
			/* 1st 512 bytes */
			byte[] buf = new byte[512]; // Sector size
			sha1.init();
			len = in.read(buf);
			sha1.update(buf, 0, len);
			byte[] byteout = sha1.digest();
			String base64out = Base64Tool.encode(byteout);
			if (log.isDebugEnabled()) {
				log.debug("1st 512 bytes " + HexTool.getHexString(byteout)
						+ " len " + len);
				log.debug("getDigest " + filename2 + " = " + base64out);
			}
			return base64out;
		}
		case 1: {
			/* 1st 2048 bytes - HP */
			byte[] buf = new byte[2048]; // Sector size
			sha1.init();
			len = in.read(buf);
			sha1.update(buf, 0, len);
			byte[] byteout = sha1.digest();
			String base64out = Base64Tool.encode(byteout);
			if (log.isDebugEnabled()) {
				log.debug("1st 2048 bytes " + HexTool.getHexString(byteout)
						+ " len " + len);
				log.debug("getDigest " + filename2 + " = " + base64out);
			}
			return base64out;
		}
		case 2:
			log.error("IBM BootImage of Knoppix511trust");
			return "ctbTWBvIB+q6eIrySOAxKIaXNec=";
		default:
			break;
		}

		in.close();
		fis.close();

		return null;
	}

	// public void verbose() {
	// _verbose = true;
	// }

}
