/*
 * 
 * Licensed Materials - Property of IBM
 *
 * Open Platform Trust Services - An open source TCG PTS
 *
 * (C) Copyright International Business Machines Corp. 2007
 *
 */

package com.ibm.trl.tcg.pts.vulnerability.rpm;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.sql.SQLException;
import java.util.List;
import java.util.Vector;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.ibatis.sqlmap.client.SqlMapClient;
import com.ibm.trl.tcg.pts.ibatis.SqlConfigIidbCreater;
import com.ibm.trl.tcg.pts.ibatis.SqlConfigVul;
import com.ibm.trl.tcg.pts.ibatis.dto.CveDefinitions;
import com.ibm.trl.tcg.pts.ibatis.dto.Packages;
import com.ibm.trl.tcg.pts.vulnerability.oval.ReadOvalRedhat;

/**
 * Validate RPM package list (rpm -qa output) get OVAL/CVE info from IIDB get
 * CVSS info from VulDB.
 * 
 * @author Megumi Nakamura, Seiji Munetoh
 * 
 */
public class CheckRpmVulnerability {

	/* Logger */
	private Log log = LogFactory.getLog(this.getClass());

	// DB
	private SqlMapClient sqlMapVul = SqlConfigVul.getSqlMapInstance();

	private SqlMapClient sqlMapIidb = null;

	// File
	private FileWriter fileWriter = null;

	private BufferedWriter bufWriter = null;

	private String verboseHeader;

	private int unknownPackageNum = 0;

	private int obsoletePackageNum = 0;

	private int goodPackageNum = 0;

	private int totalPackageNum = 0;

	private int badPackageNum = 0;

	private int highSeverityNum = 0;

	private int mediumSeverityNum = 0;

	private int lowSeverityNum = 0;

	private static int lineNum = 0;

	/**
	 * Main method to start to load the xmlfile, parse them, and store into the
	 * database.
	 * 
	 * @param args
	 *            The options are properties, input file and output file.
	 */
	public static void main(String[] args) {
		String inputFile = null;
		String outputFile = null;
		CheckRpmVulnerability checker = new CheckRpmVulnerability(6);

		if (args.length == 0) {
			usage();
		} else {
			for (int i = 0; i < args.length; i++) {
				if ("--help".equals(args[i])) {
					usage();
				} else if ("--input".equals(args[i]) || "-i".equals(args[i])) {
					inputFile = args[i + 1];
					i++;
				} else if ("--output".equals(args[i]) || "-o".equals(args[i])) {
					outputFile = args[i + 1];
					i++;
				} else {
					usage();
				}
			}
			if (inputFile == null) {
				usage();
			}
			if (outputFile == null) {
				outputFile = inputFile + "-out.txt";
			}
		}
		try {
			checker.run(inputFile, outputFile);
			checker.finish();
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	public CheckRpmVulnerability(int dbIndex) {
		sqlMapIidb = SqlConfigIidbCreater.getSqlMapInstance(dbIndex);
	}

	/**
	 * Print the usage for main method.
	 */
	private static void usage() {
		System.out.println("usage: java Checker -i rpm-qa.txt -o output.txt");
		System.out.println("--input, -i: (required)");
		System.out.println("	Input file which contains the list of rpms");
		System.out.println("	$ rpm -qa > rpm-qa.txt");
		System.out.println("--output, -o:");
		System.out.println("	Output file");
		System.out.println(" ");
	}

	/**
	 * Running the all process of preparing the file, checking the version,
	 * writing down in the file.
	 * 
	 * @param inputFile
	 *            File path, This file contains the output from the command of
	 *            rpm -qa
	 * @param outputFile
	 *            File path, This file contains the result
	 * @throws Exception
	 */
	public void run(String inputFile, String outputFile) throws Exception {
		try {
			sqlMapVul.startTransaction();
			sqlMapIidb.startTransaction();

		} catch (SQLException ex) {
			do {
				log.error("SQLSTATE: " + ex.getSQLState());
				log.error("ERR-CODE: " + ex.getErrorCode());
				log.error("ERR-MSEG: " + ex.getMessage());
				ex = ex.getNextException();
			} while (null != ex);
		}

		/* Read RPM List */
		Vector rpms = null;
		try {
			rpms = getRPMSfromFile(inputFile);
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}

		/* Validate RPMs */
		writeFileOpen(outputFile);

		if (rpms != null) {
			int rpmNum = rpms.size();
			for (int i = 0; i < rpmNum; i++) {
				writeFileLineCountup();
				validatePackage((String) rpms.get(i));
			}
		}
	}

	/**
	 * Close the file stream.
	 */
	public void finish() {
		writeFileClose();

		try {
			sqlMapVul.commitTransaction();
			sqlMapIidb.commitTransaction();

		} catch (SQLException ex) {
			do {
				log.error("SQLSTATE: " + ex.getSQLState());
				log.error("ERR-CODE: " + ex.getErrorCode());
				log.error("ERR-MSEG: " + ex.getMessage());
				ex = ex.getNextException();
			} while (null != ex);
		}

		if (log.isDebugEnabled()) {
			log.debug("-------------------");
			log.debug("good           " + goodPackageNum);
			log.debug("bad            " + badPackageNum);
			log.debug("obsolute       " + obsoletePackageNum);
			log.debug("unknown        " + unknownPackageNum);
			log.debug("-------------------");
			log.debug("Total Package  " + totalPackageNum);
			log.debug("-------------------");
			log.debug("CVSS Severity");
			log.debug("  high         " + highSeverityNum);
			log.debug("  medium       " + mediumSeverityNum);
			log.debug("  low          " + lowSeverityNum);
			log.debug("-------------------");

		}
	}

	/**
	 * Read the file, and store each line into Vector.
	 * 
	 * @param inputFile
	 *            which contains the output from the command of rpm -qa
	 * @return Vector(of String lines)
	 * @throws FileNotFoundException
	 *             No file to be checked
	 */
	private Vector getRPMSfromFile(String inputFile)
			throws FileNotFoundException {
		Vector inputRpms = new Vector();
		try {
			BufferedReader bufReader = new BufferedReader(new FileReader(
					inputFile));
			String line = null;
			// lines -> line
			while ((line = bufReader.readLine()) != null) {
				inputRpms.add(line);
			}
			bufReader.close();

		} catch (IOException e) {
			e.printStackTrace();
		}
		return inputRpms;
	}

	/**
	 * Setup the file to write.
	 * 
	 * @param outputFile
	 *            which contains the result
	 */
	private void writeFileOpen(String outputFile) {
		try {
			fileWriter = new FileWriter(outputFile);
			bufWriter = new BufferedWriter(fileWriter);
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Close the file.
	 */
	private void writeFileClose() {
		try {
			bufWriter.flush();
			bufWriter.close();
			fileWriter.close();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 * The number written in the file. Rpm order.
	 */
	private void writeFileLineCountup() {
		lineNum++;
	}

	/**
	 * Write one line with "package.i.".
	 * 
	 * @param line
	 *            The value that follows after package.i.
	 */
	private void writeFileLine(String line) {
		try {
			if (line != null) {
				bufWriter.write("package." + lineNum + "." + line);
				bufWriter.newLine();
			}
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Get the information about cve_id from the vulnerability database.
	 * 
	 * @param cveId
	 *            The CVE ID
	 */
	private void getInfoFromVULDB(String cveId) {
		try {
			CveDefinitions cveDef = (CveDefinitions) sqlMapVul.queryForObject(
					"getCveDefinitionByCveId", cveId);
			String nvdSeverity = cveDef.getNvdSeverity();
			String nvdCvssScore = cveDef.getNvdCvssScore();
			String nvdDescription = cveDef.getNvdDescription();
			String nvdSolution = cveDef.getNvdSolution();

			if (fileWriter != null && bufWriter != null) {
				writeFileLine("vulnerability.cve=" + cveId);
				writeFileLine("vulnerability.cvss.severity=" + nvdSeverity);
				writeFileLine("vulnerability.cvss.score=" + nvdCvssScore);
				writeFileLine("vulnerability.cvss.description="
						+ nvdDescription);
				writeFileLine("vulnerability.cvss.solution=" + nvdSolution);
			}

			if (nvdSeverity != null) {
				if (nvdSeverity.equals("High")) {
					highSeverityNum++;
				} else if (nvdSeverity.equals("Medium")) {
					mediumSeverityNum++;
				} else if (nvdSeverity.equals("Low")) {
					lowSeverityNum++;
				}
			}
			if (log.isDebugEnabled()) {
				log.debug("CVE             : " + cveId);
				log.debug("CVSS Level      : " + nvdSeverity);
				log.debug("CVSS Score      : " + nvdCvssScore);
				log.debug("CVSS Description: " + nvdDescription);
				log.debug("CVSS Solution   : " + nvdSolution);
			}
		} catch (SQLException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Search packages(package_id) from database, and get oval/cve information.
	 * 
	 * @param rpmName The RPM name, one line of rpm -qa
	 */
	private void validatePackage(String rpmName) {
		String[] rpmInfo = RpmVersionTool.parseRpmName(rpmName);
		String name = rpmInfo[0];
		String version = rpmInfo[1];
		String release = rpmInfo[2];

		validatePackage(name, version, release);
	}

	/**
	 * Search packages(package_id) from database, and get oval/cve information.
	 * 
	 * @param name	Package name
	 * @param version Package version
	 * @param release Package release number
	 * @return cve
	 */
	protected String validatePackage(String name, String version, String release) {
		String oval = null;
		String cve = null;
		boolean vul = false;
		String remediation = null;

		if (fileWriter != null && bufWriter != null) {
			writeFileLine("name=" + name);
			writeFileLine("version=" + version);
			writeFileLine("release=" + release);
		}

		totalPackageNum++;
		try {
			/* Get package_id from IIDB */
			int packageId = 0;
			String rpmVer = version + "-" + release;

			Packages pkgParam = new Packages();
			pkgParam.setPackageName(name);
			pkgParam.setPackageVersion(rpmVer);
			Packages pkgs = (Packages) sqlMapIidb.queryForObject(
					"getPackageByNameVersion", pkgParam);

			/* Just check the first package, ignore other archs */
			if (pkgs != null) {
				packageId = pkgs.getPackageId();
			}
			if (packageId == 0) {
				/* no hit, ckeck other versions */
				boolean hit = false;
				vul = false;
				int vulNum = 0;
				remediation = "unknown package/version. ";

				/* ignore version info, get all packages by name */
				List<Packages> pkgs2 = (List<Packages>) sqlMapIidb
						.queryForList("getPackageByName", name);
				for (Packages p : pkgs2) {
					packageId = p.getPackageId();
					String pkgVersion = p.getPackageVersion();
					int vulnerability = p.getPackageVulnerability();
					String tmpOval = p.getPackageOval();
					String tmpCve = p.getPackageCve();
					hit = true;

					if ((vulnerability == ReadOvalRedhat.VUL_DISTRO_MATCH)
							|| (vulnerability == ReadOvalRedhat.VUL_DISTRO_NOT_MATCH)
							|| (vulnerability == ReadOvalRedhat.CHECK_AGAIN
									+ ReadOvalRedhat.VUL_DISTRO_MATCH)
							|| (vulnerability == ReadOvalRedhat.CHECK_AGAIN
									+ ReadOvalRedhat.VUL_DISTRO_NOT_MATCH)) {
						/* Compare Version */
						int rtn = RpmVersionTool.compareTwoRpmVersion(
								pkgVersion, version, release);
						if (rtn == RpmVersionTool.CHECK_AGAIN) {
							remediation += "the data of version " + pkgVersion
									+ " has to be checked again. ";
						} else if (rtn == RpmVersionTool.GREATER
								|| rtn == RpmVersionTool.EQUAL) {
							vul = true;
							if (oval == null) {
								oval = "";
							}
							if (cve == null) {
								cve = "";
							}
							oval += tmpOval + "//";
							cve += tmpCve + " ";
							vulNum++;
							remediation += "version " + pkgVersion
									+ " has vul,";
						} else if (rtn == RpmVersionTool.LESS) {
							remediation += "version " + pkgVersion
									+ " had vul. but older than this pkg!,";
						}
					} else {
						vul = false;
						remediation += "version " + pkgVersion + " ok!,";
					}
				}

				// TODO
				if (vulNum > 0) {
					vul = true;
				}

				if (hit && vul) {
					badPackageNum++;
				} else if (hit && !vul) {
					goodPackageNum++;
				} else {
					unknownPackageNum++;
				}
			} else {
				/* IIDB just knows this package+version */
				/* Get OVAL&CVE info from IIDB */
				Packages pkgs3 = (Packages) sqlMapIidb.queryForObject(
						"getPackage", packageId);
				if (pkgs3 != null) {
					int vulnerability = pkgs3.getPackageVulnerability();
					int obsolete = pkgs3.getPackageObsolete();
					oval = pkgs3.getPackageOval();
					cve = pkgs3.getPackageCve();

					if (vulnerability == 0) { // Unknown
						vul = false;
						if (obsolete == 1) {
							// old version, new version exist
							remediation = "There is no known vulnerability "
									+ "but new version is available";
							obsoletePackageNum++;
						} else {
							goodPackageNum++;
						}
					} else if (vulnerability == ReadOvalRedhat.NO_VUL_DISTRO_MATCH) {
						vul = false;
						goodPackageNum++;
					} else if (vulnerability == ReadOvalRedhat.NO_VUL_DISTRO_NOT_MATCH) {
						vul = false;
						goodPackageNum++;
					} else if (vulnerability == ReadOvalRedhat.VUL_DISTRO_MATCH) {
						vul = true;
						badPackageNum++;
					} else if (vulnerability == ReadOvalRedhat.VUL_DISTRO_NOT_MATCH) {
						vul = true;
						badPackageNum++;
					} else if (vulnerability == ReadOvalRedhat.CHECK_AGAIN) {
						// Check again
						vul = true;
						remediation = "Check by yourself";
						badPackageNum++;
					} else {
						vul = true;
						remediation = "Internal error (vulnerability="
								+ vulnerability + ")";
						badPackageNum++;
					}
					// rs.first()
				} else {
					// Unknown
					vul = false;
					remediation = "Unknown package. sorry";
					unknownPackageNum++;
				}
			} // pcakage_id
		} catch (SQLException e) {
			e.printStackTrace();
		}

		/* Output */
		if (vul) {
			if (log.isDebugEnabled()) {
				log.debug("Vulnerability found");
				log.debug(verboseHeader);
				log.debug("package name    : " + name);
				log.debug("package version : " + version);
				log.debug("package release : " + release);
				log.debug("remediation     : " + remediation);
				log.debug("OVAL            : " + oval);
				log.debug("CVE             : " + cve);
				// TODO add desc sol
			}
			if (fileWriter != null && bufWriter != null) {
				writeFileLine("vulnerability=yes");
				writeFileLine("remediation=" + remediation);
				writeFileLine("vulnerability.oval=" + oval);
				writeFileLine("vulnerability.cve=" + cve);
			}
			if (cve != null) {
				String[] cves = cve.split(" ");
				for (int i = 0; i < cves.length; i++) {
					getInfoFromVULDB(cves[i].trim());
				}
			}

		} else {
			if (fileWriter != null && bufWriter != null) {
				writeFileLine("vulnerability=no");
				if (remediation != null) {
					writeFileLine("remediation=" + remediation);
				}
			}
		}
		return cve;
	}
}
