/*
 * 
 * Licensed Materials - Property of IBM
 *
 * Open Platform Trust Services - An open source TCG PTS
 *
 * (C) Copyright International Business Machines Corp. 2007
 *
 */
package com.ibm.trl.tcg.pts.vulnerability.deb;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.sql.SQLException;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.ibatis.sqlmap.client.SqlMapClient;
import com.ibm.trl.tcg.pts.ibatis.SqlConfigIidbCreater;
import com.ibm.trl.tcg.pts.ibatis.dto.Measurements;
import com.ibm.trl.tcg.pts.ibatis.dto.Packages;
import com.ibm.trl.tcg.pts.vulnerability.tool.DbOperations;

/**
 * Copy data files (.metadata, .sha1.filelist, .md5.filelist) to database. These
 * files are made by perl scripts of tools/deb/(deb-file.pl, deb-meta.pl)
 * 
 * @author Megumi Nakamura
 */
public class CopyActualDebPackageData2DB {

	/* Logger */
	private Log log = LogFactory.getLog(this.getClass());

	private SqlMapClient sqlMapIidb = null;

	private int dbIndex;

	/**
	 * @param dbIndex Index of Integrity Info DB
	 */
	public CopyActualDebPackageData2DB(int dbIndex) {
		this.dbIndex = dbIndex;
		sqlMapIidb = SqlConfigIidbCreater.getSqlMapInstance(dbIndex);
	}

	/**
	 * Run.
	 * @param dirPath Directory path of metadata and filelist 
	 * @throws Exception 
	 */
	public void run(String dirPath) throws Exception {
		run(dirPath, true);
	}
	
	/**
	 * Run.
	 * @param dirPath Directory path of metadata and filelist
	 * @param clearMeasurementsObsolete To skip making clean obsolete flag before updating 
	 * @throws Exception
	 */
	public void run(String dirPath, boolean clearMeasurementsObsolete) throws Exception {
		try {
			sqlMapIidb.startTransaction();
		} catch (SQLException e) {
			e.printStackTrace();
		}

		getFileFromDir(dirPath);

		new DbOperations().updateObsolete(dbIndex, clearMeasurementsObsolete);
	}

	/**
	 * Start from Metadata directory.
	 * 
	 * @param dirPath Directory path of metadata and filelist
	 */
	private void getFileFromDir(String dirPath) throws Exception {
		File dir = new File(dirPath);
		File[] files = dir.listFiles();

		if (files == null) {
			throw new FileNotFoundException("PackageData dir " + dirPath
					+ " is not found.");
		}

		for (int i = 0; i < files.length; i++) {
			File file = files[i];
			if (file.getName().endsWith(".metadata")) {
				if (file.exists() && file.canRead() && file.isFile()) {
					getMetadata(file);
				}
			}
		}
	}

	/**
	 * Read Metadata File and Insert into database.
	 * 
	 * @param file Metadata file
	 */
	private void getMetadata(File file) {
		Packages packages = new Packages();
		try {
			String pkgName = null;
			String pkgVersion = null;

			FileReader fileReader = new FileReader(file);
			BufferedReader bufReader = new BufferedReader(fileReader);
			String line = null;
			while ((line = bufReader.readLine()) != null) {
				if (line.startsWith("Package:")) {
					pkgName = line.substring(8).trim();
					packages.setPackageName(pkgName);

				} else if (line.startsWith("Version:")) {
					pkgVersion = line.substring(8).trim();
					packages.setPackageVersion(pkgVersion);

				} else if (line.startsWith("Architecture:")) {
					packages.setPackageArch(line.substring(13).trim());

				} else if (line.startsWith("Maintainer:")) {
					if (line.indexOf("debian") > 0) {
						packages.setPackageVendor("Debian");
					} else {
						packages.setPackageVendor("Unknown");
					}

				} else if (line.startsWith("Description:")) {
					if (line.substring(12).trim().length() > 255) {
						packages.setPackageSummary(line.substring(12, 255)
								.trim());
					} else {
						packages.setPackageSummary(line.substring(12).trim());
					}

				}
			}
			bufReader.close();
			fileReader.close();

			// Insert into database

			// If not exist, Insert. If exist, Update.
			List<Packages> pkgs = (List<Packages>) sqlMapIidb.queryForList(
					"getPackageByNameVerArchBuild", packages);
			if (pkgs.size() > 0) {
				for (Packages p : pkgs) {
					int packageId = p.getPackageId();
					packages.setPackageId(packageId);
				}
				sqlMapIidb.update("updatePackageById", packages);
				sqlMapIidb.commitTransaction();

			} else {
				sqlMapIidb.insert("insertPackage", packages);
				sqlMapIidb.commitTransaction();
				int packageId = ((Packages)sqlMapIidb.queryForObject("getPackageIdInserted", packages)).getPackageId();
				// Measurements
				String sha1filename = file.getPath().replaceAll(".metadata",
						".sha1.filelist");
				String md5filename = file.getPath().replaceAll(".metadata",
						".md5.filelist");
				getMeasurements(sha1filename, packageId);
				getMeasurements(md5filename, packageId);
			}

		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		} catch (SQLException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Get Measurements and Insert into database.
	 * 
	 * @param filename Filelist file
	 * @param packageId Package ID of filelist's parent package
	 */
	private void getMeasurements(String filename, int packageId) {
		try {
			File file = new File(filename);
			BufferedReader bufReader = null;
			if (file.exists() && file.canRead() && file.isFile()) {
				bufReader = new BufferedReader(new FileReader(file));
				Measurements measurements = new Measurements();
				String line = null;
				while ((line = bufReader.readLine()) != null) {
					String[] digestPath = line.split(" +");
					measurements.setDigest(digestPath[0]);
					measurements.setDigestName(digestPath[1]);

					if (filename.indexOf("md5") > 0) {
						measurements.setDigestAlg("md5");
					} else if (filename.indexOf("sha1") > 0) {
						measurements.setDigestAlg("sha1");
					}
					measurements.setDigestPackageId(packageId);

					sqlMapIidb.insert("insertDigest", measurements);
					sqlMapIidb.commitTransaction();
				}
			}
			if (bufReader != null) {
				bufReader.close();
			}
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		} catch (SQLException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Main.
	 * 
	 * @param args --dbindex, --inputdir, --notclear
	 */
	public static void main(String[] args) {
		String inputDir = null;
		boolean clearMeasurementsObsolete = true;
		int dbIndex = 0;
		
		for (int i = 0; i < args.length; ++i) {
			if ("--dbindex".equals(args[i])) {
				dbIndex = Integer.valueOf(args[++i]);
			} else if ("--inputdir".equals(args[i]) || "-i".equals(args[i])) {
				inputDir = args[++i];
			} else if ("--notclear".equals(args[i]) || "-n".equals(args[i])) {
				clearMeasurementsObsolete = false;
			} else {
				System.err.println("Unknown option " + args[i]);
				usage();
				return;
			}
		}
		try {
			new CopyActualDebPackageData2DB(dbIndex).run(inputDir, clearMeasurementsObsolete);

		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	/**
	 * Print the usage for main method.
	 */
	private static void usage() {
		System.out.println("usage: java debimport --dbindex 1 -i testdata/debian/all");
		System.out.println("--dbindex: (required)");
		System.out.println("--inputdir, -i: (required)");
		System.out.println("	Input file which contains the metadata files");
		System.out.println("--notclear");
	}

}
