/*
 * 
 * Licensed Materials - Property of IBM
 *
 * Open Platform Trust Services - An open source TCG PTS
 *
 * (C) Copyright International Business Machines Corp. 2007
 *
 */
package com.ibm.trl.tcg.pts.vulnerability.nvdcve;

import java.io.IOException;
import java.sql.SQLException;
import java.sql.Timestamp;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

import com.ibatis.sqlmap.client.SqlMapClient;
import com.ibm.trl.tcg.pts.ibatis.SqlConfigVul;
import com.ibm.trl.tcg.pts.ibatis.dto.CveDefinitions;
import com.ibm.trl.tcg.pts.vulnerability.tool.GetFileFromInternet;

/**
 * This class gets the information of the NVD CVE file, and stores the cve score
 * into the database.
 * 
 * @author Megumi Nakamura
 */
public class ReadNvdCveSAX extends DefaultHandler {

	/* Logger */
	private Log log = LogFactory.getLog(this.getClass());

	private String cveId = null; // entry attribute (required)

	private Timestamp published = null; // entry attribute (required)

	private Timestamp modified = null; // entry attribute

	private String severity = null; // entry attribute

	private String cvssScore = null; // entry attribute

	private String cvssVector = null; // entry attribute

	private String descript = null; // desc->descript contents

	private String sol = null; // sols->sol contents

	private String temp = null;

	private String xmlFile = null;

	private static final int DES_LIMIT = 3000;

	private static final int SOL_LIMIT = 500;

	private SqlMapClient sqlMap = SqlConfigVul.getSqlMapInstance();

	/**
	 * Main method to start to load the xmlfile, parse them, and store into the
	 * database.
	 * 
	 * @param args --xmlfile, --outputdir
	 */
	public static void main(String[] args) {
		/* Default value */
		String xmlFile = null;
		String outputDir = null;
		// xmlFile = "http://nvd.nist.gov/download/nvdcve-modified.xml";
		// xmlFile = "http://nvd.nist.gov/download/nvdcve-recent.xml";
		// xmlFile = "http://nvd.nist.gov/download/nvdcve-2007.xml";
		// xmlFile = "http://nvd.nist.gov/download/nvdcve-2006.xml";
		// xmlFile = "http://nvd.nist.gov/download/nvdcve-2005.xml";
		// xmlFile = "http://nvd.nist.gov/download/nvdcve-2004.xml";
		// xmlFile = "http://nvd.nist.gov/download/nvdcve-2003.xml";
		// xmlFile = "http://nvd.nist.gov/download/nvdcve-2002.xml";

		for (int i = 0; i < args.length; ++i) {
			if ("--xmlfile".equals(args[i]) || "-x".equals(args[i])) {
				xmlFile = args[++i];
			} else if ("--outputdir".equals(args[i]) || "-d".equals(args[i])) {
				outputDir = args[++i];
			} else {
				System.err.println("Unknown option " + args[i]);
				usage();
				return;
			}
		}
		try {
			ReadNvdCveSAX cve = new ReadNvdCveSAX(xmlFile, outputDir);
			cve.run();
			cve.finish();
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	/**
	 * Print the usage for main method.
	 */
	private static void usage() {
		System.out.println("usage: cve [option]");
		System.out.println("--xmlfile, -x:");
		System.out.println("	XML file path or url of NVD/CVE file");
		System.out.println("--outputdir, -d (optional):");
		System.out.println("	Directory to store the xml file");
		System.exit(0);
	}

	/**
	 * This is used for calling SAXParser.
	 */
	public ReadNvdCveSAX() {
	}

	/**
	 * Prepare xml file.
	 * 
	 * @param file
	 *            The file path or url of the NVD xml data.
	 * @throws Exception
	 *             All exceptions
	 */
	public ReadNvdCveSAX(String file) throws Exception {
		if (file.startsWith("http:") || file.startsWith("ftp:")) {
			xmlFile = new GetFileFromInternet().getFile(file, null, true);
		} else {
			xmlFile = file;
		}
		if (log.isDebugEnabled()) {
			log.debug("XML File: " + xmlFile);
		}

		try {
			sqlMap.startTransaction();
		} catch (SQLException ex) {
			do {
				log.error("SQLSTATE: " + ex.getSQLState());
				log.error("ERR-CODE: " + ex.getErrorCode());
				log.error("ERR-MSEG: " + ex.getMessage());
				ex = ex.getNextException();
			} while (null != ex);
		}
	}

	/**
	 * Prepare xml file.
	 * 
	 * @param file The file path or url of the NVD xml data.
	 * @param dir Directory for downloaded file
	 * @throws Exception  All exceptions
	 */
	public ReadNvdCveSAX(String file, String dir) throws Exception {
		if (file.startsWith("http:") || file.startsWith("ftp:")) {
			xmlFile = new GetFileFromInternet().getFile(file, dir, true);
		} else {
			xmlFile = file;
		}
		if (log.isDebugEnabled()) {
			log.debug("XML File: " + xmlFile);
		}
		try {
			sqlMap.startTransaction();
		} catch (SQLException ex) {
			do {
				log.error("SQLSTATE: " + ex.getSQLState());
				log.error("ERR-CODE: " + ex.getErrorCode());
				log.error("ERR-MSEG: " + ex.getMessage());
				ex = ex.getNextException();
			} while (null != ex);
		}

	}

	/**
	 * Running the all process of preparing the file, parsing the xml, inserting
	 * into the database.
	 */
	public void run() {
		readXmlSax(xmlFile);
	}

	/**
	 * Close the database connection.
	 */
	public void finish() {
		try {
			sqlMap.commitTransaction();
		} catch (SQLException ex) {
			do {
				log.error("SQLSTATE: " + ex.getSQLState());
				log.error("ERR-CODE: " + ex.getErrorCode());
				log.error("ERR-MSEG: " + ex.getMessage());
				ex = ex.getNextException();
			} while (null != ex);
		}

		if (log.isDebugEnabled()) {
			log.debug("# END");
		}
	}

	/**
	 * Read XML File, and start parser.
	 * 
	 * @param file
	 *            The file path or url of the NVD xml data.
	 */
	protected void readXmlSax(String file) {
		if (log.isDebugEnabled()) {
			log.debug("Reading XML File.");
		}
		try {
			SAXParserFactory factory = SAXParserFactory.newInstance();
			SAXParser parser = factory.newSAXParser();
			parser.parse(file, new ReadNvdCveSAX());
		} catch (ParserConfigurationException e) {
			e.printStackTrace();
		} catch (SAXException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 * The event called at starting the document.
	 */
	public void startDocument() {
	}

	/**
	 * The event called at starting the element.
	 * 
	 * @param uri
	 *            namespace uri
	 * @param name
	 *            local name
	 * @param qualifiedName
	 *            qname
	 * @param attributes
	 *            atts
	 */
	public void startElement(String uri, String name, String qualifiedName,
			Attributes attributes) {
		if (qualifiedName.equalsIgnoreCase("entry")) {
			int len = attributes.getLength();

			for (int i = 0; i < len; i++) {
				if (attributes.getQName(i).equalsIgnoreCase("name")) {
					cveId = filter(attributes.getValue(i));
				} else if (attributes.getQName(i).equalsIgnoreCase("seq")) {
					cveId = filter("CVE-" + attributes.getValue(i));
				} else if (attributes.getQName(i).equalsIgnoreCase("published")) {
					published = published.valueOf(filter(attributes.getValue(i)
							+ " 00:00:00"));
				} else if (attributes.getQName(i).equalsIgnoreCase("modified")) {
					modified = modified.valueOf(filter(attributes.getValue(i)
							+ " 00:00:00"));
				} else if (attributes.getQName(i).equalsIgnoreCase("severity")) {
					severity = filter(attributes.getValue(i));
				} else if (attributes.getQName(i)
						.equalsIgnoreCase("cvss_score")
						|| attributes.getQName(i).equalsIgnoreCase(
								"CVSS_base_score")) {
					cvssScore = filter(attributes.getValue(i));
				} else if (attributes.getQName(i).equalsIgnoreCase(
						"cvss_vector")) {
					cvssVector = filter(attributes.getValue(i));
				}
			}
			// } else if (qualifiedName.equalsIgnoreCase("descript")) {
			// } else if (qualifiedName.equalsIgnoreCase("sol")) {
		}
	}

	/**
	 * The event called at ending the element.
	 * 
	 * @param uri
	 *            namespace uri
	 * @param name
	 *            local name
	 * @param qualifiedName
	 *            qname
	 */
	public void endElement(String uri, String name, String qualifiedName) {
		if (qualifiedName.equalsIgnoreCase("entry")) {
			dbentry();
			clearString();
		} else if (qualifiedName.equalsIgnoreCase("descript")) {
			descript = filter(temp);
			temp = null;
		} else if (qualifiedName.equalsIgnoreCase("sol")) {
			sol = filter(temp);
			temp = null;
		}
	}

	/**
	 * The event called at ending the document.
	 */
	public void endDocument() {
	}

	/**
	 * The event called at reading the characters.
	 * 
	 * @param ch
	 *            Character in the xml
	 * @param start
	 *            The starting location
	 * @param length
	 *            The length of characters to be read
	 */
	public void characters(char[] ch, int start, int length) {
		String now = new String(ch, start, length);
		if (temp != null) {
			temp = temp + now;
		} else {
			temp = now;
		}
	}

	/**
	 * Clear the value stored in String.
	 */
	private void clearString() {
		cveId = null; // entry attribute (required)
		published = null; // entry attribute (required)
		modified = null; // entry attribute
		severity = null; // entry attribute
		cvssScore = null; // entry attribute
		cvssVector = null; // entry attribute
		descript = null; // desc->descript contents
		sol = null; // sols->sol contents
		temp = null;
	}

	/**
	 * Print the elements to display for test.
	 */
	private void printEntry() {
		System.out.println("cve_id:\t" + cveId);
		System.out.println("published:\t" + published);
		System.out.println("modified:\t" + modified);
		System.out.println("severity:\t" + severity);
		System.out.println("cvss_score:\t" + cvssScore);
		System.out.println("cvss_vector:\t" + cvssVector);
		System.out.println("descript:\t" + descript);
		System.out.println("sol:\t" + sol);
	}

	/**
	 * Add one entry to database.
	 */
	private void dbentry() {
		try {
			// 2007-2-12 SM the size of biggest one is 2710, now
			if (descript == null) {
				descript = "n/a";
			}
			if (descript.length() > DES_LIMIT) {
				if (log.isDebugEnabled()) {
					log.debug("NVD description is too big, size = "
							+ descript.length());
				}
				descript = descript.substring(0, DES_LIMIT);
			}
			if (sol == null) {
				sol = "n/a";
			}
			if (sol.length() > SOL_LIMIT) {
				if (log.isDebugEnabled()) {
					log
							.debug("NVD solution is too big, size = "
									+ sol.length());
				}
				sol = sol.substring(0, SOL_LIMIT);
			}

			CveDefinitions cveDef = new CveDefinitions();
			cveDef.setCveId(cveId);
			cveDef.setCveUrl("http://cve.mitre.org/cgi-bin/cvename.cgi?name=" + cveId);
			cveDef.setCvePublishedOn(published);
			cveDef.setNvdSeverity(severity);
			cveDef.setNvdCvssScore(cvssScore);
			cveDef.setNvdCvssVector(cvssVector);
			cveDef.setNvdDescription(descript);
			cveDef.setNvdSolution(sol);
			if (modified != null) {
				cveDef.setCveModifiedOn(modified);
			}
			sqlMap.insert("insertCveDefinitionNvd", cveDef);
			sqlMap.commitTransaction();

		} catch (SQLException ex) {
			do {
				log.error("SQLSTATE: " + ex.getSQLState());
				log.error("ERR-CODE: " + ex.getErrorCode());
				log.error("ERR-MSEG: " + ex.getMessage());
				ex = ex.getNextException();
			} while (null != ex);
		}
	}

	/**
	 * remove ' " from sql.
	 * 
	 * @param line
	 *            String
	 * @return String removed the specific characters
	 */
	private String filter(String line) {
		if (line != null) {
			return line.replaceAll("['\"]", "");
		} else {
			return null;
		}
	}

}
