/*
 * 
 * Licensed Materials - Property of IBM
 *
 * Open Platform Trust Services - An open source TCG PTS
 *
 * (C) Copyright International Business Machines Corp. 2007
 *
 */
package com.ibm.trl.tcg.pts.integrity;

// import java.io.BufferedWriter;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.security.KeyPair;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;
import java.util.UUID;
import java.util.zip.GZIPInputStream;

import javax.xml.crypto.dsig.CanonicalizationMethod;
import javax.xml.crypto.dsig.DigestMethod;
import javax.xml.crypto.dsig.Reference;
import javax.xml.crypto.dsig.SignatureMethod;
import javax.xml.crypto.dsig.SignedInfo;
import javax.xml.crypto.dsig.Transform;
import javax.xml.crypto.dsig.XMLSignature;
import javax.xml.crypto.dsig.XMLSignatureFactory;
import javax.xml.crypto.dsig.dom.DOMSignContext;
import javax.xml.crypto.dsig.dom.DOMValidateContext;
import javax.xml.crypto.dsig.keyinfo.KeyInfo;
import javax.xml.crypto.dsig.keyinfo.KeyInfoFactory;
import javax.xml.crypto.dsig.keyinfo.KeyValue;
import javax.xml.crypto.dsig.keyinfo.X509Data;
import javax.xml.crypto.dsig.spec.C14NMethodParameterSpec;
import javax.xml.crypto.dsig.spec.TransformParameterSpec;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.XMLReader;

import com.ibm.trl.tcg.pts.engine.Event;
import com.ibm.trl.tcg.pts.engine.FiniteStateMachine;
import com.ibm.trl.tcg.pts.eventlog.IML;
import com.ibm.trl.tcg.pts.eventlog.RuntimeDigest;
import com.ibm.trl.tcg.pts.eventlog.SMBIOS;
import com.ibm.trl.tcg.pts.integrity.iidb.IIDB;
import com.ibm.trl.tcg.pts.tools.Base64Tool;
import com.ibm.trl.tcg.pts.tools.HexTool;
import com.ibm.trl.tcg.pts.tools.XmlTool;

/**
 * 
 * Generate RIMM by using DOM
 * 
 * 2007-06-04 Munetoh - copy from RIMM.java - TCG RM + UML2 Stage Diagram
 * 
 * for GCJ Client
 *  
 * @author Seiji Munetoh
 * 
 */
public class ReferenceManifest {
	
	/* Logger */
	private Log log = LogFactory.getLog(this.getClass());
	
	private String _namespace = "http://www.trustedcomputinggroup.org/XML/SCHEMA/1_0/rimm#";

	private Element _manifest;

	private String _head;

	private String _tail;

	private String _ComponentID;

	private String _DigestMethod;

	private int cid = 0;

	// private Vector _c;

	private Document _doc;

	private String _filename;

	// private String _schemasPath="../schemas/";
	//private String _schemasPath = "../../pvs/tcg/schemas/";

	//private String[] _schemas = {
	//		_schemasPath + "SimpleObject_v8_ibm.xsd",
	//		_schemasPath + "Core_Integrity_Manifest_v16_ibm.xsd",
	//		_schemasPath + "Reference_Integrity_Measurement_Manifest_v13_ibm.xsd",
	//		_schemasPath + "xmldsig-core-schema.xsd" };
	//private TCGSchemas _tcgSchemas;
	
	//private boolean _verbose = false;

	private boolean _indent = false;

	private Element _cid;

	private String _simpleName;

	private int _signatureOption = 1; // 0: pubkey, 1: X509

	private Element _cai;

	private Element _vms;
	
	// private Element _cv;
	
	/* Select IIDB */
	private int dbIndex = 0 ;

	public ReferenceManifest() {
	}
	
	public ReferenceManifest(String xmlFilename) throws Exception {
		loadFile(xmlFilename);
	}
	
	public ReferenceManifest(String xmlFilename, int option) throws Exception {
		if (option == 1) {
			loadGzip(xmlFilename);
		}
		else {
			loadFile(xmlFilename);			
		}
	}

	/**
	 * Create new RIMM
	 */
	public void init() {
		// http://java.sun.com/j2se/1.5.0/ja/docs/ja/api/java/util/UUID.html
		UUID uuid = UUID.randomUUID();

		try {
			DocumentBuilderFactory factory = DocumentBuilderFactory
					.newInstance();
			DocumentBuilder builder = factory.newDocumentBuilder();
			DOMImplementation domImpl = builder.getDOMImplementation();
			_doc = domImpl.createDocument(_namespace, "Rimm", null);

			_manifest = _doc.getDocumentElement();
			_manifest.setAttribute("Id", "_Rimm");
			_manifest
					.setAttribute("xmlns:core",
							"http://www.trustedcomputinggroup.org/XML/SCHEMA/1_0_1/core_integrity#");
			_manifest
					.setAttribute("xmlns:stuff",
							"http://www.trustedcomputinggroup.org/XML/SCHEMA/1_0/simple_object#");
			_manifest.setAttribute("xmlns:xsi",
					"http://www.w3.org/2001/XMLSchema-instance");
			_manifest.setAttribute("RevLevel", "0");
			_manifest.setAttribute("UUID", uuid.toString());

		} catch (ParserConfigurationException e) {
			e.printStackTrace();
		}
	}

	/**
	 * 
	 * 
	 * @param rimmFilename
	 * @throws Exception 
	 */
	public void loadFile(String rimmFilename) throws Exception {
		_filename = rimmFilename;
		//try {
			DocumentBuilderFactory dbfactory = DocumentBuilderFactory
					.newInstance();
			DocumentBuilder builder = dbfactory.newDocumentBuilder();
			_doc = builder.parse(new File(_filename));

			_manifest = _doc.getDocumentElement();
			NodeList nl = _manifest.getElementsByTagName("core:ComponentID");
			_cid = (Element) nl.item(0); // TODO check size 
			_simpleName = _cid.getAttribute("SimpleName");
			// System.out.print(_root.getFirstChild().getNodeValue());
		//} catch (Exception e) {
		//	e.printStackTrace();
		//}
	}
	private void loadGzip(String rimmFilename) {
		_filename = rimmFilename;
		try {
	    	FileInputStream fis = new FileInputStream(rimmFilename);
	    	BufferedInputStream bis = new BufferedInputStream(fis);
	    	GZIPInputStream gis = new GZIPInputStream(bis);
			
			
			DocumentBuilderFactory dbfactory = DocumentBuilderFactory
					.newInstance();
			DocumentBuilder builder = dbfactory.newDocumentBuilder();
			//_doc = builder.parse(new File(_filename));
			_doc = builder.parse(gis);

			_manifest = _doc.getDocumentElement();
			NodeList nl = _manifest.getElementsByTagName("core:ComponentID");
			_cid = (Element) nl.item(0); // TODO check size 
			_simpleName = _cid.getAttribute("SimpleName");
			// System.out.print(_root.getFirstChild().getNodeValue());
		} catch (Exception e) {
			e.printStackTrace();
		}
	}
	
	/**
	 * 
	 * setAttribute of core:ComponentID
	 * 
	 * Reference: 
	 *  TCG Infrastructure Working Group Reference Manifest (RM) Schema Specification 
	 *  Specification Version 1.0 Revision 1.0 17 November 2006 FINAL, p12
	 *  
	 *  core:ComponentID
	 *   Id Record instance identifier - recommended globally unique
	 *     UUID?
	 *   SimpleName String-ified version information for simple compare operations
	 *     BIOS vendorname + version?
	 *     P/N + BIOS version? 
	 *         
	 *   ModelName Model name with which the component is marketed
	 *     SMBIOS: System Information-> Version: ThinkPad T43 
	 *       
	 *   ModelNumber Alphanumeric model number with which the component is identified
	 *     
	 *   ModelSerialNumber Alphanumeric model serial number with which the component is identified
	 *   ModelSystemClass Vendor-specific system type or environment with which the component is associated
	 *     SMBIOS: System Information -> Product Name: 266872J
	 *   
	 *   VersionMajor Major version number of the component
	 *   VersionMinor Minor version number of the component
	 *   VersionBuild Build number of the component
	 *   VersionString String with which the components version may be identified
	 *     SMBIOS: BIOS Information -> Version: V1.00L11
	 *     
	 *   BuildDate Date on which the component was manufactured
	 *     SMBIOS: BIOS Information -> Release Date: 08/21/2006
	 *   
	 *   PatchLevel Patch level of the component
	 *   DiscretePatches Token strings enumerating each discrete patch that has been applied to the component; that is
	 *   not also represented by PatchLevel or other attributes in ComponentType
	 *   
	 * core:ComponentID -> VendorID p24
	 *   Name Familiar name associated with the component manufacturer or vendor
	 *     SMBIOS: BIOS Information -> Vendor: IBM
	 *     SMBIOS: System Information -> Manufacturer: IBM
	 *     SMBIOS: BIOS Information -> Vendor: Phoenix Technologies Ltd.
	 *     SMBIOS: System Information -> Manufacturer: Matsushita Electric Industrial Co.,Ltd.
	 *   
	 *  element core:VendorIdType/TcgVendorId p40
	 *   4 bytes string, issed by TCG e.g. "ATML","IBM "
	 *   
	 *   
	 * @param simpleName
	 * @param VersionString  e.g. BIOS Version
	 * @param VersionBuild   e.g. BIOS release ate
	 * @param ModelSystemClass e.g. P/N
	 * @param VendorName  Manufacturer or Vendor?
	 */
	public void setAttribute(
			String simpleName, 
			String VersionString,
			String VersionBuild, 
			String ModelSystemClass, 
			String VendorName) {

		_cid = _doc.createElement("core:ComponentID");
		_cid.setAttribute("Id", "_ComponentID");
		_cid.setAttribute("SimpleName", simpleName);
		_cid.setAttribute("VersionString", VersionString);
		_cid.setAttribute("VersionBuild", VersionBuild);
		_cid.setAttribute("ModelSystemClass", ModelSystemClass);

		Element vid = _doc.createElement("core:VendorID");
		vid.setAttribute("Name", VendorName);

		// TODO How to get the SmiVendorId?
		Element svid = _doc.createElement("core:SmiVendorId");
		svid.appendChild(_doc.createTextNode("0")); // int

		// TODO How to get the TcgVendorId?
		Element tvid = _doc.createElement("core:TcgVendorId");
		//tvid.appendChild(_doc.createTextNode("0"));
		tvid.appendChild(_doc.createTextNode("DEMO")); // max 4 byte

		vid.appendChild(svid);
		vid.appendChild(tvid);
		_cid.appendChild(vid);
		_manifest.appendChild(_cid);

		Element dm1 = _doc.createElement("core:DigestMethod");
		dm1.setAttribute("Id", "md5");
		dm1.setAttribute("Algorithm", "http://www.w3.org/2000/09/xmldsig#md5");
		_manifest.appendChild(dm1);

		Element dm2 = _doc.createElement("core:DigestMethod");
		dm2.setAttribute("Id", "sha1");
		dm2.setAttribute("Algorithm", "http://www.w3.org/2000/09/xmldsig#sha1");
		_manifest.appendChild(dm2);

		// AssertionType
		// _cv = _doc.createElement("core:Values");
		// _manifest.appendChild(_cv);

	}

	public String toString() {
		return "N/A Sorry";
	}

	/**
	 * @param bs
	 */
	public void addComponent(String name, byte[] md5digest, byte[] sha1digest) {

		Element cv = _doc.createElement("core:Values");
		Element sso = _doc.createElement("stuff:SimpleObject");
		Element so = _doc.createElement("stuff:Objects");
		so.setAttribute("Name", name);

		int hid = cid;
		if (md5digest != null) {
			Element md5 = _doc.createElement("stuff:Hash");
			md5.setAttribute("Id", "_c" + hid + "md5");
			md5.setAttribute("AlgRef", "md5");
			md5.appendChild(_doc.createTextNode(Base64Tool.encode(md5digest)));
			so.appendChild(md5);
		}
		if (sha1digest != null) {
			Element sha1 = _doc.createElement("stuff:Hash");
			sha1.setAttribute("Id", "_c" + hid + "sha1");
			sha1.setAttribute("AlgRef", "sha1");
			sha1
					.appendChild(_doc.createTextNode(Base64Tool
							.encode(sha1digest)));
			so.appendChild(sha1);
		}
		sso.appendChild(so);
		cv.appendChild(sso);
		_manifest.appendChild(cv);

		cid++;
	}

	public void addComponent(String name, String md5Base64digest,
			String sha1Base64digest) {

		Element cv = _doc.createElement("core:Values");
		Element sso = _doc.createElement("stuff:SimpleObject");
		Element so = _doc.createElement("stuff:Objects");
		so.setAttribute("Name", name);

		int hid = cid;
		if (md5Base64digest != null) {
			Element md5 = _doc.createElement("stuff:Hash");
			md5.setAttribute("Id", "_c" + hid + "md5");
			md5.setAttribute("AlgRef", "md5");
			md5.appendChild(_doc.createTextNode(md5Base64digest));
			so.appendChild(md5);
		}
		if (sha1Base64digest != null) {
			Element sha1 = _doc.createElement("stuff:Hash");
			sha1.setAttribute("Id", "_c" + hid + "sha1");
			sha1.setAttribute("AlgRef", "sha1");
			sha1.appendChild(_doc.createTextNode(sha1Base64digest));
			so.appendChild(sha1);
		}
		sso.appendChild(so);
		cv.appendChild(sso);
		_manifest.appendChild(cv);

		cid++;

	}

	public void indent() {
		_indent=true;
		
	}

	/**
	 * @param rimmFilename
	 * @throws Exception
	 */
	public void save(String rimmFilename) throws Exception {
		if (rimmFilename == null) {
			throw new Exception("rimmFilename == null");
		}
		_filename = rimmFilename;
		XmlTool.saveToFile(rimmFilename,_doc,_indent);

		if (log.isDebugEnabled())
			log.debug("toFile done");
	}

	/**
	 * @deprecated
	 * @param propFilename
	 * @param string
	 * @throws NumberFormatException
	 * @throws Exception
	 * @throws NumberFormatException
	 */
	public void addToIIDB(String propFilename, String schema) throws Exception {
		IIDB iidb = new IIDB(dbIndex); //(propFilename, schema);
//		iidb.connect();

		/* add package */
		NodeList nl = _manifest.getElementsByTagName("core:ComponentID");
		Element ccid = (Element) nl.item(0);

		int package_id;
		String packageName = ccid.getAttribute("SimpleName");
		String version = ccid.getAttribute("VersionString");
		package_id = iidb.addPackage(packageName, version, ccid
				.getAttribute("ModelSystemClass"), // arch
				ccid.getAttribute(""), // vendor
				Long.parseLong(ccid.getAttribute("VersionBuild")), // time
				""); // summary

		/* add digest */
		if (package_id > 0) {
			NodeList sonl = _manifest.getElementsByTagName("stuff:Objects");
			for (int i = 0; i < sonl.getLength(); i++) {
				Element so = (Element) sonl.item(i);
				String filename = so.getAttribute("Name");
				// log.debug(so.getAttribute("Name"));

				NodeList shnl = so.getElementsByTagName("stuff:Hash");
				for (int j = 0; j < shnl.getLength(); j++) {
					Element sh = (Element) shnl.item(j);
					log.debug("+" + j + sh.getAttribute("AlgRef"));

					if (sh.getAttribute("AlgRef").equals("sha1")) {
						// log.debug("+" + j +
						// sh.getAttribute("AlgRef"));
						String sha1 = sh.getFirstChild().getNodeValue();
						log.debug("sha1 " + sha1);
//						byte[] digest = null;
//						try {
//							digest = Base64Tool.decode(sha1);
//						} catch (IOException e) {
//							// TODO Auto-generated catch block
//							e.printStackTrace();
//						}
						int digest_id = iidb.addDigest(package_id, packageName,
								version, filename, "sha1", sha1);
					}
					if (sh.getAttribute("AlgRef").equals("md5")) {
						String md5 = sh.getFirstChild().getNodeValue();
						log.debug(so.getAttribute("md5 " + md5));
//						byte[] digest = null;
//						try {
//							digest = Base64Tool.decode(md5);
//						} catch (IOException e) {
//							// TODO Auto-generated catch block
//							e.printStackTrace();
//						}
						int digest_id = iidb.addDigest(package_id, packageName,
								version, filename, "md5", md5);
					}
				}
			}
		} else {
			log.debug("Package exist - Skip");
		}

//		iidb.close();

	}

	public boolean validateXML() throws Exception {		
		TCGSchemas schema = new TCGSchemas();		
		return schema.validate(_filename);
	}




	public void addModel(int pcrindex, FiniteStateMachine fsm) {
		/* add FSM */
		if (_cai == null) {
			_cai = _doc.createElement("core:AssertionInfo");
			_manifest.appendChild(_cai);
		}
		if (_vms == null) {
			_vms = _doc.createElement("ValidationModels");
			_cai.appendChild(_vms);			
		}
		
		Element vm = _doc.createElement("ValidationModel");
		vm.setAttribute("pcrindex", Integer.valueOf(pcrindex).toString());
		_vms.appendChild(vm);
			
		Element fsmRootOld = fsm.getRoot();
		Node fsmRootNew = _doc.importNode(fsmRootOld, true);
		vm.appendChild(fsmRootNew);
	}



	/**
	 * 
	 * get fsm[24] array 
	 * 
	 * 
	 * @return
	 * @throws Exception 
	 */
	public FiniteStateMachine[] getValidationModel() throws Exception {
		FiniteStateMachine[] fsms = new FiniteStateMachine[24];
		
		NodeList nl = _doc.getElementsByTagName("ValidationModel");
		if (nl.getLength() < 1) {
			throw new Exception("nl.getLength()!=1");
		}
	
		//log.debug("Models num " + nl.getLength());
		
		for (int i=0;i<nl.getLength();i++) {
			Element vm = (Element) nl.item(i);
			int pcrindex = new Integer(vm.getAttribute("pcrindex"));

			//log.debug("pcrindex  " + pcrindex);

			NodeList nlUM = vm.getElementsByTagName("uml:Model");
			if (nlUM.getLength() != 1) {
				throw new Exception("nl.getLength()!=1");
			} else {
				fsms[pcrindex] = new FiniteStateMachine((Element) nlUM.item(0));
			}
			
		}
		return fsms;
	}

	//public void verbose() {
	//	_verbose = true;
//
	//}

	/**
	 * 
	 * SignerInfo 
	 * 
	 * @param keyPair
	 * @throws Exception
	 */
	public void sign(KeyPair keyPair,X509Certificate cert) throws Exception {
		
		

		/* setup document */
		DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
		dbf.setNamespaceAware(true);
		DocumentBuilder builder = dbf.newDocumentBuilder();
		
		Document doc = builder.parse(new FileInputStream(_filename));
		Element root = doc.getDocumentElement();
		String uri= "#" + root.getAttribute("Id");
		
		//NodeList nl = root.getElementsByTagName("core:ComponentID");
		NodeList nl = root.getElementsByTagName("core:DigestMethod");
		//NodeList nl = root.getChildNodes();
		Element te = (Element) nl.item(0);
		Element sf = doc.createElement("core:SignerInfo");
		
		root.insertBefore(sf, te);
		
		DOMSignContext dsc = new DOMSignContext(keyPair.getPrivate(), sf);
		//DOMSignContext dsc = new DOMSignContext(keyPair.getPrivate(), root);
		

		XMLSignatureFactory fac = XMLSignatureFactory.getInstance("DOM");

		Reference ref = fac.newReference(uri, fac.newDigestMethod(
				DigestMethod.SHA1, null), Collections.singletonList(fac
				.newTransform(Transform.ENVELOPED,
						(TransformParameterSpec) null)), null, null);
		
	      SignedInfo si = fac.newSignedInfo
	        (fac.newCanonicalizationMethod
	          (CanonicalizationMethod.INCLUSIVE_WITH_COMMENTS,
	            (C14NMethodParameterSpec) null),
	          fac.newSignatureMethod(SignatureMethod.DSA_SHA1, null),
	          Collections.singletonList(ref)); 

	      KeyInfoFactory kif = fac.getKeyInfoFactory(); 
	      
	      KeyInfo ki;
	      if (_signatureOption==1) {
	    	  /* embed X509 Certificate */
	    	  List x509Content = new ArrayList();
	    	  x509Content.add(cert.getSubjectX500Principal().getName());
	    	  x509Content.add(cert);
	    	  X509Data xd = kif.newX509Data(x509Content);
	    	  ki = kif.newKeyInfo(Collections.singletonList(xd));
	      }
	      else if (_signatureOption ==0) {
	    	  /* put pub key */
		      KeyValue kv = kif.newKeyValue(keyPair.getPublic());
		      ki = kif.newKeyInfo(Collections.singletonList(kv)); 	    	  
	      }
	      else {
	    	  throw new Exception("Wrong option");
	      }

	      
	      XMLSignature signature = fac.newXMLSignature(si, ki); 
	      
	      signature.sign(dsc); 


	      _doc = null;
	      _doc = doc;


	}

	/**
	 * Validate XML Signature 
	 * http://www.techscore.com/tech/J2SE/JavaSE6/4.html#mustang4-4 << NG
	 * http://e-class.center.yuge.ac.jp/jdk_docs/ja/technotes/guides/security/xmldsig/XMLDigitalSignature.html << Good
	 * @return
	 * @throws Exception 
	 */
	public boolean validateSignature() throws Exception {
		//_verbose=true;
		
		if (log.isDebugEnabled()) log.debug("validateSignature");
		
		DocumentBuilderFactory dbf = 
			  DocumentBuilderFactory.newInstance(); 
		dbf.setNamespaceAware(true); 
		
		DocumentBuilder builder = dbf.newDocumentBuilder();  
		Document doc = builder.parse(new FileInputStream(_filename));
		
		if (log.isDebugEnabled()) log.debug("SM DEBUG" + _filename);
		
		NodeList nl = doc.getElementsByTagNameNS
		  (XMLSignature.XMLNS, "Signature");
		
		if (nl.getLength() == 0) {
		  throw new Exception("Cannot find Signature element");
		} 
		
		
		boolean coreValidity = false;
		DOMValidateContext valContext = null;
		XMLSignature signature = null;
		
		XMLSignatureFactory factory = 
			  XMLSignatureFactory.getInstance("DOM"); 
		
		/*
		// TODO GCJ Does not support XML Sig, so, v0.1.0 does not support XML Sig
		//
		if (_signatureOption == 0) {
			// TODO use bubkey in KeyInfo?
			// PubKey Lookup by PVSKeySelector
			valContext = new DOMValidateContext
			  (new PTSKeySelector(), nl.item(0)); 	
			
			
			signature = 
				  factory.unmarshalXMLSignature(valContext); 
			
			if (signature == null)
				throw new Exception("XMLSignature - missing");
			
			coreValidity = signature.validate(valContext); 			
		}
		else if (_signatureOption == 1) {
			// use Embeddded Cert
			valContext = new DOMValidateContext
		    (new PTSX509KeySelector(), nl.item(0));
			signature = factory.unmarshalXMLSignature(valContext);
			coreValidity = signature.validate(valContext);

			
		}
		else {
			throw new Exception("wrong _signatureOption");
		}
		*/
		
		if (coreValidity == false) {
			/* Why? */
			
			boolean sv = 
				  signature.getSignatureValue().validate(valContext);
				log.debug("signature validation status: " + sv); 


				Iterator i =
				  signature.getSignedInfo().getReferences().iterator();
				for (int j=0; i.hasNext(); j++) {
				  boolean refValid = ((Reference) 
				    i.next()).validate(valContext);
				  log.debug("ref["+j+"] validity status: " + 
				    refValid);
				} 

			
		}
		
		
		if (log.isDebugEnabled()) log.debug("validateSignature - done");
		return coreValidity;
	}

	public Element getCoreComponentID() {
		return _cid;
	}

	public String getName() {
		return _filename;
	}

	public String getSimpleName() {
		return _simpleName;
	}

	public String getSignerDN() throws Exception {
		NodeList nl = _manifest.getElementsByTagName("X509SubjectName");
		if (nl.getLength() == 0) {
			//throw new Exception("No X509SubjectName");
			return null;
		}
		if (nl.getLength() > 1) {
			throw new Exception("Multiple X509SubjectName");
		}

		return nl.item(0).getTextContent();
	}
	
	//public void setSchemaPath(String path) throws Exception {
	//	_tcgSchemas = new TCGSchemas(path);
	//	//_tcgSchemas.setSchemaDir(path);	
	//}

	public String getBase64String() throws Exception {
		return XmlTool.getBase64String(_doc);
	}

	/**
	 * Select IIDB index will be 0-7 (default is 0)
	 * @param dbIndex
	 */
	public void selectIIDB(int index) {
		this.dbIndex = index;
	}

	static int MAX_PCRS_NUM = 24; // TODO
	
	public ReferenceManifest generatePlatformReferenceManifestByProp(
			String eventlogFilename,
			SMBIOS smbios,
			String modelPropFilename
			) throws Exception {
		
		String modelDir = "";
		/* Check Properties */
		File file = new File(modelPropFilename);
		if (file.exists() == false) {
			throw new Exception("Model properties file is not found " + modelPropFilename);
		}
		else {
			modelDir = file.getParent();			
		}
		/* Load Properties */
		Properties prop = new Properties();
		FileInputStream fis = new FileInputStream(modelPropFilename);
		prop.load(fis);
		
		
		// Count models
		int pCount=0;
		int rCount=0;
		for (int i=0;i<MAX_PCRS_NUM ;i++) {
			String pModel = prop.getProperty("platform.model.pcr." + i);
			String rModel = prop.getProperty("runtime.model.pcr." + i);			
			if (pModel != null) pCount++;
			if (rModel != null) rCount++;			
		}
		
		// Create model list
		String[] behaviorModelFilenames = null;
		String[] runtimeModelFilenames = null;
		if (pCount>0) {
			behaviorModelFilenames = new String[pCount];
			int j=0;
			for (int i=0;i<MAX_PCRS_NUM ;i++) {
				String pModel = prop.getProperty("platform.model.pcr." + i);
				if (pModel != null) {
					//behaviorModelFilenames[j] = modelDir + "/" + pModel;
					behaviorModelFilenames[j] = pModel;
					j++;
				}
			}
		}
		if (rCount>0) {
			runtimeModelFilenames = new String[rCount];
			for (int i=0;i<MAX_PCRS_NUM ;i++) {
				String rModel = prop.getProperty("runtime.model.pcr." + i);			
				int j=0;
				if (rModel != null) {
					//runtimeModelFilenames[j] = modelDir + "/" + rModel;
					runtimeModelFilenames[j] = rModel;
					j++;
				}
			}
		}
		
		fis.close();
		
		/* Call */
		return generatePlatformReferenceManifest(
				eventlogFilename,
				behaviorModelFilenames, // in
				smbios,
				runtimeModelFilenames); // o
	}

	/**
	 * Create BIOS Runtime FSM and Reference manifest (RM)
	 * 
	 * 
	 * @param eventlogFilename
	 * @param behaviorModelFilenames
	 *            PCR0-7
	 * @param runtimeModelFilenames
	 *            PCR0-7
	 * @return
	 * @throws Exception
	 */
	public ReferenceManifest generatePlatformReferenceManifest(
			String eventlogFilename, // in
			String[] behaviorModelFilenames, // in
			SMBIOS smbios, // in
			String[] runtimeModelFilenames // out(as file)
	) throws Exception {
	
		// _verbose = true;
		boolean createRuntimeModelFiles = false;
		
		/* Check array size of FSM files */
		if (behaviorModelFilenames.length != 8) {
			throw new Exception("Wrong behaviorModelFilenames");
		}
		if (runtimeModelFilenames != null) {
			createRuntimeModelFiles = true;
			if (runtimeModelFilenames.length != 8) {
				throw new Exception("Wrong runtimeModelFilenames");
			}
		}
		
		/* ReferenceManifest */
		//ReferenceManifest rm = new ReferenceManifest();		
		//rm.init();
		this.init();
		
		/* set Attribute from SMBIOS info */
		if (smbios == null) {
			smbios = new SMBIOS(); // unknown state
		}
		
		//rm.selectIIDB(dbIndex); // TODO need?
		/* get time */
		//long date = new Date().getTime();
		long date = smbios.getBIOSDate().getTime();
		String simpleName =
			smbios.getProductName() + " " +
			smbios.getBIOSVersion();
 
		this.setAttribute(
				simpleName,
				smbios.getBIOSVersion(),
				Long.valueOf(date).toString(),
				smbios.getProductName(),
				smbios.getManufacturerName()
				); // TODO dynamic
	
		/* Load IML */
		IML iml = new IML();
		iml.loadBIOSIML(eventlogFilename);
	
		int pcrindex;
		for (pcrindex = 0; pcrindex < 8; pcrindex++) {
			
			Event[] event = iml.getEventList(pcrindex);
	
			/* FSM */
			FiniteStateMachine fsmBehavior = new FiniteStateMachine(
				behaviorModelFilenames[pcrindex]);
			//if (_verbose)
				//fsmBehavior.verbose();
	
			if (log.isDebugEnabled()) {
				/* print Eventlog */
				for (int i = 0; i < event.length; i++) {
					log.debug(i + " " + event[i].toString());
				}
			}
	
	
			/* Validate Events and Generate Runtime Model and Reference Manifest */
			int rc = 0;
			for (int i = 0; i < event.length; i++) {
				if (log.isDebugEnabled()) {
					log.debug(event[i].toString());
					log.debug("----------------------------------------"
						+ i);
				}
	
				rc = fsmBehavior.generateRuntimeModelByEventlog(event[i]);
				if (rc == 2) { 
					/* code */
					String name = fsmBehavior.getName();
					this.addComponent(name, null, event[i].getDigest());
				} else if (rc == 1) { 
					// infomative
				} else if (rc == 3) {
					// terminate
					break;
				} else {
					throw new Exception("FSM Trans was Failed at "
							+ event[i].toString() + ", FSM : "
							+ fsmBehavior.getName());
				}
	
			} // for event
	
			/* Check last state, muste be Final */
			if (rc != 3) {
				/* trans by dummy event, in case of no event in PCR8- */
				if (log.isDebugEnabled())
					log.debug("PUSH!! PUSH!!");
				Event ed = new Event();
				rc = fsmBehavior.generateRuntimeModelByEventlog(ed);
				
				if (rc == 1) {
					// TODO WORK AROUND for T60p PCR[1]
					//  
					log.worn("Known proplem on Thinkpad T60 and X60, FSM Trans was Failed, PUSH PUSH  rc=" + rc);
				}
				else if (rc != 3) {
					log.error("FSM Trans was Failed rc=" + rc);
					throw new Exception("FSM Trans was Failed. try verbose mode");
				} else {
					if (log.isDebugEnabled())
						log.debug(" OK? ");
				}
			}
	
			if (createRuntimeModelFiles) {
				/* save runtime model file */
				if (log.isDebugEnabled())
					log.debug("save Runtime Model ");
				fsmBehavior.toFile(runtimeModelFilenames[pcrindex]);				
			}
			
			/* add FSM to RM */
			this.addModel(pcrindex,fsmBehavior);
			
		} /* pcrindex */
	
		/* save Reference Manifest file */
		// if (_verbose) System.out.println("save Reference Manifest ");
		// rm.toFile(_prop.biosRimmFilename);
		// rm.addModel(fsmBehavior);
		return this;
		//return null;
	}

	/**
	 * 
	 * RM for Knoppix CD Boot and USB Boot
	 * 
	 * @param biosImlFilename
	 * @param runtimeImlFilename
	 * @param modelPropFilename
	 * @param grubDir
	 * @param object
	 * @throws Exception 
	 */
	private void generateKnoppixReferenceManifestByProp(
			String biosImlFilename, 
			String runtimeImlFilename, 
			String modelPropFilename, 
			String grubCdromDir, 
			String grubRootfsDir ) throws Exception {
		String modelDir = "";
		/* Check Properties */
		File file = new File(modelPropFilename);
		if (file.exists() == false) {
			throw new Exception("Model properties file is not found " + modelPropFilename);
		}
		else {
			modelDir = file.getParent();			
		}
		/* Load Properties */
		Properties prop = new Properties();
		FileInputStream fis = new FileInputStream(modelPropFilename);
		prop.load(fis);
		
		
		// Count models
		int rCount=0;
		for (int i=0;i<MAX_PCRS_NUM ;i++) {
			String rModel = prop.getProperty("runtime.model.pcr." + i);			
			if (rModel != null) rCount++;			
		}
		
		// Create model list
		String[] behaviorModelFilenames = null;
		String[] runtimeModelFilenames = null;

		if (rCount>0) {
			runtimeModelFilenames = new String[rCount];
			int j=0;
			for (int i=0;i<MAX_PCRS_NUM ;i++) {
				String rModel = prop.getProperty("runtime.model.pcr." + i);			
				if (rModel != null) {
					//runtimeModelFilenames[j] = modelDir + "/" + rModel;
					runtimeModelFilenames[j] = rModel;
					if (log.isDebugEnabled()) {
						log.debug("Model file " + runtimeModelFilenames[j]);
					}
					j++;
				}
			}
		}
		
		fis.close();
		
		/* Call */
		generateKnoppixReferenceManifest(
				biosImlFilename, 
				runtimeImlFilename, 
				grubCdromDir, 
				grubRootfsDir, 
				behaviorModelFilenames, 
				runtimeModelFilenames);
		
	}

	/*
	 * 
	 * RM for Knoppix CD Boot and USB Boot
	 * 
	 * Required Runtime PCR 4 5 8 10 model
	 * 
	 * @param biosImlFilename
	 * @param runtimeImlFilename (option)
	 * @param grubCdromDir (option)
	 * @param grubRootfsDir (option)
	 * @param behaviorModelFilenames
	 * @param runtimeModelFilenames
	 * @throws Exception
	 */
	public void generateKnoppixReferenceManifest(
			String biosImlFilename, 
			String runtimeImlFilename, 
			String grubCdromDir, 
			String grubRootfsDir,
			String[] behaviorModelFilenames, 
			String[] runtimeModelFilenames) throws Exception {

		//boolean createRuntimeModelFiles = false; // TODO need?
		
		/* Check array size of FSM files */
		// TODO Size mustbe  4
		//if (behaviorModelFilenames.length != 2) {
		//	throw new Exception("Wrong behaviorModelFilenames size " + behaviorModelFilenames.length);
		//}
		
		if (runtimeModelFilenames != null) {
			//createRuntimeModelFiles = true;
			// TODO add good messages 
			// properties file must 
			if (runtimeModelFilenames.length != 5) {
				throw new Exception("Wrong runtimeModelFilenames size " + runtimeModelFilenames.length);
			}
		} 
		else {
			throw new Exception("Models are requires");
		}
		if (log.isDebugEnabled()) {
			log.debug("0 " + runtimeModelFilenames[0]);
			log.debug("1 " + runtimeModelFilenames[1]);
			log.debug("2 " + runtimeModelFilenames[2]);
			log.debug("3 " + runtimeModelFilenames[3]);
			log.debug("3 " + runtimeModelFilenames[4]);
		}
		
		/* get time */
		long date = new Date().getTime();
		
		/* Check Grub dirs */
		if (grubCdromDir == null) {
			grubCdromDir = "/cdrom/boot/grub";
		}
		if (grubRootfsDir == null) {
			grubRootfsDir = "/user/share/grub/i386-pc";
		}
		
		/* IML */
		IML iml = new IML();
		iml.loadBIOSIML(biosImlFilename);
		
		/* ReferenceManifest */
		this.init();

		this.setAttribute(
				"Knoppix", // TODO
				"unknown", // TODO
				Long.valueOf(date).toString(),
				"unknown", // TODO
				"unknown"); // TODO dynamic
	
		
		/* Setup Behavior Models */
		FiniteStateMachine fsmPcr4 = new FiniteStateMachine(runtimeModelFilenames[0]);
		FiniteStateMachine fsmPcr5 = new FiniteStateMachine(runtimeModelFilenames[1]);
		FiniteStateMachine fsmPcr8 = new FiniteStateMachine(runtimeModelFilenames[2]);
		FiniteStateMachine fsmPcr10 = new FiniteStateMachine(runtimeModelFilenames[3]);
		FiniteStateMachine fsmPcr17 = new FiniteStateMachine(runtimeModelFilenames[4]);
	
		
		/* get digest values from local file */
		// TODO this for IBM's GRUB 
		RuntimeDigest rdRootfs = new RuntimeDigest(grubRootfsDir);
		RuntimeDigest rdCdrom = new RuntimeDigest(grubCdromDir);
		//if (_verbose) rd.verbose();
		
		/* stage 1 masked digest */
		String stage1Digest = rdRootfs.getStage1MaskedDigest("stage1");
		fsmPcr4.setReferenceAssertionDigest("SetMBRReferenceDigest", stage1Digest);
		
		if (log.isDebugEnabled()) {
			//byte[] buf = ;
			log.debug("GRUB Stage1 0 : " + 
					HexTool.getHexString(Base64Tool.decode(rdRootfs.getStage1Digest("stage1", 0))));
			log.debug("GRUB Stage1 1 : " + 
					HexTool.getHexString(Base64Tool.decode(rdRootfs.getStage1Digest("stage1", 1))));
			log.debug("GRUB Stage1 2 : " + 
					HexTool.getHexString(Base64Tool.decode(rdRootfs.getStage1Digest("stage1", 2))));
		}
		
					
		/* stage 1_5 */
		fsmPcr4.setReferenceDigest("stage1",  "stage1_5", rdRootfs.getStage15Digest("e2fs_stage1_5"));
		this.addComponent("stage1_5", null, rdRootfs.getStage15Digest("e2fs_stage1_5"));
		
		/* Stage 1_5 -> fs -> State2 */
		String[] fsNames = {
				"e2fs_stage1_5","fat_stage1_5","ffs_stage1_5","jfs_stage1_5",
				"minix_stage1_5","ufs2_stage1_5","vstafs_stage1_5","xfs_stage1_5",
				"iso9660_stage1_5","reiserfs_stage1_5"};
		String stage2Base64Digest = rdRootfs.getDigest("stage2");
		this.addComponent("stage2", null, stage2Base64Digest);
		for (int i=0;i<fsNames.length;i++) {
			String fsbase64Digest = rdRootfs.getStage15fsDigest(fsNames[i]);
			this.addComponent(fsNames[i], null, fsbase64Digest);
			fsmPcr4.setReferenceDigest("stage1_5",fsNames[i],fsbase64Digest);
			fsmPcr4.setReferenceDigest(fsNames[i],"stage2",stage2Base64Digest);
			if (log.isDebugEnabled()) {
				String hex = HexTool.getHexString(Base64Tool.decode(fsbase64Digest));
				log.debug("GRUB stage1 " + fsNames[i] + " " + hex);
			}
		}
		
		/* El Torito BootImage  */
		fsmPcr4.setReferenceAssertionDigest(
				"SetEltoritoReferenceDigest", 
				rdCdrom.getEltoritoDigest("stage2_eltorito", 0));
		fsmPcr4.setReferenceAssertionDigest(
				"SetEltoritoHPReferenceDigest", 
				rdCdrom.getEltoritoDigest("stage2_eltorito", 1));
		fsmPcr4.setReferenceAssertionDigest(
				"SetEltoritoIBMReferenceDigest", 
				rdCdrom.getEltoritoDigest("stage2_eltorito", 2));
				
		/* El Torito */
		/* Depend on the mkisofs */
		fsmPcr4.setReferenceDigest("stage1","stage2_eltorito",rdCdrom.getDigest("stage2_eltorito"));
	
		
		/* Kernel/Initrd Images get digest from IML */
		
		fsmPcr8.setReferenceDigest(
				"Grub_PCR8_START",
				"Kernel",
				iml.getDigestByType(0x1205));
		fsmPcr8.setReferenceDigest(
				"Kernel",
				"Initrd",
				iml.getDigestByType(0x1405));
		
		// For IMA too
		fsmPcr10.setVerifyPropertyDigest("Kernel", iml.getDigestByType(0x1205));
		fsmPcr10.setVerifyPropertyDigest("Initrd", iml.getDigestByType(0x1405));
		
		/* FSM -> RM */
	
		this.addModel(4, fsmPcr4);
		this.addModel(5, fsmPcr5);
		this.addModel(8, fsmPcr8);
		this.addModel(10, fsmPcr10);		
		this.addModel(17, fsmPcr17);		
		
		
		
	}

	
	private void generateRuntimeReferenceManifestByProp(
			String biosImlFilename, 
			String runtimeImlFilename, 
			String modelPropFilename, 
			String grubCdromDir, 
			String grubRootfsDir ) throws Exception {
		
		//String modelDir = "";
		/* Check Properties */
		File file = new File(modelPropFilename);
		if (file.exists() == false) {
			throw new Exception("Model properties file is not found " + modelPropFilename);
		}
		else {
			//modelDir = file.getParent();			
		}
		/* Load Properties */
		Properties prop = new Properties();
		FileInputStream fis = new FileInputStream(modelPropFilename);
		prop.load(fis);
		
		
		// Count models
		int rCount=0;
		for (int i=0;i<MAX_PCRS_NUM ;i++) {
			String rModel = prop.getProperty("runtime.model.pcr." + i);			
			if (rModel != null) rCount++;			
		}
		
		// Create model list
		String[] behaviorModelFilenames = null;
		String[] runtimeModelFilenames = null;

		if (rCount>0) {
			runtimeModelFilenames = new String[rCount];
			int j=0;
			for (int i=0;i<MAX_PCRS_NUM ;i++) {
				String rModel = prop.getProperty("runtime.model.pcr." + i);			
				if (rModel != null) {
					//runtimeModelFilenames[j] = modelDir + "/" + rModel;
					runtimeModelFilenames[j] = rModel;
					if (log.isDebugEnabled()) {
						log.debug("Model file " + runtimeModelFilenames[j]);
					}
					j++;
				}
			}
		}
		
		fis.close();
		
		/* Call */
		generateGrubReferenceManifest(
				biosImlFilename, 
				grubCdromDir,
				grubRootfsDir, 
				runtimeModelFilenames,
				null);
				//behaviorModelFilenames, 
				//runtimeModelFilenames);
		
	}

	/**
	 * 
	 * Ref ModelTransform.java
	 * PCR 4,5,8,10(ima)
	 * 
	 * @param localPath
	 * @param behaviorModelFilenames
	 * @param runtimeModelFilenames
	 * @return
	 * @throws Exception
	 */
	public void generateGrubReferenceManifest(
			String biosImlFilename, // in 1117 SM ADD
			String grubCdromDir,
			String localPath, // in
			String[] behaviorModelFilenames, // in
			String[] runtimeModelFilenames // out
	) throws Exception {
	
		boolean createRuntimeModelFiles = false;
		
		/* Check array size of FSM files */
		if (behaviorModelFilenames.length != 4) {
			throw new Exception("Wrong behaviorModelFilenames size " + behaviorModelFilenames.length + " !=4");
		}
		if (runtimeModelFilenames != null) {
			createRuntimeModelFiles = true;
			if (runtimeModelFilenames.length != 4) {
				throw new Exception("Wrong runtimeModelFilenames size");
			}
		}
		
		
		
		/* get time */
		long date = new Date().getTime();
		
		/* ReferenceManifest */
		//ReferenceManifest rm = new ReferenceManifest();
		this.init();
		//rm.selectIIDB(dbIndex); TODO 
		this.setAttribute(
				"Lenovo", // TODO
				"unknown", // TODO
				Long.valueOf(date).toString(),
				"unknown", // TODO
				"Lenovo"); // TODO dynamic
	
		
		/* Behavior Model*/
		FiniteStateMachine fsmPcr4 = new FiniteStateMachine(behaviorModelFilenames[0]);
		FiniteStateMachine fsmPcr5 = new FiniteStateMachine(behaviorModelFilenames[1]);
		FiniteStateMachine fsmPcr8 = new FiniteStateMachine(behaviorModelFilenames[2]);
		FiniteStateMachine fsmPcr10 = new FiniteStateMachine(behaviorModelFilenames[3]);
	
		
		/* get digest values from local file */
		// TODO this for IBM's GRUB 
		RuntimeDigest rd = new RuntimeDigest(localPath);
		RuntimeDigest rd2 = new RuntimeDigest(grubCdromDir);
		//if (_verbose) rd.verbose();grubCdromDir
		
		/* stage 1 masked digest */
		String stage1Digest = rd.getStage1MaskedDigest("stage1");
		fsmPcr4.setReferenceAssertionDigest("SetMBRReferenceDigest", stage1Digest);
					
		/* stage 1_5 */
		fsmPcr4.setReferenceDigest("stage1",  "stage1_5", rd.getStage15Digest("e2fs_stage1_5"));
		this.addComponent("stage1_5", null, rd.getStage15Digest("e2fs_stage1_5"));
		
		/* Stage 1_5 -> fs -> State2 */
		String[] fsNames = {
				"e2fs_stage1_5","fat_stage1_5","ffs_stage1_5","jfs_stage1_5",
				"minix_stage1_5","ufs2_stage1_5","vstafs_stage1_5","xfs_stage1_5",
				"iso9660_stage1_5","reiserfs_stage1_5"};
		String stage2Base64Digest = rd.getDigest("stage2");
		this.addComponent("stage2", null, stage2Base64Digest);
		for (int i=0;i<fsNames.length;i++) {
			String fsbase64Digest = rd.getStage15fsDigest(fsNames[i]);
			this.addComponent(fsNames[i], null, fsbase64Digest);
			fsmPcr4.setReferenceDigest("stage1_5",fsNames[i],fsbase64Digest);
			fsmPcr4.setReferenceDigest(fsNames[i],"stage2",stage2Base64Digest);			
		}
		
		/* El Torito */
		/* Depend on the mkisofs */
		fsmPcr4.setReferenceDigest("stage1","stage2_eltorito",rd2.getDigest("stage2_eltorito"));
	
		/* Kernel Digest */
		
		/* IML */
		IML iml = new IML();
		iml.loadBIOSIML(biosImlFilename);
		
		fsmPcr8.setReferenceDigest(
				"Grub_PCR8_START",
				"Kernel",
				iml.getDigestByType(0x1205));
		fsmPcr8.setReferenceDigest(
				"Kernel",
				"Initrd",
				iml.getDigestByType(0x1305)); // 
		
		// For IMA too
		
		fsmPcr10.setVerifyPropertyDigest("Kernel", iml.getDigestByType(0x1205));
		fsmPcr10.setVerifyPropertyDigest("Initrd", iml.getDigestByType(0x1305));
		
		/* FSM -> RM */
	
		this.addModel(4, fsmPcr4);
		this.addModel(5, fsmPcr5);
		this.addModel(8, fsmPcr8);
		this.addModel(10, fsmPcr10);
	
	}

	static int MAX_FILE_COUNT = 2;

	private static boolean verbose;
	
	private static void usage() {
		System.out.println("Usage: manifest [OPTIONS]");
		System.out.println(" OPTIONS");
		System.out.println(" --create --platform --model propfile --iml imlfile --out rmfile");
		System.out.println("     create platform(BIOS) Reference Manifest from IML");
		System.out.println(" --create --knoppix --model propfile --grubcd grubdir --grubrootfs grubdir --iml imlfile --out rmfile");
		System.out.println("     create knoppix Reference Manifest from IML");
		System.out.println(" --create --knoppix --model propfile --prop propfile2");
		System.out.println("     create knoppix Reference Manifest from IML");
		System.out.println(" --create --runtime --model propfile --prop propfile2");
		System.out.println("     create runtime(e.g. Redhat) Reference Manifest from IML");
	}

	public static void main(String[] args) {
		int opType = 0; // 1:create
		int opMode = 0; // 0:platform 1:runtime
		int fileCount = 0;
		String[] inputFilename = new String[MAX_FILE_COUNT];
		String outputFilename = null;
		String modelPropFilename = null;
		String propFilename = null;
		String grubCdromDir = null;
		String grubRootfsDir = null;
		SMBIOS smbios = null;
		boolean verbose = false;
		
		for (int i=0; i<args.length; ++i) {
            if ("--create".equals(args[i])) {
            	opType = 1;
            } 
            else if ("--platform".equals(args[i])) {
            	opMode = 0;
            } 
            else if ("--knoppix".equals(args[i])) {
            	opMode = 1;
            } 
            else if ("--runtime".equals(args[i])) {
            	opMode = 2;
            } 
            else if ("--iml".equals(args[i])) {
            	if (fileCount<MAX_FILE_COUNT) {
            		inputFilename[fileCount] = args[++i];
            		fileCount++;
            	}
            	else {
            		System.err.println("ERROR Supported eventlog file number is  " + MAX_FILE_COUNT);
            		usage();
            	}
            } 
            else if ("--model".equals(args[i])) {
            	modelPropFilename = args[++i];
            }
            else if ("--prop".equals(args[i])) {
            	propFilename = args[++i];
            }
            else if ("--grubcd".equals(args[i])) {
            	grubCdromDir = args[++i];
            }
            else if ("--grubrootfs".equals(args[i])) {
            	grubRootfsDir = args[++i];
            }
            else if ("--out".equals(args[i])) {
            	outputFilename = args[++i];
            }
            else if ("--verbose".equals(args[i])) {
            	verbose = true;
            }
            else {
            	System.err.println("Bad option " + args[i]);
            	usage();
            	return;
            }
        }
		
		/* Properties  */
		// TODO prop also used by IR, move to new class or method 
		if (propFilename == null) {
			//System.err.println("require properties file ");
			//usage();
			//return;	
		}
		else { 
			String propDir = "";
			File file = new File(propFilename);
			if (file.exists() == false) {
				System.err.println("properties file is not found " + propFilename);
				usage();
				return;			
				//throw new Exception("properties file is not found " + propFilename);
			}
			else {
				String parent = file.getParent();
				if (parent != null) {
					propDir = file.getParent() + "/";					
				}
				else {
					// NULL
					propDir = "/opt/OpenPlatformTrustServices/tcdemo/";					
				}

			}
			
			if (verbose) {
				System.out.println("Properties " + propFilename + " at " + propDir);
				// TODO getParent did not work on Linux :-( 
				//System.out.println(" getAbsolutePath " + file.getAbsolutePath());
				//System.out.println(" getParentFile " + file.getParentFile());
				//System.out.println(" getPath " + file.getPath());
				//System.out.println(" getParent " + file.getParent());
			}
			
			/* Load Properties */
			try {
				Properties prop = new Properties();
				prop.load(new FileInputStream(propFilename));

				grubCdromDir = propDir + prop.getProperty("runtime.grub.cdrom.dir");	
				grubRootfsDir = propDir + prop.getProperty("runtime.grub.rootfs.dir");	
				inputFilename[0] = propDir + prop.getProperty("platform.iml");	
				inputFilename[1] = propDir + prop.getProperty("runtime.iml");	
				if (opMode == 0) { // platform(BIOS)
					outputFilename = propDir + prop.getProperty("platform.manifest");				
				}
				else if (opMode == 1) { // runtime(Knoppix)
					outputFilename = propDir + prop.getProperty("runtime.manifest");				
				}
				else if (opMode == 2) { // runtime(Knoppix)
					outputFilename = propDir + prop.getProperty("runtime.manifest");				
				}
				String dmifilename = propDir + prop.getProperty("dmidecode");
				if (dmifilename != null) {
					smbios = new SMBIOS();
					smbios.loadDmidecodeOutput(dmifilename);
				}
			} catch (Exception e1) {
				e1.printStackTrace();
				System.err.println("Cannot load properties file " + propFilename);
				usage();
				return;
			}
		}

		
		
		/* Execute */
		try {
			switch(opType) {
			case 1: // create RM from IML
				switch(opMode) {
				case 0: // Platform
					// TODO validate inputs
					
					/* Load IML */
					//IML iml = new IML();	
					//iml.loadBIOSIML(inputFilename[0]);
					
					/* Generate Platform RM from BIOS IML */
					ReferenceManifest rm0 = new ReferenceManifest();
					rm0.generatePlatformReferenceManifestByProp(
							inputFilename[0],
							smbios,
							modelPropFilename);
					
					/* save to file */
					//rm0.indent();
					rm0.save(outputFilename);				
					break;			
				case 1: // Runtime (Knoppix)
					/* Runtime, Grub+IMA */
					ReferenceManifest rm1 = new ReferenceManifest();
					rm1.generateKnoppixReferenceManifestByProp(
							inputFilename[0],//biosImlFilename,
							inputFilename[1],//runtimeImlFilename, // TBD
							modelPropFilename,
							grubCdromDir, //grubBinariesDir, 
							grubRootfsDir
							//runtimeBehaviorModelFilenames,
							);
					
					//rm1.indent();
					rm1.save(outputFilename);										
					break;			
				case 2: // Runtime (Knoppix)
					/* Runtime, Grub+IMA */
					ReferenceManifest rm2 = new ReferenceManifest();
					rm2.generateRuntimeReferenceManifestByProp(
							inputFilename[0],//biosImlFilename,
							inputFilename[1],//runtimeImlFilename, // TBD
							modelPropFilename,
							grubCdromDir, //grubBinariesDir, 
							grubRootfsDir
							//runtimeBehaviorModelFilenames,
							);
					
					//rm1.indent();
					rm2.save(outputFilename);										
					break;	
				}
				break;			
			default:
				usage();
				break;
				
			}		
		} catch (Exception e) {
			System.err.println("Internal Error");
			e.printStackTrace();
		}
		

		
		// TODO Auto-generated method stub
		
	}

	public String getGipedBase64String() throws Exception {
		return XmlTool.getGipedBase64String(_doc);
	}

}
