/*
 * 
 * Licensed Materials - Property of IBM
 *
 * Open Platform Trust Services - An open source TCG PTS
 *
 * (C) Copyright International Business Machines Corp. 2007
 *
 */
package com.ibm.trl.tcg.pts.vulnerability.deb;

import java.sql.SQLException;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.ibatis.sqlmap.client.SqlMapClient;
import com.ibm.trl.tcg.pts.ibatis.SqlConfigIidbCreater;
import com.ibm.trl.tcg.pts.ibatis.SqlConfigVul;
import com.ibm.trl.tcg.pts.ibatis.dto.CveDefinitions;
import com.ibm.trl.tcg.pts.ibatis.dto.Debians;
import com.ibm.trl.tcg.pts.ibatis.dto.Measurements;
import com.ibm.trl.tcg.pts.ibatis.dto.Packages;
import com.ibm.trl.tcg.pts.vulnerability.oval.ReadOvalRedhat;
import com.ibm.trl.tcg.pts.vulnerability.tool.PackageVersionTool;

/**
 * Sync IIDB with DSA(VulDB).
 * 
 * @author Megumi Nakamura
 */
public class SyncWithDSA {

	/* Logger */
	private Log log = LogFactory.getLog(this.getClass());

	private int defaultValue = -2;

	// DB
	private SqlMapClient sqlMapVul = SqlConfigVul.getSqlMapInstance();

	private SqlMapClient sqlMapIidb = null;

	public SyncWithDSA(int dbIndex) {
		sqlMapIidb = SqlConfigIidbCreater.getSqlMapInstance(dbIndex);
	}

	/**
	 * Sync main.
	 */
	public void sync() {
		try {
			sqlMapIidb.startTransaction();
			sqlMapVul.startTransaction();

			String etchFixedVersion = null;
			String sargeFixedVersion = null;
			String sidFixedVersion = null;
			String version = null;
			String cveId = null;
			String[] cvssMinMax = null;

			Packages pkg = null;
			Measurements mgmt = null;

			List<Debians> deball = (List<Debians>) sqlMapVul
					.queryForList("getDebianDistinct");
			for (Debians ds : deball) {
				List<Debians> debs = (List<Debians>) sqlMapVul.queryForList(
						"getDebianByPackageName", ds.getDebianPackageName());
				List<Packages> pkgs = (List<Packages>) sqlMapIidb.queryForList(
						"getPackageByName", ds.getDebianPackageName());

				for (Debians d : debs) {
					etchFixedVersion = d.getDebianFixedEtch();
					sargeFixedVersion = d.getDebianFixedSarge();
					sidFixedVersion = d.getDebianFixedSid();

					for (Packages p : pkgs) {
						version = p.getPackageVersion();

						pkg = new Packages();
						pkg.setPackageOval(d.getDebianDsaId());
						pkg.setPackageId(p.getPackageId());
						cveId = d.getDebianCve();
						pkg.setPackageCve(cveId);
						cvssMinMax = changeCve2CvssMinMax(cveId);

						mgmt = new Measurements();
						mgmt.setDigestPackageId(p.getPackageId());

						// check version
						// 0(safe), 1(vul), -1(couldn't check), -2(null)
						int etchStatus = checkVulnerability(etchFixedVersion,
								version);
						int sargeStatus = checkVulnerability(sargeFixedVersion,
								version);
						int sidStatus = checkVulnerability(sidFixedVersion,
								version);

						if ((etchStatus == 0 && sargeStatus != 1 && sidStatus != 1)
								|| (etchStatus != 1 && sargeStatus == 0 && sidStatus != 1)
								|| (etchStatus != 1 && sargeStatus != 1 && sidStatus == 0)
								|| ((version.indexOf("etch") > -1) && (etchStatus == 0))
								|| ((version.indexOf("sarge") > -1) && (sargeStatus == 0))
								|| ((version.indexOf("sid") > -1) && (sidStatus == 0))) {
							// safe
							pkg.setPackageVulnerability(ReadOvalRedhat.NO_VUL_DISTRO_MATCH); // no vul
							mgmt.setDigestVulnerability(ReadOvalRedhat.NO_VUL_DISTRO_MATCH); // no vul
							pkg.setPackageCvssMin("0");
							pkg.setPackageCvssMax("0");
							// Update
							sqlMapIidb.update("updatePackageVulnerabilities",
									pkg);
							sqlMapIidb.commitTransaction();
							sqlMapIidb.update(
									"updateDigestVulnerabilityByPackageId",
									mgmt);
							sqlMapIidb.commitTransaction();

							log.debug("OK\t" + etchFixedVersion + "(etch), "
									+ sargeFixedVersion + "(sarge), "
									+ sidFixedVersion + "(sid) \t <= "
									+ version);

						} else if ((etchStatus == 1 || sargeStatus == 1 || sidStatus == 1)
								&& ((etchStatus != 0 && sargeStatus != 0 && sidStatus != 0))
								|| ((version.indexOf("etch") > -1) && (etchStatus == 1))
								|| ((version.indexOf("sarge") > -1) && (sargeStatus == 1))
								|| ((version.indexOf("sid") > -1) && (sidStatus == 1)))  {
							// vulnerable
							pkg.setPackageVulnerability(ReadOvalRedhat.VUL_DISTRO_MATCH); // vul
							mgmt.setDigestVulnerability(ReadOvalRedhat.VUL_DISTRO_MATCH); // vul
							pkg.setPackageCvssMin(cvssMinMax[0]);
							pkg.setPackageCvssMax(cvssMinMax[1]);
							// Update
							sqlMapIidb.update("updatePackageVulnerabilities",
									pkg);
							sqlMapIidb.commitTransaction();
							sqlMapIidb.update(
									"updateDigestVulnerabilityByPackageId",
									mgmt);
							sqlMapIidb.commitTransaction();

							log.debug("VUL\t" + etchFixedVersion + "(etch), "
									+ sargeFixedVersion + "(sarge), "
									+ sidFixedVersion + "(sid) \t <= "
									+ version);

						} else if (((etchStatus == -1 && etchFixedVersion != null)
								|| (sargeStatus == -1 && sargeFixedVersion != null) || (sidStatus == -1 && sidFixedVersion != null))
								|| (etchStatus != defaultValue
										&& sargeStatus != defaultValue && sidStatus != defaultValue)
								|| ((etchStatus == 1 || sargeStatus == 1 || sidStatus == 1) && (etchStatus == 0
										|| sargeStatus == 0 || sidStatus == 1))) {
							// unknown
							pkg
									.setPackageVulnerability(ReadOvalRedhat.CHECK_AGAIN); // recheck
							mgmt
									.setDigestVulnerability(ReadOvalRedhat.CHECK_AGAIN); // recheck
							pkg.setPackageCvssMin(cvssMinMax[0]);
							pkg.setPackageCvssMax(cvssMinMax[1]);
							// Update
							sqlMapIidb.update("updatePackageVulnerabilities",
									pkg);
							sqlMapIidb.commitTransaction();
							sqlMapIidb.update(
									"updateDigestVulnerabilityByPackageId",
									mgmt);
							sqlMapIidb.commitTransaction();

							log.debug("-?-\t" + etchFixedVersion + "(etch), "
									+ sargeFixedVersion + "(sarge), "
									+ sidFixedVersion + "(sid) \t <= "
									+ version);
						}
					}
				}
			}
		} catch (SQLException ex) {
			do {
				log.error("SQLSTATE: " + ex.getSQLState());
				log.error("ERR-CODE: " + ex.getErrorCode());
				log.error("ERR-MSEG: " + ex.getMessage());
				ex = ex.getNextException();
			} while (null != ex);
		}
	}

	/**
	 * Compare two versions.
	 * 
	 * @param fixedVersion
	 *            The version number from database
	 * @param currentVersion
	 *            The version number from input file
	 * @return 0(safe), 1(vulnerable), -1(couldn't check), -2(null param)
	 */
	private int checkVulnerability(String fixedVersion, String currentVersion) {
		if (currentVersion == null || fixedVersion == null) {
			return defaultValue; // Null Error
		}
		int status = PackageVersionTool.compareTwoVersion(fixedVersion,
				currentVersion);
		if (status == 0) { // EQUAL
			return 0; // Safe
		} else if (status == 1) { // GREATER
			return 1; // Vulnerable
		} else if (status == -1) { // LESS
			return 0; // Safe
		} else { // status = 100(check again)
			return -1; // Couldn't check
		}
	}

	/**
	 * Cve to Cvss{min,max}.
	 * 
	 * @param cveId CVE string
	 * @return String[] String[0]=cvss_min, String[1]=cvss_max
	 */
	private String[] changeCve2CvssMinMax(String cveId) {
		String[] cvssMinMax = new String[2];
		try {
			String cvss = "";
			double cvssScore = 0;
			double cvssScoreMin = 0;
			double cvssScoreMax = 0;
			if (cveId != null) {
				String[] cves = cveId.split(" ");
				for (int i = 0; i < cves.length; i++) {
					String cveIdString = cves[i].trim();
					CveDefinitions cveDef = (CveDefinitions) sqlMapVul
							.queryForObject("getCveDefinitionByCveId",
									cveIdString);
					if (cveDef != null) {
						cvss = cveDef.getNvdCvssScore();
						if (cvss != null) {
							cvssScore = new Double(cvss).doubleValue();

							if (cvssScoreMin == 0 && cvssScoreMax == 0) {
								cvssScoreMin = cvssScore;
								cvssScoreMax = cvssScore;
							} else if (cvssScore < cvssScoreMin) {
								cvssScoreMin = cvssScore;
							} else if (cvssScore > cvssScoreMax) {
								cvssScoreMax = cvssScore;
							}
						}
					}
				}
			}
			cvssMinMax[0] = new Double(cvssScoreMin).toString();
			cvssMinMax[1] = new Double(cvssScoreMax).toString();

		} catch (NumberFormatException e) {
			e.printStackTrace();
		} catch (SQLException e) {
			e.printStackTrace();
		}
		return cvssMinMax;
	}

	/**
	 * @param args --outdir
	 */
	public static void main (String[] args) {
		int dbIndex = 0;
		
		for (int i = 0; i < args.length; ++i) {
			if ("--dbindex".equals(args[i])) {
				dbIndex = Integer.valueOf(args[++i]);
			} else {
				System.err.println("Unknown option " + args[i]);
				usage();
				return;
			}
		}
		new SyncWithDSA(dbIndex).sync();
	}
	
	/**
	 * Usage.
	 */
	private static void usage() {
		System.out.println("Usage: dsasync [OPTIONS]");
		System.out.println(" OPTIONS");
		System.out.println(" --dbindex 0-7 (the database number of integrity info db)");
	}


}
