/*
 * 
 * Licensed Materials - Property of IBM
 *
 * Open Platform Trust Services - An open source TCG PTS
 *
 * (C) Copyright International Business Machines Corp. 2007
 *
 */
package com.ibm.trl.tcg.pts.integrity.iidb;

import java.io.IOException;
import java.util.List;
import java.util.Vector;
import java.util.regex.Pattern;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import sun.misc.BASE64Decoder;

import com.ibatis.sqlmap.client.SqlMapClient;
import com.ibm.trl.tcg.pts.engine.Event;
import com.ibm.trl.tcg.pts.ibatis.SqlConfigIidbCreater;
import com.ibm.trl.tcg.pts.ibatis.dto.Measurements;
import com.ibm.trl.tcg.pts.ibatis.dto.Packages;
import com.ibm.trl.tcg.pts.vulnerability.tool.DbOperations;

/**
 * 
 * TODO IIDB Check does not work e.g. Miss 0.6.13-6 != 0.6.15 - 0.1.i586.rpm
 * 
 * @author Seiji Munetoh
 * 
 */
public class IIDB {

	/* Logger */
	private Log log = LogFactory.getLog(this.getClass());

	/* TCG result */
	static final int VALID = 0;

	static final int INVALID = 1;

	static final int UNVERIFIED = 2;

	/* IIDB Vul Flag */
	static final int IIDB_VUL_NOINFO = 0;

	static final int IIDB_VUL_VALID = 1;

	static final int IIDB_VUL_INVALID = 3;

	static final int IIDB_VUL_RECHECK = 10;

	static final int IIDB_VUL_VALID_BY_ADMIN = 11;

	static final int IIDB_VUL_INVALID_BY_ADMIN = 13;

	static String _schemaName = null;

	final int _sleeptime = 500; // msec

	private String _log;

	private String _advice;

	private String _filename;

	// private String _behaviorModel;

	// private String _runtimeModel;

	private String _javaVEClass = null;

	// private String _cVElib = null;

	// private boolean _verbose = false;

	private String _remediation;

	/* Database */
	private SqlMapClient sqlMapIidb = null;

	/**
	 * @throws Exception
	 * 
	 */
	public IIDB(int dbIndex) throws Exception {
		sqlMapIidb = SqlConfigIidbCreater.getSqlMapInstance(dbIndex);
		if (sqlMapIidb == null) {
			throw new Exception("IIDB not found, index " + dbIndex);
		}
	}

	/**
	 * 
	 */
	private static void usage() {
		// System.out.println("Usage: iidbtool [OPTIONS]");
		// System.out.println("Usage: --list_all");
		// System.err.println("Usage: --list");
		System.out.println("Usage: iidb [OPTIONS]");
		System.out.println(" OPTIONS");
		System.out.println(" --list");
		System.out.println("     list summary");
		System.out.println(" --list --vul --index db_index");
		System.out.println("     list vulnerable packages");
		System.out.println(" --check --index db_index");
		System.out.println("     ckeck/update uncertain packages");
	}

	public static void main(String[] args) {
		int opType = 0; // 1:create
		int opMode = 0; // 0:platform 1:runtime
		boolean verbose = false;
		int iidbIndex = -1;

		for (int i = 0; i < args.length; ++i) {
			if ("--list".equals(args[i])) {
				opType = 1;
			} else if ("--check".equals(args[i])) {
				opType = 2;
			} else if ("--vul".equals(args[i])) {
				opMode = 1;
			} else if ("--index".equals(args[i])) {
				iidbIndex = Integer.valueOf(args[++i]).intValue();// args[++i];
			} else {
				System.err.println("Unknown option " + args[i]);
				usage();
				return;
			}
		}

		try {
			DbOperations dbo = null;

			switch (opType) {
			case 1: // List
				switch (opMode) {
				case 0: // Summary
					dbo = new DbOperations();
					dbo.listIidbSummary(iidbIndex);
					break;
				case 1: // vul
					dbo = new DbOperations();
					dbo.listVulnerablePackage(iidbIndex, 3);
					dbo.listVulnerablePackage(iidbIndex, 13);
					break;
				default:
					break;
				}
				break;
			case 2: // Check
				if (iidbIndex < 0) {
					System.err.println("ERROR, IIDB index is missing.");
					usage();
					return;
				}
				dbo = new DbOperations();
				dbo.fixUnclearPackage(iidbIndex);
				break;
			default:
				usage();
				break;

			}
		} catch (Exception e) {
			System.err.println("Internal Error");
			e.printStackTrace();
		}

		// String schemaName = null;
		// int dbIndex = 0;
		// int jobNum = 0;
		//
		// if (args.length == 0) {
		// usage();
		// } else {
		// for (int i = 0; i < args.length; i++) {
		// // log.debug("argv[" + i + "] = " + args[i] );
		// if ("--list_all".equals(args[i])) {
		// try {
		// new IIDB(dbIndex).listAll();
		// } catch (Exception e) {
		// // TODO error message
		// e.printStackTrace();
		// }
		// } else if ("--schema".equals(args[i])) {
		// schemaName = args[i + 1];
		// i++;
		// } else {
		// usage();
		// }
		// }
		// }
	}

	/**
	 * 
	 */
	private void listAll() {
		try {
			List<Packages> packages = (List<Packages>) sqlMapIidb
					.queryForList("getPackageAll");
			for (Packages p : packages) {
				int id = p.getPackageId();
				String filename = p.getPackageName();
				String version = p.getPackageVersion();
				if (log.isDebugEnabled()) {
					log.debug(id + " " + filename + " " + version);
				}
			}
			List<Measurements> digests = (List<Measurements>) sqlMapIidb
					.queryForList("getDigestAll");
			for (Measurements m : digests) {
				int id = m.getDigestId();
				String filename = m.getDigestName();
				String digest = m.getDigest();// .toString();
				if (log.isDebugEnabled()) {
					log.debug(id + " " + filename + " " + digest);
				}
			}

		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	/**
	 * @param package_id
	 * @param name
	 * @param version
	 * @param filename
	 * @param alg
	 * @param digest
	 */
	public int addDigest(int package_id, String name, String version,
			String filename, String alg, String digest) {

		int digest_id = -1;

		try {
			List<Measurements> list = (List<Measurements>) sqlMapIidb
					.queryForList("getDigestByDigest", digest);
			for (Measurements m : list) {
				int id = m.getDigestId();
				int pid = m.getDigestPackageId();
				String n = m.getDigestName();
				// log.debug(id + " " + pid);
				if (pid == package_id) { // Exist
					if (filename.equals(n)) {
						// log.debug("FIND " + name);
						digest_id = id;
						break;
					}
				}
			}

			/* if not exist Add Digest */
			if (digest_id < 0) {
				Measurements measurements = new Measurements();
				measurements.setDigest(digest);
				measurements.setDigestAlg(alg);
				measurements.setDigestName(filename);
				measurements.setDigestPackageId(package_id);
				sqlMapIidb.startTransaction();
				sqlMapIidb.insert("insertDigest", measurements);
				sqlMapIidb.commitTransaction();

				Thread.sleep(100); // TODO
			}

		} catch (Exception e) {
			e.printStackTrace();
		}

		return digest_id;
	}

	/**
	 * @param package_id
	 * @param name
	 * @param version
	 * @param filename
	 * @param alg
	 * @param digest
	 */
	public int addDigestByHexString(int package_id, String name,
			String version, String filename, String alg, String digest) {

		int digest_id = -1;

		try {
			List<Measurements> list = (List<Measurements>) sqlMapIidb
					.queryForList("getDigestByDigest", digest);
			for (Measurements m : list) {
				int id = m.getDigestId();
				int pid = m.getDigestPackageId();
				String n = m.getDigestName();
				// log.debug(id + " " + pid);
				if (pid == package_id) { // Exist
					if (filename.equals(n)) {
						// log.debug("FIND " + name);
						digest_id = id;
						break;
					}
				}
			}
			if (digest_id < 0) {
				Measurements measurements = new Measurements();
				measurements.setDigest(digest);
				measurements.setDigestAlg(alg);
				measurements.setDigestName(filename);
				measurements.setDigestPackageId(package_id);
				sqlMapIidb.startTransaction();
				sqlMapIidb.insert("insertDigest", measurements);
				sqlMapIidb.commitTransaction();
				Thread.sleep(100); // TODO
			}
		} catch (Exception e) {
			e.printStackTrace();
		}

		return digest_id;
	}

	/**
	 * add new package if new package name & new version & new arch
	 * 
	 * @param name
	 * @param version
	 * @param arch
	 * @param vendor
	 * @param build
	 * @param summary
	 * @return package_id 0 if same package is already exist
	 * @throws Exception
	 * 
	 */
	public int addPackage(String name, String version, String arch,
			String vendor, long build, String summary) throws Exception {

		return addPackage(name, version, arch, vendor, build, summary, "n/a",
				"n/a", "n/a", "n/a", "n/a");
	}

	public int addPackage(String name, String version, String arch,
			String vendor, long build, String summary, String rpmreponame,
			String rpmfilename, String rpmfilepath, String rimmfilename,
			String rimmfilepath) throws Exception {

		int package_id = -1;

		List<Packages> list = (List<Packages>) sqlMapIidb.queryForList(
				"getPackageByName", name);
		Vector ids = new Vector();
		for (Packages p : list) {
			int id = p.getPackageId();
			String ver = p.getPackageVersion();
			String arc = p.getPackageArch();
			long bui = p.getPackageBuild();

			// log.debug(id + " " + ver);
			// if (version.equals(ver)) { // Exist
			if ((version.equals(ver)) && (arch.equals(arc))) {
				/* Same version exist */
				package_id = 0;
			}
			if (arch.equals(arc) && (bui < build)) { // OLD BUILD
				ids.add(Integer.valueOf(id));
			}
		}

		/* obsolete */
		for (int i = 0; i < ids.size(); i++) {
			// Update PACKAGE TABLE
			Integer id0 = (Integer) ids.get(i);
			int id = id0.intValue();

			sqlMapIidb.startTransaction();
			sqlMapIidb.update("updatePackageObsolete", id);
			sqlMapIidb.commitTransaction();

			sqlMapIidb.startTransaction();
			sqlMapIidb.update("updateDigestObsolete", id);
			sqlMapIidb.commitTransaction();

		}

		/* if not exist Add Package */
		if (package_id < 0) {
			Packages packages = new Packages();
			packages.setPackageName(name);
			packages.setPackageVersion(version);
			packages.setPackageArch(arch);
			packages.setPackageVendor(vendor);
			packages.setPackageBuild(build);
			packages.setPackageSummary(summary);
			packages.setPackageRpmReponame(rpmreponame);
			packages.setPackageRpmFilename(rpmfilename);
			packages.setPackageRpmFilepath(rpmfilepath);
			packages.setPackageRimmFilename(rimmfilename);
			packages.setPackageRimmFilepath(rimmfilepath);
			sqlMapIidb.startTransaction();
			sqlMapIidb.update("insertPackage", packages);
			sqlMapIidb.commitTransaction();

			list = (List<Packages>) sqlMapIidb.queryForList("getPackageByName",
					name);
			for (Packages p : list) {
				int id = p.getPackageId();
				String ver = p.getPackageVersion();
				if (version.equals(ver)) { // Exist
					package_id = id;
					break;
				}
			}
		}

		return package_id;
	}

	/**
	 * @param digest2
	 * @return
	 */
	public static String getHexString(byte[] data) {
		String str = "";
		for (int i = 0; i < data.length; i++) {
			int d = data[i];
			if (d < 0) {
				d += 256;
			}
			if (d < 16) {
				str += "0";
			}
			str += Integer.toString(d, 16);
		}
		return str;
	}

	/**
	 * @param digest2
	 * @return
	 */
	public static byte[] getByteFromHexString(String data) {
		byte ba[] = new byte[data.length() / 2];
		char ca[] = data.toCharArray();
		int j = 0;
		for (int i = 0; i < ba.length; i++) {
			ba[i] = (byte) ((Character.digit(ca[j], 16)) * 16 + (Character
					.digit(ca[j + 1], 16)));
			j += 2;
		}
		return ba;
	}

	/**
	 * @param string
	 * @return
	 */
	public static byte[] decodeBase64(String string) {
		BASE64Decoder decoder = new BASE64Decoder();
		try {
			byte[] out = decoder.decodeBuffer(string);
			// log.debug("base64 : " + string);
			// log.debug("hex : " + getHexString(out));
			return out;
			// return decoder.decodeBuffer(string);
		} catch (IOException e) {
			e.printStackTrace();
		}
		return null;
	}

	/**
	 * Check the vulnerability from given digestvalue. if INVALID, remediations
	 * are available, use getRemediationXXX() methods to retrive them.
	 * 
	 * DB    VUL     RESULT
	 * -------------------------- 
	 * HIT   NO      VALID
	 * HIT   YES     INVALID
	 * MISS          UNVERIFIED
	 * 
	 * @param digest
	 * @return 0 VALID 1 INVALID 2 UNVERIFIED
	 * @throws Exception
	 */
	public int queryByDigest(byte[] binDigest) throws Exception {
		// resetLog();
		String hexDigest = getHexString(binDigest);
		// addLog("digest=" + getHexString(binDigest));
		_remediation = "";

		if (log.isDebugEnabled()) {
			log.debug("queryByDigest  by " + getHexString(binDigest));
		}

		int rc = UNVERIFIED;
		try {
			/* SQL Query */
			// List<Measurements> list =
			// (List<Measurements>)sqlMapIidb.queryForList("getPackageDigestByDigest",
			// hexDigest);
			// for(Measurements m : list){
			// Packages p = m.getPackages();
			// }
			List<Measurements> list = (List<Measurements>) sqlMapIidb
					.queryForList("getDigestByDigest", hexDigest);

			int pid = 0; // package ID
			// int index = 0;
			// int vulnerability = 0;
			// int obsolete = 0;

			int countValid = 0;
			int countInvalid = 0;
			int countUnverified = 0;
			for (Measurements m : list) {
				int id = m.getDigestId();
				pid = m.getDigestPackageId();
				// hits.add(new Integer(pid));
				String filename = m.getDigestName();
				_filename = filename;
				int obso = m.getDigestObsolete();
				int vol = m.getDigestVulnerability();
				// obsolete += obso;
				// vulnerability += vol;
				boolean getPackageinfinfo = false;
				String msg = "";

				if (log.isDebugEnabled()) {
					log.debug(id + " " + filename + " "
							+ getHexString(binDigest));
					log.debug("obsolete : " + obso);
					log.debug("vulnerability : " + vol);
				}

				/* logging */
				// addLog("hit." + index + ".name=" + filename);
				if (vol == IIDB_VUL_NOINFO) {
					/* no vul info */
					countValid++;
					if (obso == 0) {
						msg = "latest";
					} else {
						msg = "update available";
						getPackageinfinfo = true;
					}
					// rc = VALID;
					// addLog("hit." + index + ".recommend=VALID");
					// _remediation += "Good,";
				} else if ((vol == IIDB_VUL_VALID)
						|| (vol == IIDB_VUL_VALID_BY_ADMIN)) {
					/* no vul */
					countValid++;
					if (obso == 0) {
						msg = "latest";
					} else {
						msg = "update available";
						getPackageinfinfo = true;
					}
					// rc = VALID;
					// addLog("hit." + index + ".recommend=VALID");
					// _remediation += "Good,";
				} else if (vol == IIDB_VUL_RECHECK) {
					/* VUL info is ambiguous, ask to DB admin */
					countUnverified++;
					if (obso == 0) {
						msg = "ambiguous but latest";
						getPackageinfinfo = true;
					} else {
						msg = "ambiguous but update available";
						getPackageinfinfo = true;
					}
					// addLog("hit." + index + ".recommend=UNKNOWN");
					// addLog("hit." + index + ".comments=UNKNOWN PACKAGE");
					// _remediation += " is unknown package. Please ask your
					// administrator,";

				} else if ((vol == IIDB_VUL_INVALID)
						|| (vol == IIDB_VUL_INVALID_BY_ADMIN)) { // Vulnerable
					/* Vulnerable */
					countInvalid++;
					// addLog("hit." + index + ".recommend=INVALID");
					// addLog("hit." + index + ".comments=VULNERABILITY EXIST");
					if (obso == 0) {
						msg = "is vulnelable!!! but no update";
						getPackageinfinfo = true;
						// addLog("hit." + index + ".comments=OLD VERSION");
						// addLog("hit." + index + ".comments=USE LATEST VERSION
						// SO FAR");
					} else {
						msg = "is vulnelable & obsolete, PLEASE UPDATE!";
						getPackageinfinfo = true;
					}
				} else {
					msg = " INTERNAL ERROR (unknown vul flag)";
				}
				// else if (true) { //
				// //addLog("hit." + index + ".recommend=INVALID");
				// if (obso != 0) {
				// addLog("hit." + index + ".comments=OLD VERSION");
				// addLog("hit." + index + ".comments=USE LATEST VERSION SO
				// FAR");
				// _remediation += " is obsolute, UPDATE!,";
				// }
				// }
				// else {
				// /* Get Package Info */
				// //Packages p =
				// (Packages)sqlMapIidb.queryForList("getPackage", pid);
				// Packages p =
				// (Packages)sqlMapIidb.queryForObject("getPackage", pid);
				// String name = p.getPackageName();
				// String ver = p.getPackageVersion();
				// String arc = p.getPackageArch();
				// String oval = p.getPackageOval();
				// String cve = p.getPackageCve();
				//	
				// String packageName = name + "-" + ver + "." + arc;
				//	
				// //_remediation += packageName + ",";
				// _remediation = packageName + ",";
				// /*Get Vul Info*/
				// if (oval != null) {
				// log.warn("OVAL/DSA(" + oval.trim() + ")");
				// _remediation += "OVAL/DSA(" + oval.trim() + "), ";
				// }
				// if (cve != null) {
				// log.warn("CVE(" + cve.trim() + ")");
				// _remediation += "CVE(" + cve.trim() + "), ";
				// //VULDB vuldb = new VULDB(_propFilename);
				// //vuldb.connect();
				// //_remediation += vuldb.getRemediation(cve);
				// //vuldb.close();
				// }
				//	
				// /* Add redimation message */
				// vulnerability++;
				// rc = INVALID;
				// }

				if (getPackageinfinfo) {
					/* Package */
					Packages p = (Packages) sqlMapIidb.queryForObject(
							"getPackage", pid);
					// String filename2 = p.getPackageName();
					// String version = p.getPackageVersion();
					String name = p.getPackageName();
					String ver = p.getPackageVersion();
					String arc = p.getPackageArch();
					String oval = p.getPackageOval();
					String cve = p.getPackageCve();

					String packageName = name + "-" + ver + "." + arc;

					_remediation += packageName + ",";
					/* Get Vul Info */
					if (oval != null) {
						// log.warn("OVAL/DSA(" + oval.trim() + ")");
						_remediation += "OVAL/DSA(" + oval.trim() + "), ";
					}
					if (cve != null) {
						// log.warn("CVE(" + cve.trim() + ")");
						_remediation += "CVE(" + cve.trim() + "), ";
						// //VULDB vuldb = new VULDB(_propFilename);
						// //vuldb.connect();
						// //_remediation += vuldb.getRemediation(cve);
						// //vuldb.close();
					}
					// int pObso = p.getPackageObsolete();
					// int pVol = p.getPackageVulnerability();
					// TODO we assume single pkg here
					//
					// _behaviorModel = p.getPackageBehaviorModel();
					// _runtimeModel = p.getPackageRuntimeModel();
					// _javaVEClass = p.getPackageJavaVeModule();
					// _cVElib = p.getPackageCVeModule();
					// obsolete += pObso;
					// vulnerability += pVol;
					// addLog("hit." + index + ".package=" + filename2);
					// addLog("hit." + index + ".version=" + version);
					_remediation += " " + msg;

				} else {
					// no vul info
					_remediation = msg;
				}

			} // for list

			if (countInvalid > 0) {
				rc = INVALID;
			} else if (countUnverified > 0) {
				rc = UNVERIFIED;
			} else if (countValid > 0) {
				rc = VALID;
			} else {
				// MISS
				if (log.isDebugEnabled()){
					log.debug(" Unknown digest");
				}
				// NG rc = INVALID;
				rc = UNVERIFIED;
			}

		} catch (Exception e) {
			e.printStackTrace();
			throw new Exception("DB Error? at queryByDigest");
		}

		// int countValid = 0;
		// int countInvalid = 0;
		// int countUnverified = 0;
		// if ((obsolete != 0) || (vulnerability != 0)) {
		// rc = INVALID;
		// }
		// if (rc == UNVERIFIED) {
		// _remediation = "IIDB does not support this digest/package";
		// }

		return rc;
	}

	/**
	 * Return 0 VALID 1 INVALID 2 UNVERIFIED
	 * 
	 * @param digest
	 * @return
	 * @throws Exception
	 */
	public int queryByBase64Digest(String digest) throws Exception {
		byte[] binDigest = decodeBase64(digest);
		return queryByDigest(binDigest);
	}

	/**
	 * Return 0 VALID 1 INVALID 2 UNVERIFIED
	 * 
	 * @param digest
	 * @return
	 * @throws Exception
	 */
	public int queryByHexStringDigest(String digest) throws Exception {
		byte[] binDigest = getByteFromHexString(digest);
		return queryByDigest(binDigest);
	}

	/**
	 * @param string
	 * @return
	 */
	public String queryByDigestName(String name) {
		int vulnerability = 0;
		int obsolete = 0;
		// resetLog();
		if (log.isDebugEnabled())
			log.debug("name=" + name);

		try {
			List<Measurements> list = (List<Measurements>) sqlMapIidb
					.queryForList("getDigestByName", name);

			int pid = 0;
			// Vector hits = new Vector(); // package list
			int index = 0;

			for (Measurements m : list) {
				int id = m.getDigestId();
				pid = m.getDigestPackageId();
				String digest = m.getDigest();
				// hits.add(new Integer(pid));
				String filename = m.getDigestName();
				int obso = m.getDigestObsolete();
				int vol = m.getDigestVulnerability();

				if (log.isDebugEnabled()) {
					log.debug("digest_id : " + id);
					log.debug("name      : " + filename);
					log.debug("digest    : " + digest);
				}
				index++;
			}

		} catch (Exception e) {
			e.printStackTrace();
		}

		return null;
	}

	/**
	 * 
	 */
	// private void resetLog() {
	// _log = "";
	// }
	/**
	 * @param string
	 */
	// private void addLog(String string) {
	// _log += string + "\n";
	// }
	/**
	 * @return
	 */
	// public Object getLastLogNG() {
	// if (_log != null)
	// return _log;
	// else
	// return "na";
	// }
	/**
	 * @param name
	 */
	public int searchLatestPackageByFilename(String name) {
		int rc = -1;
		// resetLog();
		resetAdvice();
		try {
			List<Measurements> list = (List<Measurements>) sqlMapIidb
					.queryForList("getDigestByName", name);

			int pid = 0;
			int index = 0;

			for (Measurements m : list) {
				int id = m.getDigestId();
				pid = m.getDigestPackageId();
				String filename = m.getDigestName();
				int obso = m.getDigestObsolete();
				int vol = m.getDigestVulnerability();

				/* logging */
				// addLog("hit." + index + ".name=" + filename);
				if ((obso == 0) && (vol == 0)) {
					rc = 0;
					// addLog("hit." + index + ".recommend=VALID");

					/* Package List */
					// Packages p =
					// (Packages)sqlMapIidb.queryForList("getPackage", pid);
					Packages p = (Packages) sqlMapIidb.queryForObject(
							"getPackage", pid);
					String filename2 = p.getPackageName();
					String version = p.getPackageVersion();
					int pObso = p.getPackageObsolete();
					int pVol = p.getPackageVulnerability();
					if ((pObso == 0) && (pVol == 0)) {
						// addLog("hit." + index + ".package=" + filename2);
						// addLog("hit." + index + ".version=" + version);
						setAdvice("The latest version is " + filename2 + " "
								+ version);
						rc = 0;
					}
				}
				index++;
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
		return rc;
	}

	/**
	 * 
	 */
	private void resetAdvice() {
		_advice = null;
	}

	/**
	 * @param string
	 */
	private void setAdvice(String string) {
		_advice = string + "\n";

	}

	/**
	 * @return
	 */
	public String getAdvice() {
		if (_advice != null)
			return _advice;
		else
			return "na";
	}

	/**
	 * @param value
	 * @return
	 * @deprecated
	 */
	public int queryByHexStringDigestFast(String digest) {
		int vulnerability = 0;
		int obsolete = 0;
		// resetLog();
		// addLog("digest=" + digest);

		// TODO byte[] binDigest = getByteFromHexString(digest);

		int rc = 2;
		try {
			List<Measurements> list = (List<Measurements>) sqlMapIidb
					.queryForList("getDigestByDigest", digest);

			if (true) {
				int pid = 0;
				// Vector hits = new Vector(); // package list
				int index = 0;
				for (Measurements m : list) {
					int id = m.getDigestId();
					pid = m.getDigestPackageId();
					// hits.add(new Integer(pid));
					String filename = m.getDigestName();
					_filename = filename;

					int obso = m.getDigestObsolete();
					int vol = m.getDigestVulnerability();
					obsolete += obso;
					vulnerability += vol;

					/* logging */
					// addLog("hit." + index + ".name=" + filename);
					if ((obso == 0) && (vol == 0)) {
						rc = 0;
						// addLog("hit." + index + ".recommend=VALID");
					} else {
						vulnerability++;
						rc = 1;
						// addLog("hit." + index + ".recommend=INVALID");
						if (vol != 0)
							// addLog("hit." + index
							// + ".comments=VULNERABILITY EXIST");
							if (obso != 0) {
								// addLog("hit." + index + ".comments=OLD
								// VERSION");
								// addLog("hit." + index
								// + ".comments=USE LATEST VERSION SO FAR");
							}
					}

					index++;
				} // while
			} // false

		} catch (Exception e) {
			e.printStackTrace();
		}

		if ((obsolete != 0) || (vulnerability != 0)) {
			rc = 1;
		}

		return rc;
	}

	/**
	 * @return
	 */
	public String getFileName() {
		return _filename;
	}

	/**
	 * @return
	 */
	public String getJavaVEName() {
		// System.err.println("DEBUG " + _javaVEClass);

		return _javaVEClass;
	}

	// public void setVerbose() {
	// _verbose = true;
	// }

	// TODO Copied From VIM

	/**
	 * Split the string by pattern. ex. patternCheck(".-", "x.y.z-wv") ==>
	 * String[]{x,y,z,wv}
	 * 
	 * @param pattern
	 * @param target
	 * @return
	 */
	private String[] patternCheck(String pattern, String target) {
		Pattern ptn = Pattern.compile(pattern);
		return ptn.split(target);
	}

	/**
	 * Parse one line of "rpm -qa" to name, version, and release number.
	 * 
	 * @param rpmName
	 * @return String[3]{Name,Version,Release}
	 */
	private String[] parseVersion(String rpmName) {
		String[] rpmInfo = new String[3];
		rpmInfo[0] = null; // Name
		rpmInfo[1] = null; // Version
		rpmInfo[2] = null; // Release

		// Name
		String[] parts = patternCheck("-v[0-9]", rpmName);
		if (parts.length > 1) { // -v3.0
			rpmInfo[0] = parts[0];
		} else {
			parts = patternCheck("-alpha[0-9]", rpmName);
			if (parts.length > 1) { // -alpha3.0
				rpmInfo[0] = parts[0];
			} else { // General pattern
				parts = patternCheck("-[0-9]", rpmName);
				if (parts[0].startsWith("java") && parts.length > 2) {
					rpmInfo[0] = rpmName.substring(0, parts[0].length()
							+ parts[1].length() + 2);
				} else {
					rpmInfo[0] = parts[0];
				}
			}
		}
		// Version
		String left = rpmName.substring(rpmInfo[0].length() + 1);
		Pattern patternVersion = Pattern.compile("-");
		parts = patternVersion.split(left);
		rpmInfo[1] = (parts.length > 0) ? parts[0] : null;

		// Release
		if (parts.length > 1) {
			rpmInfo[2] = parts[1];
			for (int i = 2; i < parts.length; i++) {
				rpmInfo[2] += "-" + parts[i];
			}
		}

		return rpmInfo;
	}

	/**
	 * Check RPM package
	 * 
	 * @param rpm_name
	 *            RPM File name
	 * @return
	 */
	public boolean existPackage(String rpmName) {
		boolean rc = false;
		String[] rpmInfo = parseVersion(rpmName);
		String versionRef = rpmInfo[1] + "-##-" + rpmInfo[2];

		/* Query */
		int package_id = -1;
		try {
			List<Packages> list = (List<Packages>) sqlMapIidb.queryForList(
					"getPackageByName", rpmInfo[0]);
			int result;
			for (Packages p : list) {
				String version = p.getPackageVersion();
				if (version.equals(versionRef)) {
					/* Same version exist */
					if (log.isDebugEnabled()) {
						log.debug("Hit  " + version + " == " + versionRef);
					}
					rc = true;
				} else {
					if (log.isDebugEnabled()) {
						log.debug("Miss " + version + " != " + versionRef);
					}
				}
			}

		} catch (Exception e) {
			e.printStackTrace();
		}
		return rc;
	}

	/**
	 * return remediation string
	 * 
	 * @return
	 */
	public String getRemediation() {
		return _remediation;
	}

	/**
	 * check existing entry by RPM filename and RIMM filename
	 * 
	 * 
	 * @param rpmfilename
	 * @param rimmfilename
	 * @return
	 * @throws Exception
	 */
	public boolean exist(String rpmfilename, String rimmfilename)
			throws Exception {
		boolean rc = false;

		/* Query */
		int package_id = -1;
		try {
			/* ask by package name */
			List<Packages> list = (List<Packages>) sqlMapIidb.queryForList(
					"getPackageByRpmfilename", rpmfilename);
			for (Packages p : list) {
				String rimm = p.getPackageRimmFilename();
				if (rimm.equals(rimmfilename)) {
					rc = true;
				} else {
					if (log.isDebugEnabled()) {
						log.debug("Miss " + rimmfilename);
					}
				}
			}

		} catch (Exception e) {
			e.printStackTrace();
		}

		return rc;
	}

	public void printStatus() {
		try {
			/* Package List */
			List<Packages> list = (List<Packages>) sqlMapIidb
					.queryForList("getPackageAll");
			for (Packages p : list) {
				int id = p.getPackageId();
				String filename = p.getPackageName();
				String version = p.getPackageVersion();
				int vul = (int) p.getPackageVulnerability();

				System.out.print(id + " " + filename + " " + version);
				switch (vul) {
				case 3:
					System.out
							.print("  VUL <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<");
					break;
				case 4:
					System.out
							.print("  VUL? <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<");
					break;
				case 10:
					System.out
							.print("  ???? <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<");
					break;
				default:
					System.out.print(" OK");
					break;
				}
			}

			/* Digest List */
			if (true) {
				List<Measurements> measurements = (List<Measurements>) sqlMapIidb
						.queryForList("getDigestAll");
				for (Measurements m : measurements) {
					int id = m.getDigestId();
					String filename = m.getDigestName();
					// String digest = rs.getString("DIGEST");
					if (log.isDebugEnabled()) {
						log.debug(id + " " + filename + " ");
					}
				}
			}

		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	/**
	 * 
	 * Look for the VM Class name 1) i=0 2) select digest[i] 3) hit single ->
	 * BINGO hit multiple - i++ goto 2) miss -> unknown i>limit -> unknown
	 * 
	 * @param event1
	 * @return
	 * @throws Exception
	 */
	public String getVmClassName(Event[] events, int offset, int limit)
			throws Exception {

		for (int i = offset; i < offset + limit; i++) {
			byte[] digest = events[i].getDigest();// digest;
			limit--;

			int rc = 2;
			int hitNum = 0;
			String packageName = null;
			String packageVersion = null;
			int pNum = 0;
			/* look for digest */

			List<Measurements> list = (List<Measurements>) sqlMapIidb
					.queryForList("getDigestByDigest", digest);
			for (Measurements m : list) {
				int id = m.getDigestId();
				int pid = m.getDigestPackageId();

				hitNum++;

				/* Package List */
				// Packages p = (Packages)sqlMapIidb.queryForList("getPackage",
				// pid);
				Packages p = (Packages) sqlMapIidb.queryForObject("getPackage",
						pid);
				packageName = p.getPackageName();
				packageVersion = p.getPackageVersion();

				// _behaviorModel = p.getPackageBehaviorModel();
				// _runtimeModel = p.getPackageRuntimeModel();
				_javaVEClass = p.getPackageJavaVeModule();

				pNum++;
			}
			if (pNum != 1) {
				throw new Exception("pNum !=1 " + pNum);
			}

			if (hitNum == 1) {
				/* single hit */
				if (_javaVEClass == null) {
					// com.ibm.trl.pvs.module.bios.ibm.thinkpad_1KET46WW;
					return null;// "com.ibm.trl.pvs.module.bios."; //
								// ibm.thinkpad_1KET46WW;";
				}

				return _javaVEClass;
				// return packageName + " " + packageVersion;
			}
		} // for

		return null;
	}

}
