/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.db.AbstractRenderer;
import org.opengion.hayabusa.db.CellRenderer;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.Selection;
import org.opengion.hayabusa.db.SelectionFactory;		// 5.7.3.0 (2014/02/07)

/**
 * BITBOX レンデラーは、カラムのデータをコードリソースに対応した
 * チェックボックスで表示する場合に使用するクラスです。
 *
 * カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * 一覧表示する場合は、通常は、□(0)か■(1)のみ表示されます。
 * ラベル表示が必要な場合は、表示パラメータに、"useLabel"と
 * 記述しておくことで、ラベルを出力することが可能です。
 *
 * @og.rev 6.2.2.4 (2015/04/24) 新規作成
 * @og.group データ表示
 *
 * @version  6.2
 * @author	 Kazuhiko Hasegawa
 * @since    JDK8.0,
 */
public class Renderer_BITBOX extends AbstractRenderer {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.6.0 (2016/05/27)" ;

	private final Selection selection	;
	private final boolean useLabel	;		// 4.3.3.0 (2008/10/01)
	private final boolean useKeyLabel ;		// 6.2.0.0 (2015/02/27) キー:ラベル形式

	private String errMsg	;				// 6.0.4.0 (2014/11/28)

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 */
	public Renderer_BITBOX() {
		super();					// 6.4.1.1 (2016/01/16) PMD refactoring. It is a good practice to call super() in a constructor
		selection	= null;
		useLabel	= false;		// 4.3.3.0 (2008/10/01)
		useKeyLabel = true;
	}

	/**
	 * DBColumnオブジェクトを指定したprivateコンストラクター。
	 *
	 * @og.rev 6.2.2.4 (2015/04/24) 新規追加
	 * @og.rev 6.4.6.0 (2016/05/27) getEditorParam → getRendererParam に修正
	 *
	 * @param	clm	DBColumnオブジェクト
	 */
	private Renderer_BITBOX( final DBColumn clm ) {
		super();		// 6.4.1.1 (2016/01/16) PMD refactoring. It is a good practice to call super() in a constructor
		useKeyLabel = "true".equalsIgnoreCase( clm.getAddKeyLabel() ) ;		// 値:ラベル形式

		// 5.7.3.0 (2014/02/07) SelectionFactory 対応
		final String addKeyLabel = clm.getAddKeyLabel();					// 6.2.0.0 (2015/02/27) キー:ラベル形式
		selection = SelectionFactory.newSelection( "BITBOX" , clm.getCodeData() , addKeyLabel );

		// 6.0.4.0 (2014/11/28) selection が null の場合、Selection_NULL を作成します。
		if( selection == null ) {
			errMsg = "codeData が未設定です。"
								+ " name="  + clm.getName()
								+ " label=" + clm.getLabel()
								+ " rendType="  + clm.getRenderer() ;
			System.out.println( errMsg );
		}

		useLabel = "useLabel".equalsIgnoreCase( clm.getRendererParam() ) ;			// 6.4.6.0 (2016/05/27)
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @og.rev 6.2.2.4 (2015/04/24) 新規追加
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellRendererオブジェクト
	 * @og.rtnNotNull
	 */
	public CellRenderer newInstance( final DBColumn clm ) {
		return new Renderer_BITBOX( clm );
	}

	/**
	 * データの表示用文字列を返します。
	 *
	 * @og.rev 6.2.2.4 (2015/04/24) 新規追加
	 *
	 * @param   value 入力値
	 *
	 * @return  データの表示用文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final String value ) {
		// 6.0.4.0 (2014/11/28) selection が null の場合、警告表示します。
		if( selection == null ) {
			return "<span class=\"error\">" + errMsg + " value=" + value + "</span>";
		}

		return "<pre class=\"BITBOX\">" + selection.getValueLabel( value,true ) + "</pre>" ;
	}

	/**
	 * データの一覧表示用文字列を返します。
	 * 一覧表示のため、useLabel が有効です。
	 *
	 * @og.rev 6.2.2.4 (2015/04/24) 新規追加
	 *
	 * @param	row   行番号
	 * @param	value 入力値
	 *
	 * @return	データ表示/編集用の文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final int row,final String value ) {
		// 6.4.1.1 (2016/01/16) PMD refactoring. A method should have only one exit point, and that should be the last statement in the method
		return selection == null
					? "<span class=\"error\">" + errMsg + " value=" + value + " row=" + row + "</span>"
					: "<pre class=\"BITBOX\">" + selection.getValueLabel( value,useLabel ) + "</pre>" ;
	}

	/**
	 * データ出力用の文字列を作成します。
	 * ファイル等に出力する形式を想定しますので、HTMLタグを含まない
	 * データを返します。
	 * 基本は、#getValue( String ) をそのまま返します。
	 *
	 * @og.rev 6.2.2.4 (2015/04/24) 新規追加
	 *
	 * @param   value 入力値
	 *
	 * @return  データ出力用の文字列
	 * @see		#getValue( String )
	 */
	@Override
	public String getWriteValue( final String value ) {
		// 6.4.1.1 (2016/01/16) PMD refactoring. A method should have only one exit point, and that should be the last statement in the method
		return selection == null
					? value
					: useKeyLabel
						? value + ':' + selection.getValueLabel( value,useLabel )
						: selection.getValueLabel( value,useLabel );
	}
}
