/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.fukurou.system.OgBuilder ;				// 6.4.4.1 (2016/03/18)
import org.opengion.fukurou.util.ToString;					// 6.1.1.0 (2015/01/17)
import org.opengion.fukurou.util.StringUtil ;				// 6.2.2.0 (2015/03/27)

import static org.opengion.fukurou.util.StringUtil.nval;

import java.util.concurrent.atomic.AtomicInteger;			// 5.5.2.6 (2012/05/25) findbugs対応

/**
 * Query画面の拡張検索領域を指定用のタグです。
 *
 * 通常は属性を何もセットせずにQuery画面の隠しメニューに利用します。
 * queryButtonタグが存在しない場合はボタンを出力したい場所にIDをセットしたspanタグを
 * 配置し、そのIDをこのタグのbuttonId属性で指定して下さい。
 * タグの動きとしてはにはbody内に書いた内容をbuttonIdで指定した領域に動的に作成した
 * ボタン(初期値はシステムパラメータの画像)でON/OFF制御可能とします。
 * なお、body部分にはog:firldsetタグも利用可能です。
 *
 * @og.formSample
 * ●形式：
 *       ・&lt;og:hideMenu
 *                   [ id           = "uniqHideId"   ] タグで囲われた部分に付けるID
 *                   [ buttonId     = "targetId"     ] ボタンを出力する場所のIDを指定
 *                   [ defaultNone  = "[true|false]" ] 初期状態の開閉指定
 *                   [ openButton   = "open.png"     ] 開く(＋)ボタンの画像
 *                   [ closeButton  = "close.png"    ] 閉じる(－)ボタンの画像
 *                   [ lbl          = "MSG0076"      ] 画像のalt属性に付けるメッセージ(初期値:MSG0076)
 *         &gt;
 * ●body：あり(EVAL_BODY_INCLUDE:BODYをインクルードし、{&#064;XXXX} は解析しません)
 *
 * ●Tag定義：
 *   &lt;og:hideMenu
 *       id                 【TAG】このタグのBODY部分にIDを指定します
 *       buttonId           【TAG】画像ボタン出力先のIDを指定します
 *       defaultNone        【TAG】初期状態の開閉指定[[true:閉じている/false:開いている]を指定します(初期値:true)
 *       openButton         【TAG】開くボタンの画像を指定します(初期値:"jsp/image/" + SystemData#HIDE_OPEN_BUTTON[={@og.value SystemData#HIDE_OPEN_BUTTON}])
 *       closeButton        【TAG】閉じるボタンの画像を指定します(初期値:"jsp/image/" + SystemData#HIDE_CLOSE_BUTTON[={@og.value SystemData#HIDE_CLOSE_BUTTON}])
 *       lbl                【TAG】ラベルリソースのラベルIDを指定します
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:hideMenu&gt;
 *
 * ●使用例
 * 例１:通常のQUERY画面での利用
 *         &lt;og:hideMenu&gt;
 *           &lt;table&gt;&lt;tr&gt;
 *             &lt;og:column name="CLM"/&gt;
 *           &lt;/tr&gt;&lt;/table&gt;
 *         &lt;/og:hideMenu&gt;
 *
 * 例２:queryButtonタグが存在しない状況での利用(buttonIdの明示的な指定)
 *         &lt;span id="hideButtonSet"&gt; &lt;/span&gt;
 *         &lt;og:hideMenu&gt;
 *             ......
 *         &lt;/og:hideMenu&gt;
 *
 * @og.rev 4.3.6.0 (2009/04/01) 新規作成
 * @og.rev 7.3.2.3 (2021/04/09) ｼｽﾃﾑ定数のICON_DIRやIMAGE_DIRを使用します。
 * @og.group 画面部品
 *
 * @version  4.0
 * @author   Takahashi Masakazu
 * @since    JDK5.0,
 */
public class HideMenuTag extends CommonTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "8.4.3.0 (2023/04/07)" ;
	private static final long serialVersionUID = 843020230407L ;

	private static final AtomicInteger UNIQ_ID = new AtomicInteger(1);		// 5.5.2.6 (2012/05/25) findbugs対応		// 6.4.1.1 (2016/01/16) uniqID  → UNIQ_ID  refactoring

	private String id			;											// この領域に付けるid
	private String buttonId		= "query-button-append";					// ボタンを出力する場所のid
	private boolean defaultNone = true;										// 開閉初期状態

	// 7.3.2.3 (2021/04/09) ｼｽﾃﾑ定数のICON_DIRやIMAGE_DIRを使用します。
	private static final String JSP_ICON = HybsSystem.sys( "JSP_ICON" ) + "/" ;

//	private String openButton	= HybsSystem.sys( "JSP" ) + "/image/" + HybsSystem.sys( "HIDE_OPEN_BUTTON" );	// 開くボタン
	private String openButton	= JSP_ICON + HybsSystem.sys( "HIDE_OPEN_BUTTON" );	// 開くボタン
//	private String closeButton	= HybsSystem.sys( "JSP" ) + "/image/" + HybsSystem.sys( "HIDE_CLOSE_BUTTON" );	// 閉じるボタン
	private String closeButton	= JSP_ICON + HybsSystem.sys( "HIDE_CLOSE_BUTTON" );	// 閉じるボタン
	private String hideClass	;		// 5.5.4.4 (2012/07/20)

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public HideMenuTag() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 5.5.2.6 (2012/05/25) findbugs対応。staticフィールドへの書き込みに、AtomicInteger を利用します。
	 * @og.rev 5.5.3.2 (2012/06/08) 画像にalt属性を追加します。
	 * @og.rev 5.5.4.4 (2012/07/20) hideClassを外部から指定可能にします
	 * @og.rev 5.7.1.2 (2013/12/20) msg ⇒ msglbl 変更
	 * @og.rev 6.0.2.4 (2014/10/17) img タグに、title 属性追記
	 * @og.rev 6.2.2.0 (2015/03/27) BRと\nを相互に変換する処理を追加
	 * @og.rev 6.2.2.3 (2015/04/10) htmlフィルターに、BR→改行処理機能を追加。
	 * @og.rev 5.9.2.1 (2015/11/12) , 6.3.9.0 (2015/11/06) id修正
	 * @og.rev 6.4.4.1 (2016/03/18) StringBuilderの代わりに、OgBuilderを使用する。
	 * @og.rev 7.0.1.0 (2018/10/15) XHTML → HTML5 対応(空要素の、"／＞" 止めを、"＞" に変更します)。
	 * @og.rev 8.4.3.0 (2023/04/07) script タグから、language 属性を削除します。
	 *
	 * @return	後続処理の指示( EVAL_BODY_INCLUDE )
	 */
	@Override
	public int doStartTag() {
		// 6.3.9.0 (2015/11/06) Variables should start with a lowercase character(PMD)
		// IMG を final 化するに当たり、実際に使用する直前で生成するように変更。
		if( hideClass == null || hideClass.isEmpty() ){ // 5.5.4.4 (2012/07/20) 外部指定対応
			hideClass = "hideMenu" + ( UNIQ_ID.getAndIncrement() );
		}

		// 5.5.3.2 (2012/06/08) 画像にalt属性を追加します。
		String msglbl = getMsglbl();	// 5.7.1.2 (2013/12/20) msg ⇒ msglbl 変更
		if( msglbl == null ) { setLbl( "MSG0076" ); msglbl = getMsglbl(); }			// 初期値は、MSG0076:拡張検索条件

		final OgBuilder buf = new OgBuilder()
				.append( "<div class=\"" , hideClass , "\" " )
				.appendIf( id != null								// if
						, "id=\"" , id , "\" " )
				.appendIf( defaultNone								// if
						, "style=\"display: none;\"" )
				.append( ">" );

		// 指定したIDにボタン画像を放り込むスクリプトを出力
		if( buttonId != null && buttonId.length() > 0){
			final String IMG = "  <img id=\"queryHide\" src=\""
							+	( defaultNone ? openButton : closeButton )
							+	"\" alt=\"" + msglbl + "\" border=\"0px\""
							+	" title=\"" + StringUtil.htmlFilter( msglbl,true ) + "\""			// 6.0.2.4 (2014/10/17) img タグに、title 属性追記
//							+	" onClick=\"hideClass( event, this, '" + hideClass + "', '" + openButton + "', '" + closeButton + "' )\"/> " ;
							+	" onClick=\"hideClass( event, this, '" + hideClass + "', '" + openButton + "', '" + closeButton + "' )\"> " ;		// 7.0.1.0 (2018/10/15)

//			buf.appendCR( "<script language=\"javascript\">" )
			buf.appendCR( "<script>" )											// 8.4.3.0 (2023/04/07)
				.append( "$(function(){$('#" , buttonId , "').append(\"" )
				.append( IMG.replace( "\"", "\\\"" ) )
				.appendCR( "\");})" )
				.appendCR( "</script>");
		}

		jspPrint( buf.toString() );
		return EVAL_BODY_INCLUDE ;		// Body インクルード( extends TagSupport 時)
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();

		jspPrint( CR + "</div>" );

		return EVAL_PAGE ;		// ページの残りを評価する。
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 5.5.4.4 (2012/07/20) hideClass対応
	 * @og.rev 7.3.2.3 (2021/04/09) ｼｽﾃﾑ定数のICON_DIRやIMAGE_DIRを使用します。
	 */
	@Override
	protected void release2() {
		super.release2();
		id			= null;
		buttonId	= "query-button-append";
		defaultNone = true;
//		openButton  = HybsSystem.sys( "JSP" ) + "/image/" + HybsSystem.sys( "HIDE_OPEN_BUTTON" );	// 開くボタン
		openButton  = JSP_ICON + HybsSystem.sys( "HIDE_OPEN_BUTTON" );		// 開くボタン
//		closeButton = HybsSystem.sys( "JSP" ) + "/image/" + HybsSystem.sys( "HIDE_CLOSE_BUTTON" );	// 閉じるボタン
		closeButton = JSP_ICON + HybsSystem.sys( "HIDE_CLOSE_BUTTON" );	// 閉じるボタン
		hideClass	= null;
	}

	/**
	 * 【HTML】要素に対して固有の名前(id)をつける場合に設定します。
	 *
	 * @og.tag
	 * 開閉される部分を囲っているDIVタグのBODY部分にIDを指定します。
	 * 特定のhideMenuにスタイルを指定したい場合等に利用して下さい。
	 *
	 * @param	input	BODY部分のID
	 */
	@Override
	public void setId( final String input) {
		id = nval( getRequestParameter( input ), id );
	}

	/**
	 * 【TAG】画像ボタン出力先のIDを指定します。
	 *
	 * @og.tag
	 * 初期値は"query-button-append"です。
	 * queryButtonタグが存在する画面では通常指定する必要はありません。
	 * queryButtonタグが存在しない場合は、別途用意したspanタグの
	 * IDを指定する事でボタンが表示されます。
	 *
	 * @param	input	出力先のID
	 *
	 */
	public void setButtonId( final String input) {
		buttonId = nval( getRequestParameter( input ), buttonId );
	}

	/**
	 * 【TAG】初期状態の開閉指定[[true:閉じている/false:開いている]を指定します(初期値:true)。
	 *
	 * @og.tag
	 * 初期状態が開いているか、閉じているかを指定します。
	 * 初期値は閉じる(true)です。
	 *
	 * @param flag 開閉指定 [true:閉じている/false:開いている]
	 *
	 */
	public void setDefaultNone( final String flag) {
		defaultNone = nval( getRequestParameter( flag ), defaultNone );
	}

	/**
	 * 【TAG】開くボタンの画像を指定します
	 * 		(初期値:"jsp/image/" + SystemData#HIDE_OPEN_BUTTON[={@og.value SystemData#HIDE_OPEN_BUTTON}])。
	 *
	 * @og.tag
	 * ボタンの画像を指定します。
	 * 絶対パスの場合は、システムの先頭から、相対パスの場合、基底は各画面のフォルダとなります。
	 * 初期値はjsp/imageフォルダ内のシステムパラメータHIDE_OPEN_BUTTONのファイルです。
	 * (/システムID/jsp/image/expand_u.png)
	 *
	 * @param	name	画像ファイル名
	 */
	public void setOpenButton( final String name) {
		openButton = nval( getRequestParameter( name ), openButton );
	}

	/**
	 * 【TAG】閉じるボタンの画像を指定します
	 * 		(初期値:"jsp/image/" + SystemData#HIDE_CLOSE_BUTTON[={@og.value SystemData#HIDE_CLOSE_BUTTON}])。
	 *
	 * @og.tag
	 * ボタンの画像を指定します。
	 * 絶対パスの場合は、システムの先頭から、相対パスの場合、基底は各画面のフォルダとなります。
	 * 初期値はjsp/imageフォルダ内のシステムパラメータHIDE_CLOSE_BUTTONのファイルです。
	 * (/システムID/jsp/image/shrink_u.png)
	 *
	 * @param	name	画像ファイル名
	 */
	public void setCloseButton( final String name) {
		closeButton = nval( getRequestParameter( name ), closeButton );
	}

	/**
	 * 【TAG】開閉するclassを指定します。
	 *
	 * @og.tag
	 * 開閉を制御するためのclass属性を外部から指定します。
	 * 指定しない場合は自動でユニークなClassがセットされます。
	 *
	 * @param	name	クラス
	 *
	 * @og.rev 5.5.4.4 (2012/07/20) 新規追加
	 */
	public void setHideClass( final String name) {
		hideClass = nval( getRequestParameter( name ), hideClass );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION						)
				.println( "id"				,id								)
				.println( "buttonId"		,buttonId						)
				.println( "defaultNone"		,defaultNone					)
				.println( "openButton"		,openButton						)
				.println( "closeButton"		,closeButton					)
				.println( "Other..."		,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
