/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.fukurou.system.OgBuilder ;					// 6.4.4.2 (2016/04/01)
import org.opengion.fukurou.system.DateSet;						// 6.4.2.0 (2016/01/29)
import org.opengion.fukurou.security.HybsCryptography;			// 5.9.6.3 (2016/03/18)
import org.opengion.fukurou.util.ToString;						// 6.8.5.0 (2018/01/09)
import org.opengion.fukurou.util.HybsDateUtil;					// 6.4.2.0 (2016/01/29)
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.ArraySet;						// 6.4.3.4 (2016/03/11)
import org.opengion.fukurou.db.Transaction;
import static org.opengion.fukurou.util.StringUtil.nval;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.db.Query;
import org.opengion.hayabusa.db.QueryFactory;

import java.util.Calendar;
import java.util.Enumeration;
import java.util.LinkedHashSet;
import java.util.Locale;
import java.util.List;											// 7.3.1.1 (2021/02/25)
import java.util.ArrayList;
import java.util.Arrays;										// 7.3.1.1 (2021/02/25)
import java.util.Set;
import java.util.Map;
import java.util.LinkedHashMap;									// 6.4.2.1 (2016/02/05)
import java.util.stream.Collectors;								// 8.1.0.1 (2022/01/07)
import java.util.function.Predicate;							// 8.0.1.2 (2021/11/19)
import java.text.NumberFormat;									// 6.2.4.2 (2015/05/29)
import java.text.DateFormat;									// 8.0.1.2 (2021/11/19)
import java.text.SimpleDateFormat;								// 8.0.1.2 (2021/11/19)

import jakarta.servlet.jsp.PageContext;
import javax.script.ScriptEngine;
import javax.script.ScriptEngineManager;
import javax.script.ScriptException;

/**
 * JSP上からｷｰ､値を設定することにより､ﾊﾟﾗﾒｰﾀとして､値を
 * 取出し可能にするﾀｸﾞです｡
 *
 * 通常のﾘｸｴｽﾄ情報と同じ扱いができます(優先順位は､ﾘｸｴｽﾄが上位)｡
 *
 * 設定した値は､{&#064;XXXX} 形式で 取り出すことができます｡
 * また､command ="GET" で 直接画面に値を書き出すことも可能です｡
 *
 * ※ このﾀｸﾞは､Transaction ﾀｸﾞの対象です｡
 *
 * &lt;style&gt;
 *   #valueJavaDoc th { vertical-align:top; margin:0px; padding:0px; }
 *   #valueJavaDoc td { text-align:center;  margin:0px; padding:0px; }
 *  .act {
 *      background-color: #ccffcc;
 *      height : 160px;
 *  }
 *  .cmd {
 *      background-color: #ffcccc;
 *      width  : 80px;
 *  }
 *  .arrow {
 *     border-top:  160px solid #ccffcc;
 *     border-left:  80px solid #ffcccc;
 *     position: absolute;
 *  }
 *  .actarw { margin-left:20px; position: absolute; }
 *  .cmdarw { margin-top:140px; position: absolute; }
 * &lt;/style&gt;
 *
 * @og.formSample
 * ●形式：&lt;og:value command="SET" key="ABC" value="123" /&gt;
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し､{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:value
 *       command            【TAG】ｺﾏﾝﾄﾞ (SET,GET,REMOVE,CLEAR,SQL,SETTBL,GETTBL,KEYTBL,SETMEM,SQLGET,DEBUG)をｾｯﾄします(初期値:SET)
 *       action             【TAG】ｱｸｼｮﾝ(UPPER,LOWER,MESSAGE,APPEND,ALL_APPEND,MAP,ALL_MAP,MAPOBJ,ALL_MAPOBJ,LIST,ALL_LIST,LIST_CSV,ALL_LIST_CSV,
 *                                  MERGE,FIRST,VIEWFIRST,ROW_APPEND,REPLACE,INDEX_OF,LASTINDEX_OF,SUBSTR,LENGTH,SET,ALL_SET,SET_CSV,ALL_SET_CSV,
 *                                  SPLIT,SPLIT_LIST,FOR,FOR_LIST,FORTM,FORTM_LIST,MAX_MIN,ALL_MAX_MIN,ADD,CASE,ESCAPE,TO_DATE,DAY_WEEK,
 *                                  URL_ENCODE,URL_ENCODE2,URL_DECODE,TRIM,HASH,TO_NUMBER,HMCSV,CRYPT_ENC,CRYPT_DEC,REQUEST,FORMAT)をｾｯﾄします
 *                                  7.3.0.0 (2021/01/06) EXEC 廃止
 *       key                【TAG】ﾊﾟﾗﾒｰﾀ に登録するｷｰをｾｯﾄします
 *       value              【TAG】ﾊﾟﾗﾒｰﾀ に登録する値をｾｯﾄします
 *       defaultVal         【TAG】value値がNULLの場合に､この初期値を設定します
 *       separator          【TAG】各種ｱｸｼｮﾝの文字列を連結/分解する項目区切り文字をｾｯﾄします(初期値:",")
 *       useMultiRows       【TAG】ﾏﾙﾁﾃﾞｰﾀ(複数件検索)を使用するかどうか[true/false]を指定します(初期値:false)
 *       scope              【TAG】ｷｬｯｼｭする場合のｽｺｰﾌﾟ(request,session)を指定します(初期値:request)
 *       tableId            【TAG】sessionから取得する DBTableModelｵﾌﾞｼﾞｪｸﾄの ID
 *       tblScope           【TAG】DBTableModel から取得する場合のｽｺｰﾌﾟ(request,session)を指定します(初期値:session)
 *       dbid               【TAG】(通常は使いません)Queryｵﾌﾞｼﾞｪｸﾄを作成する時のDB接続IDを指定します
 *       xssCheck           【TAG】ﾊﾟﾗﾒｰﾀの HTMLTag開始/終了文字(&gt;&lt;) 存在ﾁｪｯｸを実施するかどうか[true/false]を設定します (初期値:USE_XSS_CHECK[=true])
 *       caseKey            【TAG】このﾀｸﾞ自体を利用するかどうかの条件ｷｰを指定します(初期値:null)
 *       caseVal            【TAG】このﾀｸﾞ自体を利用するかどうかの条件値を指定します(初期値:null)
 *       caseNN             【TAG】指定の値が､null/ｾﾞﾛ文字列 でない場合(Not Null=NN)は､このﾀｸﾞは使用されます(初期値:判定しない)
 *       caseNull           【TAG】指定の値が､null/ｾﾞﾛ文字列 の場合は､このﾀｸﾞは使用されます(初期値:判定しない)
 *       caseIf             【TAG】指定の値が､true/TRUE文字列の場合は､このﾀｸﾞは使用されます(初期値:判定しない)
 *       fromVal            【TAG】部分文字列置換の 置換え前の部分文字列(from)を指定します･･･HMCSVの開始時刻にも使用します
 *       toVal              【TAG】部分文字列置換の 置換え後の部分文字列(to)を指定します･･･HMCSVの終了時刻にも使用します
 *       skipVal            【TAG】HMCSVで使用する､ｽﾃｯﾌﾟ時間を分で指定します(例：１時間=60 , ２時間=120 , 45分=45 など) 6.8.4.1 (2017/12/18)
 *       format             【TAG】actionが、FORMAT,FOR,FOR_LIST,FORTM,FORTM_LIST の場合の文字列変換ﾌｫｰﾏｯﾄを指定します。8.0.1.2 (2021/11/19)
 *       cryptKey           【TAG】暗号化と復号化で利用する､暗号化ｷｰを指定します｡
 *       debug              【TAG】ﾃﾞﾊﾞｯｸﾞ情報を出力するかどうか[true/false]を指定します(初期値:false)  5.10.4.0 (2018/10/05)
 *   &gt;   ... Body ...
 *   &lt;/og:value&gt;
 *
 * 【廃止】5.1.8.0 (2010/07/01) 物理削除
 *  //   nullSet            【廃止】value が NULL の時に､設定するかどうか[true/false]を指定します(初期値:true)
 *
 * ●使用例
 *     ･ &lt;og:value command="SET" key="ABC" value="123" /&gt;
 *     ･ &lt;og:value command="SQL"&gt;SELECT 1 TEST FROM DUAL&lt;/og:value&gt;
 *     ･ &lt;og:value command="SET" key="DEF" value="{&#064;NOCOMMAND}" defaultVal="0000" /&gt;
 *     ･ &lt;og:value command="GET" key="NOKEY" defaultVal="NODATA" /&gt;
 *     ･ &lt;og:value command="SET" key="{&#064;ABC}4" value="TEST1234" /&gt;
 *     ･ &lt;og:value command="GET" key="ABC" /&gt;
 *     ･ &lt;og:value command="SET" action="LOWER" key="LOWERTEST" value="ABCDEF" /&gt; ⇒ abcdef
 *     ･ &lt;og:value command="GET" key="LOWERTEST" /&gt;
 *     ･ &lt;og:value command="GET" action="UPPER" key="LOWERTEST" /&gt;
 *     ･ &lt;og:value command="REMOVE" key="ABC" /&gt;
 *     ･ &lt;og:value command="SET" action="FORMAT" key="ABC" value="15" format="%05d" <del>fromVal="%05d"</del> /&gt; ⇒ 00015
 *
 * <table class="plain" id="valueJavaDoc">
 *   <caption>command 一覧と使用可能 action の対応表</caption>
 *   <tr>
 *       <th><span class="arrow">Key</span><span class="actarw">action</span><span class="cmdarw">command</span></th>
 *       <th class="act">UPPER<br>LOWER<br>MESSAGE<br>DAY_WEEK<br>MERGE<br>REPLACE<br>
 *                      INDEX_OF<br>LASTINDEX_OF<br>SUBSTR<br>LENGTH<br>ADD<br>CASE<br>
 *                      ESCAPE<br>URL_ENCODE<br>URL_ENCODE2<br>URL_DECODE<br>TRIM<br>HASH<br>
 *                      TO_NUMBER<br>FORMAT</th>
 *       <th class="act">APPEND<br>ALL_APPEND</th>
 *       <th class="act">MAP<br>ALL_MAP<br>MAPOBJ<br>ALL_MAPOBJ<br>LIST<br>LIST_CSV<br>ALL_LIST<br>ALL_LIST_CSV<br>SET<br>SET_CSV<br>ALL_SET<br>ALL_SET_CSV</th>
 *       <th class="act">FIRST<br>VIEWFIRST</th>
 *       <th class="act">ROW_APPEND</th>
 *       <th class="act">SPLIT<br>SPLIT_LIST<br>FOR<br>FOR_LIST<br><br>FORTM<br>FORTM_LIST<br>MAX_MIN<br>ALL_MAX_MIN</th>
 *       <th class="act">HMCSV</th>										<!-- 6.8.4.1 (2017/12/18) -->
 *   </tr>
 *   <tr><td class="cmd">SET    </td><td>○</td><td>○</td><td>×</td><td>×</td><td>×</td><td>○</td><td>○</td></tr>
 *   <tr><td class="cmd">GET    </td><td>○</td><td>×</td><td>×</td><td>×</td><td>×</td><td>×</td><td>○</td></tr>
 *   <tr><td class="cmd">REMOVE </td><td>×</td><td>×</td><td>×</td><td>×</td><td>×</td><td>×</td><td>×</td></tr>
 *   <tr><td class="cmd">CLEAR  </td><td>×</td><td>×</td><td>×</td><td>×</td><td>×</td><td>×</td><td>×</td></tr>
 *   <tr><td class="cmd">SQL    </td><td>×</td><td>×</td><td>○</td><td>×</td><td>○</td><td>×</td><td>×</td></tr>
 *   <tr><td class="cmd">SETTBL </td><td>○</td><td>○</td><td>○</td><td>○</td><td>×</td><td>○</td><td>×</td></tr>
 *   <tr><td class="cmd">GETTBL </td><td>○</td><td>○</td><td>×</td><td>○</td><td>×</td><td>×</td><td>×</td></tr>
 *   <tr><td class="cmd">KEYTBL </td><td>○</td><td>○</td><td>×</td><td>○</td><td>×</td><td>○</td><td>×</td></tr>
 *   <tr><td class="cmd">SETMEM </td><td>○</td><td>○</td><td>×</td><td>×</td><td>×</td><td>○</td><td>×</td></tr>
 *   <tr><td class="cmd">SQLGET </td><td>×</td><td>×</td><td>×</td><td>×</td><td>○</td><td>×</td><td>×</td></tr>
 * </table>
 *
 * @og.group その他部品
 *
 * @version  4.0
 * @author   M.Endou
 * @since    JDK5.0,
 */
public class ValueTag extends CommonTagSupport {
	/** このﾌﾟﾛｸﾞﾗﾑのVERSION文字列を設定します｡	{@value} */
	private static final String VERSION = "8.2.0.1 (2022/06/17)" ;
	private static final long serialVersionUID = 820120220617L ;

	/** command 引数に渡す事の出来る ｺﾏﾝﾄﾞ  ｾｯﾄ {@value} */
	public static final String CMD_SET		= "SET";
	/** command 引数に渡す事の出来る ｺﾏﾝﾄﾞ  ｹﾞｯﾄ {@value} */
	public static final String CMD_GET		= "GET";
	/** command 引数に渡す事の出来る ｺﾏﾝﾄﾞ  ﾘﾑｰﾌﾞ {@value} */
	public static final String CMD_REMOVE	= "REMOVE";
	/** command 引数に渡す事の出来る ｺﾏﾝﾄﾞ  ｸﾘｱ {@value} */
	public static final String CMD_CLEAR	= "CLEAR";
	/** command 引数に渡す事の出来る ｺﾏﾝﾄﾞ  ＳＱＬ {@value} */
	public static final String CMD_SQL		= "SQL";
	/** command 引数に渡す事の出来る ｺﾏﾝﾄﾞ  ｾｯﾄﾃｰﾌﾞﾙ {@value} */
	public static final String CMD_SETTBL	= "SETTBL";
	/** command 引数に渡す事の出来る ｺﾏﾝﾄﾞ  ｹﾞｯﾄﾃｰﾌﾞﾙ {@value} */
	public static final String CMD_GETTBL	= "GETTBL";
	/** command 引数に渡す事の出来る ｺﾏﾝﾄﾞ  ｷｰﾃｰﾌﾞﾙ {@value} */
	public static final String CMD_KEYTBL	= "KEYTBL";
	/** command 引数に渡す事の出来る ｺﾏﾝﾄﾞ  ｾｯﾄﾒﾓﾘ {@value} */
	public static final String CMD_SETMEM	= "SETMEM";			// 3.7.1.0 (2005/04/15)
	/** command 引数に渡す事の出来る ｺﾏﾝﾄﾞ  ＳＱＬｹﾞｯﾄ {@value} */
	public static final String CMD_SQLGET	= "SQLGET";			// 5.1.7.0 (2010/06/01) SQLGET対応
	/** command 引数に渡す事の出来る ｺﾏﾝﾄﾞ  ﾃﾞﾊﾞｯｸﾞ {@value} */
	public static final String CMD_DEBUG	= "DEBUG";			// 6.7.7.2 (2017/04/14)

	// 6.4.3.4 (2016/03/11) String配列 から､Setに置き換えます｡
	private static final Set<String> COMMAND_SET = new ArraySet<>( CMD_SET , CMD_GET , CMD_REMOVE , CMD_SQL , CMD_SETTBL , CMD_GETTBL ,
																   CMD_KEYTBL , CMD_CLEAR , CMD_SETMEM, CMD_SQLGET );

	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  ｱｯﾊﾟｰ(大文字化) {@value} */
	public static final String ACT_UPPER		= "UPPER" ;
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  ﾛｰﾜｰ(小文字化) {@value} */
	public static final String ACT_LOWER		= "LOWER" ;
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  ﾒｯｾｰｼﾞ変換 {@value} */
	public static final String ACT_MESSAGE		= "MESSAGE" ;
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  ﾃﾞｰﾀｱﾍﾟﾝﾄﾞ {@value} */
	public static final String ACT_APPEND		= "APPEND" ;
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  ｵｰﾙｱﾍﾟﾝﾄﾞ {@value} */
	public static final String ACT_ALL_APPEND	= "ALL_APPEND" ;				// 3.6.1.0 (2005/01/05)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  MAP {@value} */
	public static final String ACT_MAP			= "MAP" ;						// 5.5.0.3 (2012/03/12)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  ALL_MAP {@value} */
	public static final String ACT_ALL_MAP		= "ALL_MAP" ;					// 5.5.0.3 (2012/03/12)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  MAP {@value} */
	public static final String ACT_MAPOBJ		= "MAPOBJ" ;					// 6.2.4.2 (2015/05/29)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  ALL_MAP {@value} */
	public static final String ACT_ALL_MAPOBJ	= "ALL_MAPOBJ" ;				// 6.2.4.2 (2015/05/29)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  LIST {@value} */
	public static final String ACT_LIST			= "LIST" ;						// 4.3.7.5 (2009/07/13)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  ALL_LIST {@value} */
	public static final String ACT_ALL_LIST		= "ALL_LIST" ;					// 4.3.7.5 (2009/07/13)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  LIST_CSV {@value} */
	public static final String ACT_LIST_CSV		= "LIST_CSV" ;					// 8.1.0.1 (2022/01/07)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  ALL_LIST_CSV {@value} */
	public static final String ACT_ALL_LIST_CSV	= "ALL_LIST_CSV" ;				// 8.1.0.1 (2022/01/07)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  SET {@value} */
	public static final String ACT_SET			= "SET" ;						// 8.1.0.1 (2022/01/07)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  ALL_SET {@value} */
	public static final String ACT_ALL_SET		= "ALL_SET" ;					// 8.1.0.1 (2022/01/07)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  SET_CSV {@value} */
	public static final String ACT_SET_CSV		= "SET_CSV" ;					// 8.1.0.1 (2022/01/07)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  ALL_SET_CSV {@value} */
	public static final String ACT_ALL_SET_CSV	= "ALL_SET_CSV" ;				// 8.1.0.1 (2022/01/07)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  日付前方まるめ {@value} */
	public static final String ACT_DAY_WEEK		= "DAY_WEEK" ;					// 3.7.1.0 (2005/04/15)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  ﾏｰｼﾞ {@value} */
	public static final String ACT_MERGE		= "MERGE" ;						// 3.7.1.1 (2005/05/23)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  ﾌｧｰｽﾄ {@value} */
	public static final String ACT_FIRST		= "FIRST" ;						// 3.8.0.4 (2005/08/08)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  ﾋﾞｭｰﾌｧｰｽﾄ {@value} */
	public static final String ACT_VIEWFIRST	= "VIEWFIRST" ;					// 7.3.1.1 (2021/02/25)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  縦横回転 {@value} */
	public static final String ACT_ROW_APPEND	= "ROW_APPEND" ;				// 3.8.9.2 (2007/07/28)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  文字列置換 {@value} */
	public static final String ACT_REPLACE		= "REPLACE" ;					// 5.2.2.0 (2010/11/01)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  部分文字列 {@value} */
	public static final String ACT_SUBSTR		= "SUBSTR" ;					// 5.2.2.0 (2010/11/01)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  文字数(char数) {@value} */
	public static final String ACT_LENGTH		= "LENGTH" ;					// 7.2.6.0 (2020/06/30)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  INDEX_OF数 {@value} */
	public static final String ACT_INDEX_OF		= "INDEX_OF" ;					// 7.2.6.0 (2020/06/30)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  LASTINDEX_OF数 {@value} */
	public static final String ACT_LASTINDEX_OF	= "LASTINDEX_OF" ;				// 7.2.6.0 (2020/06/30)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  文字列分割 {@value} */
	public static final String ACT_SPLIT		= "SPLIT" ;						// 5.2.2.0 (2010/11/01)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  文字列分割後LIST登録 {@value} */
	public static final String ACT_SPLIT_LIST		= "SPLIT_LIST" ;			// 7.3.1.1 (2021/02/25)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  fromVal,toVal,skipVal を使用した連結文字列を作成する {@value} */
	public static final String ACT_FOR			= "FOR" ;						// 7.3.1.1 (2021/02/25)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  fromVal,toVal,skipVal を使用した文字列をLIST登録 {@value} */
	public static final String ACT_FOR_LIST		= "FOR_LIST" ;					// 7.3.1.1 (2021/02/25)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  fromVal,toVal,skipVal を使用した日付文字列を作成する {@value} */
	public static final String ACT_FORTM		= "FORTM" ;						// 8.0.1.2 (2021/11/19)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  fromVal,toVal,skipVal を使用した日付文字列をLIST登録 {@value} */
	public static final String ACT_FORTM_LIST	= "FORTM_LIST" ;				// 8.0.1.2 (2021/11/19)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  最大_最小値 {@value} */
	public static final String ACT_MAX_MIN		= "MAX_MIN" ;					// 5.6.4.3 (2013/05/24)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  ALL_最大_最小値 {@value} */
	public static final String ACT_ALL_MAX_MIN	= "ALL_MAX_MIN" ;				// 5.6.4.3 (2013/05/24)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  計算処理結果 {@value} */
	public static final String ACT_EXEC			= "EXEC" ;						// 5.7.7.2 (2014/06/20)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  値をﾌﾟﾗｽする {@value} */
	public static final String ACT_ADD			= "ADD" ;						// 7.3.0.0 (2021/01/06)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  条件分岐 {@value} */
	public static final String ACT_CASE			= "CASE" ;						// 5.7.7.2 (2014/06/20)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  HTML上のｴｽｹｰﾌﾟ文字 {@value} */
	public static final String ACT_ESCAPE		= "ESCAPE" ;					// 6.3.5.0 (2015/08/08)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  UTF-8 でURLｴﾝｺｰﾄﾞ {@value} */
	public static final String ACT_ENCODE		= "URL_ENCODE" ;					// 6.9.7.0 (2018/05/14)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  UTF-8 で部分URLｴﾝｺｰﾄﾞ {@value} */
	public static final String ACT_ENCODE2		= "URL_ENCODE2" ;					// 6.9.7.0 (2018/05/14)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  UTF-8 でURLﾃﾞｺｰﾄﾞ {@value} */
	public static final String ACT_DECODE		= "URL_DECODE" ;					// 6.9.7.0 (2018/05/14)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  ｽﾍﾟｰｽ削除 {@value} */
	public static final String ACT_TRIM			= "TRIM" ;						// 6.4.1.2 (2016/01/22)
//	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  MD5 {@value} */
//	public static final String ACT_MD5			= "MD5" ;						// 5.9.6.3 (2016/03/18) 8.1.2.0 (2022/03/10) Delete
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  HASH {@value} */
	public static final String ACT_HASH			= "HASH" ;						// 8.1.2.0 (2022/03/10) Add
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  TO_NUMBER {@value} */
	public static final String ACT_TO_NUMBER	= "TO_NUMBER" ;					// 6.7.4.0 (2017/02/10)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  TO_DATE {@value} */
	public static final String ACT_TO_DATE		= "TO_DATE" ;					// 8.0.1.2 (2021/11/19)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  HMCSV {@value} */
	public static final String ACT_HMCSV		= "HMCSV" ;						// 6.8.4.1 (2017/12/18)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  CRYPT_ENC {@value} */
	public static final String ACT_CRYPT_ENC	= "CRYPT_ENC";					// 5.10.4.0 (2018/10/05)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  CRYPT_DEC {@value} */
	public static final String ACT_CRYPT_DEC	= "CRYPT_DEC";					// 5.10.4.0 (2018/10/05)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  REQUEST {@value} */
	public static final String ACT_REQUEST		= "REQUEST";					// 7.2.6.1 (2020/07/17)
	/** action 引数に渡す事の出来る ｱｸｼｮﾝ  FORMAT {@value} */
	public static final String ACT_FORMAT		= "FORMAT";						// 7.4.2.0 (2021/05/08)

	/** action 引数に渡す事の出来る ｱｸｼｮﾝ ﾘｽﾄ  */
	// 6.4.3.4 (2016/03/11) String配列 から､Setに置き換えます｡
	// 6.8.4.1 (2017/12/18) ACT_HMCSV
	// 6.9.7.0 (2018/05/14) ACT_ENCODE , ACT_ENCODE2 , ACT_DECODE
	// 5.10.4.0 (2018/10/05) ACT_CRYPT_ENC, ACT_CRYPT_DEC
	// 7.2.6.0 (2020/06/30) ACT_LENGTH
	// 7.3.0.0 (2021/01/06) ACT_ADD , (ACT_EXECは基本廃止)
	// 7.4.2.0 (2021/05/08) ACT_FORMAT追加
	// 8.0.1.2 (2021/11/19) ACT_TO_DATE,ACT_FORTM,ACT_FORTM_LIST 追加
	// 8.1.2.0 (2022/03/10) ACT_MD5 を ACT_HASH に変更
	private static final Set<String> ACTION_SET = new ArraySet<>(
				ACT_UPPER,ACT_LOWER,ACT_MESSAGE,ACT_APPEND,ACT_ALL_APPEND,
				ACT_MAP,ACT_ALL_MAP,ACT_MAPOBJ,ACT_ALL_MAPOBJ,ACT_LIST,ACT_LIST_CSV,ACT_ALL_LIST,ACT_ALL_LIST_CSV,ACT_SET,ACT_SET_CSV,ACT_ALL_SET,ACT_ALL_SET_CSV,
				ACT_DAY_WEEK,ACT_MERGE,ACT_FIRST,ACT_VIEWFIRST,ACT_ROW_APPEND,ACT_REPLACE,ACT_INDEX_OF,ACT_LASTINDEX_OF,ACT_SUBSTR,ACT_LENGTH,
				ACT_SPLIT,ACT_SPLIT_LIST,ACT_FOR,ACT_FOR_LIST,ACT_FORTM,ACT_FORTM_LIST,ACT_MAX_MIN,ACT_ALL_MAX_MIN,ACT_EXEC,
				ACT_CASE,ACT_ESCAPE,ACT_ENCODE,ACT_ENCODE2,ACT_DECODE,
				ACT_TRIM,ACT_HASH,ACT_TO_NUMBER,ACT_TO_DATE,ACT_HMCSV,ACT_CRYPT_ENC,ACT_CRYPT_DEC,ACT_REQUEST,ACT_ADD,ACT_FORMAT
	);

//	/** 6.8.4.1 (2017/12/18) action="HMCSV" で､ﾙｰﾌﾟ制限します｡ */
//	8.2.0.1 (2022/06/17) MAX_CNT 廃止
//	public static final int MAX_CNT = 361;	// 8.0.1.2 (2021/11/19) 300 → 361 に変更(360まで許可する)

	/** 8.0.1.2 (2021/11/19) 日付のﾌｫｰﾏｯﾄの初期値 */
	public static final String YMDHMS = "yyyyMMddHHmmss" ;
	private static final String	HASH_CODE = HybsSystem.sys( "FILE_HASH_CODE" );	// 8.1.2.0 (2022/03/10)

	private transient DBTableModel table;

	private String		tableId		= HybsSystem.TBL_MDL_KEY;
	private String		command		= CMD_SET;
	private String		key			;
	private String		inValue		;		// 3.5.4.0 (2003/11/25)
	private String		value		;
	private String		defaultVal	;
	private String		action		;
	private String		dbid		;				// 4.0.0.0 (2007/10/10) dbid の初期値を､"DEFAULT" から null に変更
	private String		scope		= "request";	// "request","session"
	private String		tblScope	= "session";	// 5.1.2.0 (2010/01/01) DBTableModel の取得先のscope
	private String		separator	= ",";			// 項目区切り文字
	private boolean		useMultiRows ;				// 3.2.4.0 (2003/06/12) ﾏﾙﾁﾃﾞｰﾀ(複数件検索)を使用するかしないか｡初期値:使用せず
	private boolean		xssCheck	= HybsSystem.sysBool( "USE_XSS_CHECK" ); // 5.1.7.0 (2010/06/01) XSS対策

	private String	fromVal			;					// 5.2.2.0 (2010/11/01)
	private String	toVal			;					// 5.2.2.0 (2010/11/01)
	private String	skipVal			;					// 6.8.4.1 (2017/12/18)
	private String	format			;					// 8.0.1.2 (2021/11/19)
	private String	cryptKey		;					// 5.10.4.0 (2018/10/05) 暗号化ｷｰ

	private boolean	tmpSelectedAll	;					// 5.6.4.3 (2013/05/24) action="ALL_xxx"を tmpSelectedAll="true" で内部処理します｡

	/**
	 * ﾃﾞﾌｫﾙﾄｺﾝｽﾄﾗｸﾀｰ
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public ValueTag() { super(); }		// これも､自動的に呼ばれるが､空のﾒｿｯﾄﾞを作成すると警告されるので､明示的にしておきます｡

	/**
	 * Taglibの開始ﾀｸﾞが見つかったときに処理する doStartTag() を ｵｰﾊﾞｰﾗｲﾄﾞします｡
	 *
	 * @og.rev 3.1.7.0 (2003/05/02) isNullSet 属性が true(初期値)のときは､ﾘｸｴｽﾄ情報から値を取得｡
	 * @og.rev 3.1.7.0 (2003/05/02) scope 属性を設定するﾀｲﾐﾝｸﾞを､早くする｡
	 * @og.rev 3.5.4.0 (2003/11/25) getRequestParameter( value ) ﾒｿｯﾄﾞを setValue に移動｡
	 * @og.rev 5.1.7.0 (2010/06/01) SQLGET対応
	 * @og.rev 5.1.8.0 (2010/07/01) isNullSet 属性 廃止
	 * @og.rev 5.2.2.0 (2010/11/01) caseKey ､caseVal 属性対応
	 * @og.rev 6.4.8.1 (2016/07/02) xssCheckを､doStartTag に移動
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doStartTag() {
		// 5.2.2.0 (2010/11/01) caseKey ､caseVal 属性対応
		if( useTag() ) {
			useXssCheck( xssCheck );		// 6.4.8.1 (2016/07/02)

			super.setScope( scope );		// ﾃﾞﾌｫﾙﾄscope が "request" なので､再設定している｡

	//		if( isNullSet ) {
	//			setUseValue( false );
	//		}

			// 5.1.7.0 (2010/06/01) SQLGET対応
			if( CMD_SQL.equals( command ) || CMD_SET.equals( command ) || CMD_SQLGET.equals( command ) ) {
				return EVAL_BODY_BUFFERED ;		// Body を評価する
			}
		}
		return SKIP_BODY ;				// Body を評価しない
	}

	/**
	 * Taglibのﾀｸﾞ本体を処理する doAfterBody() を ｵｰﾊﾞｰﾗｲﾄﾞします｡
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) ﾎﾞﾃﾞｨの内容を取得する処理を､CommonTagSupport で行う｡
	 * @og.rev 3.6.0.8 (2004/11/19) ｴﾗｰ発生時に確実にﾘﾘｰｽされるように try finally 追加
	 * @og.rev 3.8.7.0 (2006/12/15) ｱｸｾｽﾛｸﾞ取得の為,ApplicationInfoｵﾌﾞｼﾞｪｸﾄを設定
	 * @og.rev 4.0.0.0 (2005/01/31) lang ⇒ ResourceManager へ変更
	 * @og.rev 5.1.7.0 (2010/06/01) SQLGET対応
	 * @og.rev 5.1.7.0 (2010/06/01) XSS解除対応
	 * @og.rev 5.1.9.0 (2010/08/01) TransactionTag 対応｡上位に TransactionTag があれば､そこからConnection をもらう｡
	 * @og.rev 5.2.1.0 (2010/10/01) command="SET" action="APPEND"でvalueをbody部に書いた場合に動作しないﾊﾞｸﾞを修正
	 * @og.rev 5.3.7.0 (2011/07/01) TransactionReal の引数変更
	 * @og.rev 5.3.8.0 (2011/08/01) Transaction発生箇所でclose()
	 * @og.rev 6.3.6.1 (2015/08/28) Transaction でAutoCloseableを使用したtry-with-resources構築に対応｡
	 * @og.rev 6.3.6.1 (2015/08/28) QueryFactory.close( Query ) 廃止｡Queryはｷｬｯｼｭしません｡
	 * @og.rev 6.4.8.1 (2016/07/02) xssCheckを､doStartTag に移動
	 * @og.rev 8.0.2.0 (2021/11/30) 検索実行前に、SQL文字をdebugPrint出来るように修正
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		if( value == null || value.length() <= 0 ) {
			// 5.1.7.0 (2010/06/01) XSS対策

			value = getBodyString();
			// 5.2.1.0 (2010/10/01)
			inValue = getBodyRawString();

			// 5.1.7.0 (2010/06/01) SQLGET対応
			if( CMD_SQL.equals( command ) || CMD_SQLGET.equals( command ) ) {
				// 6.3.6.1 (2015/08/28) Transaction でAutoCloseableを使用したtry-with-resources構築に対応｡
				try( Transaction tran = getTransaction() ) {
					final Query query = QueryFactory.newInstance();			// 4.0.0 (2005/01/31)
					value = value.trim();
					debugPrint( value );									// 8.0.2.0 (2021/11/30)

					query.setConnection( tran.getConnection( dbid ) );		// 6.3.6.1 (2015/08/28)
					query.setResourceManager( getResource() );				// 4.0.0 (2005/01/31)
					query.setStatement( value );
					query.execute();

					table = query.getDBTableModel();
					tran.commit();											// 6.3.6.1 (2015/08/28)
				}	// 5.3.8.0 (2011/08/01) Transaction発生箇所でclose()	(6.3.6.1 (2015/08/28) Transaction でAutoCloseableを使用)
			}
		}

		return SKIP_BODY ;
	}

	/**
	 * Taglibの終了ﾀｸﾞが見つかったときに処理する doEndTag() を ｵｰﾊﾞｰﾗｲﾄﾞします｡
	 *
	 * @og.rev 3.1.0.1 (2003/03/26) DBTableModelの値をSET/GETできる command , action を追加｡
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応｡release2() を doEndTag()で呼ぶ｡
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応｡release2() を doEndTag()で呼ぶ｡
	 * @og.rev 3.1.5.0 (2003/04/22) DBTableModel が存在するときのみ実行するﾛｼﾞｯｸになっていたﾊﾞｸﾞ対応｡
	 * @og.rev 5.1.2.0 (2010/01/01) DBTableModel の取得先の tblScope を追加｡
	 * @og.rev 5.1.7.0 (2010/06/01) SQLGET対応
	 * @og.rev 5.2.2.0 (2010/11/01) caseKey ､caseVal 属性対応
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		// 5.2.2.0 (2010/11/01) caseKey ､caseVal 属性対応
		if( useTag() ) {
			if( check( command, COMMAND_SET ) ) {
				if( CMD_SETTBL.equals( command ) ||
					CMD_GETTBL.equals( command ) ||
					CMD_KEYTBL.equals( command ) ) {
	//					table = (DBTableModel)getSessionAttribute( tableId );
						// 5.1.2.0 (2010/01/01) DBTableModel の取得先の tblScope を追加｡
						if( "session".equals( tblScope ) ) { table = (DBTableModel) getSessionAttribute( tableId ); }
						else if( "request".equals( tblScope ) ) { table = (DBTableModel) getRequestAttribute( tableId ); }
						else {
							final String errMsg = "このｽｺｰﾌﾟはｻﾎﾟｰﾄされていません｡[" + tblScope + "]";
							throw new IllegalArgumentException( errMsg );
						}
				}
			}

			commandExec( command );

			// 5.1.7.0 (2010/06/01) SQLGET対応
			if( CMD_GET.equals( command ) || CMD_GETTBL.equals( command ) || CMD_SQLGET.equals( command ) ) {
				if( value != null ) { jspPrint( value ); }
			}
		}
		return EVAL_PAGE ;
	}

	/**
	 * ﾀｸﾞﾘﾌﾞｵﾌﾞｼﾞｪｸﾄをﾘﾘｰｽします｡
	 * ｷｬｯｼｭされて再利用されるので､ﾌｨｰﾙﾄﾞの初期設定を行います｡
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) ｶｽﾀﾑﾀｸﾞの release() ﾒｿｯﾄﾞを､追加
	 * @og.rev 3.1.0.1 (2003/03/26) DBTableModelの値をSET/GETできる command , action を追加｡
	 * @og.rev 3.1.0.1 (2003/03/26) query 属性を削除します｡
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応｡release2() を doEndTag()で呼ぶ｡
	 * @og.rev 3.2.4.0 (2003/06/12) ﾏﾙﾁﾃﾞｰﾀ(複数件検索)を使用するかしないか｡
	 * @og.rev 3.5.4.0 (2003/11/25) inValue 変数の追加
	 * @og.rev 4.0.0.0 (2007/10/10) dbid の初期値を､"DEFAULT" から null に変更
	 * @og.rev 5.1.2.0 (2010/01/01) DBTableModel の取得先の tblScope を追加｡
	 * @og.rev 5.1.7.0 (2010/06/01) XSS解除対応
	 * @og.rev 5.1.8.0 (2010/07/01) isNullSet 属性 廃止
	 * @og.rev 5.2.2.0 (2010/11/01) fromVal , toVal 属性 追加
	 * @og.rev 5.6.4.3 (2013/05/24) parameter 属性は､未使用なので削除
	 * @og.rev 5.6.4.3 (2013/05/24) tmpSelectedAll 属性追加｡action="ALL_xxx"を tmpSelectedAll="true" に置き換えます｡
	 * @og.rev 6.8.4.1 (2017/12/18) skipVal 属性 追加
	 * @og.rev 5.10.4.0 (2018/10/03) CRYPT追加
	 * @og.rev 8.0.1.2 (2021/11/19) format 属性 追加
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		tableId			= HybsSystem.TBL_MDL_KEY;
		command			= CMD_SET;
		key				= null;
		value			= null;
		defaultVal		= null;
		action			= null;
		table			= null;
		dbid			= null;
		scope			= "request";	// "request","session"
		tblScope		= "session";	// 5.1.2.0 (2010/01/01) DBTableModel の取得先のscope
		separator		= ",";
		useMultiRows	= false;
		inValue			= null;			// 3.5.4.0 (2003/11/25)
		xssCheck		= HybsSystem.sysBool( "USE_XSS_CHECK" );	// 5.1.7.0 (2010/06/01) XSS解除対応
		fromVal			= null;			// 5.2.2.0 (2010/11/01)
		toVal			= null;			// 5.2.2.0 (2010/11/01)
		skipVal			= null;			// 6.8.4.1 (2017/12/18)
		format			= null;			// 8.0.1.2 (2021/11/19)
		cryptKey		= null;			// 5.10.4.0 (2018/10/05)
		tmpSelectedAll	= false;		// 5.6.4.3 (2013/05/24)
	}

	/**
	 * ｺﾏﾝﾄﾞを実行します｡
	 *
	 * ｺﾏﾝﾄﾞは,HTMLから(get/post)指定されますので,CMD_xxx で設定される
	 * ﾌｨｰﾙﾄﾞ定数値のいづれかを､指定できます｡
	 * ｺﾏﾝﾄﾞを登録すると同時に,実行も行ないます｡
	 *
	 * @og.rev 3.1.0.1 (2003/03/26) command に､SETTBL / GETTBL / KEYTBL / CLEAR を追加｡
	 * @og.rev 3.1.0.1 (2003/03/26) query 属性を削除します｡
	 * @og.rev 3.1.7.0 (2003/05/02) scope 属性を設定するﾀｲﾐﾝｸﾞを､早くする｡
	 * @og.rev 3.7.1.0 (2005/04/15) command に､SETMEM を追加｡
	 * @og.rev 5.1.7.0 (2010/06/01) SQLGET対応
	 * @og.rev 6.7.7.2 (2017/04/14) command に､DEBUG 追加
	 * @og.rev 8.0.0.0 (2021/07/09) action="ADD" の時は､valueのnull時にdefaultValを設定しない｡
	 *
	 * @param	command	ｺﾏﾝﾄﾞ (public static final 宣言されている文字列)
	 * @see		<a href="../../../../constant-values.html#org.opengion.hayabusa.taglib.ValueTag.CMD_GET">ｺﾏﾝﾄﾞ定数</a>
	 */
	private void commandExec( final String command ) {

		if( CMD_SQL.equals( command ) ) {
			setSQLAttribute( table );
		}
		else if( CMD_SQLGET.equals( command ) ) {
			value = getSQLAttribute( table );
		}
		else if( CMD_SET.equals( command ) ) {
			// 8.0.0.0 (2021/07/09) action="ADD" の時は､valueのnull時にdefaultValを設定しない｡
			if( !ACT_ADD.equals( action ) ) {			// つまり､ADD 以外は､設定する｡
				value = nval( value, defaultVal );
			}
			setAttribute( key,value,action );
		}
		else if( CMD_GET.equals( command ) ) {
			value = getAttribute( key,action );
		}
		else if( CMD_REMOVE.equals( command ) ) {
			removeAttribute( key );
		}
		else if( CMD_CLEAR.equals( command ) ) {
			clearAttribute( key );
		}
		else if( CMD_SETTBL.equals( command ) ) {
			setTableAttribute( table,key,action );
		}
		else if( CMD_GETTBL.equals( command ) ) {
			value = getTableAttribute( table,key,action );
		}
		else if( CMD_KEYTBL.equals( command ) ) {
			setKeyTableAttribute( table,key,value,action );
		}
		else if( CMD_SETMEM.equals( command ) ) {		// 3.7.1.0 (2005/04/15)
			// 8.0.0.0 (2021/07/09) action="ADD" の時は､valueのnull時にdefaultValを設定しない｡
			if( !ACT_ADD.equals( action ) ) {			// つまり､ADD 以外は､設定する｡
				value = nval( value, defaultVal );
			}
			setAttribute( key,value,action );
			setRequestCacheData( key,(String)getObject( key ) );
		}
		else if( CMD_DEBUG.equals( command ) ) {
			value = scopeAttributePrint();				// 6.7.7.2 (2017/04/14)
		}
	}

	/**
	 * ｱｸｼｮﾝを実行します｡
	 *
	 * ｺﾏﾝﾄﾞは action 属性で指定します｡
	 * action ｺﾏﾝﾄﾞ が､ null の場合は､なにも実行しません｡
	 *
	 * @og.rev 3.0.1.3 (2003/03/11) MESSAGE action を追加
	 * @og.rev 3.1.0.1 (2003/03/26) 引数を与えて処理する様に変更する｡
	 * @og.rev 3.7.1.0 (2005/04/15) action に､DAY_WEEK を追加｡
	 * @og.rev 3.7.1.1 (2005/05/23) action に､MERGE を追加｡
	 * @og.rev 4.0.0.0 (2007/10/18) ﾒｯｾｰｼﾞﾘｿｰｽ統合( getResource().getMessage ⇒ getResource().getLabel )
	 * @og.rev 5.2.2.0 (2010/11/01) ACT_MERGE 時には､ｶﾝﾏで分解､separator で合成を行います｡
	 * @og.rev 5.2.2.0 (2010/11/01) ACT_REPLACE 処理を新規追加します｡
	 * @og.rev 5.5.7.2 (2012/10/09) HybsDateUtil を利用するように修正します｡
	 * @og.rev 5.6.6.1 (2013/07/12) value が null の場合は､無視します｡
	 * @og.rev 5.7.7.2 (2014/06/20) EXEC と CASE ｱｸｼｮﾝを追加
	 * @og.rev 6.3.1.1 (2015/07/10) action="REPLACE" の､fromVal,toVal は､CSV形式で指定可能にする｡
	 * @og.rev 6.3.1.1 (2015/07/10) action="ESCAPE" 追加
	 * @og.rev 6.3.5.0 (2015/08/08) action="ESCAPE" 追加
	 * @og.rev 6.3.9.0 (2015/11/06) 文字列配列を toString() する｡
	 * @og.rev 6.4.1.2 (2016/01/22) TRIM ｱｸｼｮﾝを追加｡
	 * @og.rev 6.4.2.0 (2016/01/29) HybsDateUtil.getCalendar( String ) を直接利用するように修正します｡
	 * @og.rev 5.9.6.3 (2016/02/18) MD5 ｱｸｼｮﾝを追加
	 * @og.rev 6.7.4.0 (2017/02/10) TO_NUMBER ｱｸｼｮﾝを追加
	 * @og.rev 6.8.4.1 (2017/12/18) HMCSV ｱｸｼｮﾝを追加
	 * @og.rev 6.9.1.0 (2018/02/26) HMCSV ｱｸｼｮﾝに､MAX,MIN,HMF を追加
	 * @og.rev 6.9.2.1 (2018/03/12) HybsDateUtil.getDateFormat の ﾒｿｯﾄﾞ変更(引数の簡素化)
	 * @og.rev 6.9.7.0 (2018/05/14) URL_ENCODE,URL_ENCODE2,URL_DECODE を追加
	 * @og.rev 5.10.4.0 (2018/10/05) CRYPT追加
	 * @og.rev 7.2.6.0 (2020/06/30) INDEX_OF ,LASTINDEX_OF, LENGTH追加
	 * @og.rev 7.2.6.1 (2020/07/17) ACT_REQUEST追加
	 * @og.rev 7.3.0.0 (2021/01/06) JavaScriptｴﾝｼﾞﾝをNashornからGraalJSに移行(ただし､未対応)
	 * @og.rev 7.3.0.0 (2021/01/06) ADDｱｸｼｮﾝを追加
	 * @og.rev 7.3.1.3 (2021/03/09) REPLACEｱｸｼｮﾝで､fromVal,toValの分割を､ｶﾝﾏ固定から､separator 使用に変更
	 * @og.rev 7.4.2.0 (2021/05/08) FORMAT追加
	 * @og.rev 8.0.1.2 (2021/11/19) TO_DATE,FORTM,FORTM_LIST追加
	 * @og.rev 8.1.2.0 (2022/03/10) ACT_MD5 を ACT_HASH に変更
	 * @og.rev 8.2.0.1 (2022/06/17) ACT_HMCSV ｱｸｼｮﾝで、fromVal,toVal に日付指定を可能にする。
	 *
	 * @param action	ｺﾏﾝﾄﾞ (public static final 宣言されている文字列)
	 * @param value		旧の値
	 *
	 * @return	処理後の値
	 */
	private String actionExec( final String action,final String value ) {
		String rtn = value;

		// 8.2.0.1 (2022/06/17) ACT_HMCSV ｱｸｼｮﾝで、fromVal が null でなく、8桁以上(つまり時:分:秒 でなく年月日を含むｹｰｽ)
		final boolean hmCSVfrom = ACT_HMCSV.equals( action ) && fromVal != null && fromVal.length() > 8 ;

		// 5.6.6.1 (2013/07/12) value が null の場合は､無視します｡
//		if( action == null || value == null ) { return rtn; }
		if( action == null || value == null && !hmCSVfrom ) { return rtn; }

		if( ACT_UPPER.equals( action ) ) {
			// Localeを共通管理するようになった場合､String.toUpperCase( Locale locale )使用の事
			rtn = value.toUpperCase(Locale.JAPAN);
		}
		else if( ACT_LOWER.equals( action ) ) {
			// Localeを共通管理するようになった場合､String.toLowerCase( Locale locale )使用の事
			rtn = value.toLowerCase(Locale.JAPAN);
		}
		else if( ACT_MESSAGE.equals( action ) ) {
			// 引数をﾒｯｾｰｼﾞﾘｿｰｽのｷｰとして､ﾒｯｾｰｼﾞ変換する｡
			rtn = getResource().getLabel( value );
		}
		else if( ACT_DAY_WEEK.equals( action ) ) {
			// 日付型文字列(YYYYMMDD) の入力ﾃﾞｰﾀを､開始日を月曜日にｾｯﾄします｡
			// SUNDAY=1 , MONDAY=2 になります｡月曜日との差だけ､前に戻します｡
			// 指定日が日曜日の場合は､次の日(月曜日)に進めます｡
			final Calendar ymd = HybsDateUtil.getCalendar( value );				// 6.4.2.0 (2016/01/29)
			final int shu = ymd.get( Calendar.DAY_OF_WEEK ) - Calendar.MONDAY ;

			if( shu != 0 ) { ymd.add( Calendar.DATE, -shu ); }

			rtn = DateSet.getDate( ymd.getTimeInMillis() , "yyyyMMdd" );		// 5.5.7.2 (2012/10/09) HybsDateUtil を利用
		}
		// 3.7.1.1 (2005/05/23)
		else if( ACT_MERGE.equals( action ) ) {
			final Set<String> set = new LinkedHashSet<>();
			// 引数をｶﾝﾏで文字列配列に分解します｡
			final String[] vals = StringUtil.csv2Array( value );
			// 7.2.9.4 (2020/11/20) PMD:This for loop can be replaced by a foreach loop
			for( final String str : vals ) {
				if( str != null && str.length() > 0 ) {
					set.add( str );
				}
			}
//			final String[] str = StringUtil.csv2Array( value );
//			for( int i=0; i<str.length; i++ ) {
//				if( str[i] != null && str[i].length() > 0 ) {
//					set.add( str[i] );
//				}
//			}

			// 分解後､ﾏｰｼﾞ(Setで)されます｡ 登録順は､ｷｰﾌﾟします｡
			rtn = StringUtil.iterator2line( set.iterator(),separator );	// 5.2.2.0 (2010/11/01) separator 使用
		}
		// 5.2.2.0 (2010/11/01) ACT_REPLACE 処理を新規追加
		else if( ACT_REPLACE.equals( action ) ) {
			// value.replaceAll( from, to ) という文法で処理します｡
			// 6.0.2.5 (2014/10/31) null でないことがわかっている値の冗長な null ﾁｪｯｸがあります｡
			if( fromVal != null && toVal != null ) {
				// 6.3.1.1 (2015/07/10) CSV形式可能にする｡
				if( fromVal.indexOf( ',' ) < 0 ) {
					rtn = value.replaceAll( fromVal, toVal );
				}
				else {
//					final String[] frmStr = fromVal.split( "," );
//					final String[] toStr  = toVal.split( "," );
					final String[] frmStr = fromVal.split( separator );			// 7.3.1.3 (2021/03/09) ｶﾝﾏ固定から､separator 使用に変更
					final String[] toStr  = toVal.split( separator );			// 7.3.1.3 (2021/03/09) ｶﾝﾏ固定から､separator 使用に変更
					if( frmStr.length == toStr.length ) {
						for( int i=0; i<frmStr.length; i++ ) {
							rtn = rtn.replaceAll( frmStr[i], toStr[i] );
						}
					}
					else {
//						final String errMsg = "REPLACE時の fromValのｶﾝﾏの数が､toValの数と一致しません｡"
						final String errMsg = "REPLACE時の fromValとtoValの数と一致しません｡(separator=[" + separator + "])" + CR
										// 6.3.9.0 (2015/11/06) 文字列配列を toString() する｡
										+ " fromVal=[" + String.join( ",",frmStr ) + "] , toVal=[" + toVal + "] , value=[" + value + "]";
						throw new HybsSystemException( errMsg );
					}
				}
			}
		}
		// 5.2.2.0 (2010/11/01) SUBSTR 処理を新規追加
		else if( ACT_SUBSTR.equals( action ) ) {
			// value.substring( from, to ) という文法で処理します｡
			// 6.0.2.5 (2014/10/31) null でないことがわかっている値の冗長な null ﾁｪｯｸがあります｡
				final int from = fromVal == null || fromVal.isEmpty() ? 0              : Integer.parseInt( fromVal );	// 6.4.2.1 (2016/02/05) PMD refactoring. Useless parentheses.
				final int to   =   toVal == null || toVal.isEmpty()   ? value.length() : Integer.parseInt( toVal );		// 6.4.2.1 (2016/02/05) PMD refactoring. Useless parentheses.

				if(   from >= value.length() ) { rtn = ""; }
				else if( to > value.length() ) { rtn = value.substring( from ); }
				else {
					rtn = value.substring( from, to );
				}
		}
		// 7.2.6.0 (2020/06/30) INDEX_OF ,LASTINDEX_OF, LENGTH追加
		else if( ACT_INDEX_OF.equals( action ) ) {
			final int from = fromVal == null || fromVal.isEmpty() ? 0 : Integer.parseInt( fromVal );
			rtn = String.valueOf( value.indexOf( separator,from ) );
		}
		// 7.2.6.0 (2020/06/30) INDEX_OF ,LASTINDEX_OF, LENGTH追加
		else if( ACT_LASTINDEX_OF.equals( action ) ) {
			final int to = toVal == null || toVal.isEmpty() ? value.length() : Integer.parseInt( toVal );
			rtn = String.valueOf( value.lastIndexOf( separator,to ) );
		}
		// 7.2.6.0 (2020/06/30) INDEX_OF ,LASTINDEX_OF, LENGTH追加
		else if( ACT_LENGTH.equals( action ) ) {
			rtn = String.valueOf( value.length() );
		}
		// 5.7.7.2 (2014/06/20) CASE 処理を新規追加
		else if( ACT_CASE.equals( action ) ) {
			// fromVal="A:1 B:2 C:3 D:4" 形式
			if( fromVal != null && fromVal.length()>0 ) {
				final String[] keys = StringUtil.csv2Array( fromVal , ' ' );
				// 7.2.9.4 (2020/11/20) PMD:This for loop can be replaced by a foreach loop
				for( final String fkey : keys ) {
					final int idx = fkey.indexOf( ':' );
					if( idx >= 0 ) {									// 分解した値にｺﾛﾝ(:)がなければ､ﾊﾟｽ
						final String key = fkey.substring( 0,idx );		// 左辺(ｷｰ)
						if( value.equalsIgnoreCase( key ) ) {
							rtn = fkey.substring( idx+1 );				// 右辺(値)
							break;
						}
					}
				}
//				for( int i=0; i<keys.length; i++ ) {
//					final int idx = keys[i].indexOf( ':' );
//					if( idx >= 0 ) {									// 分解した値にｺﾛﾝ(:)がなければ､ﾊﾟｽ
//						final String key = keys[i].substring( 0,idx );	// 左辺(ｷｰ)
//						if( value.equalsIgnoreCase( key ) ) {
//							rtn = keys[i].substring( idx+1 );			// 右辺(値)
//							break;
//						}
//					}
//				}
			}
		}
		// 5.7.7.2 (2014/06/20) EXEC 処理を新規追加
		else if( ACT_EXEC.equals( action ) ) {
//			final ScriptEngine jsEngine = new ScriptEngineManager().getEngineByName( "JavaScript" );
			ScriptEngine jsEngine = new ScriptEngineManager().getEngineByName( "graal.js" );		// 7.3.0.0 (2021/01/06)

			if( jsEngine == null ) {
				jsEngine = new ScriptEngineManager().getEngineByName( "nashorn" );		// 7.3.0.0 (2021/01/06)
				if( jsEngine == null ) {
					final String errMsg = "ScriptEngine(Nashorn) は､廃止されました｡" + CR
								+ " 別の方法に変更するか､GraalJSに移行してください｡" + CR
								+ " command=" + command ;
					throw new HybsSystemException( errMsg );
				}
			}

			try {
				final Object obj = jsEngine.eval( value );
				rtn = String.valueOf( obj );
			}
			catch( final ScriptException ex ) {
				final String errMsg = "JavaScript式のﾊﾟｰｽに失敗しました｡[" + value + "]";
				throw new HybsSystemException( errMsg , ex );
			}
		}
		// 7.3.0.0 (2021/01/06) ADDｱｸｼｮﾝを追加
		else if( ACT_ADD.equals( action ) ) {
			// defaultVal の 値と､(value またはBODY部の)値を､加算した結果をｷｰ変数にｾｯﾄします｡
			final int addVal = nval( defaultVal,0 ) + nval( value,0 );
			rtn = String.valueOf( addVal );
		}
		// 6.3.5.0 (2015/08/08) action="ESCAPE" 追加
		else if( ACT_ESCAPE.equals( action ) ) {
			rtn = StringUtil.htmlFilter( value );
		}
		// 6.9.7.0 (2018/05/14) action="URL_ENCODE" 追加
		else if( ACT_ENCODE.equals( action ) ) {
			rtn = StringUtil.urlEncode( value );
		}
		// 6.9.7.0 (2018/05/14) action="URL_ENCODE" 追加
		else if( ACT_ENCODE2.equals( action ) ) {
			rtn = StringUtil.urlEncode2( value );
		}
		// 6.9.7.0 (2018/05/14) action="URL_DECODE" 追加
		else if( ACT_DECODE.equals( action ) ) {
			rtn = StringUtil.urlDecode( value );
		}
		// 6.4.1.2 (2016/01/22) TRIM ｱｸｼｮﾝを追加｡
		else if( ACT_TRIM.equals( action ) ) {
			rtn = value.trim();
		}
		// 5.9.6.3 (2016/02/18) 8.1.2.0 (2022/03/10) Delete
//		else if( ACT_MD5.equals( action ) ){
//			rtn = HybsCryptography.getMD5( value );
//		}
		// action="HASH" 追加 8.1.2.0 (2022/03/10) Add
		else if( ACT_HASH.equals( action ) ){
			final String hash = format != null ? format : HASH_CODE ;
			rtn = HybsCryptography.getHash( hash, value );
		}

		// 6.7.4.0 (2017/02/10)
		else if( ACT_TO_NUMBER.equals( action ) ){
			rtn = delChar( value , ch -> ( '-' == ch || '0'<=ch && ch<='9' ) );

	//		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
	//		for( int i=0; i<value.length(); i++ ) {
	//			final char ch = value.charAt( i );
	//			if( '-' == ch || '0'<=ch && ch<='9' ) {
	//				buf.append( ch );
	//			}
	//		}
	//		rtn = buf.toString();
		}
		// 8.0.1.2 (2021/11/19) ACT_TO_NUMBER との違いは、ﾏｲﾅｽ記号も削除します。
		else if( ACT_TO_DATE.equals( action ) ){
			rtn = delChar( value , ch -> ( '0'<=ch && ch<='9' ) );

//			final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
//			for( int i=0; i<value.length(); i++ ) {
//				final char ch = value.charAt( i );
//				if( '0'<=ch && ch<='9' ) {
//					buf.append( ch );
//				}
//			}
//			rtn = buf.toString();
		}
		// 6.8.4.1 (2017/12/18)
		// 6.9.1.0 (2018/02/26) HMCSV ｱｸｼｮﾝに､MAX,MIN,HMF を追加
		// 8.2.0.1 (2022/06/17) ACT_HMCSV ｱｸｼｮﾝで、fromVal,toVal に日付指定を可能にする。
		else if( ACT_HMCSV.equals( action ) && fromVal != null && fromVal.length()>0 ) {
			fromVal = delChar( fromVal , ch -> ( '0'<=ch && ch<='9' ) );	// ﾌｫｰﾏｯﾄされていた場合の処理
			toVal   = delChar( toVal   , ch -> ( '0'<=ch && ch<='9' ) );	//

			// 8.2.0.1 (2022/06/17) ACT_HMCSV ｱｸｼｮﾝで、fromVal が null でなく、8桁以上(つまり時:分:秒 でなく年月日を含むｹｰｽ)
			if( !hmCSVfrom ) {												// 上記の否定形なので従来と同じ処理
				final int fromNum = Integer.parseInt( fromVal );
				final int toNum   = Integer.parseInt( toVal   );

				// value には､基準となる日付ﾃﾞｰﾀ
				fromVal = HybsDateUtil.getDateFormat( "YMDHMS" , value, "HM" , fromNum );
				toVal   = HybsDateUtil.getDateFormat( "YMDHMS" , value, "HM" , toNum   );	// HM (時分形式の加算)
			}

			final Calendar stCal  = HybsDateUtil.getCalendar( fromVal );
			final Calendar edCal  = HybsDateUtil.getCalendar( toVal );

			final int field  = Calendar.MINUTE ;				// 加算するﾌｨｰﾙﾄﾞ(単位は分)
			final int amount = nval( skipVal , 60 );			// 初期値の間隔は､60分

			final DateFormat ymdhms  = new SimpleDateFormat( "yyyyMMddHHmmss",Locale.JAPAN );
			final DateFormat hmFmt   = new SimpleDateFormat( "HH:mm",Locale.JAPAN );
			final DateFormat mdhmFmt = new SimpleDateFormat( "M/d HH:mm",Locale.JAPAN );

			final StringBuilder buf  = new StringBuilder( BUFFER_MIDDLE );
			final StringBuilder slbl = new StringBuilder( BUFFER_MIDDLE );	// 6.9.1.0 (2018/02/26) HMF を追加

			int lastDay = 0;									// 日付変更時のﾁｪｯｸ用
			while( stCal.before( edCal ) ) {
				buf.append( ymdhms.format( stCal.getTime() ) ).append( separator );

				final int nowDay = stCal.get( Calendar.DATE ) ;
				if( lastDay == nowDay ) {						// 日付が同じ場合
					slbl.append( hmFmt.format( stCal.getTime() ) ).append( separator );
				}
				else {
					slbl.append( mdhmFmt.format( stCal.getTime() ) ).append( separator );
				}
				lastDay = nowDay;
				stCal.add( field, amount );
			}
			if( buf.length()  > 0 ) { buf.setLength(  buf.length()-separator.length()); }	// 最後のseparatorを削除する｡ 8.2.0.1 (2022/06/17)
			if( slbl.length() > 0 ) { slbl.setLength( slbl.length()-separator.length()); }	// 最後のseparatorを削除する｡ 8.2.0.1 (2022/06/17)
			rtn = buf.toString();

			// MAX,MIN を追加
			setObject( "MIN." + key , fromVal );
			setObject( "MAX." + key , toVal   );
			setObject( "HMF." + key , slbl.toString() );

//			final int fromNum = Integer.parseInt( StringUtil.deleteChar( fromVal , ':' ) );
//			final int toNum   = Integer.parseInt( StringUtil.deleteChar( toVal   , ':' ) );
//
//			// value には､基準となる日付ﾃﾞｰﾀ
//			final String fromS = HybsDateUtil.getDateFormat( "YMDHMS" , value, "HM" , fromNum );
//			final String toS   = HybsDateUtil.getDateFormat( "YMDHMS" , value, "HM" , toNum   );	// HM (時分形式の加算)
//
//			final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
//			buf.append( fromS );
//			final StringBuilder slbl = new StringBuilder( BUFFER_MIDDLE );	// 6.9.1.0 (2018/02/26) HMF を追加
//			slbl.append( HybsDateUtil.getDateFormat( "HMF",fromS ) );		// 6.9.2.1 (2018/03/12)
//
//			final int skip = nval( skipVal , 60 );	// 初期値の間隔は､60分
//			int cnt = 1;
//			String timeS = fromS;
//			String lastS = timeS.substring( 0,8 );	// 日付部分の比較のための変数
//			while( timeS.compareTo( toS ) < 0 && cnt < MAX_CNT ) {		// 最大件数で制限しておきます｡
//				timeS = HybsDateUtil.getDateFormat( "YMDHMS" , fromS , "MI" , skip*cnt );			// MI (分の数値の加算)
//				cnt++;
//
//				buf.append( separator ).append( timeS );		// 区切り文字で､連結
//
//				if( lastS.equals( timeS.substring( 0,8 ) ) ) {	// 日付が同じ場合
//					slbl.append( separator ).append( HybsDateUtil.getDateFormat( "HMF",timeS ) );		// 6.9.2.1 (2018/03/12)
//				}
//				else {
//					slbl.append( separator ).append( HybsDateUtil.getDateFormat( "M/d HH:mm",timeS ) );			// 6.9.2.1 (2018/03/12)
//				}
//				lastS = timeS.substring( 0,8 );	// 日付部分の比較のための変数
//			}
//			rtn = buf.toString();

//			// 8.2.0.1 (2022/06/17) MAX_CNT 廃止
//			if( cnt >= MAX_CNT ) {
//				final String errMsg = "valueﾀｸﾞで､action=\"HMCSV\" で､最大件数をｵｰﾊﾞｰしました｡ MAX_CNT=[" + MAX_CNT + "]" + CR
//								+ " fromVal=[" + fromVal + "] , toVal=[" + toVal + "] , skipVal=[" + skipVal
//								+ "] , key=[" + key + "] , value=[" + value + "]" ;
////								+ " rtn=[" + rtn + "]" ;	// 8.0.1.2 (2021/11/19) rtn → key に変更
//				System.err.println( errMsg );
//			}
//
//			// 6.9.1.0 (2018/02/26) MAX,MIN を追加
//			setObject( "MIN." + key , fromS );
//			setObject( "MAX." + key , toS   );
//			setObject( "HMF." + key , slbl.toString() );
		}
		// 6.9.1.0 (2018/02/26) HMCSV ｱｸｼｮﾝで､from～to ではないｹｰｽ
		// 6.9.8.0 (2018/05/28) FindBugs:null でないことがわかっている値の冗長な null ﾁｪｯｸ
		else if( ACT_HMCSV.equals( action ) && value.contains( "," ) ){		// valueは､nullﾁｪｯｸ済み
			final String[] ymdhms = StringUtil.csv2Array( value );

			final StringBuilder slbl = new StringBuilder( BUFFER_MIDDLE );	// 6.9.1.0 (2018/02/26) HMF を追加

//			String lastS = ymdhms[0].substring( 0,8 );			// 日付部分の比較のための変数
			String lastS = "";									// 8.2.0.1 (2022/06/17) 日付部分の比較のための変数(初回は空文字列)
			for( final String timeS : ymdhms ) {
				final String nowS = timeS.substring( 0,8 );
				if( lastS.equals( nowS ) ) {					// 日付が同じ場合
					slbl.append( HybsDateUtil.getDateFormat( "HMF",timeS ) ).append( separator );			// 6.9.2.1 (2018/03/12)
				}
				else {
					slbl.append( HybsDateUtil.getDateFormat( "M/d HH:mm",timeS ) ).append( separator );		// 6.9.2.1 (2018/03/12)
				}
				lastS = nowS;		// 日付部分の比較のための変数(前回の日付)
			}
			if( slbl.length() > 0 ) { slbl.setLength( slbl.length()-separator.length()); }	// 最後のseparatorを削除する｡ 8.2.0.1 (2022/06/17)

			rtn = value;		// 元のvalue のまま､返します｡

			// 6.9.1.0 (2018/02/26) MAX,MIN を追加
			setObject( "MIN." + key , ymdhms[0] );
			setObject( "MAX." + key , ymdhms[ymdhms.length-1] );
			setObject( "HMF." + key , slbl.toString() );
		}
		// 5.10.4.0 (2018/10/05)
		else if( ACT_CRYPT_ENC.equals( action ) ) {
			final HybsCryptography hc = new HybsCryptography(cryptKey);
			rtn = hc.encrypt( value );
		}
		else if( ACT_CRYPT_DEC.equals( action ) ) {
			final HybsCryptography hc = new HybsCryptography(cryptKey);
			rtn = hc.decrypt( value );
		}
		// 7.2.6.1 (2020/07/17)
		else if( ACT_REQUEST.equals( action ) ) {
			rtn = getRequestParameter( value );			// setValue で変換された結果を､変換します｡
		}
		// 7.4.2.0 (2021/05/08) FORMAT追加
//		else if( ACT_FORMAT.equals( action ) ) {
		else if( ACT_FORMAT.equals( action ) && format != null ) {		// 8.0.1.2 (2021/11/19) fromVal → format に変更
			try {
				final int val = Integer.parseInt(value);
//				rtn = String.format(fromVal, val);
				rtn = String.format(format, val);						// 8.0.1.2 (2021/11/19)
			}
			catch( final NumberFormatException ex ) {
				final String errMsg = "value が数値ではありません｡ value=[" + value + "]" ;
				throw new HybsSystemException( errMsg );
			}
		}

		return rtn;
	}

	/**
	 * 指定の文字列から、特定のｷｬﾗｸﾀを削除します。
	 *
	 * @og.rev 8.0.1.2 (2021/11/19) 新規追加
	 *
	 * @param org		ｷｬﾗｸﾀを削除するｵﾘｼﾞﾅﾙの文字列
	 * @param isOmit	ｷｬﾗｸﾀ削除の判定処理(Predicate)
	 */
	private String delChar( final String org , final Predicate<Character> isOmit ) {
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
		for( int i=0; i<org.length(); i++ ) {
			final char ch = org.charAt( i );
			if( isOmit.test( ch ) ) {
				buf.append( ch );
			}
		}
		return buf.toString();
	}

	/**
	 * 指定のｽｺｰﾌﾟの内部ｷｬｯｼｭ情報を､ｷｰで登録します｡
	 *
	 * @og.rev 3.1.0.1 (2003/03/26) 引数を与えて処理する様に変更する｡
	 * @og.rev 3.5.4.0 (2003/11/25) APPENDｱｸｼｮﾝを有効にします｡
	 * @og.rev 3.5.6.5 (2004/08/09) APPEND時のｾﾊﾟﾚｰﾀを外部指定の変数を使用
	 * @og.rev 5.1.8.0 (2010/07/01) isNullSet 属性 廃止
	 * @og.rev 5.2.2.0 (2010/11/01) ACT_SPLIT 追加
	 * @og.rev 5.6.4.3 (2013/05/24) ACT_MAX_MIN ｱｸｼｮﾝの追加
	 * @og.rev 5.6.6.1 (2013/07/12) value が null の場合は､無視します｡
	 * @og.rev 7.3.1.1 (2021/02/25) ACT_SPLIT_LIST,ACT_FOR,ACT_FOR_LIST 追加
	 * @og.rev 8.0.1.2 (2021/11/19) ACT_TO_DATE,ACT_FORTM,ACT_FORTM_LIST 追加
	 *
	 * @param key		ｷｰ
	 * @param value		値
	 * @param action	ｱｸｼｮﾝ
	 */
	private void setAttribute( final String key,final String value,final String action ) {
		if( key == null || key.isEmpty() ) {
			final String errMsg = "key がｾｯﾄされていません｡"
						+ " command=" + command + " , action=" + action
						+ " , value=" + value ;			// 5.1.8.0 (2010/07/01) errMsg 修正
			throw new HybsSystemException( errMsg );
		}

		// 7.3.1.1 (2021/02/25) ACT_FOR,ACT_FOR_LIST 追加
//		final boolean useFor = ACT_FOR.equals( action ) || ACT_FOR_LIST.equals( action ) ;
		// 8.0.1.2 (2021/11/19) ACT_FORTM,ACT_FORTM_LIST 追加
		final boolean useFor = ACT_FOR.equals( action ) || ACT_FOR_LIST.equals( action ) || ACT_FORTM.equals( action ) || ACT_FORTM_LIST.equals( action ) ;
		// 8.2.0.1 (2022/06/17) ACT_HMCSV ｱｸｼｮﾝも追加
		final boolean useHmcsv = ACT_HMCSV.equals( action ) ;

		// 5.1.8.0 (2010/07/01) isNullSet 属性 廃止
		// 5.6.6.1 (2013/07/12) value が null の場合は､無視します｡
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
//		if( value == null ) {
//		if( value == null && !useFor ) {	// 7.3.1.1 (2021/02/25) ACT_FOR,ACT_FOR_LIST は､value 未指定も可能
		if( value == null && !useFor && !useHmcsv ) {	// 8.2.0.1 (2022/06/17) ACT_HMCSV ｱｸｼｮﾝも追加
			setObject( key, null );			// 5.6.6.1 (2013/07/12) value が null の場合
		}
		else {
			if( ACT_APPEND.equals( action ) ) {
				final String[] array = getRequestParameterValues( inValue );
				setObject( key, StringUtil.array2line( array,separator ) );
			}
			// 5.2.2.0 (2010/11/01) ACT_SPLIT 追加
			else if( ACT_SPLIT.equals( action ) )  {
				final String[] array = value.split( separator );
				setObject( key , array[0] );		// ｷｰ自体には､分割時の先頭の文字列を設定しておく｡
				for( int i=0; i<array.length; i++ ) {
					setObject( key + i , array[i] );
				}
			}
			// 7.3.1.1 (2021/02/25) ACT_SPLIT_LIST 追加
			else if( ACT_SPLIT_LIST.equals( action ) )  {
				final String[] array = value.split( separator );
				final List<String> list = Arrays.asList( array );
				setObject( key , list );
			}

			// 7.3.1.1 (2021/02/25) ACT_FOR,ACT_FOR_LIST 追加
			else if( ACT_FOR.equals( action ) || ACT_FOR_LIST.equals( action ) )  {
				final String valKey = value == null ? "" : value ;

				final String stStr  = nval( fromVal , "0" );
				final String edStr  = nval( toVal   , "10" );
				final String stpStr = nval( skipVal , "1" );

				final int start = Integer.parseInt( stStr );
				final int end   = Integer.parseInt( edStr );
				final int step  = Integer.parseInt( stpStr );

				// 8.0.1.2 (2021/11/19) format属性追加に伴い、format指定を優先させる。
				if( format == null ) {
					// ｾﾞﾛ埋めﾌｫｰﾏｯﾄを判定する｡
					final int len = stStr.length();
//					final String numFmt = len>1 && stStr.charAt(0) == '0' ? "%0" + len + "d" : "%d" ;
					format = len>1 && stStr.charAt(0) == '0' ? "%0" + len + "d" : "%d" ;
				}

				if( ACT_FOR.equals( action ) ) {					// ACT_FOR
					final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
					for( int val=start; val<=end; val+=step ) {		// 終了は､endを含む
//						final String val2 = String.format( numFmt , val );
						final String val2 = String.format( format , val );		// 8.0.1.2 (2021/11/19) format属性追加
						buf.append( valKey ).append( val2 ).append( separator );
					}
//					if( buf.length() > 0 ) { buf.setLength( buf.length()-1); }	// 最後のseparatorを削除する｡
					if( buf.length() > 0 ) { buf.setLength( buf.length()-separator.length()); }	// 最後のseparatorを削除する｡ 8.2.0.1 (2022/06/17)

					setObject( key , buf.toString() );
				}
				else {												// ACT_FOR_LIST
					final List<String> list = new ArrayList<>();
					for( int val=start; val<=end; val+=step ) {		// 終了は､endを含む
//						final String val2 = String.format( numFmt , val );
						final String val2 = String.format( format , val );		// 8.0.1.2 (2021/11/19) format属性追加
						list.add( valKey + val2 );
					}

					setObject( key , list );
				}
			}
			// 8.0.1.2 (2021/11/19) ACT_FORTM,ACT_FORTM_LIST 追加
			else if( ACT_FORTM.equals( action ) || ACT_FORTM_LIST.equals( action ) )  {
				// FORTM,FORTM_LIST では、fromVal,toVal は、必須になる。
				if( StringUtil.isNull(fromVal,toVal) ) {
					final String errMsg = "fromVal,toVal は必須です。"
								+ " command=" + command + " , action=" + action;
					throw new HybsSystemException( errMsg );
				}

				fromVal = delChar( fromVal , ch -> ( '0'<=ch && ch<='9' ) );	// ﾌｫｰﾏｯﾄされていた場合の処理
				toVal   = delChar( toVal   , ch -> ( '0'<=ch && ch<='9' ) );	//

				final Calendar stCal  = HybsDateUtil.getCalendar( fromVal );
				final Calendar edCal  = HybsDateUtil.getCalendar( toVal );

				final String   stpStr = nval( skipVal , "1" );		// null や、"M" などがあり得るため
				final int field = "M".equals( stpStr ) ? Calendar.MONTH : Calendar.DATE ;	// 加算するフィールド
				final int amount= "M".equals( stpStr ) ? 1 : Integer.parseInt( stpStr );	// 追加量

				if( StringUtil.isNull( format) ) {
					format = YMDHMS.substring( 0,fromVal.length() ) ;	// format の初期値は、開始日付の長さ
				}
				final DateFormat formatter = new SimpleDateFormat( format,Locale.JAPAN );

				if( ACT_FORTM.equals( action ) ) {					// ACT_FORTM
					final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
					while( stCal.before( edCal ) ) {
						buf.append( formatter.format( stCal.getTime() ) ).append( separator );
						stCal.add( field, amount );
					}
//					if( buf.length() > 0 ) { buf.setLength( buf.length()-1); }	// 最後のseparatorを削除する｡
					if( buf.length() > 0 ) { buf.setLength( buf.length()-separator.length()); }	// 最後のseparatorを削除する｡ 8.2.0.1 (2022/06/17)

					setObject( key , buf.toString() );
				}
				else {												// ACT_FORTM_LIST
					final List<String> list = new ArrayList<>();
					while( stCal.before( edCal ) ) {
						list.add( formatter.format( stCal.getTime() ) );
						stCal.add( field, amount );
					}

					setObject( key , list );
				}
			}
			// 5.6.4.3 (2013/05/24) ACT_MAX_MIN ｱｸｼｮﾝの追加
			else if( ACT_MAX_MIN.equals( action ) )  {
				final String[] array = value.split( separator );

				// command="SET" では､数字型としてのみ処理します｡
				double minNum = Double.MAX_VALUE ;
				double maxNum = Double.MIN_VALUE ;
				double sumNum = 0d ;
				int    cntSum = 0;		// 平均計算に有効な件数

				for( int i=0; i<array.length; i++ ) {
					final String val = array[i].trim().replaceAll( ",","" );		// 数字型ﾌｫｰﾏｯﾄのｶﾝﾏを取り除く
					if( val.isEmpty() ) { continue; }		// ｾﾞﾛ文字列の場合は､取り直し

					final double tmp = Double.parseDouble( val );
					if( minNum > tmp ) { minNum = tmp; }
					if( maxNum < tmp ) { maxNum = tmp; }
					sumNum += tmp ;  cntSum++ ;
				}

				// command="SET" の場合は､数字型の場合のみ使用します｡
				if( cntSum > 0 ) {
					final NumberFormat nf = NumberFormat.getInstance();
					nf.setGroupingUsed( false );					// ｶﾝﾏ編集なし
					nf.setMaximumFractionDigits(2);					// 最大小数部は２桁
					nf.setMinimumFractionDigits(0);					// できれば､整数表示

					setObject( "MIN." + key,nf.format( minNum ) );		// Double.toString( minNum ) の代わり｡
					setObject( "MAX." + key,nf.format( maxNum ) );
					setObject( "SUM." + key,nf.format( sumNum ) );
					setObject( "AVG." + key,nf.format( sumNum/cntSum ) );
				}
				else {
					setObject( "MIN." + key,nval( defaultVal , "" ) );
					setObject( "MAX." + key,nval( defaultVal , "" ) );
					setObject( "SUM." + key,nval( defaultVal , "" ) );
					setObject( "AVG." + key,nval( defaultVal , "" ) );
				}
			}
			else {
				setObject( key, actionExec( action,value ) );
			}
		}
	}

	/**
	 * 指定のｽｺｰﾌﾟの内部ｷｬｯｼｭ情報を､ｷｰで取得します｡
	 *
	 * @og.rev 3.1.0.1 (2003/03/26) 引数を与えて処理する様に変更する｡
	 *
	 * @param key		ｷｰ
	 * @param action	ｱｸｼｮﾝ
	 *
	 * @return	ｷｰに対する内部ｷｬｯｼｭ情報
	 */
	private String getAttribute( final String key,final String action ) {
		if( key == null || key.isEmpty() ) {
			final String errMsg = "key がｾｯﾄされていません｡"
						+ " command=" + command + " , action=" + action;	// 5.1.8.0 (2010/07/01) errMsg 修正
			throw new HybsSystemException( errMsg );
		}

		String rtn = defaultVal;
		final Object obj = pageContext.findAttribute( key );
		if( obj != null ) { rtn = obj.toString(); }

		return actionExec( action,rtn );
	}

	/**
	 * 指定のｽｺｰﾌﾟの内部ｷｬｯｼｭ情報を削除します｡
	 *
	 * @og.rev 3.1.0.1 (2003/03/26) 引数を与えて処理する様に変更する｡
	 *
	 * @param key		ｷｰ
	 */
	private void removeAttribute( final String key ) {
		if( key == null || key.isEmpty() ) {
			final String errMsg = "key がｾｯﾄされていません｡"
						+ " command=" + command ;			// 5.1.8.0 (2010/07/01) errMsg 修正
			throw new HybsSystemException( errMsg );
		}
		removeObject( key );
	}

	/**
	 * ｾｯｼｮﾝ/ｱﾌﾟﾘｹｰｼｮﾝｽｺｰﾌﾟのｷｬｯｼｭ情報をｸﾘｱします｡
	 *
	 * このｸﾘｱは､ｷｰの前方一致で､大文字小文字の区別をせずにｸﾘｱします｡
	 * また､ｷｰが null の場合は､"X_" で始めるもの以外のすべての値をｸﾘｱします｡
	 * また､Ｗｅｂｴﾝｼﾞﾝ内部で使用しているｷｰは､ここではｸﾘｱできません｡
	 *
	 * @og.rev 3.1.0.1 (2003/03/26) ｸﾘｱｺﾏﾝﾄﾞの追加｡
	 * @og.rev 4.3.4.0 (2008/12/01) PageContextのｽｺｰﾌﾟをｸﾗｽ変数としてｱｸｾｽ
	 *
	 * @param key		ｷｰ
	 */
	private void clearAttribute( final String key ) {

		String lowKey = null;
		if( key != null ) { lowKey = key.toLowerCase(Locale.JAPAN); }

		Enumeration<String> ekeys = pageContext.getAttributeNamesInScope( PageContext.APPLICATION_SCOPE );		// 4.3.3.6 (2008/11/15) Generics警告対応
		while( ekeys.hasMoreElements() ) {
			final String ekey = ekeys.nextElement().toLowerCase(Locale.JAPAN);		// 4.3.3.6 (2008/11/15) Generics警告対応
			if( ! ekey.startsWith( "h_") && ! ekey.startsWith( "x_") &&				// 6.9.7.0 (2018/05/14) PMD Useless parentheses.
				( lowKey == null || ekey.startsWith( key ) ) ) {
					pageContext.removeAttribute( ekey, PageContext.APPLICATION_SCOPE ) ;
			}
		}

		ekeys = pageContext.getAttributeNamesInScope( PageContext.SESSION_SCOPE );
		while( ekeys.hasMoreElements() ) {
			final String ekey = String.valueOf( ekeys.nextElement() ).toLowerCase(Locale.JAPAN);
			if( ! ekey.startsWith( "h_") && ! ekey.startsWith( "x_") &&				// 6.9.7.0 (2018/05/14) PMD Useless parentheses.
				( lowKey == null || ekey.startsWith( key ) ) ) {
					pageContext.removeAttribute( ekey, PageContext.SESSION_SCOPE ) ;
			}
		}
	}

	/**
	 * 指定のｽｺｰﾌﾟの内部ｷｬｯｼｭ情報を､指定のSQL文より作成します｡
	 *
	 * @og.rev 3.1.0.1 (2003/03/26) 引数を与えて処理する様に変更する｡
	 * @og.rev 3.2.4.0 (2003/06/12) ﾏﾙﾁﾃﾞｰﾀ(複数件検索)を使用するかしないか｡
	 * @og.rev 3.8.6.0 (2006/08/07) nullSet="true"(初期値)の時は､検索結果がｾﾞﾛ件時に "" をｾｯﾄする｡
	 * @og.rev 3.8.9.2 (2007/07/28) action="ROW_APPEND" 追加
	 * @og.rev 4.3.7.5 (2009/07/13) ACT_LIST､ACT_ALL_LIST ｱｸｼｮﾝの追加
	 * @og.rev 5.1.8.0 (2010/07/01) isNullSet 属性 廃止
	 * @og.rev 5.5.0.3 (2012/03/12) ACT_MAP ｱｸｼｮﾝの追加
	 * @og.rev 5.6.4.3 (2013/05/24) tmpSelectedAll 属性追加｡action="ALL_xxx"を tmpSelectedAll="true" に置き換えます｡
	 * @og.rev 6.2.4.2 (2015/05/29) ACT_MAPOBJ ｱｸｼｮﾝの追加
	 * @og.rev 6.2.6.0 (2015/06/19) ｷｰの判定に､isEmpty() が漏れていた｡
	 * @og.rev 6.4.2.1 (2016/02/05) ACT_MAPOBJ ｱｸｼｮﾝのMapを､TreeMap から LinkedHashMap に変更｡
	 * @og.rev 6.4.4.2 (2016/04/01) StringBuilderの代わりに､OgBuilderを使用する｡
	 * @og.rev 6.9.0.2 (2018/02/13) ACT_MAP,(ACT_ALL_MAP) で､第３ｶﾗﾑ以降も､ｷｰ + '_番号(1～)' で､ｷｰを作成します｡
	 * @og.rev 7.2.6.1 (2020/07/17) ACT_REQUEST追加
	 * @og.rev 7.3.1.1 (2021/02/25) LIST ｱｸｼｮﾝを､ArrayListからListに変更｡
	 * @og.rev 8.1.0.1 (2022/01/07) ACT_LIST_CSV､ACT_ALL_LIST_CSV,ACT_SET,ACT_ALL_SET,ACT_SET_CSV,ACT_ALL_SET_CSV ｱｸｼｮﾝの追加
	 *
	 * @param table DBTableModelｵﾌﾞｼﾞｪｸﾄ
	 */
	private void setSQLAttribute( final DBTableModel table ) {
		if( table == null ) {		// 3.8.6.0 (2006/08/07)
			return;
		}

		final int clmCnt = table.getColumnCount();
		final int rowCnt = table.getRowCount();
		String sqlkey ;
		String sqlval ;
		String sufix = "";

		if( ACT_ROW_APPEND.equals( action ) ) {
			final OgBuilder buf = new OgBuilder();
			for( int clmNo=0; clmNo<clmCnt; clmNo++ ) {
				// 6.4.4.2 (2016/04/01)
				final int clm = clmNo;			// ﾗﾑﾀﾞ式の引数に使えるのは､final変数のみ｡
				buf.clear();
				buf.appendRoop( 0,rowCnt,separator
									, row -> table.getValue( row, clm ) );
				sqlkey = table.getColumnName( clm );
				setObject( sqlkey , buf.toString() );
			}
		}
		// 4.3.7.5 (2009/07/13) ACT_LIST,ACT_LIST_CSV,ACT_ALL_LIST,ACT_ALL_LIST_CSV ｱｸｼｮﾝの追加
		// 8.1.0.1 (2022/01/07) ACT_LIST_CSV,ACT_ALL_LIST_CSV 追加
//		else if( ACT_LIST.equals( action ) ) {
		else if( ACT_LIST.equals( action ) || ACT_LIST_CSV.equals( action ) ) {
			final boolean useCSV = ACT_LIST_CSV.equals( action );
			for( int clm=0; clm<clmCnt; clm++ ) {
//				final ArrayList<String> list = new ArrayList<>();
				final List<String> list = new ArrayList<>();				// 7.3.1.1 (2021/02/25)
				for( int row=0; row<rowCnt; row++ ) {
					sqlval = table.getValue( row, clm );
					list.add( sqlval );
				}
				sqlkey = table.getColumnName( clm );
//				setObject( sqlkey , list );
				if( useCSV ) {
					final String csv = list.stream().collect(Collectors.joining(","));
					setObject( sqlkey,csv );
				}
				else {
					setObject( sqlkey,list );
				}
			}
		}
		// 8.1.0.1 (2022/01/07) ACT_SET,ACT_ALL_SET,ACT_SET_CSV,ACT_ALL_SET_CSV ｱｸｼｮﾝの追加
		else if( ACT_SET.equals( action ) || ACT_SET_CSV.equals( action ) ) {
			final boolean useCSV = ACT_SET_CSV.equals( action );
			for( int clm=0; clm<clmCnt; clm++ ) {
				final Set<String> set = new LinkedHashSet<>();
				for( int row=0; row<rowCnt; row++ ) {
					sqlval = table.getValue( row, clm );
					set.add( sqlval );
				}
				sqlkey = table.getColumnName( clm );
//				setObject( sqlkey , set );
				if( useCSV ) {
					final String csv = set.stream().collect(Collectors.joining(","));
					setObject( sqlkey , csv );
				}
				else {
					setObject( sqlkey , set );
				}
			}
		}
		// 5.5.0.3 (2012/03/12) ACT_MAP ｱｸｼｮﾝの追加
		else if( ACT_MAP.equals( action )  ) {
			if( clmCnt < 2 ) {
				final String errMsg = "action=MAP 時には､ｶﾗﾑは､２つ以上必要です｡ｶﾗﾑ数=[" + clmCnt + "]";
				throw new IllegalArgumentException( errMsg );
			}
			for( int row=0; row<rowCnt; row++ ) {
				sqlkey = table.getValue( row, 0 );			// 0番目ｶﾗﾑがｷｰ
				sqlval = table.getValue( row, 1 );			// 1番目ｶﾗﾑが値
				setObject( sqlkey , sqlval );

				// 6.9.0.2 (2018/02/13) ACT_MAP,(ACT_ALL_MAP) で､第３ｶﾗﾑ以降も､ｷｰ + '_番号(1～)' で､ｷｰを作成します｡
				for( int clm=2; clm<clmCnt; clm++ ) {
					final String keyN = sqlkey + "_" + (clm-1);		// 第３ｶﾗﾑは､ｷｰ + '_1' になります｡
					sqlval = table.getValue( row, clm );			// N番目ｶﾗﾑの値
					setObject( keyN , sqlval );
				}
			}
		}
		// 6.2.4.2 (2015/05/29) ACT_MAPOBJ ｱｸｼｮﾝの追加
		else if( ACT_MAPOBJ.equals( action )  ) {
			if( clmCnt < 2 ) {
				final String errMsg = "action=MAPOBJ 時には､ｶﾗﾑは､２つ以上必要です｡ｶﾗﾑ数=[" + clmCnt + "]";
				throw new IllegalArgumentException( errMsg );
			}
			if( key == null || key.isEmpty() ) {			// 6.2.6.0 (2015/06/19)
				key = table.getColumnName( 0 );				// key が指定されていない場合は､最初のｶﾗﾑ名
			}
			final Map<String,String> mapObj = new LinkedHashMap<>();	// 6.4.2.1 (2016/02/05)

			for( int row=0; row<rowCnt; row++ ) {
				sqlkey = table.getValue( row, 0 );			// 0番目ｶﾗﾑがMapのｷｰ
				sqlval = table.getValue( row, 1 );			// 1番目ｶﾗﾑがMapの値
				mapObj.put( sqlkey,sqlval );
			}
			setObject( key , mapObj );
		}
		// 7.2.6.1 (2020/07/17) SQL処理なので､ﾙｰﾌﾟで判断させずにactionで分けておきます｡
		else if( ACT_REQUEST.equals( action ) ) {
			for( int row=0; row<rowCnt; row++ ) {
				if( useMultiRows ) { sufix = String.valueOf( row ) ; }
				for( int clm=0; clm<clmCnt; clm++ ) {
					sqlkey = table.getColumnName( clm );
					sqlval = table.getValue( row, clm );
					sqlval = getRequestParameter( sqlval );			// SQLで取得した値を､変換します｡
					setObject( sqlkey + sufix, sqlval );
				}
				if( ! useMultiRows ) { break; }
			}
		}
		else {
			// 5.1.8.0 (2010/07/01) isNullSet 属性 廃止
			if( rowCnt == 0 ) {
				if( useMultiRows ) { sufix = "0" ; }
				for( int clm=0; clm<clmCnt; clm++ ) {
					sqlkey = table.getColumnName( clm );
					sqlval = "";
					setObject( sqlkey + sufix, sqlval );
				}
			}
			else {
				for( int row=0; row<rowCnt; row++ ) {
					if( useMultiRows ) { sufix = String.valueOf( row ) ; }
					for( int clm=0; clm<clmCnt; clm++ ) {
						sqlkey = table.getColumnName( clm );
						sqlval = table.getValue( row, clm );
						setObject( sqlkey + sufix, sqlval );
					}
					if( ! useMultiRows ) { break; }
				}
			}
		}
	}

	/**
	 * 指定のSQL文の結果を文字列として画面に出力します｡
	 * 画面に出力される項目は､1項目だけで2項目以降は無視されます｡
	 *
	 * @og.rev 5.1.7.0 (2010/06/01) SQLGET対応
	 * @og.rev 6.4.4.2 (2016/04/01) StringBuilderの代わりに､OgBuilderを使用する｡
	 *
	 * @param table DBTableModelｵﾌﾞｼﾞｪｸﾄ
	 *
	 * @return SQL文の結果文字列
	 */
	private String getSQLAttribute( final DBTableModel table ) {
		if( table == null ) {		// 3.8.6.0 (2006/08/07)
			return "";
		}

		final int rowCnt = table.getRowCount();
		final String rtn;
		if( ACT_ROW_APPEND.equals( action ) ) {
			// 6.4.4.2 (2016/04/01)
			rtn = new OgBuilder()
					.appendRoop( 0,rowCnt,separator
									, row -> table.getValue( row, 0 ) )
					.toString();
		}
		else if( rowCnt == 0 ) {
			rtn = "";
		}
		else {
			rtn = table.getValue( 0, 0 );
		}

		return rtn;
	}

	/**
	 * 指定のｽｺｰﾌﾟの内部ｷｬｯｼｭ情報に､DBTableModel の選択された値を登録します｡
	 *
	 * 複数選択行が存在する場合は､先頭行を処理します｡ただし､action="APPEND"の
	 * 場合は､separator属性で指定された文字を使用して､連結します｡
	 *
	 * @og.rev 3.1.0.1 (2003/03/26) 新規作成
	 * @og.rev 3.5.6.5 (2004/08/09) ACT_APPEND 時の処理変更
	 * @og.rev 3.6.1.0 (2005/01/05) ACT_ALL_APPEND ｱｸｼｮﾝの追加
	 * @og.rev 4.3.7.5 (2009/07/13) ACT_LIST､ACT_ALL_LIST ｱｸｼｮﾝの追加
	 * @og.rev 5.1.6.0 (2010/05/01) ALL_APPENDで選択行がない場合に処理されないﾊﾞｸﾞを修正
	 * @og.rev 5.5.0.3 (2012/03/12) ACT_MAP ｱｸｼｮﾝの追加
	 * @og.rev 5.6.4.3 (2013/05/24) ACT_MAX_MIN,ACT_ALL_MAX_MIN ｱｸｼｮﾝの追加
	 * @og.rev 5.6.4.3 (2013/05/24) tmpSelectedAll 属性追加｡action="ALL_xxx"を tmpSelectedAll="true" に置き換えます｡
	 * @og.rev 6.2.4.2 (2015/05/29) ACT_MAPOBJ ｱｸｼｮﾝの追加
	 * @og.rev 6.4.2.1 (2016/02/05) ACT_MAPOBJ ｱｸｼｮﾝのMapを､TreeMap から LinkedHashMap に変更｡
	 * @og.rev 6.4.4.2 (2016/04/01) StringBuilderの代わりに､OgBuilderを使用する｡
	 * @og.rev 6.4.4.2 (2016/04/01) contains 判定を行う新しいﾒｿｯﾄﾞを使用します｡
	 * @og.rev 6.4.6.0 (2016/05/27) isNumber , isDate 追加｡
	 * @og.rev 7.3.1.1 (2021/02/25) LIST ｱｸｼｮﾝを､ArrayListからListに変更｡
	 * @og.rev 8.1.0.1 (2022/01/07) ACT_LIST_CSV､ACT_ALL_LIST_CSV,ACT_SET,ACT_ALL_SET,ACT_SET_CSV,ACT_ALL_SET_CSV ｱｸｼｮﾝの追加
	 *
	 * @param table		DBTableModelｵﾌﾞｼﾞｪｸﾄ
	 * @param key		ｷｰ
	 * @param action	ｱｸｼｮﾝ
	 */
	private void setTableAttribute( final DBTableModel table,final String key,final String action ) {
		if( table == null || table.getRowCount() == 0 || table.getColumnCount() == 0 ) { return ; }

		final int[] rowNo = getParameterRows();
		// 5.1.6.0 (2010/05/01)
		if( rowNo.length == 0 ) { return; }				// 5.6.4.3 (2013/05/24) tmpSelectedAll 属性を追加したので､0件判定が使える｡

		final String[] keys ;
		if( key == null || key.isEmpty() ) {
			keys = table.getNames();
		}
		else if( key.indexOf( ',' ) >= 0 ) {
			keys = StringUtil.csv2Array( key , ',' );	// 8.1.0.1 (2022/01/07) ｷｰをｶﾝﾏ区切りで渡せるようにする。
		}
		else {
			keys = new String[] { key } ;
		}

		// 5.6.4.3 (2013/05/24) tmpSelectedAll で､ﾛｼﾞｯｸの共通化
		if( ACT_APPEND.equals( action ) ) {
			final OgBuilder buf = new OgBuilder();
			// 7.2.9.4 (2020/11/20) PMD:This for loop can be replaced by a foreach loop
			for( final String ky : keys ) {
				final int clm = table.getColumnNo( ky );
//			for( int i=0; i<keys.length; i++ ) {
//				final int clm = table.getColumnNo( keys[i] );
				// 6.4.4.2 (2016/04/01)
				buf.clear();
				buf.appendRoop( 0,rowNo.length,separator
									, j -> table.getValue( rowNo[j],clm ) );
				setObject( ky,buf.toString() );
//				setObject( keys[i],buf.toString() );
			}
		}

		// 5.5.0.3 (2012/03/12) ACT_MAP ｱｸｼｮﾝの追加
		else if( ACT_MAP.equals( action ) ) {
			int keyClm = table.getColumnNo( key   ,false );
			int valClm = table.getColumnNo( value ,false );

			if( keyClm < 0 ) { keyClm = 0; }	// ｷｰとなるｶﾗﾑが指定されていない場合は､最初のｶﾗﾑ
			if( valClm < 0 ) { valClm = 1; }	// 値となるｶﾗﾑが指定されていない場合は､２番目のｶﾗﾑ

			// 7.2.9.4 (2020/11/20) PMD:This for loop can be replaced by a foreach loop
			for( final int row : rowNo ) {
				final String mapkey = table.getValue( row, keyClm );
				final String mapval = table.getValue( row, valClm );
				setObject( mapkey , mapval );
			}
//			for( int j=0; j<rowNo.length; j++ ) {
//				final String mapkey = table.getValue( rowNo[j], keyClm );
//				final String mapval = table.getValue( rowNo[j], valClm );
//				setObject( mapkey , mapval );
//			}
		}

		// 6.2.4.2 (2015/05/29) ACT_MAPOBJ ｱｸｼｮﾝの追加
		else if( ACT_MAPOBJ.equals( action )  ) {
			int keyClm = table.getColumnNo( key   ,false );
			int valClm = table.getColumnNo( value ,false );

			if( keyClm < 0 ) { keyClm = 0; }	// ｷｰとなるｶﾗﾑが指定されていない場合は､最初のｶﾗﾑ
			if( valClm < 0 ) { valClm = 1; }	// 値となるｶﾗﾑが指定されていない場合は､２番目のｶﾗﾑ

			final Map<String,String> mapObj = new LinkedHashMap<>();	// 6.4.2.1 (2016/02/05)
			// 7.2.9.4 (2020/11/20) PMD:This for loop can be replaced by a foreach loop
			for( final int row : rowNo ) {
				final String mapkey = table.getValue( row, keyClm );
				final String mapval = table.getValue( row, valClm );
				mapObj.put( mapkey,mapval );
			}
//			for( int j=0; j<rowNo.length; j++ ) {
//				final String mapkey = table.getValue( rowNo[j], keyClm );
//				final String mapval = table.getValue( rowNo[j], valClm );
//				mapObj.put( mapkey,mapval );
//			}
			setObject( key , mapObj );
		}

		// 4.3.7.5 (2009/07/13) ACT_LIST,ACT_LIST_CSV,ACT_ALL_LIST,ACT_ALL_LIST_CSV ｱｸｼｮﾝの追加
		// 8.1.0.1 (2022/01/07) ACT_LIST_CSV,ACT_ALL_LIST_CSV 追加
//		else if( ACT_LIST.equals( action ) ) {
		else if( ACT_LIST.equals( action ) || ACT_LIST_CSV.equals( action ) ) {
			final boolean useCSV = ACT_LIST_CSV.equals( action );
			// 7.2.9.4 (2020/11/20) PMD:This for loop can be replaced by a foreach loop
			for( final String ky : keys ) {
				final int clm = table.getColumnNo( ky );
				final List<String> list = new ArrayList<>();				// 7.3.1.1 (2021/02/25)
				// 7.2.9.4 (2020/11/20) PMD:This for loop can be replaced by a foreach loop
				for( final int row : rowNo ) {
					list.add( table.getValue( row,clm ) );
				}
//				setObject( ky,list );
				if( useCSV ) {
					final String csv = list.stream().collect(Collectors.joining(","));
					setObject( ky,csv );
				}
				else {
					setObject( ky,list );
				}
			}
		}

		// 8.1.0.1 (2022/01/07) ACT_SET,ACT_ALL_SET,ACT_SET_CSV,ACT_ALL_SET_CSV ｱｸｼｮﾝの追加
		else if( ACT_SET.equals( action ) || ACT_SET_CSV.equals( action ) ) {
			final boolean useCSV = ACT_SET_CSV.equals( action );
			for( final String ky : keys ) {
				final int clm = table.getColumnNo( ky );
				final Set<String> set = new LinkedHashSet<>();
				for( final int row : rowNo ) {
					set.add( table.getValue( row,clm ) );
				}
//				setObject( ky,set );
				if( useCSV ) {
					final String csv = set.stream().collect(Collectors.joining(","));
					setObject( ky,csv );
				}
				else {
					setObject( ky,set );
				}
			}
		}

		// 5.6.4.3 (2013/05/24) ACT_MAX_MIN ｱｸｼｮﾝの追加
		else if( ACT_MAX_MIN.equals( action ) ) {
			NumberFormat nf = null;
			// 7.2.9.4 (2020/11/20) PMD:This for loop can be replaced by a foreach loop
			for( final String ky : keys ) {
				final int clm = table.getColumnNo( ky );
//			for( int i=0; i<keys.length; i++ ) {
//				final int clm = table.getColumnNo( keys[i] );
				final DBColumn clmObj = table.getDBColumn( clm );
				// 6.4.4.2 (2016/04/01) contains 判定を行う新しいﾒｿｯﾄﾞを使用します｡
				final boolean numFlag = clmObj.isNumberType();		// 6.4.6.0 (2016/05/27) 数字型:true

				String minVal = null ;
				String maxVal = null ;
				double minNum = Double.MAX_VALUE ;
				double maxNum = Double.MIN_VALUE ;
				double sumNum = 0d ;
				int    cntSum = 0;		// 平均計算に有効な件数

				// 7.2.9.4 (2020/11/20) PMD:This for loop can be replaced by a foreach loop
				for( final int row : rowNo ) {
					final String val = table.getValue( row,clm );
//				for( int j=0; j<rowNo.length; j++ ) {
//					final String val = table.getValue( rowNo[j],clm );
					if( val != null && val.length() > 0 ) {
						// 数字として比較
						if( numFlag ) {
							final double tmp = Double.parseDouble( val );
							if( minNum > tmp ) { minNum = tmp; minVal = val; }
							if( maxNum < tmp ) { maxNum = tmp; maxVal = val; }
							sumNum += tmp ;  cntSum++ ;
						}
						else {
							// 文字列として比較
							if( minVal == null || minVal.compareTo( val ) > 0 ) { minVal = val; }	// 最初のﾃﾞｰﾀは､必ずｾｯﾄする｡
							if( maxVal == null || maxVal.compareTo( val ) < 0 ) { maxVal = val; }	// 最初のﾃﾞｰﾀは､必ずｾｯﾄする｡
						}
					}
				}
				if( minVal == null ) { minVal = nval( defaultVal , "" ); }		// 最後まで値が決まらなければ､""(空文字列) にする｡
				if( maxVal == null ) { maxVal = nval( defaultVal , "" ); }		// 最後まで値が決まらなければ､""(空文字列) にする｡
				setObject( "MIN." + ky,minVal );
				setObject( "MAX." + ky,maxVal );
//				setObject( "MIN." + keys[i],minVal );
//				setObject( "MAX." + keys[i],maxVal );

				// SUM と AVG は､数字型の場合のみｾｯﾄします｡
				if( cntSum > 0 ) {
					if( nf == null ) {
						nf = NumberFormat.getInstance();
						nf.setGroupingUsed( false );					// ｶﾝﾏ編集なし
						nf.setMaximumFractionDigits(2);					// 最大小数部は２桁
						nf.setMinimumFractionDigits(0);					// できれば､整数表示
					}
					setObject( "SUM." + ky,nf.format( sumNum ) );		// Double.toString( sumNum ) の代わり｡
					setObject( "AVG." + ky,nf.format( sumNum/cntSum ) );
//					setObject( "SUM." + keys[i],nf.format( sumNum ) );		// Double.toString( sumNum ) の代わり｡
//					setObject( "AVG." + keys[i],nf.format( sumNum/cntSum ) );
				}
				else {
					setObject( "SUM." + ky,nval( defaultVal , "" ) );
					setObject( "AVG." + ky,nval( defaultVal , "" ) );
//					setObject( "SUM." + keys[i],nval( defaultVal , "" ) );
//					setObject( "AVG." + keys[i],nval( defaultVal , "" ) );
				}
			}
		}
		else {
			// 7.2.9.4 (2020/11/20) PMD:This for loop can be replaced by a foreach loop
			for( final String ky : keys ) {
				final int clm = table.getColumnNo( ky );
				setAttribute( key,table.getValue( rowNo[0],clm ),action );
			}
//			for( int i=0; i<keys.length; i++ ) {
//				final int clm = table.getColumnNo( keys[i] );
//				setAttribute( keys[i],table.getValue( rowNo[0],clm ),action );
//			}
		}
	}

	/**
	 * DBTableModel の選択された値を取得します｡
	 *
	 * 複数選択行が存在する場合は､先頭行を処理します｡ただし､action="APPEND"の
	 * 場合は､separator属性で指定された文字を使用して､連結します｡
	 *
	 * @og.rev 3.1.0.1 (2003/03/26) 新規作成
	 * @og.rev 3.5.6.5 (2004/08/09) ACT_APPEND 時の処理変更
	 * @og.rev 3.6.1.0 (2005/01/05) ACT_ALL_APPEND ｱｸｼｮﾝの追加
	 * @og.rev 5.1.6.0 (2010/05/01) ALL_APPENDで選択行がない場合に処理されないﾊﾞｸﾞを修正
	 * @og.rev 5.6.4.3 (2013/05/24) tmpSelectedAll 属性追加｡action="ALL_xxx"を tmpSelectedAll="true" に置き換えます｡
	 * @og.rev 6.4.4.2 (2016/04/01) StringBuilderの代わりに､OgBuilderを使用する｡
	 *
	 * @param	table	DBTableModelｵﾌﾞｼﾞｪｸﾄ
	 * @param	key		ｷｰ
	 * @param	action	ｱｸｼｮﾝ
	 *
	 * @return	DBTableModelの選択された値
	 */
	private String getTableAttribute( final DBTableModel table,final String key,final String action ) {
		if( table == null ) {
			final String errMsg = "table がｾｯﾄされていません｡"
						+ " command=" + command + " , action=" + action
						+ " , key=" + key ;			// 5.1.8.0 (2010/07/01) errMsg 修正
			throw new HybsSystemException( errMsg );
		}

		if( key == null || key.isEmpty() ) {
			final String errMsg = "key がｾｯﾄされていません｡"
						+ " command=" + command + " , action=" + action;			// 5.1.8.0 (2010/07/01) errMsg 修正
			throw new HybsSystemException( errMsg );
		}

		final int[] rowNo = getParameterRows();
		// 5.1.6.0 (2010/05/01)
		if( rowNo.length == 0 ) { return "" ; }				// 5.6.4.3 (2013/05/24) tmpSelectedAll 属性を追加したので､0件判定が使える｡

		// 3.6.1.0 (2005/01/05) ACT_ALL_APPEND ｱｸｼｮﾝの追加
		final int clm = table.getColumnNo( key );
		if( ACT_APPEND.equals( action ) ) {
			// 6.4.4.2 (2016/04/01)
			return new OgBuilder()
					.appendRoop( 0,rowNo.length,separator
									, j -> table.getValue( rowNo[j],clm ) )
					.toString();
		}
		else {
			return actionExec( action,table.getValue( rowNo[0],clm ) );
		}
	}

	/**
	 * 指定のｽｺｰﾌﾟの内部ｷｬｯｼｭ情報に､DBTableModel の選択された値を登録します｡
	 *
	 * これは､key で指定したｶﾗﾑの値をｷｰとして､value で指定したｶﾗﾑの値を
	 * value 値として設定します｡
	 * setTableAttribute が､ｶﾗﾑ(横持ち)ﾃﾞｰﾀを処理するのに対して､
	 * ﾛｳ(縦持ち)ﾃﾞｰﾀを処理することが出来ます｡
	 *
	 * 8.1.0.1 (2022/01/07)
	 *  command="SETTBL" の場合の action="APPEND,LIST,SET,MAX_MIN" の場合は、
	 *  key をｶﾝﾏで分解して、複数指定できるようにします。
	 *
	 * @og.rev 3.1.0.1 (2003/03/26) 新規作成
	 * @og.rev 3.3.3.3 (2003/08/06) key 情報がﾃﾞｰﾀの値になっていた｡ﾊﾞｸﾞ修正｡
	 * @og.rev 3.5.6.5 (2004/08/09) ACT_APPEND 時の処理変更
	 * @og.rev 6.4.4.2 (2016/04/01) StringBuilderの代わりに､OgBuilderを使用する｡
	 *
	 * @param table		DBTableModelｵﾌﾞｼﾞｪｸﾄ
	 * @param key		ｷｰ
	 * @param value		値
	 * @param action	ｱｸｼｮﾝ
	 */
	private void setKeyTableAttribute( final DBTableModel table,final String key,final String value,final String action ) {
		if( table == null ) { return ; }

		if( key == null || key.isEmpty() ) {
			final String errMsg = "key がｾｯﾄされていません｡"
						+ " command=" + command + " , action=" + action
						+ " , value=" + value ;			// 5.1.8.0 (2010/07/01) errMsg 修正
			throw new HybsSystemException( errMsg );
		}

		final int[] rowNo = getParameterRows();
		if( rowNo.length == 0 ) { return ; }

		final int[] valClm ;
		if( value == null || value.isEmpty() ) {
			final String[] vals = table.getNames();
			valClm = new int[vals.length];
			for( int i=0; i<vals.length; i++ ) {
				valClm[i] = table.getColumnNo( vals[i] );
			}
		}
		else {
			valClm = new int[] { table.getColumnNo( value ) } ;
		}

		// 6.4.4.2 (2016/04/01)
		final OgBuilder buf = new OgBuilder();
		for( int j=0; j<rowNo.length; j++ ) {
			final String rowKey = key + j ;
			final int rNo = rowNo[j];
			if( ACT_APPEND.equals( action ) ) {
				buf.clear();
				buf.appendRoop( 0,valClm.length,separator
									, i -> table.getValue( rNo,valClm[i] ) );
				setObject( rowKey,buf.toString() );
			}
			else {
				setAttribute( rowKey,table.getValue( rNo,valClm[0] ),action );
			}
		}
	}

	/**
	 * 指定のｽｺｰﾌﾟに登録されているﾊﾟﾗﾒｰﾀ（ｱﾄﾘﾋﾞｭｰﾄ）を返します｡
	 *
	 * 従来は､debug="true" で出力していましたが､情報量が多すぎるため､
	 * それぞれ指定することが出来るようにします｡
	 *
	 * @og.rev 6.7.7.2 (2017/04/14) command に､DEBUG 追加
	 *
	 * @return	scopeに応じた値
	 */
	private String scopeAttributePrint() {
		final StringBuilder rtn = new StringBuilder( BUFFER_MIDDLE )
			.append( "====================================" ).append( CR );

		if( "application".equals( scope ) ) {
			rtn.append( "APPLICATION_SCOPE:" ).append( CR );

			final Enumeration<String> ekeys = pageContext.getAttributeNamesInScope( PageContext.APPLICATION_SCOPE );	// 4.3.3.6 (2008/11/15) Generics警告対応
			while( ekeys.hasMoreElements() ) {
				final String ekey = ekeys.nextElement();		// 4.3.3.6 (2008/11/15) Generics警告対応
				// 6.0.2.5 (2014/10/31) char を append する｡
				rtn.append( "  key=[" ).append( ekey )
					.append( "] value=[" )
					.append( pageContext.getAttribute( ekey,PageContext.APPLICATION_SCOPE ) )
					.append( ']' ).append( CR );
			}
		}

		if( "session".equals( scope ) ) {
			rtn.append( "SESSION_SCOPE:" ).append( CR );
			final Enumeration<String> ekeys = pageContext.getAttributeNamesInScope( PageContext.SESSION_SCOPE );
			while( ekeys.hasMoreElements() ) {
				final String ekey = String.valueOf( ekeys.nextElement() );
				// 6.0.2.5 (2014/10/31) char を append する｡
				rtn.append( "  key=[" ).append( ekey )
					.append( "] value=[" )
					.append( pageContext.getAttribute( ekey,PageContext.SESSION_SCOPE ) )
					.append( ']' ).append( CR );
			}
		}

		if( "request".equals( scope ) ) {
			rtn.append( "REQUEST_SCOPE:" ).append( CR );
			final Enumeration<String> ekeys = pageContext.getAttributeNamesInScope( PageContext.REQUEST_SCOPE );
			while( ekeys.hasMoreElements() ) {
				final String ekey = String.valueOf( ekeys.nextElement() );
				// 6.0.2.5 (2014/10/31) char を append する｡
				rtn.append( "  key=[" ).append( ekey )
					.append( "] value=[" )
					.append( pageContext.getAttribute( ekey,PageContext.REQUEST_SCOPE ) )
					.append( ']' ).append( CR );
			}
		}

		if( "page".equals( scope ) ) {
			rtn.append( "PAGE_SCOPE:" ).append( CR );
			final Enumeration<String> ekeys = pageContext.getAttributeNamesInScope( PageContext.PAGE_SCOPE );
			while( ekeys.hasMoreElements() ) {
				final String ekey = String.valueOf( ekeys.nextElement() );
				// 6.0.2.5 (2014/10/31) char を append する｡
				rtn.append( "  key=[" ).append( ekey )
					.append( "] value=[" )
					.append( pageContext.getAttribute( ekey,PageContext.PAGE_SCOPE ) )
					.append( ']' ).append( CR );
			}
		}

		rtn.append( "====================================" ).append( CR );

		return rtn.toString();
	}

	/**
	 * 表示ﾃﾞｰﾀの HybsSystem.ROW_SEL_KEY を元に､選ばれた 行を処理の対象とします｡
	 *
	 * action="FIRST" 処理と､tmpSelectedAll 属性追加に伴う処理を､Override します｡
	 *
	 * @og.rev 3.1.0.1 (2003/03/26) 新規作成
	 * @og.rev 4.0.0.0 (2005/01/31) ﾒｲﾝ処理を､super class で対応
	 * @og.rev 3.8.0.4 (2005/08/08) action="FIRST" 機能の追加
	 * @og.rev 5.6.4.3 (2013/05/24) tmpSelectedAll 属性の追加に伴い､全件選択状態のﾃﾞｰﾀを作成します｡
	 * @og.rev 7.3.1.1 (2021/02/25) action="VIEWFIRST" 機能の追加
	 *
	 * @return	選択行の配列
	 * @og.rtnNotNull
	 */
	@Override
	protected int[] getParameterRows() {
		final int[] rowNo ;
		// 3.8.0.4 (2005/08/08) action="FIRST" 機能の追加
		if( ACT_FIRST.equals( action ) ) {
			rowNo = new int[] { 0 };
		}
		// 7.3.1.1 (2021/02/25) action="VIEWFIRST" 機能の追加
		else if( ACT_VIEWFIRST.equals( action ) ) {
			final int startNo = nval( getRequestCacheData( "startNo" ),0 );		// viewFormでﾘｸｴｽﾄｷｬｯｼｭに登録している値を取り出す｡
			rowNo = new int[] { startNo };
		}
		// 5.6.4.3 (2013/05/24) tmpSelectedAll 属性の追加
		else if( tmpSelectedAll ) {
			final int rowCnt = table.getRowCount();
			rowNo = new int[ rowCnt ];
			for( int i=0; i<rowCnt; i++ ) {
				rowNo[i] = i;
			}
		}
		else {
			rowNo = super.getParameterRows() ;
		}
		return rowNo ;
	}

	/**
	 * 【TAG】(通常は使いません)結果のDBTableModelを､sessionに登録するときのｷｰを指定します
	 *		(初期値:HybsSystem#TBL_MDL_KEY[={@og.value HybsSystem#TBL_MDL_KEY}])｡
	 *
	 * @og.tag
	 * 検索結果より､DBTableModelｵﾌﾞｼﾞｪｸﾄを作成します｡これを､下流のviewﾀｸﾞ等に
	 * 渡す場合に､通常は､session を利用します｡その場合の登録ｷｰです｡
	 * query ﾀｸﾞを同時に実行して､結果を求める場合､同一ﾒﾓﾘに配置される為､
	 * この tableId 属性を利用して､ﾒﾓﾘ空間を分けます｡
	 *		(初期値:HybsSystem#TBL_MDL_KEY[={@og.value HybsSystem#TBL_MDL_KEY}])｡
	 *
	 * @param	id ﾃｰﾌﾞﾙID (sessionに登録する時のID)
	 */
	public void setTableId( final String id ) {
		tableId   = nval( getRequestParameter( id ),tableId );
	}

	/**
	 * 【TAG】ｺﾏﾝﾄﾞ (SET,GET,REMOVE,SQL,SETTBL,GETTBL,KEYTBL,CLEAR,SETMEM)をｾｯﾄします(初期値:SET)｡
	 *
	 * @og.tag
	 * ｺﾏﾝﾄﾞは,HTMLから(get/post)指定されますので,CMD_xxx で設定される
	 * ﾌｨｰﾙﾄﾞ定数値のいづれかを､指定できます｡
	 * 何も設定されない､または､null の場合は､"SET" が初期値にｾｯﾄされます｡
	 *
	 * CLEAR 以外のすべての処理は､指定のｽｺｰﾌﾟの内部ｷｬｯｼｭ情報に対して行われます｡
	 * <table class="plain">
	 *   <caption>ｺﾏﾝﾄﾞ 一覧</caption>
	 *   <tr><th>command </th><th>名称          </th><th>機能</th></tr>
	 *   <tr><td>SET    </td><td>ｾｯﾄ         </td><td>指定のｷｰに､value値を登録します｡</td></tr>
	 *   <tr><td>GET    </td><td>ｹﾞｯﾄ         </td><td>指定のｷｰの値を画面に出力します｡</td></tr>
	 *   <tr><td>REMOVE </td><td>ﾘﾑｰﾌﾞ       </td><td>指定のｷｰの値を削除します｡</td></tr>
	 *   <tr><td>CLEAR  </td><td>ｸﾘｱ         </td><td>ｾｯｼｮﾝ/ｱﾌﾟﾘｹｰｼｮﾝｽｺｰﾌﾟのｷｬｯｼｭ情報をｸﾘｱします｡</td></tr>
	 *   <tr><td>SQL    </td><td>ＳＱＬ         </td><td>指定のSQL文の実行結果を､ｶﾗﾑ名をｷｰとして設定します｡(useMultiRows="false" 時は､最初のﾃﾞｰﾀを使用)</td></tr>
	 *   <tr><td>SQLGET </td><td>ＳＱＬｹﾞｯﾄ   </td><td>指定のSQL文の実行結果を､画面に出力します｡(2項目以降は無視されます)｡</td></tr>
	 *   <tr><td>SETTBL </td><td>ｾｯﾄﾃｰﾌﾞﾙ </td><td>指定のｷｰに､DBTableModel の選択されたｶﾗﾑ(横持ち)の値を登録します｡</td></tr>
	 *   <tr><td>GETTBL </td><td>ｹﾞｯﾄﾃｰﾌﾞﾙ </td><td>指定のｷｰに､DBTableModel の選択されたｶﾗﾑ(横持ち)の値を画面に出力します｡</td></tr>
	 *   <tr><td>KEYTBL </td><td>ｷｰﾃｰﾌﾞﾙ   </td><td>指定のｷｰに､DBTableModel の選択されたﾛｳ(縦持ち)の値を登録します｡</td></tr>
	 *   <tr><td>SETMEM </td><td>ｾｯﾄﾒﾓﾘ   </td><td>指定のｷｰに､value値を内部ｷｬｯｼｭに登録します｡</td></tr>
	 *   <tr><td>DEBUG  </td><td>ﾃﾞﾊﾞｯｸﾞ       </td><td>指定のｽｺｰﾌﾟに登録されているﾊﾟﾗﾒｰﾀ（ｱﾄﾘﾋﾞｭｰﾄ）を返します｡</td></tr>
	 * </table>
	 *
	 * @og.rev 3.1.0.1 (2003/03/26) 指定のｺﾏﾝﾄﾞ以外は､ｴﾗｰとするように変更｡
	 * @og.rev 3.5.6.2 (2004/07/05) 文字列の連結にStringBuilderを使用します｡
	 * @og.rev 6.3.4.0 (2015/08/01) Arrays.toString から String.join に置き換え｡
	 * @og.rev 6.7.7.2 (2017/04/14) command に､DEBUG 追加
	 *
	 * @param	cmd ｺﾏﾝﾄﾞ (public static final 宣言されている文字列)
	 * @see		<a href="../../../../constant-values.html#org.opengion.hayabusa.taglib.ValueTag.CMD_GET">ｺﾏﾝﾄﾞ定数</a>
	 */
	public void setCommand( final String cmd ) {
		command = nval( getRequestParameter( cmd ),command ).toUpperCase(Locale.JAPAN);

		if( !check( command, COMMAND_SET ) ) {
			final String errMsg = "指定のｺﾏﾝﾄﾞは実行できません｡ｺﾏﾝﾄﾞｴﾗｰ"	+ CR
								+ "command=[" + command + "] "							+ CR
								+ "commandList=[" + String.join( ", " , COMMAND_SET ) + "]" ;
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【TAG】ｱｸｼｮﾝ(UPPER,LOWER,MESSAGE,APPEND,ALL_APPEND,MAP,ALL_MAP,MAPOBJ,ALL_MAPOBJ,LIST,ALL_LIST,SET,ALL_SET,DAY_WEEK,
	 *        MERGE,FIRST,VIEWFIRST,ROW_APPEND,REPLACE,INDEX_OF,LASTINDEX_OF,SUBSTR,LENGTH,
	 *        SPLIT,SPLIT_LIST,FOR,FOR_LIST,FORTM,FORTM_LIST,MAX_MIN,ALL_MAX_MIN,ADD,CASE,ESCAPE,
	 *        URL_ENCODE,URL_ENCODE2,URL_DECODE,TRIM,HASH,TO_NUMBER,TO_DATE,HMCSV,CRYPT_ENC,CRYPT_DEC,REQUEST,FORMAT)をｾｯﾄします｡
	 *
	 * @og.tag
	 * ｱｸｼｮﾝは,HTMLから(get/post)指定されますので,ACT_xxx で設定される
	 * ﾌｨｰﾙﾄﾞ定数値のいづれかを､指定できます｡
	 * 無指定の場合は､なにもしません｡
	 *
	 * <table class="plain">
	 *   <caption>ｱｸｼｮﾝ 一覧</caption>
	 *   <tr><th>action      </th><th>名称              </th><th>機能</th></tr>
	 *   <tr><td>UPPER       </td><td>ｱｯﾊﾟｰ(大文字化)   </td><td>value値を大文字に変換します｡</td></tr>
	 *   <tr><td>LOWER       </td><td>ﾛｰﾜｰ(小文字化)    </td><td>value値を小文字に変換します｡</td></tr>
	 *   <tr><td>MESSAGE     </td><td>ﾒｯｾｰｼﾞ変換        </td><td>引数をﾒｯｾｰｼﾞﾘｿｰｽのｷｰとして､ﾒｯｾｰｼﾞ変換します｡</td></tr>
	 *   <tr><td>APPEND      </td><td>ﾃﾞｰﾀｱﾍﾟﾝﾄﾞ        </td><td>複数ﾘｸｴｽﾄや複数選択時に値を連結します｡</td></tr>
	 *   <tr><td>ALL_APPEND  </td><td>ｵｰﾙｱﾍﾟﾝﾄﾞ         </td><td>SETTBL,GETTBL 時に､ﾁｪｯｸ行以外の全行を対象に値の連結を行います｡</td></tr>
	 *   <tr><td>MAP         </td><td>ﾏｯﾌﾟ              </td><td>検索結果やTableModelのkeyをｷｰに､valueを値に一括設定します｡※③</td></tr>
	 *   <tr><td>ALL_MAP     </td><td>ｵｰﾙﾏｯﾌﾟ           </td><td>全行を対象に検索結果やTableModelのkeyをｷｰに､valueを値に一括設定します｡※③</td></tr>
	 *   <tr><td>MAPOBJ      </td><td>ﾏｯﾌﾟｵﾌﾞｼﾞｪｸﾄ      </td><td>検索結果やTableModelのkeyに､ｷｰと値のMapを設定します｡6.2.4.2 (2015/05/29)</td></tr>
	 *   <tr><td>ALL_MAPOBJ  </td><td>ｵｰﾙﾏｯﾌﾟｵﾌﾞｼﾞｪｸﾄ   </td><td>全行を対象に検索結果やTableModelのkeyに､ｷｰと値のMapを設定します｡6.2.4.2 (2015/05/29)</td></tr>
	 *   <tr><td>LIST        </td><td>ﾘｽﾄ               </td><td>複数ﾘｸｴｽﾄや複数選択時に値をArrayListにｾｯﾄします｡4.3.7.5 (2009/07/13)</td></tr>
	 *   <tr><td>LIST_CSV    </td><td>ﾘｽﾄCSV            </td><td>複数ﾘｸｴｽﾄや複数選択時に値をArrayListにｾｯﾄ後CSV化します｡8.1.0.1 (2022/01/07)</td></tr>
	 *   <tr><td>ALL_LIST    </td><td>ｵｰﾙﾘｽﾄ            </td><td>全行を対象に値をArrayListにｾｯﾄします｡4.3.7.5 (2009/07/13)</td></tr>
	 *   <tr><td>ALL_LIST_CSV</td><td>ｵｰﾙﾘｽﾄCSV         </td><td>全行を対象に値をArrayListにｾｯﾄｾｯﾄ後CSV化します｡8.1.0.1 (2022/01/07)</td></tr>
	 *   <tr><td>SET         </td><td>ｾｯﾄ               </td><td>複数ﾘｸｴｽﾄや複数選択時に値を順序を保持したﾕﾆｰｸな集合(LinkedHashSet)にします｡8.1.0.1 (2022/01/07)</td></tr>
	 *   <tr><td>SET_CSV     </td><td>ｾｯﾄCSV            </td><td>複数ﾘｸｴｽﾄや複数選択時に値を順序を保持したﾕﾆｰｸな集合(LinkedHashSet)後CSV化します｡8.1.0.1 (2022/01/07)</td></tr>
	 *   <tr><td>ALL_SET     </td><td>ｵｰﾙｾｯﾄ            </td><td>全行を対象に値を順序を保持したﾕﾆｰｸな集合(LinkedHashSet)にします｡8.1.0.1 (2022/01/07)</td></tr>
	 *   <tr><td>ALL_SET_CSV </td><td>ｵｰﾙｾｯﾄCSV         </td><td>全行を対象に値を順序を保持したﾕﾆｰｸな集合(LinkedHashSet)後CSV化にします｡8.1.0.1 (2022/01/07)</td></tr>
	 *   <tr><td>DAY_WEEK    </td><td>日付前方まるめ    </td><td>日付型文字列(YYYYMMDD) の値を､月曜日に変換します｡</td></tr>
	 *   <tr><td>            </td><td>                  </td><td>指定日が日曜日の場合は､次の日(月曜日)に進めます｡その他は､週始めに戻します｡</td></tr>
	 *   <tr><td>MERGE       </td><td>ﾃﾞｰﾀのﾏｰｼﾞ        </td><td>重複を除く､ﾕﾆｰｸな値に､ﾏｰｼﾞします｡(ｶﾝﾏで分解､separatorで合成)</td></tr>
	 *   <tr><td>FIRST       </td><td>１件目取得        </td><td>最初の１件目を強制的に選択状態にして､処理を行います｡</td></tr>
	 *   <tr><td>VIEWFIRST   </td><td>表示１件目取得    </td><td>7.3.1.1 (2021/02/25) 現在表示の１件目を強制的に選択状態にして､処理を行います｡</td></tr>
	 *   <tr><td>ROW_APPEND  </td><td>検索結果の連結    </td><td>検索結果の行方向のﾃﾞｰﾀを連結します｡</td></tr>
	 *   <tr><td>REPLACE     </td><td>文字列置換        </td><td>value の値から､指定された正規表現(from)の部分文字列を､部分文字列(to)で置換します｡</td></tr>
	 *   <tr><td>INDEX_OF    </td><td>最初に出現する位置</td><td>value の値から､指定された(separator)が､fromVal以降に最初に出現する位置｡※⑤</td></tr>
	 *   <tr><td>LASTINDEX_OF</td><td>最後に出現する位置</td><td>value の値から､指定された(separator)が､toVal以前に最後に出現する位置｡※⑤</td></tr>
	 *   <tr><td>SUBSTR      </td><td>部分文字列        </td><td>value の値から､指定された(fromVal)から(toVal)の部分文字列を作成します｡</td></tr>
	 *   <tr><td>LENGTH      </td><td>文字数(char数)    </td><td>value の文字数(char数)をｶｳﾝﾄします｡</td></tr>
	 *   <tr><td>SPLIT       </td><td>文字列分割        </td><td>value の値から､指定されたseparatorで分割した文字列を作成します｡(key+0～連番)</td></tr>
	 *   <tr><td>SPLIT_LIST  </td><td>文字列分割LIST登録</td><td>value の値から､指定されたseparatorで分割した文字列をLISTに登録します｡ 7.3.1.1 (2021/02/25)</td></tr>
	 *   <tr><td>FOR         </td><td>繰返文字合成      </td><td>value の値に､fromVal,toVal,skipVal を使用した連結文字列を作成する｡7.3.1.1 (2021/02/25)※⑥</td></tr>
	 *   <tr><td>FOR_LIST    </td><td>繰返文字合成LIST登録</td><td>value の値に､fromVal,toVal,skipVal を使用した文字列をLISTに登録します｡ 7.3.1.1 (2021/02/25)※⑥</td></tr>
	 *   <tr><td>FORTM       </td><td>繰返日付文字合成  </td><td>fromVal,toVal,skipVal を使用した連結日付文字列(format指定可)を作成する｡8.0.1.2 (2021/11/19)※⑦</td></tr>
	 *   <tr><td>FORTM_LIST  </td><td>繰返日付文字合成LIST登録</td><td>fromVal,toVal,skipVal を使用した日付文字列(format指定可)をLISTに登録します｡ 8.0.1.2 (2021/11/19)※⑦</td></tr>
	 *   <tr><td>MAX_MIN     </td><td>最大_最小値       </td><td>選択行のkeyのｶﾗﾑの最小値,最大値を､MIN.ｷｰ､MAX.ｷｰ の値にｾｯﾄします｡ 5.6.4.3 (2013/05/24) ※①</td></tr>
	 *   <tr><td>ALL_MAX_MIN </td><td>最大_最小値       </td><td>全行を対象にkeyのｶﾗﾑの最小値,最大値を､MIN.ｷｰ､MAX.ｷｰ の値にｾｯﾄします｡ 5.6.4.3 (2013/05/24) ※①</td></tr>
	 *   <tr><td>EXEC 廃止   </td><td>計算結果          </td><td>【廃止】value またはBODY部の値を､JavaScriptで計算した結果をｾｯﾄします｡7.3.0.0 (2021/01/06)</td></tr>
	 *   <tr><td>ADD         </td><td>計算結果          </td><td>defaultVal の 値と､(value またはBODY部の)値を､加算した結果をｷｰ変数にｾｯﾄします｡7.3.0.0 (2021/01/06)</td></tr>
	 *   <tr><td>CASE        </td><td>条件分岐          </td><td>fromVal="A:1 B:2 C:3 D:4" 形式の値を元に､value値(左辺)に一致する右辺をｾｯﾄします｡ 5.7.7.2 (2014/06/20) ※②</td></tr>
	 *   <tr><td>ESCAPE      </td><td>ｴｽｹｰﾌﾟ文字        </td><td>HTML上のｴｽｹｰﾌﾟ文字(&amp;lt;,&amp;gt;,&amp;quot;,&amp;apos;,&amp;amp;)を変換します｡</td></tr>
	 *   <tr><td>URL_ENCODE  </td><td>URLｴﾝｺｰﾄﾞ         </td><td>UTF-8 で､URLｴﾝｺｰﾄﾞを行います｡</td></tr>
	 *   <tr><td>URL_ENCODE2 </td><td>ASCII以外の文字のURLｴﾝｺｰﾄﾞ</td><td>UTF-8 で､ASCII以外の文字の､URLｴﾝｺｰﾄﾞします｡</td></tr>
	 *   <tr><td>URL_DECODE  </td><td>URLﾃﾞｺｰﾄﾞ         </td><td>UTF-8 でURLｴﾝｺｰﾄﾞされた文字列をﾃﾞｺｰﾄﾞします｡</td></tr>
	 *   <tr><td>TRIM        </td><td>ｽﾍﾟｰｽ削除         </td><td>両端空白文字(ｽﾍﾟｰｽ､ﾀﾌﾞ､改行等)を削除します｡6.4.1.2 (2016/01/22)</td></tr>
	 *   <tr><td>HASH        </td><td>ﾊｯｼｭ変換          </td><td>値をformatに指定されたﾊｯｼｭｺｰﾄﾞで計算します｡8.1.2.0 (2022/03/10) Add</td></tr>
	 *   <tr><td>TO_NUMBER   </td><td>数値変換          </td><td>値から､数値以外の文字を削除します｡例えば､日付､金額など､区切り記号等を削除します｡6.7.4.0 (2017/02/10)</td></tr>
	 *   <tr><td>TO_DATE     </td><td>日付変換          </td><td>日付文字列から､数値以外の文字を削除します｡(/,-,:,' ',Tなどを削除) 8.0.1.2 (2021/11/19)※⑧</td></tr>
	 *   <tr><td>HMCSV       </td><td>時間(分)分割CSV   </td><td>ｽﾃｯﾌﾟ時間を分で指定します(例:1時間=60,2時間=120,45分=45など)｡
	 *   <tr><td>            </td><td>                  </td><td>valueには､初期日付(YMD形式)を設定します｡6.8.4.1 (2017/12/18)※④</td></tr>
	 *   <tr><td>CRYPT_ENC   </td><td>暗号化変換        </td><td>HybsCryptographyによる暗号化を行います｡暗号化ｷｰは､cryptKey 属性で指定します｡5.10.4.0 (2018/10/05)</td></tr>
	 *   <tr><td>CRYPT_DEC   </td><td>暗号化復号        </td><td>HybsCryptographyによる復号化を行います｡復号化ｷｰは､cryptKey 属性で指定します｡5.10.4.0 (2018/10/05)</td></tr>
	 *   <tr><td>REQUEST     </td><td>ﾘｸｴｽﾄ変数変換     </td><td>SQLなどで直接取得したﾃﾞｰﾀを､ﾘｸｴｽﾄ変換({&#064;XXXX}変換)します｡7.2.6.1 (2020/07/17)</td></tr>
	 *   <tr><td>FORMAT      </td><td>数字ﾌｫｰﾏｯﾄ変換    </td><td>value(数字)の値から､format<del>fromVal</del>に指定された書式文字列に変換します｡7.4.2.0 (2021/05/08)</td></tr>
	 * </table>
//	 *           MD5                  MD5変換                    値をMD5変換します｡ 8.1.2.0 (2022/03/10) Delete
	 *
	 * ※① MAX_MIN,ALL_MAX_MIN  については､DBTableModelのｶﾗﾑのCLS_NAMEが､NUMBER,INTEGER,DECIMAL の場合は､数字型(double)に変換して比較します｡
	 *     数字型(double)の場合だけ､合計(SUM.ｷｰ) と 平均(AVG.ｷｰ)計算も同時に行います｡
	 *     それ以外の場合は､文字列比較を行います｡(command="SETTBL" のみ)
	 *     command="SET" の場合は､数字型(double)として､value の値を､separator で分解したうえで､最大､最小､合計､平均 を計算し､
	 *     それぞれを､MIN.ｷｰ､MAX.ｷｰ､SUM.ｷｰ､AVG.ｷｰ 変数にｾｯﾄします｡
	 *     数字に変換できなかった場合は､ｴﾗｰになります｡
	 *     null は､無視され(平均を求める場合の母数としても無視)､すべてが nulll の場合は､ｾﾞﾛ文字列("")になります｡
	 *     ｾﾞﾛ文字列("") の場合に､defaultVal 属性が設定されていると､defaultVal が設定されます｡
	 * ※② CASE は､fromVal="A:1 B:2 C:3 D:4" 形式の値に対して､ｺﾛﾝ(:)で分解し､value値(左辺)が､一致する右辺を返します｡
	 *     fromVal の値は､ｽﾍﾟｰｽで分割され､ｺﾛﾝ(:)が存在する左辺(ｷｰ)と右辺(値)に分解し､value値とｷｰを比較し､
	 *     一致すれば､右辺の値を返します｡
	 *     valueがnullの場合､defaultVal が設定されており､その値で､fromVal の値と比較します｡
	 *     一致する値がなければ､元の value が返されます｡
	 * ※③ command="SQL"のMAP,ALL_MAP で､select key,val from TABLE のように､ｷｰと値のみのMapを作成していましたが､
	 *     第3引数以降も､Mapを作成するようにします｡ 6.9.0.2 (2018/02/13)
	 *     具体的には､select key,val,val1,val2,val3 ･･･ from TABLE とすると､key,val の関係は､従来どおりで､
	 *     val1 は､key + '_1' , val2 は､key + '_2' ･･･ という風に登録します｡
	 * ※④ HMCSV ｱｸｼｮﾝを実行すると､MIN(fromValの日付),MAX(toValの日付),HMF(日付の時:分ﾗﾍﾞﾙ) を追加します｡
	 *     fromVal,toVal を指定せず､valueに､CSV形式の日付ﾃﾞｰﾀを渡した場合は､MIN,MAX,HMF のみ求めます｡ 6.9.1.0 (2018/02/26)
	 * ※⑤ INDEX_OF( value.indexOf(separator,fromVal); )は､検索対象の部分文字列をseparatorに､検索開始位置のｲﾝﾃﾞｯｸｽをfromVal にｾｯﾄします｡
	 *     LASTINDEX_OF( value.lastIndexOf​(separator,toVal); )は､検索対象の部分文字列をseparatorに､
	 *     検索開始位置のｲﾝﾃﾞｯｸｽをtoVal にｾｯﾄします｡(fromValではないことを注意) 7.2.6.0 (2020/06/30)
	 * ※⑥ FOR は､valueに指定の文字列(未指定時は空文字列)に､fromVal ～ toVal を skipVal づつの数字文字列を合成した文字列を､separator で連結します｡
	 *     fromVal に､"000" など､先頭0で始まる2桁以上の文字列を指定した場合は､先頭ｾﾞﾛ埋めの文字列を生成します｡
	 *     FOR_LIST は､同じ処理の結果を､Listｵﾌﾞｼﾞｪｸﾄに登録します｡( c:forEach で取り出すことができます) 7.3.1.1 (2021/02/25)
	 *     format属性で、生成する文字列をﾌｫｰﾏｯﾄ(String#formatﾒｿｯﾄﾞ)できます。
	 * ※⑦ FORTM、FORTM_LIST は､FOR、FOR_LISTと同じく繰り返し文字列を生成しますが、文字列は日付ﾌｫｰﾏｯﾄになります。
	 *     skipValは、数字の場合は、その数だけ日付送りします。"M" 記号だけ特殊で、月送りします。 8.0.1.2 (2021/11/19)
	 *     format属性で、生成する日付文字列をﾌｫｰﾏｯﾄ(HybsDateUtil#getDateFormatﾒｿｯﾄﾞ)できます。
	 * ※⑧ TO_DATE は､日付ﾌｫｰﾏｯﾄから、不要な文字を削除します。(/,-,:,' ',Tなどを削除)
	 *     日付文字列としての整合性は、保証していません。 8.0.1.2 (2021/11/19)
	 *
	 * @og.rev 3.1.0.1 (2003/03/26) 指定のｱｸｼｮﾝ以外は､ｴﾗｰとするように変更｡
	 * @og.rev 3.5.6.2 (2004/07/05) 文字列の連結にStringBuilderを使用します｡
	 * @og.rev 4.3.7.5 (2009/07/13) ACT_LIST､ACT_ALL_LIST ｱｸｼｮﾝの追加(JavaDocのみ修正)
	 * @og.rev 5.6.4.3 (2013/05/24) ACT_MAX_MIN ｱｸｼｮﾝの追加(JavaDocのみ修正)
	 * @og.rev 5.6.4.3 (2013/05/24) tmpSelectedAll 属性追加にともなう修正｡action="ALL_xxx"は､tmpSelectedAll=true にします｡
	 * @og.rev 5.7.7.2 (2014/06/20) EXEC と CASE ｱｸｼｮﾝを追加
	 * @og.rev 6.2.4.2 (2015/05/29) MAPOBJ , ALL_MAPOBJ ｱｸｼｮﾝの追加
	 * @og.rev 6.3.4.0 (2015/08/01) Arrays.toString から String.join に置き換え｡
	 * @og.rev 6.4.1.2 (2016/01/22) TRIM ｱｸｼｮﾝを追加｡
	 * @og.rev 6.4.3.4 (2016/03/11) String配列 から､Setに置き換えます｡
	 * @og.rev 6.7.4.0 (2017/02/10) TO_NUMBER ｱｸｼｮﾝを追加｡
	 * @og.rev 6.8.4.1 (2017/12/18) HMCSV ｱｸｼｮﾝを追加
	 * @og.rev 8.0.1.2 (2021/11/19) ACT_TO_DATE,ACT_FORTM,ACT_FORTM_LIST 追加
	 * @og.rev 8.1.0.1 (2022/01/07) ACT_LIST_CSV､ACT_ALL_LIST_CSV,ACT_SET,ACT_ALL_SET,ACT_SET_CSV,ACT_ALL_SET_CSV ｱｸｼｮﾝの追加
	 *
	 * @param	act ｱｸｼｮﾝ (public static final 宣言されている文字列)
	 * @see		<a href="../../../../constant-values.html#org.opengion.hayabusa.taglib.ValueTag.ACT_APPEND">ｱｸｼｮﾝ定数</a>
	 */
	public void setAction( final String act ) {
		action = nval( getRequestParameter( act ),action );

		if( action != null && !check( action, ACTION_SET ) ) {
			final String errMsg = "指定のｱｸｼｮﾝは実行できません｡ｱｸｼｮﾝｴﾗｰ"	+ CR
								+ "action=[" + action + "] "							+ CR
								+ "actionList=[" + String.join( ", " , ACTION_SET ) + "]" ;
			throw new HybsSystemException( errMsg );
		}

		// 5.6.4.3 (2013/05/24) tmpSelectedAll 属性追加｡action="ALL_xxx"を tmpSelectedAll="true" に置き換えます｡
		if( action != null && action.startsWith( "ALL_" ) ) {
			action = action.substring( 4 );		// "ALL_" の４文字分削除
			tmpSelectedAll = true;
		}
	}

	/**
	 * 【TAG】ﾊﾟﾗﾒｰﾀ に登録するｷｰをｾｯﾄします｡
	 *
	 * @og.tag ﾊﾟﾗﾒｰﾀ に登録するｷｰをｾｯﾄします｡
	 *
	 * @og.rev 3.0.1.3 (2003/03/11) ｷｰを toUpperCase している箇所を削除
	 * @og.rev 6.2.6.0 (2015/06/19) ｷｰのnull判定は行わない｡(個別のﾒｿｯﾄﾞ内で実施している｡)
	 *
	 * @param	key1 登録ｷｰ
	 */
	public void setKey( final String key1 ) {
		key = nval( getRequestParameter( key1 ),key ) ;
	}

	/**
	 * 【TAG】ﾊﾟﾗﾒｰﾀ に登録する値をｾｯﾄします｡
	 *
	 * @og.tag ﾊﾟﾗﾒｰﾀ に登録する値をｾｯﾄします｡
	 *
	 * @og.rev 3.1.7.0 (2003/05/02) isNullSet 属性が true(初期値)のときは､ﾘｸｴｽﾄ情報から値を取得｡
	 * @og.rev 3.5.4.0 (2003/11/25) getRequestParameter( value ) ﾒｿｯﾄﾞを setValue に移動｡
	 * @og.rev 5.1.8.0 (2010/07/01) isNullSet 属性 廃止
	 *
	 * @param	val	登録値
	 */
	public void setValue( final String val ) {
		inValue = val;		// 3.5.4.0 (2003/11/25) 入力変数も､ｷｰﾌﾟしておく｡
		value = getRequestParameter( inValue );
	}

	/**
	 * 【TAG】value属性に値がｾｯﾄされていないとき使用する､初期値を指定します｡
	 *
	 * @og.tag
	 * value属性に値がｾｯﾄされていないときに､この初期値を使用します｡
	 *
	 * @param	val 初期値
	 */
	public void setDefaultVal( final String val ) {
		defaultVal = getRequestParameter( val );
	}

//	/**
//	 * 【廃止】value が NULL の時に､設定するかどうか[true/false]を指定します(初期値:true)｡
//	 *
//	 * @og.tag
//	 * true の場合は, null のときでもｾｯﾄします｡
//	 * false の場合は, null のときは､既存の値を置き換えません｡
//	 * 初期値は､null のときでもｾｯﾄするです｡ ("true")
//	 *
//	 * @og.rev 5.1.8.0 (2010/07/01) isNullSet 属性 廃止
//	 * @og.rev 6.9.5.0 (2018/04/23) 物理的に廃止
//	 *
//	 * @param   flag NULL時設定有無 [true:設定する/それ以外:設定しない]
//	 * @deprecated 5.1.8.0 (2010/07/01) 廃止｡
//	 */
//	@Deprecated public void setNullSet( final String flag ) {
//		// 廃止予定
//	}

	/**
	 * 【TAG】(通常は使いません)Queryｵﾌﾞｼﾞｪｸﾄを作成する時のDB接続IDを指定します｡
	 *
	 * @og.tag Queryｵﾌﾞｼﾞｪｸﾄを作成する時のDB接続IDを指定します｡
	 *
	 * @param	id ﾃﾞｰﾀﾍﾞｰｽ接続ID
	 */
	public void setDbid( final String id ) {
		dbid = nval( getRequestParameter( id ),dbid );
	}

	/**
	 * 【TAG】ｷｬｯｼｭする場合のｽｺｰﾌﾟ(request,session)を指定します(初期値:request)｡
	 *
	 * @og.tag
	 * "request","session" が指定できます｡
	 * 初期値は､ "request" です｡
	 *
	 * @param	scp	ｽｺｰﾌﾟ
	 */
	@Override
	public void setScope( final String scp ) {
		scope = nval( getRequestParameter( scp ),scope );
	}

	/**
	 * 【TAG】DBTableModel から取得する場合のｽｺｰﾌﾟ(request,session)を指定します(初期値:session)｡
	 *
	 * @og.tag
	 * "request","session" が指定できます｡
	 * 初期値は､ "session" です｡
	 *
	 * @og.rev 5.1.2.0 (2010/01/01) DBTableModel の取得先のscope
	 *
	 * @param	scp	ｽｺｰﾌﾟ
	 */
	public void setTblScope( final String scp ) {
		tblScope = nval( getRequestParameter( scp ),tblScope );
	}

	/**
	 * 【TAG】ﾏﾙﾁﾃﾞｰﾀ(複数件検索)を使用するかどうか[true/false]を指定します(初期値:false)｡
	 *
	 * @og.tag
	 * command="SQL" の場合に､複数行検索した結果を､ｷｰ＋行番号 というｷｰを作成して
	 * 値を設定するかどうかを指定します｡
	 * false の場合は､従来どおり､検索ｶﾗﾑ名がｷｰになります｡
	 * 初期値は､false です｡
	 *
	 * @og.rev 3.2.4.0 (2003/06/12) 新規追加
	 *
	 * @param   flag 複数件検索 [true:使用する/false:使用しない]
	 */
	public void setUseMultiRows( final String flag ) {
		useMultiRows = nval( getRequestParameter( flag ),useMultiRows );
	}

	/**
	 * 【TAG】各種ｱｸｼｮﾝの文字列を連結/分解する項目区切り文字をｾｯﾄします(初期値:",")｡
	 *
	 * @og.tag
	 * 各種ｱｸｼｮﾝに基づく処理において､文字列の区切りを指定するのに使用します｡
	 * APPEND､ROW_APPEND 時には､文字列の連結に使用します｡
	 * MERGE の場合は､ｶﾝﾏで分解後､このｾﾊﾟﾚｰﾀでMERGE処理を行い､再び､連結します｡
	 * 初期値は､"," に設定されています｡
	 *
	 * @og.rev 3.5.6.5 (2004/08/09) 新規追加
	 *
	 * @param   sepa 項目区切り文字 (初期値:",")
	 */
	public void setSeparator( final String sepa ) {
		separator = nval( getRequestParameter( sepa ),separator );
	}

	/**
	 * 【TAG】ﾊﾟﾗﾒｰﾀの HTMLTag開始/終了文字(&gt;&lt;) 存在ﾁｪｯｸを実施するかどうか[true/false]を設定します
	 *		(初期値:USE_XSS_CHECK[={@og.value SystemData#USE_XSS_CHECK}])｡
	 *
	 * @og.tag
	 * ｸﾛｽｻｲﾄｽｸﾘﾌﾟﾃｨﾝｸﾞ(XSS)対策の一環としてless/greater than signについてのﾁｪｯｸを行います｡
	 * (&gt;&lt;) が含まれていたｴﾗｰにする(true)／かﾉｰﾁｪｯｸか(false)を指定します｡
	 * (初期値:ｼｽﾃﾑ定数のUSE_XSS_CHECK[={@og.value SystemData#USE_XSS_CHECK}])｡
	 *
	 * @og.rev 5.1.7.0 (2010/06/01) 新規追加
	 *
	 * @param	flag	XSSﾁｪｯｸ [true:する/false:しない]
	 * @see		org.opengion.hayabusa.common.SystemData#USE_XSS_CHECK
	 */
	public void setXssCheck( final String flag ) {
		xssCheck = nval( getRequestParameter( flag ),xssCheck );
	}

	/**
	 * 【TAG】部分文字列置換の 置換え前の部分文字列(from)を指定します｡
	 *
	 * @og.tag
	 * value の値から､指定された正規表現(from)に一致する､この文字列の各部分文字列に対し､
	 * 指定された文字列(to)で置換します｡
	 * value.replaceAll( from, to ) という文法で処理します｡
	 *
	 * ※ action="CASE" 時は､fromVal="A:1 B:2 C:3 D:4" 形式で指定可能です｡
	 * ※ action="SUBSTR" 時は､fromVal は､数字 のみ設定可能です｡
	 * ※ action="REPLACE" 時は､fromVal は､CSV形式で､複数のﾘﾌﾟﾚｰｽｷｰを指定できます｡
	 *            例: @1,@2,@3･･
	 * ※ action="FOR,FOR_LIST" 時は､繰り返し文字列の作成開始文字を指定します。
	 * ※ action="FORTM,FORTM_LIST" 時は､繰り返し日付文字列の作成開始日付を指定します。
	 * ※ action="HMCSV" 時は､時刻のCSVﾘｽﾄの開始日付を指定します。
	 *
	 * @og.rev 5.2.2.0 (2010/11/01) 新規追加
	 *
	 * @param	from	置換え前の部分文字列
	 * @see		#setToVal(String)
	 */
	public void setFromVal( final String from ) {
		fromVal = nval( getRequestParameter( from ),fromVal );
	}

	/**
	 * 【TAG】部分文字列置換の 置換え後の部分文字列(to)を指定します｡
	 *
	 * @og.tag
	 * value の値から､指定された正規表現(from)に一致する､この文字列の各部分文字列に対し､
	 * 指定された文字列(to)で置換します｡
	 * value.replaceAll( from, to ) という文法で処理します｡
	 *
	 * ※ action="SUBSTR" 時は､toVal は､数字 のみ設定可能です｡
	 * ※ action="REPLACE" 時は､toVal は､CSV形式で､複数の置換え後の文字列を指定できます｡
	 *            例: VLA1,VAL2,VAL･･
	 * ※ action="REPLACE" 時に､toVal='""' を指定すると､ｾﾞﾛ文字列としてfromValの削除になります｡
	 * ※ action="FOR,FOR_LIST" 時は､繰り返し文字列の作成終了文字を指定します。
	 * ※ action="FORTM,FORTM_LIST" 時は､繰り返し日付文字列の作成終了日付を指定します。
	 * ※ action="HMCSV" 時は､時刻のCSVﾘｽﾄの終了日付を指定します。
	 *
	 * @og.rev 5.2.2.0 (2010/11/01) 新規追加
	 * @og.rev 7.3.2.2 (2021/03/31) REPLACEｱｸｼｮﾝで､toVal に､長さｾﾞﾛの文字列が扱えるように変更｡
	 *
	 * @param	to	置換え後の部分文字列
	 * @see		#setFromVal(String)
	 */
	public void setToVal( final String to ) {
		toVal = nval( getRequestParameter( to ),toVal );

		if( "\"\"".equals( toVal ) ) { toVal = ""; }	// 7.3.2.2 (2021/03/31) 長さｾﾞﾛの文字列対応
	}

	/**
	 * 【TAG】HMCSV,FOR,FOR_LIST,FORTM,FORTM_LISTで使用する､ｽﾃｯﾌﾟ数を指定します｡
	 *
	 * @og.tag
	 * fromVal と､toVal を組み合わせて､使用します｡
	 * action="HMCSV" で､時刻のCSVﾘｽﾄを作成する場合は、ｽﾃｯﾌﾟ時間を分で指定します(例:1時間=60 , 2時間=120 , 45分=45 など)
	 * 例えば､value="20171213" fromVal="0600" toVal="1000" skipVal="45" とすると､
	 * 201712130600,201712130645,201712130730,201712130815,201712130900,201712130945 という文字列を作成します｡
	 * ﾃﾞﾌｫﾙﾄは､60分(１時間)です｡
	 *
	 * action=FOR,FOR_LIST のｽﾃｯﾌﾟ数の指定時は、一般的な数字です。
	 *
	 * action=FORTM,FORTM_LIST のｽﾃｯﾌﾟ数の指定時は、日数になります。(HMCSVの基準と異なるので注意)
	 * 特殊系として、"M" を指定した場合は、月毎になります。
	 *
	 * skipVal には､ﾏｲﾅｽやｾﾞﾛは､指定できません｡
	 *
	 * @og.rev 6.8.4.1 (2017/12/18) skipVal 属性 追加
	 *
	 * @param	skip	HMCSVで使用する､ｽﾃｯﾌﾟ時間を分で指定
	 */
	public void setSkipVal( final String skip ) {
		skipVal = nval( getRequestParameter( skip ),skipVal );
		if( !StringUtil.isNull( skipVal ) ) {
			final char ch = skipVal.charAt(0);
			if( ch == '-' || ch == '0' ) {
				final String errMsg = "skipVal属性には､ﾏｲﾅｽやｾﾞﾛは､指定できません｡"
							+ " skipVal=[" + skipVal + "]" + CR ;
				throw new HybsSystemException( errMsg );
			}
		}
	}

	/**
	 * 【TAG】actionが、FORMAT,FOR,FOR_LIST,FORTM,FORTM_LIST の場合の文字列変換ﾌｫｰﾏｯﾄを指定します｡
	 *
	 * @og.tag
	 * FORMAT,FOR,FOR_LIST の場合に生成する文字列のﾌｫｰﾏｯﾄは、String#formatﾒｿｯﾄﾞ を使用します。
	 * FORTM,FORTM_LIST の場合に生成する日付文字列のﾌｫｰﾏｯﾄは、HybsDateUtil#getDateFormatﾒｿｯﾄﾞ を使用します。
	 *
	 * @og.rev 8.0.1.2 (2021/11/19) format 属性 追加
	 *
	 * @param	fmt	数値や日付に変換する場合のﾌｫｰﾏｯﾄを指定
	 */
	public void setFormat( final String fmt ) {
		format = nval( getRequestParameter( fmt ),format );
	}

	/**
	 * 【TAG】暗号化､復号化に利用する暗号化ｷｰを指定します｡
	 *
	 * @og.tag
	 * 暗号化ｷｰをしています｡
	 * 未設定の場合は､ｴﾝｼﾞﾝで用意されているｷｰを利用します｡
	 * 暗号化ｷｰは8の倍数の文字数が推奨されています｡
	 *
	 * @og.rev 5.10.4.0 (2018/10/05) 新規追加
	 *
	 * @param cKey	暗号化ｷｰ
	 */
	public void setCryptKey( final String cKey ) {
		cryptKey = nval( getRequestParameter( cKey ), cryptKey);
	}

	/**
	 * このｵﾌﾞｼﾞｪｸﾄの文字列表現を返します｡
	 * 基本的にﾃﾞﾊﾞｯｸﾞ目的に使用します｡
	 *
	 * @og.rev 4.3.4.0 (2008/12/01) PageContextのｽｺｰﾌﾟをｸﾗｽ変数としてｱｸｾｽ
	 * @og.rev 5.1.8.0 (2010/07/01) isNullSet 属性 廃止
	 * @og.rev 6.7.7.2 (2017/04/14) 属性の追加と､各ｽｺｰﾌﾟ変数の表示廃止
	 *
	 * @return このｸﾗｽの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION		)
				.println( "tableId"			,tableId		)
				.println( "command"			,command		)
				.println( "action"			,action			)
				.println( "key"				,key			)
				.println( "inValue"			,inValue		)
				.println( "value"			,value			)
				.println( "defaultVal"		,defaultVal		)
				.println( "dbid"			,dbid			)
				.println( "scope"			,scope			)
				.println( "tblScope"		,tblScope		)				// 6.7.7.2 (2017/04/14)
				.println( "separator"		,separator		)
				.println( "useMultiRows"	,useMultiRows	)
				.println( "xssCheck"		,xssCheck		)				// 6.7.7.2 (2017/04/14)
				.println( "fromVal"			,fromVal		)				// 6.7.7.2 (2017/04/14)
				.println( "toVal"			,toVal			)				// 6.7.7.2 (2017/04/14)
				.println( "skipVal"			,skipVal		)				// 6.8.4.1 (2017/12/18)
				.println( "tmpSelectedAll"	,tmpSelectedAll	)				// 6.7.7.2 (2017/04/14)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
