/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import java.util.Locale;
import java.util.Set;											// 6.4.3.4 (2016/03/11)
import java.util.Enumeration;
import java.util.concurrent.ConcurrentMap;						// 6.4.3.3 (2016/03/04)
import java.util.concurrent.ConcurrentHashMap;					// 6.4.3.3 (2016/03/04)

import jakarta.servlet.ServletRequest ;

import org.opengion.fukurou.system.DateSet;						// 6.4.2.0 (2016/01/29)
import org.opengion.fukurou.system.ThrowUtil;					// 6.4.2.0 (2016/01/29)
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
import org.opengion.fukurou.util.ArraySet;						// 6.4.3.4 (2016/03/11)
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.mail.MailManager_DIRECT;
import org.opengion.hayabusa.db.DBTableModel;

import static org.opengion.fukurou.util.StringUtil.nval;

/**
 * 定型文およびパラメータの設定によるメールを送信するためのタグです。
 *
 * @og.formSample
 * ●形式：&lt;og:mailSender ptnId="…" action="…" from="…" to="…" /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:mailSender2
 *       ptnId            ○【TAG】メール定型文のIDを指定します(必須)。
 *       from             ○【TAG】送信元(ＦＲＯＭ)の社員IDを指定します(必須)。
 *       action           ○【TAG】アクション[CHECK/SEND/NOCHECK]をセットします(必須)。
 *       addrCheck          【TAG】メールアドレスの構文とメールアカウントのチェックをするかどうか[true/false]を指定します
 *       to                 【TAG】送信先(ＴＯ)の社員ID、グループIDをCSV形式で指定します
 *       cc                 【TAG】送信先(ＣＣ)の社員ID、グループIDをCSV形式で指定します
 *       bcc                【TAG】送信先(ＢＣＣ)の社員ID、グループIDをCSV形式で指定します
 *       tableId            【TAG】(通常は使いません)宛先のDBTableModelを、sessionに登録するときのキーを指定します
 *       scope              【TAG】キャッシュする場合のスコープ[request/page/session/application]を指定します(初期値:session)
 *       fileURL            【TAG】添付ファイルのセーブディレクトリを指定します (初期値:FILE_URL[=filetemp/])
 *       filename           【TAG】添付ファイル名をCSV形式で指定します
 *       useStop            【TAG】例外発生した場合、後続JSPの評価を中止するかどうか[true:中止/false:継続]を指定します
 *       useSLabel          【TAG】7.0.7.0 (2019/12/13) ｴﾗｰﾒｯｾｰｼﾞにSLABELを利用するかどうか[true/false]を指定します(初期値:false)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *     &lt;og:mailSender2 &gt;
 *        ptnId     = PtnId      定型文ID(定型文マスタに登録されている定型文ID)
 *        action    = Action     アクション(CHECK:確認あり/SEND:確認後の送信/NOCHECK:確認なし)
 *        from      = From       送信元(送信者社員ID)
 *        to        = To         送信先(コンマ区切りで複数セット可能、社員ID、グループID)
 *        cc        = Cc         送信先(コンマ区切りで複数セット可能、社員ID、グループID)
 *        bcc       = Bcc        送信先(コンマ区切りで複数セット可能、社員ID、グループID)
 *        fileURL   = 添付ファイルのセーブディレクトリ
 *        filename  = 添付ファイル名(ローカルにセーブされたファイル名)(コンマ区切りで複数登録可能)
 *        addrCheck = true/false(メールアカウントの有効チェック)
 *        useStop   = true/false　エラー発生時に後続JSPの評価を中止する(true)/中止しない(false)
 *        scope     = request/session 宛先テーブルの格納スコープ(デフォルト：session)
 *        tableId   = TableId    宛先テーブルのID(通常はデフォルトのテーブルモデルID名称を利用します)
 *        debug     = true/false
 *     &lt;/og:mailSender &gt;
 *
 * from には社員IDしかセットできません。
 * to,cc,bccには社員ID、またはグループIDをコンマ区切りで複数セットできます。
 * action:CHECK は送信前に、一度送信内容を確認したい場合に利用します。action=CHECKの場合、scopeにはsessionしかセットできません。
 * action:SEND は確認済のメール文を送信する場合に利用します。
 * action:NOCHECK は確認なしで送信したい場合に利用します。
 *
 * @og.group その他出力
 *
 * @version  4.0
 * @author   Sen.Li
 * @since    JDK1.6
 */
public class MailSenderTag2 extends CommonTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "7.0.7.0 (2019/12/13)" ;
	private static final long serialVersionUID = 707020191213L ;

	private static final String 	ACT_CHECK			= "CHECK" ;
	private static final String 	ACT_SEND			= "SEND" ;
	private static final String 	ACT_NOCHECK			= "NOCHECK" ;
	// 6.4.3.4 (2016/03/11) String配列 から、Setに置き換えます。
	private static final Set<String> ACTION_SET = new ArraySet<>( ACT_CHECK , ACT_SEND, ACT_NOCHECK );

	private static final int 		MAX_FILE_COUNT		= 5 ;
	private String		ptnId		;
	private String		action		;
	private String 		from		;
	private String		to			;
	private String		cc			;
	private String		bcc			;
	private String		fileURL		= HybsSystem.sys( "FILE_URL" );
	private String[]	filename	;
	private String		tableId	 	= HybsSystem.TBL_MDL_KEY ;
	private boolean		addrCheck	;
	private boolean		useStop		= true;
	private boolean		useSLabel	;		// 7.0.7.0 (2019/12/13) ｴﾗｰﾒｯｾｰｼﾞにSLABELを利用するかどうか[true/false]を指定します(初期値:false)

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public MailSenderTag2() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 6.3.4.0 (2015/08/01) Arrays.toString から String.join に置き換え。
	 * @og.rev 6.4.2.0 (2016/01/29) ex.printStackTrace() を、ThrowUtil#ogStackTrace(Throwable) に置き換え。
	 * @og.rev 6.4.3.3 (2016/03/04) ConcurrentHashMap を受け取ることを明確にするため、I/FをConcurrentMapに変更します。
	 * @og.rev 6.4.3.4 (2016/03/11) String配列 から、Setに置き換えます。
	 * @og.rev 6.5.0.1 (2016/10/21) ErrorMessage をまとめるのと、直接 Throwable を渡します。
	 * @og.rev 7.0.7.0 (2019/12/13) useSLabel 属性を追加。
	 *
	 * @return	後続処理の指示
	*/
	@Override
	public int doEndTag() {
		debugPrint();
		int rtnCode = EVAL_PAGE;
		int	errCode	= ErrorMessage.OK;

		if( check( action, ACTION_SET ) ) {
			try {
				tableId = ( tableId == null ) ? HybsSystem.TBL_MDL_KEY:tableId;
				final MailManager_DIRECT manager = new MailManager_DIRECT();
				manager.setResourceManager( getResource() );
				DBTableModel table = null;

				if( ACT_NOCHECK.equals( action ) || ACT_CHECK.equals( action ) ){
					final ConcurrentMap<String,String> initParamMap = makeParamMap();		// 6.4.3.3 (2016/03/04)
					manager.create( initParamMap );
				}
				if( ACT_NOCHECK.equals( action ) ) {
					manager.setDebug( isDebug() );
					manager.send();
				}
				else if( ACT_CHECK.equals( action ) ) {
					setSessionAttribute( "MAIL.FROM_ADDR", manager.getFromAddr() );
					setSessionAttribute( "MAIL.PTN_ID", ptnId );
					setSessionAttribute( "MAIL.TITLE", manager.getTitle() );
					setSessionAttribute( "MAIL.CONTENT", manager.getContent() );
				}
				else if( ACT_SEND.equals( action ) ) {
					ptnId = (String) getSessionAttribute( "MAIL.PTN_ID" );
					final ConcurrentMap<String,String> initParamMap = makeParamMap();		// 6.4.3.3 (2016/03/04)
					manager.setFromAddr( (String) getSessionAttribute( "MAIL.FROM_ADDR" ) );
					manager.setTitle( (String) getSessionAttribute( "MAIL.TITLE" ) );
					manager.setContent( (String) getSessionAttribute( "MAIL.CONTENT" ) );
					table = ( DBTableModel )getObject( tableId );
					manager.create( initParamMap, table );
					manager.setDebug( isDebug() );
					manager.send();
				}
				startQueryTransaction( tableId );
				table = manager.makeDstTable();
				if( ! commitTableObject( tableId, table ) ) {
					jspPrint( "DBTableModel は登録しません。" );
				}
			}
			catch( final RuntimeException rex ){
				if( useStop ) {
					final ErrorMessage errMsg = new ErrorMessage();
					// 6.5.0.1 (2016/10/21) ErrorMessage をまとめるのと、直接 Throwable を渡します。
					errMsg.addMessage( 0, ErrorMessage.NG, "ERR0040", rex.getMessage() )
						.addMessage( rex );

//					jspPrint( TaglibUtil.makeHTMLErrorTable( errMsg, getResource() ) );
					jspPrint( TaglibUtil.makeHTMLErrorTable( errMsg, getResource(),useSLabel ) );		// 7.0.7.0 (2019/12/13)
					rtnCode = SKIP_PAGE;
				}
				System.err.println( ThrowUtil.ogStackTrace( rex ) );				// 6.4.2.0 (2016/01/29)
				errCode = ErrorMessage.WARNING;
			}
			setSessionAttribute( "MAIL.ERR_CODE", String.valueOf( errCode ) );
		}
		else {
			final String errMsg = "指定のアクションは実行できません。アクションエラー"	+ CR
							+ "action=[" + action + "] "								+ CR
							+ "actionList=" + String.join( ", " , ACTION_SET ) ;
			throw new HybsSystemException( errMsg );
		}
		return rtnCode;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 7.0.7.0 (2019/12/13) useSLabel 属性を追加。
	 */
	@Override
	protected void release2() {
		super.release2();
		from		= null;
		to			= null;
		cc			= null;
		bcc			= null;
		fileURL		= HybsSystem.sys( "FILE_URL" );
		filename	= null;
		ptnId		= null;
		action		= null;
		tableId		= HybsSystem.TBL_MDL_KEY ;
		addrCheck	= false;
		useStop		= true;
		useSLabel	= false;	// 7.0.7.0 (2019/12/13) ｴﾗｰﾒｯｾｰｼﾞにSLABELを利用するかどうか[true/false]を指定します(初期値:false)
	}

	/**
	 * 【TAG】アクション[CHECK/SEND/NOCHECK]をセットします。
	 * @og.tag
	 * 送信前に、一度送信内容を確認する場合、"CHECK "をセットします。
	 * 確認済のメール文を送信する場合、"SEND"をセットします。
	 * 確認なしで送信する場合、"NOCHECK"をセットします。
	 *
	 * @param	act アクション [CHECK/SEND/NOCHECK]
	 */
	public void setAction( final String act ) {
		final String act2 = getRequestParameter( act );
		if( act2 != null && act2.length() > 0 ) { action = act2.toUpperCase(Locale.JAPAN); }
	}

	/**
	 * 【TAG】メール定型文のIDを指定します。
	 *
	 * @og.tag
	 * 定型文マスタに定義されている定型文IDを指定します。
	 *
	 * @param   pid 定型文ID
	 */
	public void setPtnId( final String pid ) {
		ptnId = nval( getRequestParameter( pid ),null );
	}

	/**
	 * 【TAG】送信元(ＦＲＯＭ)の社員IDを指定します。
	 *
	 * @og.tag
	 * 送信元(ＦＲＯＭ)の社員IDを指定します。社員マスタに存在している社員ID(例："C12345")しかセットできません。
	 *
	 * @param   fromId 送信元(ＦＲＯＭ)の社員ID
	 */
	public void setFrom( final String fromId ) {
		from = nval( getRequestParameter( fromId ), from );
		setRequestAttribute( "FROM", from );
	}

	/**
	 * 【TAG】送信先(ＴＯ)の社員ID、グループIDをCSV形式で指定します。
	 *
	 * @og.tag
	 * 複数のID(社員ID、グループID)をCSV形式でセットできます。
	 * グループIDはグループマスタ管理画面により定義する必要があります。"GP.XXXXX"の形式でセットします。
	 *
	 * @param   toIds 送信先(ＴＯ)の社員ID、グループID(CSV形式)
	 */
	public void setTo( final String toIds ) {
		to = getRequestParameter( toIds );
	}

	/**
	 * 【TAG】送信先(ＣＣ)の社員ID、グループIDをCSV形式で指定します。
	 *
	 * @og.tag
	 * 複数のID(社員ID、グループID)をCSV形式でセットできます。
	 * グループIDはグループマスタ管理画面により定義する必要があります。"GP.XXXXX"の形式でセットします。
	 *
	 * @param   ccIds 送信先(ＣＣ)の社員ID、グループID(CSV形式)
	 */
	public void setCc( final String ccIds ) {
		cc = getRequestParameter( ccIds );
	}

	/**
	 * 【TAG】送信先(ＢＣＣ)の社員ID、グループIDをCSV形式で指定します。
	 *
	 * @og.tag
	 * 複数のID(社員ID、グループID)をCSV形式でセットできます。
	 * グループIDはグループマスタ管理画面により定義する必要があります。"GP.XXXXX"の形式でセットします。
	 *
	 * @param   bccIds 送信先(ＢＣＣ)の社員ID、グループID(CSV形式)
	 */
	public void setBcc( final String bccIds ) {
		bcc = getRequestParameter( bccIds );
	}

	/**
	 * 【TAG】添付ファイルのセーブディレクトリを指定します
	 *		(初期値:FILE_URL[={@og.value SystemData#FILE_URL}])。
	 *
	 * @og.tag
	 * この属性で指定されるディレクトリに、添付ファイルが存在すると仮定します。
	 * 指定方法は、通常の fileURL 属性と同様に、先頭が、'/' (UNIX) または、２文字目が、
	 * ":" (Windows)の場合は、指定のURLそのままのディレクトリに、そうでない場合は、
	 * fileURL = "{&#064;USER.ID}" と指定すると、FILE_URL 属性で指定のフォルダの下に、
	 * さらに、各個人ID別のフォルダを作成して、そこを使用します。
	 * (初期値:システム定数のFILE_URL[={@og.value SystemData#FILE_URL}])。
	 *
	 * @og.rev 6.4.2.1 (2016/02/05) URLの最後に、"/" を追加する処理を廃止。
	 *
	 * @param	url 添付ファイルのセーブディレクトリ
	 * @see		org.opengion.hayabusa.common.SystemData#FILE_URL
	 */
	public void setFileURL( final String url ) {
		final String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
			fileURL = StringUtil.urlAppend( fileURL,furl );
		}
	}

	/**
	 * 【TAG】添付ファイル名をCSV形式で指定します。
	 *
	 * @og.tag
	 * 複数ファイルをセットできます。
	 * 設定方法は、カンマで区切って並べ複数指定できます。
	 *
	 * @param   fname 添付ファイル名
	 */
	public void setFilename( final String fname ) {
		filename = StringUtil.csv2ArrayOnly( getRequestParameter( fname ) );
	}

	/**
	 * 【TAG】メールアドレスの構文とメールアカウントのチェックをするかどうか[true:する/false:しない]を指定します。
	 *
	 * @og.tag
	 * メールアドレスの構文とメールアカウントのチェック[true:する/false:しない]を指定します。
	 * メール文合成の段階では、メールアドレスの構文文法についてチェックします。
	 * メール送信の段階では、メールアカウントが有効かについてチェックします。
	 * "true"と指定する場合、ｴﾗｰが検出されたら、例外を投げて本タグの処理が中止されます。
	 * "false"と指定する場合、ｴﾗｰが検出されても、例外を投げません。
	 *
	 * @param   addrChk 構文,アカウントチェック可否 [true:する/false:しない]
	 */
	public void setAddrCheck( final String addrChk ) {
		addrCheck = nval( getRequestParameter( addrChk ), addrCheck );
	}

	/**
	 * 【TAG】例外発生した場合、後続JSPの評価を中止するかどうか[true:中止/false:継続]を指定します。
	 *
	 * @og.tag
	 * "true"と指定する場合、例外が発生したら、後続JSPが評価されません。
	 * "false"と指定する場合、例外が発生しても、後続JSPが評価されます。後続のJSPでは変数
	 * {&#064;MAIL.ERR_CODE}で本タグの実行状況(ｴﾗｰ発生したか)を取得できます。
	 *
	 * @param   stop 例外時に後続処理を中止可否 [true:中止/false:継続]
	 */
	public void setUseStop( final String stop ) {
		useStop = nval( getRequestParameter( stop ), useStop );
	}

	/**
	 * 【TAG】(通常は使いません)結果のDBTableModelを、sessionに登録するときのキーを指定します
	 *		(初期値:HybsSystem#TBL_MDL_KEY[={@og.value HybsSystem#TBL_MDL_KEY}])。
	 *
	 * @og.tag
	 * 検索結果より、DBTableModelオブジェクトを作成します。これを、下流のviewタグ等に
	 * 渡す場合に、通常は、session を利用します。その場合の登録キーです。
	 * query タグを同時に実行して、結果を求める場合、同一メモリに配置される為、
	 * この tableId 属性を利用して、メモリ空間を分けます。
	 *		(初期値:HybsSystem#TBL_MDL_KEY[={@og.value HybsSystem#TBL_MDL_KEY}])。
	 *
	 * @param	id テーブルID (sessionに登録する時のID)
	 */
	public void setTableId( final String id ) {
		tableId   = nval( getRequestParameter( id ),tableId );	// 3.8.0.9 (2005/10/17)
	}

	/**
	 * 【TAG】ｴﾗｰﾒｯｾｰｼﾞにSLABELを利用するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * 通常のｴﾗｰﾒｯｾｰｼﾞは、ﾗﾍﾞﾙ（長）が使われますが、これをﾗﾍﾞﾙ（短）を使いたい場合に、true にセットします。
	 * ここでのラベル（短）は、ﾀｸﾞ修飾なしの、ﾗﾍﾞﾙ（短）です。
	 * 標準はfalse:利用しない=ﾗﾍﾞﾙ（長）です。
	 * true/false以外を指定した場合はfalse扱いとします。
	 *
	 * ﾗﾍﾞﾙﾘｿｰｽの概要説明があれば表示しますが、useSLabel="true" 時は、概要説明を表示しません。
	 *
	 * @og.rev 7.0.7.0 (2019/12/13) 新規追加
	 *
	 * @param prm SLABEL利用 [true:利用する/false:利用しない]
	 */
	public void setUseSLabel( final String prm ) {
		useSLabel = nval( getRequestParameter( prm ),useSLabel );
	}

	/**
	 * リクエスト変数の値より、定型文に必要なパラメータを取得して、パレメータマップに入れます。
	 * パラメータマップは引数としてメールモジュールのマネージャに渡します。
	 * マネージャの中には、定型文を元に、パラメータマップの値とマージしてメールの各項目を合成します。
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) DateSet.getDate( String ) を利用するように修正します。
	 * @og.rev 6.4.3.3 (2016/03/04) ConcurrentHashMap を受け取ることを明確にするため、I/FをConcurrentMapに変更します。
	 *
	 * @return	定型文に必要なパレメータマップ
	 */
	private ConcurrentMap<String, String> makeParamMap() {
		final ConcurrentMap<String, String> paramMap = new ConcurrentHashMap<>();
		if( action.endsWith( ACT_NOCHECK ) || action.equals( ACT_CHECK ) ) {
			final ServletRequest request = this.getRequest();
			final Enumeration<?> enu1 = request.getAttributeNames();
			while( enu1.hasMoreElements() ) {
				final String name = (String) enu1.nextElement();
				final Object tmpObj = request.getAttribute( name );
				if( tmpObj instanceof String ) {
					putNotNull( paramMap,name,(String)tmpObj );
				}
			}
			final Enumeration<?> enu2 = request.getParameterNames();
			while( enu2.hasMoreElements() ) {
				final String name = (String) enu2.nextElement();
				putNotNull( paramMap,name,request.getParameter( name ) );
			}

			putNotNull( paramMap,"FROM"	, from );
			putNotNull( paramMap,"TO"	, to   );
			putNotNull( paramMap,"CC"	, cc   );
			putNotNull( paramMap,"BCC"	, bcc  );
		}

		putNotNull( paramMap,"PTN_ID"			, ptnId );
		putNotNull( paramMap,"SYSTEM_ID"		, HybsSystem.sys( "SYSTEM_ID" ) );
		putNotNull( paramMap,"ADDR_CHECK"		, String.valueOf( addrCheck ) );
		putNotNull( paramMap,"LOGIN_USERID"		, getRequestValue( "USER.ID" ) );
		putNotNull( paramMap,"LOGIN_USERNAME"	, getRequestValue( "USER.JNAME" ) );
		putNotNull( paramMap,"PGID"				, getRequestValue( "GUI.KEY" ) );
		putNotNull( paramMap,"DATE"				, DateSet.getDate( "yyyy/MM/dd" ) );				// 6.4.2.0 (2016/01/29)
		putNotNull( paramMap,"TIME"				, DateSet.getDate( "HH:mm:ss" ) );					// 6.4.2.0 (2016/01/29)

		String[] temp = { "", "", "", "", "" };
		if( filename != null && filename.length > 0 ) {
			final String directory = HybsSystem.url2dir( fileURL );
			final int fileCount = filename.length > MAX_FILE_COUNT ? MAX_FILE_COUNT : filename.length;
			for( int i=0; i<fileCount; i++ ) {
				temp[i] = StringUtil.urlAppend( directory, filename[i] );
			}
		}
		putNotNull( paramMap,"ATTACH1", temp[0] );
		putNotNull( paramMap,"ATTACH2", temp[1] );
		putNotNull( paramMap,"ATTACH3", temp[2] );
		putNotNull( paramMap,"ATTACH4", temp[3] );
		putNotNull( paramMap,"ATTACH5", temp[4] );

		return paramMap;
	}

	/**
	 * ConcurrentMapのnot null制限を回避するため、key,val が、not nullのときだけ、Mapにput します。
	 *
	 * @og.rev 6.4.3.3 (2016/03/04) ConcurrentHashMap を受け取ることを明確にするため、I/FをConcurrentMapに変更します。
	 *
	 * @param	cmap	putする元となるConcurrentMap
	 * @param	key		putするときのキー
	 * @param	val		putするときの値
	 */
	private void putNotNull( final ConcurrentMap<String, String> cmap , final String key , final String val ) {
		if( key != null && val != null ) { cmap.put( key,val ); }
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title(this.getClass().getName() )
		.println( "VERSION"		,VERSION 	)
		.println( "ptnId"		,ptnId		)
		.println( "action"		,action		)
		.println( "tableId"		,tableId	)
		.println( "addrCheck"	,addrCheck	)
		.println( "useStop"		,useStop	)
		.println( "from" 		,from 		)
		.println( "to"			,to			)
		.println( "cc"			,cc			)
		.println( "bcc"			,bcc		)
		.println( "filename" 	,filename 	)
		.println( "fileURL" 	,fileURL 	)
		.println( "Other...", getAttributes().getAttribute() )
		.fixForm().toString();
	}
}
