/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.fukurou.util.StringUtil ;
import org.opengion.fukurou.util.ToString;

import static org.opengion.fukurou.util.StringUtil.nval ;

/**
 * ExcelImageFileParamTag は、ExcelFileTagの画像挿入のパラメーターを渡す為のタグクラスです。
 *
 * ExcelFileTag は、EXCELを加工するための 入力、出力を備えたタグで、
 * addImageFile 属性に相当するパラメータを複数指定するためのタグです。
 *
 * @og.formSample
 * ●形式：&lt;og:excelFile action="…" fileURL="…" ･･･ &gt; lt;og:excelImageFileParam … /&gt; &lt;/og:excelFile&gt;
 * ●body：あり
 *
 * ●Tag定義：
 *  &lt;og:excelImageFileParam
 *       fileURL            【TAG】画像ファイルのディレクトリを指定します (初期値:FILE_URL[=filetemp/])
 *       imageFile        ○【TAG】挿入するイメージファイル名
 *       sheetName          【TAG】挿入するシート名(null=先頭シート)
 *       sheetNo            【TAG】挿入するシート番号(初期値:0)
 *       rowNo              【TAG】挿入する行(開始)(初期値:0)
 *       colNo              【TAG】挿入する列(開始)(初期値:0)
 *       colId              【TAG】挿入する列記号(A,B,C…)(開始)(null=先頭列)
 *       rowEndNo           【TAG】挿入する行(終了-含まず)(初期値:0)
 *       colEndNo           【TAG】挿入する列(終了-含まず)(初期値:0)
 *       colEndId           【TAG】挿入する列記号(A,B,C…)(終了-含まず)(null=先頭列)
 *       dx1                【TAG】開始セルのX軸座標のオフセット(ピクセル)(初期値:0)
 *       dy1                【TAG】開始セルのY軸座標のオフセット(ピクセル)(初期値:0)
 *       dx2                【TAG】終了セルのX軸座標のオフセット(ピクセル)(初期値:0)
 *       dy2                【TAG】終了セルのY軸座標のオフセット(ピクセル)(初期値:0)
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null)
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:判定しない)
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:判定しない)
 *       caseIf             【TAG】指定の値が、true/TRUE文字列の場合は、このタグは使用されます(初期値:判定しない)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *  /&gt;
 *
 * ●使用例
 *    ・【entry.jsp】
 *         &lt;og:excelFile file1="input.xls" file2="input.xls" addTitleSheet="Title" &gt;
 *            &lt;og:excelImageFileParam
 *               fileURL    = "{&#064;USER.ID}"
 *               imageFile  = "test.png"
 *               sheetName  = "Sheet1"
 *               rowNo      = "5"
 *               colID      = "G"
 *            /&gt;
 *         &lt;/og:tableUpdate&gt;
 *
 * @og.rev 7.2.9.0 (2020/10/12) 新規作成
 * @og.group その他部品
 *
 * @version  7.2
 * @author   Kazuhiko Hasegawa
 * @since    JDK11.0,
 */
public class ExcelImageFileParamTag extends CommonTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "7.2.9.0 (2020/10/12)" ;
	private static final long serialVersionUID = 729020201012L ;

	private String	fileURL 		= HybsSystem.sys( "FILE_URL" );		// 【TAG】画像ファイルのディレクトリを指定します (初期値:FILE_URL[=filetemp/])
	private String	imageFile		;		// 【TAG】挿入するイメージファイル名
	private String	sheetName		;		// 【TAG】挿入するシート名(null=先頭シート)
	private int		sheetNo			= -1;	// 【TAG】挿入するシート番号(初期値:0)
	private int 	rowNo			;		// 【TAG】挿入する行(開始)(初期値:0)
	private int 	colNo			= -1;	// 【TAG】挿入する列(開始)(初期値:0)
	private int 	rowEndNo		;		// 【TAG】挿入する行(終了-含まず)(初期値:0)
	private int 	colEndNo		= -1;	// 【TAG】挿入する列(終了-含まず)(初期値:0)
	private int 	dx1				;		// 【TAG】開始セルのX軸座標のオフセット(ピクセル)(初期値:0)
	private int 	dy1				;		// 【TAG】開始セルのY軸座標のオフセット(ピクセル)(初期値:0)
	private int 	dx2				;		// 【TAG】終了セルのX軸座標のオフセット(ピクセル)(初期値:0)
	private int 	dy2				;		// 【TAG】終了セルのY軸座標のオフセット(ピクセル)(初期値:0)

	/**
	 * デフォルトコンストラクター
	 *
	 */
	public ExcelImageFileParamTag() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();
		if( !useTag() || imageFile == null ) { return EVAL_PAGE ; }	// 6.3.4.0 (2015/08/01)

		final ExcelFileTag excelTag = (ExcelFileTag)findAncestorWithClass( this,ExcelFileTag.class );
		if( excelTag == null ) {
			final String errMsg = "<b>" + getTagName() + "タグは、ExcelFileTagの内側(要素)に記述してください。</b>";
			throw new HybsSystemException( errMsg );
		}

		final String directory = HybsSystem.url2dir( fileURL );
		final String fname = StringUtil.urlAppend( directory,imageFile );

		excelTag.addImgFile( fname,sheetName,sheetNo,rowNo,colNo,rowEndNo,colEndNo,dx1,dy1,dx2,dy2 ) ;

		return EVAL_PAGE ;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 */
	@Override
	protected void release2() {
		super.release2();			// 3.5.6.0 (2004/06/18) 追加(抜けていた)
		fileURL 	= HybsSystem.sys( "FILE_URL" );		// 【TAG】画像ファイルのディレクトリを指定します (初期値:FILE_URL[=filetemp/])
		imageFile	= null ;		// 【TAG】挿入するイメージファイル名
		sheetName	= null ;		// 【TAG】挿入するシート名(null=先頭シート)
		sheetNo		= -1 ;			// 【TAG】挿入するシート番号(初期値:0)
		rowNo		= 0 ;			// 【TAG】挿入する行(開始)(初期値:0)
		colNo		= -1 ;			// 【TAG】挿入する列(開始)(初期値:0)
		rowEndNo	= 0 ;			// 【TAG】挿入する行(終了-含まず)(初期値:0)
		colEndNo	= -1 ;			// 【TAG】挿入する列(終了-含まず)(初期値:0)
		dx1			= 0 ;			// 【TAG】開始セルのX軸座標のオフセット(ピクセル)(初期値:0)
		dy1			= 0 ;			// 【TAG】開始セルのY軸座標のオフセット(ピクセル)(初期値:0)
		dx2			= 0 ;			// 【TAG】終了セルのX軸座標のオフセット(ピクセル)(初期値:0)
		dy2			= 0 ;			// 【TAG】終了セルのY軸座標のオフセット(ピクセル)(初期値:0)
	}

	/**
	 * 【TAG】画像ファイルのディレクトリを指定します
	 *		(初期値:FILE_URL[={@og.value SystemData#FILE_URL}])。
	 *
	 * @og.tag
	 * この属性で指定されるディレクトリからイメージファイルを取得します。
	 * 指定方法は、通常の fileURL 属性と同様に、先頭が、'/' (UNIX) または、２文字目が、
	 * ":" (Windows)の場合は、指定のURLそのままのディレクトリに、そうでない場合は、
	 * fileURL = "{&#064;USER.ID}" と指定すると、FILE_URL 属性で指定のフォルダの下に、
	 * さらに、各個人ID別のフォルダを作成して、そこから取得します。
	 * (初期値:システム定数のFILE_URL[={@og.value SystemData#FILE_URL}])。
	 *
	 * @param	url 画像ファイルのディレクトリ名
	 * @see		org.opengion.hayabusa.common.SystemData#FILE_URL
	 */
	public void setFileURL( final String url ) {
		final String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
			fileURL = StringUtil.urlAppend( fileURL,furl );
		}
	}

	/**
	 * 【TAG】挿入するイメージファイル名をセットします(初期値:null)。
	 *
	 * @og.tag ファイル名が存在しない場合は、何も実行しません。
	 *
	 * @param   fname イメージファイル名
	 */
	public void setImageFile( final String fname ) {
		imageFile = nval( getRequestParameter( fname ),imageFile );
	}

	/**
	 * 【TAG】挿入するシート名を設定します(初期値:指定なし)。
	 *
	 * @og.tag
	 * EXCELファイルに画像を挿入する場合のシート名を指定します。
	 * sheetNo と sheetName が同時に指定された場合は、sheetNo が優先されます。
	 * エラーにはならないのでご注意ください。
	 * 初期値は、指定なしです。
	 *
	 * @param   sheet EXCELファイルのシート名
	 * @see		#setSheetNo( String )
	 */
	public void setSheetName( final String sheet ) {
		sheetName = nval( getRequestParameter( sheet ),sheetName );
	}

	/**
	 * 【TAG】挿入するシート番号を指定します(初期値:0)。
	 *
	 * @og.tag
	 * EXCELファイルに画像を挿入する場合のシート番号を指定します
	 * シート番号は、0 から始まる数字で表します。
	 *
	 * sheetNo と sheetName が同時に指定された場合は、sheetNo が優先されます。
	 * エラーにはならないのでご注意ください。
	 * 初期値は、0（第１シート） です。
	 *
	 * @param   sheet EXCELファイルのシート番号（0から始まる）
	 * @see		#setSheetName( String )
	 */
	public void setSheetNo( final String sheet ) {
		sheetNo = nval( getRequestParameter( sheet ),sheetNo );
	}

	/**
	 * 【TAG】挿入する行番号(開始)を指定します(初期値:0)。
	 *
	 * @og.tag
	 * EXCELファイルに画像を挿入する場合の左上端(アンカーの位置)の行番号を指定します
	 * 行番号は、0 から始まる数字で表します。
	 * 初期値は、0（先頭行） です。
	 *
	 * @param   row 挿入する行番号(開始)
	 * @see		#setColNo( String )
	 */
	public void setRowNo( final String row ) {
		rowNo = nval( getRequestParameter( row ),rowNo );
	}

	/**
	 * 【TAG】挿入する列番号(開始)を指定します(初期値:0)。
	 *
	 * @og.tag
	 * EXCELファイルに画像を挿入する場合の左上端(アンカーの位置)の列番号を指定します
	 * 列番号は、0 から始まる数字で表します。
	 * 初期値は、0（左先頭列） です。
	 *
	 * colNo と colID が同時に指定された場合は、colNo が優先されます。
	 * エラーにはならないのでご注意ください。
	 *
	 * @param   col 挿入する列番号(開始)
	 * @see		#setRowNo( String )
	 */
	public void setColNo( final String col ) {
		colNo = nval( getRequestParameter( col ),colNo );
	}

	/**
	 * 【TAG】挿入する列記号(A,B,C…)(開始)を指定します(初期値:0)。
	 *
	 * @og.tag
	 * EXCELファイルに画像を挿入する場合の左上端(アンカーの位置)の列記号(A,B,C…)を指定します
	 * 列記号は、A から始まる英字で表します。
	 * 初期値は、未指定です。
	 *
	 * colNo と colID が同時に指定された場合は、colNo が優先されます。
	 * エラーにはならないのでご注意ください。
	 *
	 * @param   col 挿入する列記号(A,B,C…)(開始)
	 * @see		#setColNo( String )
	 */
	public void setColId( final String col ) {
		final String colId = nval( getRequestParameter( col ),null );
		if( colId != null && colNo < 0 ) {					// colNo の初期値は、-1。 +1 して、26 かける処理をしているので、辻褄合わせ
			for( int i=0; i<colId.length(); i++ ) {
				final char ch = colId.charAt(i);
				if( 'A' <= ch && ch <= 'Z' ) { colNo = (colNo+1)*26 + ch-'A'; }
			}
		}
	}

	/**
	 * 【TAG】挿入する行番号(終了-含まず)を指定します(初期値:0)。
	 *
	 * @og.tag
	 * EXCELファイルに画像を挿入する場合の右下端の行番号を指定します
	 * 行番号は、0 から始まる数字で表します。
	 * 初期値は、0（未指定） です。
	 *
	 * @param   row 挿入する行番号(終了-含まず)
	 * @see		#setColEndNo( String )
	 */
	public void setRowEndNo( final String row ) {
		rowEndNo = nval( getRequestParameter( row ),rowEndNo );
	}

	/**
	 * 【TAG】挿入する列番号(終了-含まず)を指定します(初期値:0)。
	 *
	 * @og.tag
	 * EXCELファイルに画像を挿入する場合の右下端の列番号を指定します
	 * 列番号は、0 から始まる数字で表します。
	 * 初期値は、0（未指定） です。
	 *
	 * colNo と colID が同時に指定された場合は、colNo が優先されます。
	 * エラーにはならないのでご注意ください。
	 *
	 * @param   col 挿入する列番号(終了-含まず)
	 * @see		#setRowEndNo( String )
	 */
	public void setColEndNo( final String col ) {
		colEndNo = nval( getRequestParameter( col ),colEndNo );
	}

	/**
	 * 【TAG】挿入する列記号(A,B,C…)(終了-含まず)を指定します(初期値:0)。
	 *
	 * @og.tag
	 * EXCELファイルに画像を挿入する場合の右下端の列記号(A,B,C…)を指定します
	 * 列記号は、A から始まる英字で表します。
	 * 初期値は、未指定です。
	 *
	 * colEndNo と colID が同時に指定された場合は、colEndNo が優先されます。
	 * エラーにはならないのでご注意ください。
	 *
	 * @param   col 挿入する列記号(A,B,C…)(終了-含まず)
	 * @see		#setColEndNo( String )
	 */
	public void setColEndId( final String col ) {
		final String colId = nval( getRequestParameter( col ),null );
		if( colId != null && colEndNo < 0 ) {					// colEndNo の初期値は、-1。 +1 して、26 かける処理をしているので、辻褄合わせ
			for( int i=0; i<colId.length(); i++ ) {
				final char ch = colId.charAt(i);
				if( 'A' <= ch && ch <= 'Z' ) { colEndNo = (colEndNo+1)*26 + ch-'A'; }
			}
		}
	}

	/**
	 * 【TAG】開始セルのX軸座標のオフセット(ピクセル)を指定します(初期値:0)。
	 *
	 * @og.tag
	 * 開始セルのX軸座標のオフセット(ピクセル)を指定します(初期値:0)。
	 * 初期値は、0（未指定） です。
	 *
	 * @param   dx 開始セルのX軸座標のオフセット(ピクセル)
	 */
	public void setDx1( final String dx ) {
		dx1 = nval( getRequestParameter( dx ),dx1 );
	}

	/**
	 * 【TAG】開始セルのY軸座標のオフセット(ピクセル)を指定します(初期値:0)。
	 *
	 * @og.tag
	 * 開始セルのY軸座標のオフセット(ピクセル)を指定します(初期値:0)。
	 * 初期値は、0（未指定） です。
	 *
	 * @param   dy 開始セルのY軸座標のオフセット(ピクセル)
	 */
	public void setDy1( final String dy ) {
		dy1 = nval( getRequestParameter( dy ),dy1 );
	}

	/**
	 * 【TAG】終了セルのX軸座標のオフセット(ピクセル)を指定します(初期値:0)。
	 *
	 * @og.tag
	 * 終了セルのX軸座標のオフセット(ピクセル)を指定します(初期値:0)。
	 * 初期値は、0（未指定） です。
	 *
	 * @param   dx 終了セルのX軸座標のオフセット(ピクセル)
	 */
	public void setDx2( final String dx ) {
		dx2 = nval( getRequestParameter( dx ),dx2 );
	}

	/**
	 * 【TAG】終了セルのY軸座標のオフセット(ピクセル)を指定します(初期値:0)。
	 *
	 * @og.tag
	 * 終了セルのY軸座標のオフセット(ピクセル)を指定します(初期値:0)。
	 * 初期値は、0（未指定） です。
	 *
	 * @param   dy 終了セルのY軸座標のオフセット(ピクセル)
	 */
	public void setdy2( final String dy ) {
		dy2 = nval( getRequestParameter( dy ),dy2 );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION		)
				.println( "fileURL"			,fileURL		)
				.println( "imageFile"		,imageFile		)
				.println( "sheetName"		,sheetName		)
				.println( "sheetNo" 		,sheetNo		)
				.println( "rowNo"			,rowNo			)
				.println( "colNo"			,colNo			)
				.println( "rowEndNo"		,rowEndNo		)
				.println( "colEndNo"		,colEndNo		)
				.println( "dx1" 			,dx1			)
				.println( "dy1" 			,dy1			)
				.println( "dx2" 			,dx2			)
				.println( "dy2" 			,dy2			)
				.fixForm().toString() ;
	}
}
