/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.common;

import org.opengion.fukurou.system.OgRuntimeException ;		// 6.4.2.0 (2016/01/29)

/**
 * 有効件数が オーバーフローした場合に 発生させる RuntimeException のサブクラスです。
 *
 * RuntimeException を継承しているため、try{} catch() {} は不要です。
 * SAXパーサーなどのイベントモデルで、途中打ち切りを行うには、SAXException を発生させて、
 * それを catch する構造が一般的(http://www.ibm.com/developerworks/jp/xml/library/x-tipsaxstop/)
 * ですが、本来のエラーと区別したいのと、SAX以外の汎用的に使いたいため、HybsOverflowException を
 * 用意しました。
 * 
 * 内部処理は、コメントを発生させるだけです。
 * コンストラクタとして、オーバーした件数を取るのと、メッセージを渡せるのと、
 * ２種類しか用意していません。
 *
 * @og.group エラー処理
 * @og.rev 6.2.0.0 (2015/02/27) 新規追加
 * @og.rev 6.2.2.0 (2015/03/27) fukurou.model → hayabusa.common へ、移植
 * @og.rev 6.4.2.0 (2016/01/29) 継承元を、RuntimeException ではなく、OgRuntimeException に変更。
 *
 * @version  6.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK4.0,
 */
public class HybsOverflowException extends OgRuntimeException {
	private static final long serialVersionUID = 642020160129L ;

	/**
	 * 有効件数が オーバーフローした場合に 発生させる RuntimeException のサブクラス
	 *
	 * 引数に、有効件数(=制限件数)を取り、独自にエラーメッセージを作成します。
	 * 基本的には、このメッセージは、引数のまま、文字列にして投げるだけです。
	 *
	 * @og.rev 6.2.0.0 (2015/02/27) オーバーフロー時に起動する RuntimeException
	 */
	public HybsOverflowException() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * 有効件数が オーバーフローした場合に 発生させる RuntimeException のサブクラス
	 *
	 * 引数に、有効件数(=制限件数)を取り、独自にエラーメッセージを作成します。
	 * 『指定の制限件数( XX 件)を超えました。』
	 *
	 * @og.rev 6.2.0.0 (2015/02/27) オーバーフロー時に起動する RuntimeException
	 * @og.rev 6.4.1.2 (2016/01/22) メッセージを追加します。
	 *
	 * @param	rowCnt	件数
	 */
	public HybsOverflowException( final int rowCnt ) {
		super( "指定の制限件数( " + rowCnt + ")を超えました。" );
	}

	/**
	 * オーバーフロー時に起動する RuntimeException のサブクラス
	 *
	 * 引数に、独自のエラーメッセージを渡します。
	 * 基本的には、このメッセージは、引数のまま、投げるだけです。
	 *
	 * @og.rev 6.2.0.0 (2015/02/27) オーバーフロー時に起動する RuntimeException
	 *
	 * @param	str	詳細メッセージ
	 */
	public HybsOverflowException( final String str ) {
		super( str );
	}
}
