/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.table;

/**
 * TableFilter_INDEX_DERBY は、TableUpda インターフェースを継承した、DBTableModel 処理用の
 * 実装クラスです。とくに、MySQL用のインデックス作成スクリプトを作成します。
 *
 * ここでは、テーブル一覧の検索結果より、GF07 のインデックスカラム定義テーブルから
 * 必要な情報を取得し、テーブル作成スクリプトを作成します。
 * 出力ファイルは、テーブル名＋"I.sql" という命名規則で作成します。
 * 検索では、(SYSTEM_ID,TBLSYU,TABLE_NAME,TABLE_LABEL,INDEX_NAME,NAME_JA,INDTYPE,TABLESPACE_NAME,INITIAL_EXTENT)
 * の項目を取得する必要があります。
 *
 * @og.rev 6.4.5.0 (2016/04/08) 新規作成
 * @og.rev 6.8.1.0 (2017/07/14) JAVADB → DERBY に変更
 *
 * @version  0.9.0  2000/10/17
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.1,
 */
public class TableFilter_INDEX_DERBY extends TableFilter_INDEX {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "7.3.0.0 (2021/01/06)" ;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 * @og.rev 6.8.1.0 (2017/07/14) JAVADB → DERBY に変更
	 */
	public TableFilter_INDEX_DERBY() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * インデックス作成の処理を実行します。
	 *
	 * @og.rev 6.4.0.5 (2016/01/09) AUTO_INCREMENT 使用時は、PRIMARY KEY 制約を、TABLE 側に入れる必要がある。
	 *
	 * @param	clmNo	カラム番号配列
	 * @param	data	１行分のデータ配列
	 * @param   clms	カラム名(CSV形式)
	 *
	 * @return	作成された１行分の文字列
	 * @og.rtnNotNull
	 */
	@Override
	protected String makeLineList( final int[] clmNo,final String[] data,final String clms ) {
		final String idxtype   = data[clmNo[INDTYPE]];

		// 0:プライマリキー で、UNIQ か、UNIQSEQ の場合は、インデックスタイプを、9:未使用 にします。
		if( "0".equals( idxtype ) && ( "UNIQ".equalsIgnoreCase( clms ) || "UNIQSEQ".equalsIgnoreCase( clms ) ) ) {
			data[clmNo[INDTYPE]] = "9" ;	// 未使用
		}

		return super.makeLineList( clmNo,data,clms );
	}

	/**
	 * 定義の最後の部分の処理を実行します。
	 *
	 * @og.rev 6.0.2.3 (2014/10/10) EXEC_END_TAG を追加。キャッシュします。
	 * @og.rev 6.4.0.5 (2016/01/09) INDTYPE が 9:未使用 の場合の処理を追加。
	 *
	 * @param	clmNo	カラム番号配列
	 * @param	data	１行分のデータ配列
	 *
	 * @return	定義の最後の部分
	 */
	@Override
	protected String makeEndLine( final int[] clmNo,final String[] data ) {
		return "9".equals( data[clmNo[INDTYPE]] ) ? "" : execEndTag ;
	}

	/**
	 * インデックス削除の構文を、作成します。
	 *
	 * @og.rev 5.6.9.2 (2013/10/18) 新規作成
	 * @og.rev 6.0.2.3 (2014/10/10) EXEC_END_TAG を追加。キャッシュします。
	 * @og.rev 6.4.4.1 (2016/03/18) StringBuilderの代わりに、OgBuilderを使用する。
	 * @og.rev 6.7.2.0 (2017/01/16) DROP INDEX ｲﾝﾃﾞｯｸｽ名 に変更。プライマリキーは、削除しない。
	 *
	 * @param	clmNo	カラム番号配列
	 * @param	data	１行分のデータ配列
	 *
	 * @return	作成された１行分の文字列
	 * @og.rtnNotNull
	 */
	@Override
	protected String makeDropLine( final int[] clmNo,final String[] data ) {
		// 6.7.2.0 (2017/01/16) プライマリキーは、削除しない。
		final String idxtype   = data[clmNo[INDTYPE]];
		if( "1".equals( idxtype ) || "2".equals( idxtype ) ) {					// 1:ユニークキー 2:通常のみ。9:未使用 というのがあるので、注意
			return super.makeDropLine( clmNo,data );
		}
		else {
			return "";
		}
	}

	/**
	 * インデックスを作成するための文字列を返します。
	 * 通常、カラム名をそのまま返します。
	 *
	 * 7.3.0.0 (2021/01/06)
	 *   seqNoがマイナスの場合は、カラム名に DESC を付けて逆ｲﾝﾃﾞｯｸｽにします。
	 *
	 * ※ ORACLEの場合、逆ｲﾝﾃﾞｯｸｽはバグがあったり、正ｲﾝﾃﾞｯｸｽで自動判断するそうなので、
	 *    基本的には使わない方が良いそうです。
	 *
	 * ※ FIREBERD には、カラムごとの逆ｲﾝﾃﾞｯｸｽは定義できません。
	 *    エラーチェックは入れていませんので、ご注意ください。
	 *
	 * ※ MySQL は、8.0で採用されています。それ以前のﾊﾞｰｼﾞｮﾝでは動作しません。
	 * ※ MySQL の 500バイト以上のｶﾗﾑのｲﾝﾃﾞｯｸｽ制限が解除されているかどうかは、未調査です。
	 *
	 * @og.rev 7.3.0.0 (2021/01/06) SEQNO がマイナスの場合、カラムに DESC を付ける(逆順カラム)
	 *
	 * @param	clm		カラム名
	 * @param	useLen	カラムのバイト数
	 * @param	seqNo	カラム順(ﾏｲﾅｽの場合は、逆順)…DERBY以外には使用しないことにします。
	 *
	 * @return	インデックスカラムの文字列
	 * @see TableFilter_INDEX_MYSQL
	 */
//	protected String makeIndexClmStr( final String clm, final String useLen ) {
	protected String makeIndexClmStr( final String clm, final String useLen, final String seqNo ) {
//		return clm;
		return seqNo.charAt(0) == '-' ? clm + " DESC" : clm ;
	}
}
