/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.db.AbstractRenderer;
import org.opengion.hayabusa.db.CellRenderer;
import org.opengion.hayabusa.db.DBColumn;
import static org.opengion.fukurou.util.StringUtil.isNull;

/**
 * SUBSTR レンデラーは、カラムのデータを特定の文字列の範囲のみに分割するクラスです。
 *
 * パラメータに、開始文字列$$終了文字列　を指定すると、最初の開始文字列から
 * 最後に現れた終了文字列の間のみを抜き出します。
 *
 * 開始文字列が未設定や見つからない場合は、文字列の最初から、終了文字列が未設定や
 * 見つからない場合は、最後まで抜き出し対象とします。
 *
 * $$も存在しない場合は、開始文字列のみ指定されたとして処理します。
 * パラメータに何も指定されなかった場合は、何も加工しません。
 *
 * @og.group データ変換
 *
 * @og.rev 7.2.6.0 (2020/06/30) 行単位に抜き出す機能追加
 *
 * @version  7.2
 * @author   Kazuhiko Hasegawa
 * @since    JDK11.0,
 */
public class Renderer_SUBSTR extends AbstractRenderer {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "7.2.6.0 (2020/06/30)" ;

	private static final String SEPARATOR = "$$" ;

	private static final CellRenderer DB_CELL = new Renderer_SUBSTR( "","" ) ;

	// 7.2.6.0 (2020/06/30) 行単位に抜き出す機能追加
	private final String  stStr ;		// 開始文字列
	private final String  enStr ;		// 終了文字列
	private final boolean isUse ;		// 処理するかどうか

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public Renderer_SUBSTR() {
		this( "","" );
	}

	/**
	 * 必要なパラメータを指定したprivateコンストラクター。
	 *
	 * パラメータで、先頭からスキップする文字列と以降を無視する文字列を指定する。
	 *
	 * @og.rev 7.2.6.0 (2020/06/30) 行単位に抜き出す機能追加
	 *
	 * @param	stStr	開始文字列
	 * @param	enStr	終了文字列
	 */
	private Renderer_SUBSTR( final String stStr,final String enStr ) {
		super();
		this.stStr	= stStr;
		this.enStr	= enStr;
		this.isUse	= !isNull( stStr ) || !isNull( enStr );
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @og.rev 7.2.6.0 (2020/06/30) 行単位に抜き出す機能追加
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellRendererオブジェクト
	 * @og.rtnNotNull
	 */
	public CellRenderer newInstance( final DBColumn clm ) {
		final String param = clm.getRendererParam();
		if( isNull( param ) ) {
			return DB_CELL;
		}
		else {
			String  stStr	= "";
			String  enStr	= "";

			final int ad = param.indexOf( SEPARATOR );
			if( ad < 0 ) {
				stStr = param;
			}
			else {
				stStr = param.substring( 0,ad );
				enStr = param.substring( ad+SEPARATOR.length() );
			}
			return new Renderer_SUBSTR( stStr,enStr );
		}
	}

	/**
	 * データの表示用文字列を返します。
	 *
	 * @og.rev 7.2.6.0 (2020/06/30) 行単位に抜き出す機能追加
	 *
	 * @param   value 入力値
	 *
	 * @return  データの表示用文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final String value ) {
		final String rtn = ( value == null ) ? "" : value ;

		if( isUse ) {
			int st = rtn.indexOf(     stStr );		// 引数が空の文字列の場合も　0
			int ed = rtn.lastIndexOf( enStr );

			if( st < 0 ) { st = 0; }
			if( ed < 0 ) { ed = rtn.length(); }

			return rtn.substring( st,ed );
		}
		return rtn ;
	}

	/**
	 * データ出力用の文字列を作成します。
	 * ファイル等に出力する形式を想定しますので、HTMLタグを含まない
	 * データを返します。
	 * 基本は、#getValue( String ) をそのまま返します。
	 *
	 * @param   value 入力値
	 *
	 * @return  データ出力用の文字列
	 * @og.rtnNotNull
	 * @see		#getValue( String )
	 */
	@Override
	public String getWriteValue( final String value ) {
		return getValue( value );
	}
}
