/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.db.AbstractRenderer;
import org.opengion.hayabusa.db.CellRenderer;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.fukurou.util.StringUtil;

/**
 * QRCODE レンデラーは、カラムのデータからQRコードを表示する場合に使用するクラスです。
 * div 要素に、QRコードに変換する文字列を設定して、それを qrcode.js を使用して QRコードに
 * 変換します。一旦、画像ファイルを生成してから表示する方式とは異なり、canvas に直接描画しています。
 * その後、img に変換することで、Webの印刷が出来るようにしています。
 * パラメータは、img ﾀｸﾞに適用されます。
 *
 * 基本的な構造は、ﾃﾞｰﾀ設定用の div 要素と、QR表示用のcanvas、起動用のscript です。
 *
 *    &lt;div id='div≪ｶﾗﾑ名≫' class='QRDATA' &gt;≪値≫&lt;/div&gt;
 *    &lt;canvas id='can≪ｶﾗﾑ名≫' class='QRCANVAS' style='display:none;'&gt; &lt;/canvas&gt;
 *    &lt;img src='' id='img≪ｶﾗﾑ名≫' class='QRIMG' ≪ﾊﾟﾗﾒｰﾀ≫ '&gt;&lt;/canvas&gt;
 *    &lt;script&gt;qrView('≪ｶﾗﾑ名≫');&lt;/script&gt;
 *
 * script に CDNサービス を使うと、無線環境(iPad等)ではものすごく遅くなったため、ローカルに配置することにします。
 * &lt;script src="https://cdn.jsdelivr.net/npm/qrcode@latest/build/qrcode.min.js"&gt;&lt;!-- --&gt;&lt;/script&gt;
 *
 * これらは、使用する画面に、組み込んでください。
 * &lt;script src="{&#064;SYS.JSP}/option/qrcode.min.js"&gt;&lt;!-- --&gt;&lt;/script&gt;
 * &lt;script src="{&#064;SYS.JSP}/option/videocamera.js"&gt;&lt;!-- --&gt;&lt;/script&gt;
 *
 * @og.rev 7.4.2.1 (2021/05/21) 新規作成
 * @og.group データ表示
 *
 * @version  7.4
 * @author   Kazuhiko Hasegawa
 * @since    JDK11.0,
 */
public class Renderer_QRCODE extends AbstractRenderer {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "7.4.2.1 (2021/05/21)" ;

	private static final String BASE_HTML = "<div id='div≪ｶﾗﾑ名≫' class='QRDATA'>≪値≫</div>"
									+ CR +	"<canvas id='can≪ｶﾗﾑ名≫' class='QRCANVAS' style='display:none;'> </canvas>"
									+ CR +	"<img src='' id='img≪ｶﾗﾑ名≫' class='QRIMG' ≪ﾊﾟﾗﾒｰﾀ≫ >"
									+ CR +	"<script>qrView('≪ｶﾗﾑ名≫');</script>" ;

	private  String		name;
	private  String		param;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 7.4.2.1 (2021/05/21) 新規作成
	 */
	public Renderer_QRCODE() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * DBColumnオブジェクトを指定したprivateコンストラクター。
	 *
	 * @og.rev 7.4.2.1 (2021/05/21) 新規作成
	 *
	 * @param	clm	DBColumnオブジェクト
	 */
	private Renderer_QRCODE( final DBColumn clm ) {
		super();

		name  = clm.getName();
		param = StringUtil.nvalAdd( clm.getRendererParam() ,
									clm.getRendererAttributes().get( "optionAttributes" ) );
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @og.rev 7.4.2.1 (2021/05/21) 新規作成
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellEditorオブジェクト
	 * @og.rtnNotNull
	 */
	public CellRenderer newInstance( final DBColumn clm ) {
		return new Renderer_QRCODE( clm );
	}

	/**
	 * データの表示用文字列を返します。
	 *
	 * @og.rev 7.4.2.1 (2021/05/21) 新規作成
	 *
	 * @param   value 入力値
	 *
	 * @return  データの表示用文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final String value ) {
		return makeCanvas( name, value );
	}

	/**
	 * データの表示用文字列を返します。
	 *
	 * @og.rev 7.4.2.1 (2021/05/21) 新規作成
	 *
	 * @param   row   行番号
	 * @param   value 入力値
	 *
	 * @return  データ表示用の文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final int row,final String value ) {
		final String newName = name + HybsSystem.JOINT_STRING + row;
		return makeCanvas( newName, value );
	}

	/**
	 * データ出力用の文字列を作成します。
	 * ファイル等に出力する形式を想定しますので、HTMLタグを含まない
	 * データを返します。
	 * 基本は、#getValue( String ) をそのまま返します。
	 *
	 * @og.rev 7.4.2.1 (2021/05/21) 新規作成
	 *
	 * @param   value 入力値
	 *
	 * @return  データ出力用の文字列
	 * @og.rtnNotNull
	 * @see		#getValue( String )
	 */
	@Override
	public String getWriteValue( final String value ) {
		return value == null ? "" : value;
	}

	/**
	 * データの表示用文字列を返します。
	 *
	 * @og.rev 7.4.2.1 (2021/05/21) 新規作成
	 *
	 * @param	name	カラム名
	 * @param   value	入力値 表示するファイルのアドレス
	 *
	 * @return  データ表示用の文字列
	 * @og.rtnNotNull
	 */
	private String makeCanvas( final String name,final String value ) {
		return BASE_HTML.replaceAll( "≪ｶﾗﾑ名≫" , name ).replaceAll( "≪ﾊﾟﾗﾒｰﾀ≫" , param ).replaceAll( "≪値≫" , value );
	}
}
