/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.db.AbstractRenderer;
import org.opengion.hayabusa.db.CellRenderer;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.Selection;
import org.opengion.hayabusa.db.SelectionFactory;		// 5.7.3.0 (2014/02/07)

/**
 * CHBOX2 レンデラーは、カラムのデータをチェックボックスで表示する場合に使用するクラスです。
 *
 * カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * 一覧表示する場合は、通常は、□か■に、プラスしてラベルが表示されます。
 * 表示パラメータの、"useLabel" は、常に、指定されているのと同じ状態です。
 *
 * @og.rev 6.4.4.0 (2016/03/11) 新規作成
 * @og.group データ表示
 *
 * @version  6.2
 * @author	 Kazuhiko Hasegawa
 * @since    JDK8.0,
 */
public class Renderer_CHBOX2 extends AbstractRenderer {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.9.1 (2016/08/05)" ;

	private final Selection selection	;
	private final boolean useKeyLabel ;		// 6.2.0.0 (2015/02/27) キー:ラベル形式

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 * @og.rev 6.4.4.0 (2016/03/11) 新規追加
	 * @og.rev 6.4.9.1 (2016/08/05) useLabel パラメータは常に有効になっている。
	 */
	public Renderer_CHBOX2() {
		super();					// 6.4.1.1 (2016/01/16) PMD refactoring. It is a good practice to call super() in a constructor
		selection	= null;
		useKeyLabel = true;
	}

	/**
	 * DBColumnオブジェクトを指定したprivateコンストラクター。
	 *
	 * @og.rev 6.4.4.0 (2016/03/11) 新規追加
	 * @og.rev 6.4.6.0 (2016/05/27) getEditorParam → getRendererParam に修正
	 * @og.rev 6.4.9.1 (2016/08/05) useLabel パラメータは常に有効になっている。
	 *
	 * @param	clm	DBColumnオブジェクト
	 */
	private Renderer_CHBOX2( final DBColumn clm ) {
		super();		// 6.4.1.1 (2016/01/16) PMD refactoring. It is a good practice to call super() in a constructor
		useKeyLabel = "true".equalsIgnoreCase( clm.getAddKeyLabel() ) ;		// 値:ラベル形式

		// CHBOX2 は、コードリソース(selection)が存在しない場合もありうる。
		final String addKeyLabel = clm.getAddKeyLabel();					// 6.2.0.0 (2015/02/27) キー:ラベル形式
		selection = SelectionFactory.newSelection( "CHBOX" , clm.getCodeData() , addKeyLabel );
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @og.rev 6.4.4.0 (2016/03/11) 新規追加
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellRendererオブジェクト
	 * @og.rtnNotNull
	 */
	public CellRenderer newInstance( final DBColumn clm ) {
		return new Renderer_CHBOX2( clm );
	}

	/**
	 * データの表示用文字列を返します。
	 *
	 * @og.rev 6.4.4.0 (2016/03/11) 新規追加
	 *
	 * @param   value 入力値
	 *
	 * @return  データの表示用文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final String value ) {
		// selection が null の場合もありうる。
		final String chbox = selection == null
								? "　■" + value
								: selection.getValueLabel( value,true ) ;

		return "<pre class=\"CHBOX\">" + chbox + "</pre>" ;
	}

	/**
	 * データの一覧表示用文字列を返します。
	 * 一覧表示のため、useLabel が有効です。
	 *
	 * @og.rev 6.4.4.0 (2016/03/11) 新規追加
	 *
	 * @param	row   行番号
	 * @param	value 入力値
	 *
	 * @return	データ表示/編集用の文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final int row,final String value ) {
		// selection が null の場合もありうる。
		final String chbox = selection == null
								? "　■" + value
								: selection.getValueLabel( value,true ) ;

		return "<pre class=\"CHBOX\">" + chbox + "</pre>" ;
	}

	/**
	 * データ出力用の文字列を作成します。
	 * ファイル等に出力する形式を想定しますので、HTMLタグを含まない
	 * データを返します。
	 * 基本は、#getValue( String ) をそのまま返します。
	 *
	 * @og.rev 6.4.4.0 (2016/03/11) 新規追加
	 *
	 * @param   value 入力値
	 *
	 * @return  データ出力用の文字列
	 * @see		#getValue( String )
	 */
	@Override
	public String getWriteValue( final String value ) {
		// selection が null の場合もありうる。
		return selection == null
					? value
					: useKeyLabel
						? value + ':' + selection.getValueLabel( value,true )
						: selection.getValueLabel( value,false );
	}
}
