/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.db.AbstractEditor;
import org.opengion.hayabusa.db.CellEditor;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.fukurou.util.XHTMLTag;

import java.text.DecimalFormat;

/**
 * DECIMAL エディターは、カラムのデータをDecimal(１０進数、小数)表示する場合に
 * 使用するクラスです。
 *
 * 編集パラメータに与えられた文字列は、java.text.DecimalFormat を使用して
 * フォーマットされます。
 * フォーマット変換前に、カンマなどの数値変換時にエラーになる情報を削除しておきます。
 * 標準のフォーマットは"#,##0.#"です。
 * default値が設定されていない場合の初期値は0.0にフォーマット処理をしたものです。
 *
 * フォーマットルールは、{@link java.text.DecimalFormat} を参照願います。
 *
 * 7.4.2.3 (2021/06/09)
 *  ﾌｫｰﾏｯﾄは、表示桁数 に応じて決定します。
 *  ただし、ﾊﾟﾗﾒｰﾀ指定 が優先されます。
 *
 *  カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.group データ編集
 * @og.rev 3.8.8.5 (2007/03/09) 新規作成
 * @og.rev 5.4.3.6 (2012/01/19) コメント修正
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Editor_DECIMAL extends AbstractEditor {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "7.4.2.3 (2021/06/09)" ;

	// 7.4.2.3 (2021/06/09) ﾌｫｰﾏｯﾄは、表示桁数 に応じて決定します(小数6桁以上の精度は受け付けられない)
	private static final String[] FRMT = new String[] { "#,##0" , "#,##0.0" , "#,##0.00" , "#,##0.000" , "#,##0.0000" , "#,##0.00000" , "#,##0.000000" } ;

	private final int       minFraction;
	private final DecimalFormat format ;
	private final String	defValue ;

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 */
	public Editor_DECIMAL() {
		super();		// 6.4.1.1 (2016/01/16) PMD refactoring. It is a good practice to call super() in a constructor
		// 4.3.4.4 (2009/01/01)
		minFraction = 0;
		format = null;
		defValue   = null;
	}

	/**
	 * DBColumnオブジェクトを指定したprivateコンストラクター。
	 *
	 * @og.rev 5.3.1.0 (2009/12/01) 初期値がnullの場合は、defValueはnullとする。
	 * @og.rev 5.4.0.0 (2011/10/01) RendererParamが読み込まれているバグを修正
	 * @og.rev 7.4.2.3 (2021/06/09) ﾌｫｰﾏｯﾄは、表示桁数 に応じて決定します(ﾊﾟﾗﾒｰﾀ指定 が優先)。
	 *
	 * @param	clm	DBColumnオブジェクト
	 */
	private Editor_DECIMAL( final DBColumn clm ) {
		super( clm );
		minFraction = clm.getSizeY();

		final String fm = makeFormat( clm );
//		String fm = clm.getEditorParam();
//		if( fm == null || fm.isEmpty() || fm.equals( "_" ) ) {
//			fm = "#,##0.#";
//		}
		format = new DecimalFormat( fm );

		// 5.3.1.0 (2009/12/01)
		final String defv = clm.getDefault();
		if( defv == null || defv.isEmpty() ) {
			defValue = "";
		}
		else {
			double dd = 0.0;
			// 6.0.2.5 (2014/10/31) null でないことがわかっている値の冗長な null チェックがあります。
			if( !"_".equals( defv ) ) {
				dd  = StringUtil.parseDouble( defv );
			}
			defValue = format.format( dd );
		}

		// -XXX,XXX,XXX.YYY 形式では、カンマ、小数点、マイナスなどの項目が入ります。
		final int maxlength = (int)(clm.getSizeX() * 4.0/3.0) + minFraction + 2 ;	// +2 は、マイナスと小数点
		attributes.set( "maxlength"   ,String.valueOf( maxlength ) );
		tagBuffer.add( XHTMLTag.inputAttri( attributes ) );
	}

	/**
	 * フォーマット文字列を返します。
	 *
	 * ﾌｫｰﾏｯﾄは、表示桁数 に応じて決定します。
	 * ただし、ﾊﾟﾗﾒｰﾀ指定 が優先されます。
	 *
	 * @og.rev 7.4.2.3 (2021/06/09) ﾌｫｰﾏｯﾄは、表示桁数 に応じて決定します(ﾊﾟﾗﾒｰﾀ指定 が優先)。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	フォーマット文字列
	 */
	private String makeFormat( final DBColumn clm ) {
		String fm = clm.getEditorParam();							// ﾊﾟﾗﾒｰﾀ指定 が優先

		if( fm == null || fm.isEmpty() || fm.equals( "_" ) ) {
			fm = "#,##0.##";										// 初期値：桁を増やしておきます。

			// 7.4.2.3 (2021/06/09) ﾌｫｰﾏｯﾄは、表示桁数 に応じて決定します。
			final String viewLen = clm.getViewLength();
			if( viewLen != null ) {
				final int ad = viewLen.indexOf( '.' );
				if( ad >= 0 ) {
					final String st = viewLen.substring( ad+1 );		// 小数点以下の桁数取得
					try {
						final int su = Integer.parseInt( st );
						fm = su < FRMT.length ? FRMT[su] : FRMT[FRMT.length-1] ;
					}
					catch( final Throwable th ) {
						final String errMsg = "表示桁数の小数部の変換に失敗しました" + CR
											+ "viewLen=" + viewLen
											+ CR
											+ th.getMessage() ;
						System.err.println( errMsg );
					}
				}
			}
		}

		return fm;
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellEditorオブジェクト
	 * @og.rtnNotNull
	 */
	public CellEditor newInstance( final DBColumn clm ) {
		return new Editor_DECIMAL( clm );
	}

	/**
	 * データの編集用文字列を返します。
	 *
	 * @param   value 入力値
	 *
	 * @return  データの編集用文字列
	 */
	@Override
	public String getValue( final String value ) {
		return super.getValue( formatValue( value ) );
	}

	/**
	 * name属性を変えた、データ表示/編集用のHTML文字列を作成します。
	 * テーブル上の name に 行番号を付加して、名前_行番号 で登録するキーを作成し,
	 * リクエスト情報を１つ毎のフィールドで処理できます。
	 *
	 * @param	row   行番号
	 * @param	value 入力値
	 *
	 * @return	データ表示/編集用の文字列
	 */
	@Override
	public String getValue( final int row,final String value ) {
		return super.getValue( row,formatValue( value ) );
	}

	/**
	 * 内部的なフォーマット変換後の文字列を返します。
	 *
	 * @og.rev 5.3.1.0 (2009/12/01) 値が0の場合は、初期値を適用する。
	 *
	 * @param	value 入力値
	 *
	 * @return	データの表示用文字列
	 */
	private String formatValue( final String value ) {
		if( value == null || value.trim().isEmpty() || "0".equals( value ) ) {
			return defValue;
		}

		final double dd = StringUtil.parseDouble( value );

		final String rtn ;
		synchronized( format ) {
			rtn = format.format( dd );
		}

		return rtn;
	}
}
