/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.servlet;

// import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;

// import jakarta.mail.internet.MimeUtility;
import jakarta.servlet.ServletException;
import jakarta.servlet.ServletOutputStream;
import jakarta.servlet.http.HttpServlet;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
	// import jakarta.servlet.http.HttpSession;					// 2017/10/06 ADD bluemixのｽﾄﾚｰｼﾞに保存する場合

import jakarta.servlet.annotation.WebServlet;					// 7.3.0.0 (2021/01/06)

import org.opengion.fukurou.security.HybsCryptography;
import org.opengion.fukurou.system.Closer;
import org.opengion.fukurou.util.KanaFilter;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import static org.opengion.fukurou.system.HybsConst.FS;			// 6.1.0.0 (2014/12/26) refactoring
	// import org.opengion.hayabusa.io.StorageAPI;				// 5.9.25.0 (2017/10/06) ｸﾗｳﾄﾞｽﾄﾚｰｼﾞ対応
	// import org.opengion.hayabusa.io.StorageAPIFactory;		// 5.9.25.0 (2017/10/06) ｸﾗｳﾄﾞｽﾄﾚｰｼﾞ対応

import org.opengion.fukurou.model.FileOperation;				// 8.0.0.0 (2021/09/30)
import org.opengion.hayabusa.io.HybsFileOperationFactory;		// 8.0.0.0 (2021/09/30)

/**
 * ｻｰﾊﾞｰ管理ﾌｧｲﾙをﾀﾞｳﾝﾛｰﾄﾞする場合に使用する､ｻｰﾌﾞﾚｯﾄです｡
 *
 * 引数(URL)に指定のﾌｧｲﾙをｻｰﾊﾞｰからｸﾗｲｱﾝﾄにﾀﾞｳﾝﾛｰﾄﾞさせます｡
 * file には､ｻｰﾊﾞｰﾌｧｲﾙの物理ｱﾄﾞﾚｽを指定します｡相対ﾊﾟｽを使用する場合は､
 * ｺﾝﾃｷｽﾄﾙｰﾄ(通常､Tomcatでは､G:\webapps\dbdef2\ など)からのﾊﾟｽと判断します｡
 * name には､ｸﾗｲｱﾝﾄに送信するﾌｧｲﾙ名を指定します｡ﾌｧｲﾙ名を指定しない場合は､
 * ｻｰﾊﾞｰの物理ﾌｧｲﾙのﾌｧｲﾙ名が代わりに使用されます｡
 * 日本語ﾌｧｲﾙ名は､すべて UTF-8化して処理します｡指定するﾌｧｲﾙに日本語が含まれる
 * 場合は､URLｴﾝｺｰﾄﾞを行ってください｡変換前ｴﾝｺｰﾄﾞはﾘｸｴｽﾄ変数requestEncodingで指定可能で､標準はISO-8859-1です｡
 * 基本的にはContent-disposition属性として"attachment"が指定されます｡
 * 但し､引数に inline=true を指定することで､Content-disposition属性に"inline"が指定されます｡
 * また､ｼｽﾃﾑﾘｿｰｽのUSE_FILEDOWNLOAD_CHECKKEYをtrueに指定することで､簡易的なﾁｪｯｸを
 * 行うことができます｡
 * 具体的には､これを有効にすると､file属性の値から計算されるﾊｯｼｭｺｰﾄﾞﾁｪｯｸｻﾑと､"key"という
 * ﾊﾟﾗﾒｰﾀｰに指定された値が一致した場合のみﾀﾞｳﾝﾛｰﾄﾞが許可され､keyが指定されていない､
 * または値が異なる場合はﾀﾞｳﾝﾛｰﾄﾞｴﾗｰとなります｡
 *
 * 一般的なｻｰﾌﾞﾚｯﾄと同様に､ﾃﾞﾌﾟﾛｲﾒﾝﾄ･ﾃﾞｨｽｸﾘﾌﾟﾀ WEB-INF/web.xml に､
 * servlet 要素と そのﾏｯﾋﾟﾝｸﾞ(servlet-mapping)を定義する必要があります｡
 *
 *     &lt;servlet&gt;
 *         &lt;servlet-name&gt;fileDownload&lt;/servlet-name&gt;
 *         &lt;servlet-class&gt;org.opengion.hayabusa.servlet.FileDownload&lt;/servlet-class&gt;
 *     &lt;/servlet&gt;
 *
 *     &lt;servlet-mapping&gt;
 *         &lt;servlet-name&gt;fileDownload&lt;/servlet-name&gt;
 *         &lt;url-pattern&gt;/jsp/fileDownload&lt;/url-pattern&gt;
 *     &lt;/servlet-mapping&gt;
 *
 * 一般には､http://:ﾎﾟｰﾄ/ｼｽﾃﾑID/jsp/fileDownload?file=ｻｰﾊﾞｰ物理ﾌｧｲﾙ&amp;name=ﾌｧｲﾙ名
 * 形式のURL でｱｸｾｽします｡
 *
 * 5.9.25.0 (2017/10/06)
 * ｸﾗｳﾄﾞ上のPaaSでｵﾌﾞｼﾞｪｸﾄｽﾄﾚｰｼﾞを利用する際は以下のｼｽﾃﾑﾘｿｰｽを設定してください｡
 * CLOUD_TARGET,CLOUD_BUCKET
 * plugin/cloud内のｸﾗｽを利用してﾌｧｲﾙｱｯﾌﾟﾛｰﾄﾞ(FileUploadﾀｸﾞ)､ﾀﾞｳﾝﾛｰﾄﾞ(FileDownloadｻｰﾌﾞﾚｯﾄ)をAPI経由で行います｡
 * ﾌﾟﾗｸﾞｲﾝが利用するjarﾌｧｲﾙの配置は必要です｡
 * ｻｰﾌﾞﾚｯﾄに対して引数でstorage,bucketを与える事も可能です｡
 *
 * 5.8.1.0 (2014/11/07)
 * forwardでｱｸｾｽする場合はﾌｧｲﾙ名の文字ｺｰﾄﾞ変換が不要なため､useStringConvert=falseの
 * 引数を与えてください｡（falseとしない場合は日本語ﾌｧｲﾙ名等でｴﾗｰが発生します）
 *
 * 5.10.9.0 (2019/03/01) ｸﾗｳﾄﾞとﾊﾞｹｯﾄ名を指定するﾘｸｴｽﾄﾊﾟﾗﾒｰﾀを追加｡
 * 8.0.1.0 (2021/10/29) storageType → storage ､bucketName → bucket に変更
 *   storage  (初期値：ｼｽﾃﾑﾘｿｰｽのCLOUD_TARGET)
 *   bucket   (初期値：ｼｽﾃﾑﾘｿｰｽのCLOUD_BUCKET)
 *   useLocal (初期値：false)
 *  初期値は､ｼｽﾃﾑﾘｿｰｽ上のﾊﾟﾗﾒｰﾀで初期値を指定できます｡
 *  強制的にﾛｰｶﾙﾌｧｲﾙにｱｸｾｽする場合は､"LOCAL" を指定するか、useLocal="true"を指定してください｡
 *
 * 7.2.7.0 (2020/08/07) 相対ﾊﾟｽの場合の基準ﾌｫﾙﾀﾞ
 *  互換性確保のため､useBase ﾘｸｴｽﾄ変数を true で飛ばすと､処理を実行します｡
 *
 * 8.0.0.1 (2021/10/08)
 *  useStringConvert (ｴﾝｺｰﾄﾞ変換対応のON/OFF指定)を廃止｡
 *  漢字ﾌｧｲﾙのｴﾝｺｰﾄﾞを指定すると､文字化けするので､何も行わない｡
 *  RequestEncoding ﾊﾟﾗﾒｰﾀも､使用しません｡
 *
 * @og.rev 3.8.1.1 (2005/11/21) 新規追加
 * @og.rev 5.9.25.0 (2017/10/06) ｸﾗｳﾄﾞ対応
 * @og.rev 5.9.29.1 (2018/02/07) Azure対応追加
 * @og.rev 5.10.9.0 (2019/03/01) oota ｸﾗｳﾄﾞｽﾄﾚｰｼﾞ対応を追加｡(Fileｸﾗｽを拡張)
 * @og.group その他機能
 *
 * @version  0.9.0  2000/10/17
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.1,
 */
@WebServlet( "/jsp/fileDownload" )
public class FileDownload extends HttpServlet {
	private static final long serialVersionUID = 539020110901L ;

	// 拡張子contentType対応ﾃｰﾌﾞﾙ
	private static final String CONTENT_TYPE_TABLE[][] = {
		{"jpg", "image/pjpeg"	},
		{"gif", "image/gif"		},
		{"txt", "text/plain"	},
		// OpenDocument追加
		{"xls", "application/vnd.ms-excel"},
		{"odp", "application/vnd.oasis.opendocument.presentation"},				// 4.3.5.5 (2008/03/08)
		{"ods", "application/vnd.oasis.opendocument.spreadsheet"},				// 4.3.5.5 (2008/03/08)
		{"odt", "application/vnd.oasis.opendocument.text"}						// 4.3.5.5 (2008/03/08)
	};
	private static final int EXTENTION	 = 0;
	private static final int CONTENT_TYPE= 1;

	private static final String KEY_USELOCAL  = "useLocal";						// 8.0.1.0 (2021/10/29) 強制的にﾛｰｶﾙﾌｧｲﾙにｱｸｾｽする場合のｷｰ
//	private static final String KEY_STORAGE = "storage";						// 8.0.1.0 (2021/10/29) ｸﾗｳﾄﾞを指定するﾘｸｴｽﾄﾊﾟﾗﾒｰﾀのｷｰ
//	private static final String KEY_BUCKET  = "bucket";							// 8.0.1.0 (2021/10/29) ﾊﾞｹｯﾄ名を指定するﾘｸｴｽﾄﾊﾟﾗﾒｰﾀのｷｰ
	private static final String	HASH_CODE	= HybsSystem.sys( "FILE_HASH_CODE" );	// 8.1.2.0 (2022/03/10)

	private final String fileURL = HybsSystem.sys( "FILE_URL" );				// 7.2.7.0 (2020/08/07) 相対ﾊﾟｽの場合の基準ﾌｫﾙﾀﾞ

	/**
	 * ﾃﾞﾌｫﾙﾄｺﾝｽﾄﾗｸﾀｰ
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public FileDownload() { super(); }		// これも､自動的に呼ばれるが､空のﾒｿｯﾄﾞを作成すると警告されるので､明示的にしておきます｡

	/**
	 * GET ﾒｿｯﾄﾞが呼ばれたときに実行します｡
	 *
	 * 処理は､doPost へ振りなおしています｡
	 *
	 * @param	request	HttpServletRequestｵﾌﾞｼﾞｪｸﾄ
	 * @param	response	HttpServletResponseｵﾌﾞｼﾞｪｸﾄ
	 *
	 * @og.rev 3.8.1.2 (2005/12/19) 半角ｶﾅ-全角ｶﾅ変換機能の追加
	 *
	 * @throws ServletException ｻｰﾌﾞﾚｯﾄ関係のｴﾗｰが発生した場合､throw されます｡
	 * @throws IOException 入出力ｴﾗｰが発生したとき
	 */
	@Override
	public void doGet( final HttpServletRequest request, final HttpServletResponse response )
							throws ServletException, IOException {
		doPost( request,response );
	}

	/**
	 * POST ﾒｿｯﾄﾞが呼ばれたときに実行します｡
	 *
	 * file 引数の ｻｰﾊﾞｰ物理ﾌｧｲﾙを､ｸﾗｲｱﾝﾄにｽﾄﾘｰﾑ化して返します｡
	 * name 引数があれば､その名前のﾌｧｲﾙ名でｸﾗｲｱﾝﾄがﾌｧｲﾙｾｰﾌﾞできるように
	 * します｡name 引数がなければ､そのまま物理ﾌｧｲﾙ名が使用されます｡
	 * ｻｰﾊﾞｰ物理ﾌｧｲﾙ名が､相対ﾊﾟｽの場合､ｺﾝﾃｷｽﾄﾙｰﾄに対する相対ﾊﾟｽになります｡
	 * (例：G:\webapps\dbdef2\ など)
	 *
	 * @og.rev 5.3.2.0 (2011/02/01) 日本語ﾌｧｲﾙ名が正しく処理できないﾊﾞｸﾞを修正
	 * @og.rev 5.3.4.0 (2011/04/01) IEでﾌｧｲﾙが正しくﾀﾞｳﾝﾛｰﾄﾞできないﾊﾞｸﾞを修正
	 * @og.rev 5.3.5.0 (2011/05/01) ﾌｧｲﾙﾀﾞｳﾝﾛｰﾄﾞﾁｪｯｸｷｰ対応
	 * @og.rev 5.3.6.0 (2011/06/01) ﾌｧｲﾙﾀﾞｳﾝﾛｰﾄﾞはattachmentに変更(ﾀﾞｳﾝﾛｰﾄﾞﾀﾞｲｱﾛｸﾞを出す)
	 * @og.rev 5.3.8.0 (2011/08/01) ﾌｧｲﾙ名指定でIEの場合､URLｴﾝｺｰﾄﾞすると途中で切れるため(IE7のﾊﾞｸﾞ)､Shift_JIS(WIndows-31J)で直接指定する｡
	 * @og.rev 5.3.9.0 (2011/09/01) 引数にinline=trueを指定することで､ｲﾝﾗｲﾝ表示が出来るように対応
	 * @og.rev 5.7.1.2 (2013/12/20) 日本語ﾌｧｲﾙのIE11対応（UA変更）,msg ⇒ errMsg 変更
	 * @og.rev 5.8.1.0 (2014/11/07) forward時の文字ｺｰﾄﾞ変換不要対応
	 * @og.rev 5.9.25.0 (2017/10/06) ｸﾗｳﾄﾞｽﾄﾚｰｼﾞからﾀﾞｳﾝﾛｰﾄﾞ処理を追加対応
	 * @og.rev 5.9.27.0 (2017/12/01) Content-Lengthをhttpﾍｯﾀﾞに追加しておく
	 * @og.rev 5.9.27.2 (2017/12/15) Edgeの日本語ﾌｧｲﾙ名対応
	 * @og.rev 6.8.4.2 (2017/12/25) ｴﾝｺｰﾄﾞ変換対応のｷｰ(fileDownloadｻｰﾌﾞﾚｯﾄでｴﾝｺｰﾄﾞをON/OFF指定に利用)
	 * @og.rev 5.9.28.1 (2018/01/19) safariの日本語ﾌｧｲﾙ名対応(RFC6266方式を併記）
	 * @og.rev 6.9.4.1 (2018/04/09) 日本語ﾌｧｲﾙ名で､旧方式を入れておくと､文字化けするので､はずします｡
	 * @og.rev 5.10.12.4 (2019/06/21) ｴﾝｺｰﾃﾞｨﾝｸﾞを外部から指定可能にする
	 * @og.rev 7.2.7.0 (2020/08/07) 相対ﾊﾟｽの場合の基準ﾌｫﾙﾀﾞ(FILE_URL) 考慮
	 * @og.rev 8.0.0.1 (2021/10/08) USE_STR_CONV_KEY 廃止
	 * @og.rev 8.1.2.0 (2022/03/10) getMD5 ﾒｿｯﾄﾞを getHash ﾒｿｯﾄﾞに変更
	 *
	 * @param	request	HttpServletRequestｵﾌﾞｼﾞｪｸﾄ
	 * @param	response	HttpServletResponseｵﾌﾞｼﾞｪｸﾄ
	 *
	 * @throws ServletException ｻｰﾌﾞﾚｯﾄ関係のｴﾗｰが発生した場合､throw されます｡
	 * @throws IOException 入出力ｴﾗｰが発生したとき
	 */
	@Override
	public void doPost( final HttpServletRequest request, final HttpServletResponse response )
							throws ServletException, IOException {

		// 2017/10/06 ADD
		// ｸﾗｳﾄﾞｽﾄﾚｰｼﾞ指定
	//	final String storage = HybsSystem.sys( "CLOUD_TARGET");
		// ｸﾗｳﾄﾞｽﾄﾚｰｼﾞ指定ﾌﾗｸﾞ
	//	final boolean cloudFlag = storage != null && storage.length() > 0;

//		// 8.0.0.1 (2021/10/08) cloud対応 … USE_STR_CONV_KEY 廃止
//		// 5.8.1.0 (2014/11/07) ｴﾝｺｰﾄﾞ変換対応
//		// 6.8.4.2 (2017/12/25) ｴﾝｺｰﾄﾞ変換対応で､Attribute も確認します｡
//		// ややこしくなってますが､どちらかのｷｰﾜｰﾄﾞで､"false" が指定された場合のみ､false になります｡
//		final boolean useStrCnv = StringUtil.nval(         request.getParameter( HybsSystem.USE_STR_CONV_KEY ), true ) &&
//								  StringUtil.nval( (String)request.getAttribute( HybsSystem.USE_STR_CONV_KEY ), true ) ;

//		// 5.10.12.4 (2019/06/21)
//		final String requestEncode = StringUtil.nval( (String)request.getAttribute( "RequestEncoding" ), "ISO-8859-1" );

		// ｸﾗｲｱﾝﾄ側の文字ｴﾝｺｰﾃﾞｨﾝｸﾞをUTF-8に変換
		// 5.8.1.0 (2014/11/07) 条件追加
		String reqFilename = request.getParameter( "file" );		// 6.4.1.1 (2016/01/16) PMD refactoring.
//		// 8.0.0.1 (2021/10/08) cloud対応 … USE_STR_CONV_KEY 廃止
//		if( useStrCnv ){
//	//		reqFilename = new String( reqFilename.getBytes("ISO-8859-1"), "UTF-8" );
//			reqFilename = new String( reqFilename.getBytes(requestEncode), "UTF-8" );
//		}

		// 2017/10/06 ADD reqFilenameの保存
	//	final String cloudFilename = reqFilename;

		// 5.3.5.0 (2011/05/01) ﾌｧｲﾙﾀﾞｳﾝﾛｰﾄﾞﾁｪｯｸｷｰ対応
		final boolean useCheck = HybsSystem.sysBool( "USE_FILEDOWNLOAD_CHECKKEY" );
		if( useCheck ) {
			final String checkKey = request.getParameter( "key" );
//			if( checkKey == null || !checkKey.equals( HybsCryptography.getMD5( reqFilename ) ) ) {	// 8.1.2.0 (2022/03/10) Modify
			if( checkKey == null || !checkKey.equals( HybsCryptography.getHash( HASH_CODE, reqFilename ) ) ) {
				final String errMsg = "ｱｸｾｽが拒否されました｡(URLﾁｪｯｸ)";
				throw new HybsSystemException( errMsg );	// 5.7.1.2 (2013/12/20) msg ⇒ errMsg 変更
			}
		}

		// ※ useCheck の前か後か？ とりあえず後ろに入れておきます｡
		// 7.2.7.0 (2020/08/07) 相対ﾊﾟｽの場合の基準ﾌｫﾙﾀﾞ(FILE_URL) 考慮｡互換性の関係で､useBase で制御します｡
		final boolean useBase = StringUtil.nval( request.getParameter( "useBase" ), false );
		if( useBase ) {
			reqFilename = StringUtil.urlAppend( fileURL,reqFilename );
		}

		// 相対ﾊﾟｽを絶対ﾊﾟｽに変換｡ﾌｧｲﾙｾﾊﾟﾚｰﾀも正規化されています｡
		reqFilename = HybsSystem.url2dir( reqFilename );

		// 拡張子からcontentTypeを獲得
		final String contentType = getContentType( reqFilename );
		// contentTypeを出力
		response.setContentType( contentType );

		// 表示ﾌｧｲﾙ名の指定
		String newFilename = request.getParameter( "name" );		// 6.4.1.1 (2016/01/16) PMD refactoring.
		if( newFilename == null || newFilename.isEmpty() ) {
			newFilename = getFileName( reqFilename );
		}
//		// 8.0.0.1 (2021/10/08) cloud対応 … USE_STR_CONV_KEY 廃止
//		else if( useStrCnv ){		// 5.8.1.0 (2014/11/07) 条件追加
//	//		newFilename = new String( newFilename.getBytes("ISO-8859-1"), "UTF-8" );
//			newFilename = new String( newFilename.getBytes(requestEncode), "UTF-8" );
//		}

		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid declaring a variable if it is unreferenced before a possible exit point.
		// 3.8.1.2 (2005/12/19) 半角ｶﾅを全角ｶﾅに置き換えます｡ﾌｧｲﾙﾀﾞｲｱﾛｸﾞの文字化け仮対応
		if( HybsSystem.sysBool( "USE_FILEDOWNLOAD_HAN_ZEN" ) ) {
			newFilename = KanaFilter.han2zen( newFilename );
		}

		// 6.9.4.1 (2018/04/09) 日本語ﾌｧｲﾙ名で､旧方式を入れておくと､文字化けするので､はずします｡(StringUtil.urlEncodeだけでよい｡)
	//	// 5.7.1.2 (2013/12/20) 条件を反転させた上でIE11対応を行う
	//	final String reqHeader = request.getHeader( "User-Agent" );
	//	// 5.9.27.2 (2017/12/15) EdgeもIE同様の処理にする
	//	if( reqHeader.indexOf( "MSIE" ) >= 0 || reqHeader.indexOf( "Trident" ) >= 0 || reqHeader.indexOf( "Edge" ) >= 0 ) {
	//		newFilename = new String( newFilename.getBytes("Windows-31J"), "ISO-8859-1" );
	//	}
	//	else {
	//		newFilename = MimeUtility.encodeWord( newFilename, "UTF-8", "B" );
	//	}

		// 5.3.9.0 (2011/09/01) 引数にinline=trueを指定することで､ｲﾝﾗｲﾝ表示が出来るように対応
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid declaring a variable if it is unreferenced before a possible exit point.
		final boolean inline = StringUtil.nval( request.getParameter( "inline" ), false );
		final String dipositionType = inline ? "inline" : "attachment";

		final String newFilenameEnc = StringUtil.urlEncode( newFilename );  // 5.9.28.1 (2018/01/19)

		// ﾌｧｲﾙ名の送信( attachment部分をinlineに変更すればｲﾝﾗｲﾝ表示 )
		// 5.3.9.0 (2011/09/01) 引数にinline=trueを指定することで､ｲﾝﾗｲﾝ表示が出来るように対応
//		response.setHeader( "Content-disposition", dipositionType + "; filename=\"" + newFilename + "\"" );
		response.setHeader( "Content-disposition", dipositionType + "; filename=\"" + newFilename + "\"; "
				+ "filename*=UTF-8''" + newFilenameEnc ); // 5.9.28.1 (2018/01/19)　RFC6266方式を併記

		// 5.3.4.0 (2011/04/01) IEでﾌｧｲﾙが正しくﾀﾞｳﾝﾛｰﾄﾞできないﾊﾞｸﾞを修正
		response.setHeader( "Cache-Control", "public" );

		// ﾌｧｲﾙ内容の出力
//		FileInputStream     fin = null;
		ServletOutputStream out = null;
		// 2017/10/06 MODIFY bluemixのｽﾄﾚｰｼﾞ利用の処理を追加
		InputStream is = null;
		// 5.9.29.1 (2018/02/07) lengthのｸﾗｳﾄﾞ対応
		String filesize = null;
		try {
			// 2017/10/06 ADD bluemixのｽﾄﾚｰｼﾞに保存する場合の処理を追加
	//		if(cloudFlag){
	//			HttpSession hsession = request.getSession(true);
	//			StorageAPI storageApi = StorageAPIFactory.newStorageAPI(storage, HybsSystem.sys("CLOUD_TARGET_CONTAINER"), hsession);
	//			// ｽﾄﾘｰﾑの取得
	//			is = storageApi.get(cloudFilename, hsession);
	//
	//			// ﾌｧｲﾙｻｲｽﾞを取得
	//			Map<String,String> map = storageApi.getInfo(cloudFilename, hsession);
	//			filesize = map.get(StorageAPI.FILEINFO_SIZE);
	//		}else{
	//		// 標準のﾌｧｲﾙ保存
		//		fin = new FileInputStream( reqFilename );
		//		is = fin;

				// 8.0.0.0 (2021/09/30) ｸﾗｳﾄﾞ対応
				// 5.10.9.0 (2019/03/01) ｸﾗｳﾄﾞとﾊﾞｹｯﾄ名を指定するﾘｸｴｽﾄﾊﾟﾗﾒｰﾀを追加｡
				// 8.0.1.0 (2021/10/29) storageType , bucketName 削除
				final boolean useLocal = StringUtil.nval( request.getParameter( KEY_USELOCAL ) , false );	// 8.0.1.0 (2021/10/29)
//				final String storage   = request.getParameter( KEY_STORAGE );
//				final String bucket    = useLocal ? FileOperation.LOCAL : request.getParameter( KEY_BUCKET );	// useLocal=true で、強制ﾛｰｶﾙ

				// 5.10.9.0 (2019/03/01) MODIFY
				// 8.0.1.0 (2021/10/29) storageType , bucketName 削除
//				final FileOperation file = HybsFileOperationFactory.create( storage, bucket, reqFilename );
				final FileOperation file = HybsFileOperationFactory.create( useLocal, reqFilename );
				is = file.read();

//				filesize = String.valueOf(fin.available());
				filesize = String.valueOf(is.available());
	//		}
//			response.setHeader( "Content-Length", String.valueOf(fin.available()) );	// 5.9.27.0 (2017/12/01)
			response.setHeader( "Content-Lnegth", filesize);	// ｸﾗｳﾄﾞのｻｲｽﾞ取得対応
			out = response.getOutputStream();

			// ﾌｧｲﾙ読み込み用ﾊﾞｯﾌｧ
			final byte buffer[]  = new byte[4096];
			int size;
	//		while((size = fin.read(buffer))!=-1) {
			while((size = is.read(buffer))!=-1) {
				out.write(buffer,0, size);
				out.flush();
			}
		}
		finally {
			Closer.ioClose(is);		// 2017/10/06 ADD
//			Closer.ioClose( fin );		// 4.0.0 (2006/01/31) close 処理時の IOException を無視
			Closer.ioClose( out );		// 4.0.0 (2006/01/31) close 処理時の IOException を無視
		}
	}

	/**
	 * ｱﾄﾞﾚｽ名から拡張子を取り出します｡
	 *
	 * ｱﾄﾞﾚｽ名の後ろから､"." 以降を拡張子として切り取ります｡
	 * 拡張子が存在しない場合(指定のﾌｧｲﾙ名に "." が含まれない場合)は
	 * ｾﾞﾛ文字列("")を返します｡
	 *
	 * @param	fileAddress	ｱﾄﾞﾚｽ名
	 *
	 * @return	拡張子
	 * @og.rtnNotNull
	 */
	private String getExtention( final String fileAddress ) {
		final int idx = fileAddress.lastIndexOf( '.' );

		return idx >= 0 ? fileAddress.substring( idx+1 ) : "";		// 6.1.1.0 (2015/01/17) refactoring
	}

	/**
	 * ｱﾄﾞﾚｽ名からﾌｧｲﾙ名を取り出します｡
	 *
	 * ｱﾄﾞﾚｽ名の後ろから､ﾌｧｲﾙｾﾊﾟﾚｰﾀ以降をﾌｧｲﾙ名として切り取ります｡
	 * ﾌｧｲﾙｾﾊﾟﾚｰﾀが存在しない場合はｱﾄﾞﾚｽ名をそのまま返します｡
	 * ここでは､OS毎に異なるﾌｧｲﾙｾﾊﾟﾚｰﾀを統一後に処理してください｡
	 *
	 * @param	fileAddress	ｱﾄﾞﾚｽ名
	 *
	 * @return	ﾌｧｲﾙ名
	 */
	private String getFileName( final String fileAddress ) {
		final int idx = fileAddress.lastIndexOf( FS );

		// 6.4.1.1 (2016/01/16) PMD refactoring. A method should have only one exit point, and that should be the last statement in the method
//		// 条件変更(反転)注意
//		return idx < 0 ? fileAddress : fileAddress.substring( idx+1 );
		return idx >= 0 ? fileAddress.substring( idx+1 ) : fileAddress;
	}

	/**
	 * ｱﾄﾞﾚｽ名から対応するｺﾝﾃﾝﾂﾀｲﾌﾟを取り出します｡
	 *
	 * ｱﾄﾞﾚｽ名から､ﾌｧｲﾙ拡張子を取り出し､対応するｺﾝﾃﾝﾂﾀｲﾌﾟを返します｡
	 * ｺﾝﾃﾝﾂﾀｲﾌﾟは､CONTENT_TYPE_TABLE 配列に定義している中から検索して返します｡
	 * 存在しない場合は､"application/octet-stream" を返します｡
	 *
	 * @param	fileAddress	ｱﾄﾞﾚｽ名
	 *
	 * @return	ｺﾝﾃﾝﾂﾀｲﾌﾟ
	 */
	private String getContentType( final String fileAddress ) {
		final String extention = getExtention( fileAddress );
		for( int j=0; j<CONTENT_TYPE_TABLE.length; j++ ) {
			if( CONTENT_TYPE_TABLE[j][EXTENTION].equalsIgnoreCase( extention ) ) {
				return CONTENT_TYPE_TABLE[j][CONTENT_TYPE];
			}
		}
		return "application/octet-stream";
	}
}
