/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.table;

import java.io.File;
import java.io.PrintWriter;

import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.FixLengthData;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.AbstractTableFilter;
import org.opengion.hayabusa.db.DBTableModel;

/**
 * TableFilter_CMNT_TBL は、TableFilter インターフェースを継承した、DBTableModel 処理用の
 * 実装クラスです。
 *
 * ここでは、テーブル一覧の検索結果より、ORACLE のテーブルにコメントを作成します。
 * 構文は、「COMMENT ON TABLE テーブル名 IS 'コメント'」です。
 * このコメントを取り出す場合は、
 *「SELECT COMMENTS FROM USER_TAB_COMMENTS WHERE TABLE_NAME = 'テーブル名'」
 * とします。
 * 
 * SQLのツール類には、このコメントを使用して、テーブルの日本語名を表示させたりします。
 * 
 * 検索では、(TABLE_NAME,NAME_JA) の項目を取得する必要があります。
 *
 * パラメータは、tableFilterタグの keys, vals にそれぞれ記述するか、BODY 部にCSS形式で記述します。
 * 出力ファイル名は、通常、テーブル で１つ、カラムで１つにまとめて作成されます。
 * 【パラメータ】
 *  {
 *       DIR  : {&#064;BASE_DIR}/sql/install/08_CMNT ;    出力ファイルの基準フォルダ(必須)
 *       FILE : false ;                                   出力ファイル名(初期値:CMNT_TBL[.sql|.xml])
 *       XML  : false ;                                   XML出力を行うかどうか[true/false]を指定します(初期値:false)
 *  }
 *
 * @og.formSample
 * ●形式：
 *      select SYSTEM_ID,TABLE_NAME,NAME_JA from GF02
 * 
 *      ① &lt;og:tableFilter classId="CMNT_TBL" keys="DIR" vals='"{&#064;BASE_DIR}/sql/install/08_CMNT"' /&gt;
 *
 *      ② &lt;og:tableFilter classId="CMNT_TBL" &gt;
 *               {
 *                   DIR   : {&#064;BASE_DIR}/sql/install/08_CMNT ;
 *                   FILE  : CMNT_TBL ;
 *                   XML   : false ;
 *               }
 *         &lt;/og:tableFilter&gt;
 *
 * @og.rev 4.0.0.0 (2005/08/31) 新規作成
 *
 * @version  0.9.0  2000/10/17
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.6,
 */
public class TableFilter_CMNT_TBL extends AbstractTableFilter {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.5.0.1 (2016/10/21)" ;

	private static final String[] DBKEY = {"TABLE_NAME","NAME_JA"};

	/** データのアクセス用の配列番号 {@value} */
	protected static final int TABLE_NAME		= 0;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int NAME_JA			= 1;

 //	private static final String ENCODE = "Windows-31J" ;
	private static final String ENCODE = "UTF-8" ;

	private static final int X = FixLengthData.X ;		// type 定数
	private static final int K = FixLengthData.K ;		// type 定数

	/** 各種定数  */
	// 6.0.2.3 (2014/10/10) AbstractTableFilter へ移動

	/** XML形式かどうか */

	/** ファイル名(拡張子なし) */
	private String		fileName				= "CMNT_TBL";

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.1.1 (2016/01/16) keysMap を、サブクラスから設定させるように変更。
	 */
	public TableFilter_CMNT_TBL() {
		super();
		initSet( "DIR"	, "出力ファイルの基準フォルダ(必須)"						);
		initSet( "FILE"	, "出力ファイル名(初期値:CMNT_TBL[.sql|.xml])"				);
		initSet( "XML"	, "XML出力を行うかどうか[true/false]を指定(初期値:false)"	);
	}

	/**
	 * DBTableModel処理を実行します。
	 *
	 * @og.rev 6.0.2.3 (2014/10/10) EXEC_END_TAG を追加。キャッシュします。
	 * @og.rev 6.3.7.0 (2015/09/04) AutoCloseableを使用したtry-with-resources構築に対応。
	 * @og.rev 6.5.0.1 (2016/10/21) ErrorMessage をまとめるのと、直接 Throwable を渡します。
	 *
	 * @return	実行結果のテーブルモデル
	 */
	public DBTableModel execute() {
		isXml   = StringUtil.nval( getValue( "XML"   ), isXml   );
		execEndTag = isXml ? EXEC_END_TAG : ";" ;	// 6.0.2.3 (2014/10/10) CMNTは タグの前に改行を入れない。

		final File dir = new File( getValue( "DIR" ) );

		if( ! dir.exists() && ! dir.mkdirs() ) {
			final String errMsg = "所定のフォルダが作成できませんでした。[" + dir + "]" ;
			// 4.3.4.4 (2009/01/01)
			throw new HybsSystemException( errMsg );
		}

		fileName =  StringUtil.nval( getValue( "FILE" ), fileName );

		// COMMENT ON TABLE テーブル名 IS 'コメント'
		final int[] addLen = new int[] { 0,0,0,0 };	// 各データ間のスペース
		final int[] type   = new int[] { X,X,X,K };	// 各データの種別 X:半角 S:空白前埋め K:全角混在
		final FixLengthData fixData = new FixLengthData( addLen,type );

		String[] data  = null;
		int row = 0;
		// 6.3.7.0 (2015/09/04) AutoCloseableを使用したtry-with-resources構築に対応。
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid declaring a variable if it is unreferenced before a possible exit point.
		final DBTableModel table = getDBTableModel();
		try( final PrintWriter writer = FileUtil.getPrintWriter( new File( dir,fileName + ( isXml ? ".xml" : ".sql" ) ),ENCODE ) ) {

			if( isXml ) { writer.println( XML_START_TAG ); }

			// 全データを読み込んで、最大長の計算を行う。
			final int[] clmNo = getTableColumnNo( DBKEY );
			final int rowCnt = table.getRowCount();
			for( row=0; row<rowCnt; row++ ) {
				data      = table.getValues( row );
				final String tableName = data[clmNo[TABLE_NAME]];
				final String name_ja   = "'" + data[clmNo[NAME_JA]] + "'";

				final String[] outData = new String[] { "COMMENT ON TABLE " , tableName , " IS " , name_ja } ;
				fixData.addListData( outData );
			}

			// データの出力
			for( row=0; row<rowCnt; row++ ) {
				if( isXml ) { writer.print( EXEC_START_TAG ); }
				writer.print( fixData.getFixData( row ) );
				writer.println( execEndTag );						// 6.0.2.3 (2014/10/10)
			}

			if( isXml ) { writer.println( XML_END_TAG ); }
		}
		catch( final RuntimeException ex ) {		// catch は、close() されてから呼ばれます。
			// 6.5.0.1 (2016/10/21) ErrorMessage をまとめるのと、直接 Throwable を渡します。
			data = table.getValues( row );
			final ErrorMessage errMessage = makeErrorMessage( "TableFilter_CMNT_TBL Error",ErrorMessage.NG )
				.addMessage( row+1,ErrorMessage.NG,"CMNT_TBL"
					, "dir=[" + dir + "]" 
					, "fileName=[" + fileName + "]" 
					, StringUtil.array2csv( data )
				)
				.addMessage( ex );
			// BAT から呼び出す場合があるため、標準エラー出力にも情報を出しておきます。
			System.out.println( errMessage );
		}

		return table;
	}
}
