/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.fukurou.process.HybsProcess;
import org.opengion.fukurou.process.Process_DBParam;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)

import static org.opengion.fukurou.util.StringUtil.nval ;

/**
 * MainProcess で実行される Processクラスを構築します。
 *
 * 親クラス(Process)は、org.opengion.fukurou.process パッケージの HybsProcess
 * インターフェースを実装したクラスの、Process_****.java の **** 部分を指定します。
 * 共通的な パラメータは、このTagクラスに実装しますが、それぞれ、個別に必要な
 * パラメータは、ParamTag を使用して指定します。
 * このタグは、MainProcess タグの内部にのみ、記述可能です。
 *
 * @og.formSample
 * ●形式：&lt;og:process processID="ZZZ" &gt;
 *             &lt;og:param key="AAA" value="111" /&gt;
 *         &lt;/og:process &gt;
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し、{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:process
 *       processID        ○【TAG】リクエスト情報 に登録するキーをセットします(必須)。
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null)
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:判定しない)
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:判定しない)
 *       caseIf             【TAG】指定の値が、true/TRUE文字列の場合は、このタグは使用されます(初期値:判定しない)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:process&gt;
 *
 * ●使用例
 *     一般的な変数の渡し方
 *   &lt;og:mainProcess &gt;
 *     &lt;og:process processID="DBReader" &gt;
 *        &lt;og:param key="dbid" value="FROM" /&gt;
 *        &lt;og:param key="sql"  value="select * from GE02" /&gt;
 *     &lt;/og:process &gt;
 *     &lt;og:process processID="DBWriter" &gt;
 *        &lt;og:param key="dbid"  value="TO" /&gt;
 *        &lt;og:param key="table" value="GE02" /&gt;
 *     &lt;/og:process &gt;
 *   &lt;/og:mainProcess &gt;
 *
 *     BODY 部に記述した変数の渡し方
 *     &lt;og:process processID="DBReader" &gt;
 *        &lt;og:param key="SQL" &gt;
 *              SELECT COUNT(*) FROM GEA03
 *              WHERE SYSTEM_ID=[SYSTEM_ID]
 *              AND   CLM=[CLM]
 *              AND   FGJ = '1'
 *        &lt;/og:param&gt;
 *     &lt;/og:process &gt;
 *
 * @og.group リアルバッチ系
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class ProcessTag extends CommonTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.2.0 (2016/01/29)" ;
	private static final long serialVersionUID = 642020160129L ;

	private static final String PRCS = "org.opengion.fukurou.process.Process_" ;

	private transient HybsProcess	process	;
	private String		processID	;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public ProcessTag() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 4.3.1.1 (2008/09/04) DBParam 使用時は、専用の初期化メソッドを呼ぶ
	 * @og.rev 6.3.4.0 (2015/08/01) caseKey,caseVal,caseNN,caseNull,caseIf 属性対応
	 *
	 * @return	後続処理の指示( EVAL_BODY_BUFFERED )
	 */
	@Override
	public int doStartTag() {
		if( !useTag() ) { return SKIP_BODY ; }	// 6.3.4.0 (2015/08/01)

		final MainProcessTag mainProcess = (MainProcessTag)findAncestorWithClass( this,MainProcessTag.class );
		if( mainProcess == null ) {
			final String errMsg = "<b>" + getTagName() + "タグは、MainProcessTagの内側(要素)に記述してください。</b>"
								+ "processID=" + processID ;
			throw new HybsSystemException( errMsg );
		}

	 	// 4.3.1.1 (2008/09/04) DBParam 使用時は、専用の初期化メソッドを呼ぶ
		if( "DBParam".equals( processID ) ) {
			process = new Process_DBParam();
			((Process_DBParam)process).setAppInfo( getApplicationInfo() );
		}
		else {
			process = (HybsProcess)StringUtil.newInstance( PRCS + processID );
		}

		mainProcess.addProcess( process );

		return EVAL_BODY_BUFFERED ;		// Body を評価する
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		processID	= null;
		process		= null;
	}

	/**
	 * 【TAG】リクエスト情報 に登録するキーをセットします。
	 *
	 * @og.tag
	 * processID は、org.opengion.fukurou.process.HybsProcess インターフェースを実装した
	 * Process_**** クラスの **** を与えます。
	 * これらは、HybsProcess インターフェースを継承したサブクラスである必要があります。
	 * 標準で、org.opengion.fukurou.process 以下の Process_**** クラスが、Process_**** 宣言 と
	 * して、定義されています。
	 * 属性クラス定義の {@link org.opengion.fukurou.process.HybsProcess HybsProcess} を参照願います。
	 * {@og.doc03Link process Process_**** クラス}
	 *
	 * @param	pid リクエスト情報に登録するキー
	 * @see		org.opengion.fukurou.process.HybsProcess  HybsProcessのサブクラス
	 */
	public void setProcessID( final String pid ) {
		processID = nval( getRequestParameter( pid ),processID ) ;
	}

	/**
	 * 親クラスに登録するキーをセットします。
	 *
	 * @param	key		登録するキー
	 * @param	value	登録する値
	 */
	protected void addParam( final String key,final String value ) {
		process.putArgument( key,value );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"				,VERSION			)
				.println( "processID"			,processID			)
				.fixForm().toString() ;
	}
}
