/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.fukurou.util.Attributes;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)

import static org.opengion.fukurou.util.StringUtil.nval ;
import static org.opengion.fukurou.system.HybsConst.BR;		// 6.1.0.0 (2014/12/26) refactoring

import java.util.Locale ;

/**
 * エラーメッセージを 表形式で表示するタグです。
 *
 * Query 関係の実行時にエラー/ワーニングが発生すると、HybsSystem.ERR_MSG_KEY をキーに
 * ErrorMessage オブジェクト をセッションに登録します。
 * この情報を元に、表題(TITLE)か、内容(BODY)を表示します。
 * 基本的には，表題表示時には，リンクを張り、共通エラー表示画面をオープン
 * 出来る様になっています。
 *
 * @og.formSample
 * ●形式：
 *     &lt;og:errorMessage command=&quot;{&#064;command}&quot; clear=&quot;{&#064;clear}&quot; /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:errorMessage
 *       command            【TAG】コマンド (NEW,RENEW,RESET,REVIEW)をセットします
 *       clear              【TAG】メッセージを初期化するかどうか[true/false]を指定します(初期値:false)
 *       viewType           【TAG】表示形式『表題(TITLE)か、内容(BODY)』を指定します(初期値:TITLE)
 *       displayMsg         【TAG】plsqlUpdate の結果を画面上に表示するメッセージIDを指定します(初期値:MSG0059『登録しました』)
 *       warningMsg         【TAG】登録処理実行後のワーニング結果を画面上に表示するメッセージIDを指定します(初期値:ERR0020)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *    result.jsp 等のSQL登録実行後の戻り画面に、上記タグを配置すれば、
 *    エラーメッセージが存在すれば,リンクとなって現れ、無ければ,なにも
 *    現れません。
 *    リンクのとび先は自動的に設定されます。
 *    なお、clear=&quot;true&quot; または、command=&quot;NEW&quot; の場合に、エラーメッセージは、
 *    クリアされます。
 *
 *    [entry.jsp]
 *        &lt;% String forwardPage="result.jsp"; %&gt;
 *        &lt;jsp:forward page="&lt;%= response.encodeRedirectURL( forwardPage ) %&gt;" &gt;
 *            &lt;jsp:param name="command" value="REVIEW" /&gt;
 *            &lt;jsp:param name="clear"   value="false"  /&gt;
 *        &lt;/jsp:forward&gt;
 *
 *    [result.jsp]
 *        &lt;og:errorMessage command="{&#064;command}" clear="{&#064;clear}" /&gt;
 *
 * @og.group エラー処理
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class ErrorMessageTag extends CommonTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.4.1 (2016/03/18)" ;
	private static final long serialVersionUID = 644120160318L ;

	/** command 引数に渡す事の出来る コマンド  新規 {@value} */
	public static final String CMD_NEW	 = "NEW" ;
	/** command 引数に渡す事の出来る コマンド  再検索 {@value} */
	public static final String CMD_RENEW = "RENEW" ;
	/** command 引数に渡す事の出来る コマンド  取消 {@value} */
	public static final String CMD_RESET = "RESET" ;
	/** command 引数に渡す事の出来る コマンド  再表示 {@value} */
	public static final String CMD_REVIEW	= "REVIEW" ;

	private static final String ERR_MSG_ID	= HybsSystem.ERR_MSG_KEY;		// 6.4.1.1 (2016/01/16) errMsgId → ERR_MSG_ID  refactoring

	// 6.4.4.1 (2016/03/18) application オブジェクトに関連付ける 共通メッセージがあれば、表示します。
	private static final String CMN_MSG_ID	= HybsSystem.COMMON_MSG_KEY;

	private final String errMsgFile	= HybsSystem.sys( "ERR_MSG_FILENAME" );
	private final int	maxRowCount = HybsSystem.sysInt( "DB_MAX_ROW_COUNT" ) ;

	private transient ErrorMessage	errMessage	;
	private String			command		;
	private boolean 		msgClear	;
	private String			viewType	= "TITLE";		// TITLE/BODY

	// 2.0.1.0 (2002/10/10) デフォルト表示しないから、MSG0059=登録しました。に変更します。
	private String		displayMsg	 = "MSG0059";	// 初期値は『登録しました。』
	private String		warningMsg	 = "ERR0020";	// データ登録時にワーニングが発生しました。

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public ErrorMessageTag() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 3.5.4.0 (2003/11/25) エラーオブジェクトのクリアに、RENEW or null も追加します。
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doStartTag() {
		// クリアが指示されるか、コマンドが NEW or RESET or RENEW or null の場合は、エラーをクリアする。
		if( msgClear || CMD_NEW.equals( command ) || CMD_RESET.equals( command ) ) {
		// 3.5.4.9 (2004/02/25) RENEW の時は、エラーをクリアしない。
			removeSessionAttribute( ERR_MSG_ID );
			msgClear = true;
		}
		else {
			errMessage = (ErrorMessage)getSessionAttribute( ERR_MSG_ID );
			if( errMessage == null ) { msgClear = true; }
		}

		return SKIP_BODY ;		// Body を評価しない
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 2.1.0.3 (2002/11/08) command = NEW のときも、『登録しました。』メッセージが表示されるバグを修正
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.5.5.2 (2004/04/02) TaglibUtil.makeHTMLErrorTable メソッドを利用
	 * @og.rev 4.0.0.0 (2007/10/18) メッセージリソース統合( getResource().getMessage ⇒ getResource().getLabel )
	 * @og.rev 4.1.3.0 (2008/09/04) メッセージをspanで囲う(画面遷移なしモード対応)
	 * @og.rev 5.2.1.0 (2010/10/01) 戻るリンク時に不要な改行が出力される件に対応
	 * @og.rev 6.4.4.1 (2016/03/18) application オブジェクトに関連付ける 共通メッセージがあれば、表示します。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)

		String msg = null;
		if( errMessage == null ) {
			if( CMD_REVIEW.equals( command ) || CMD_RENEW.equals( command ) ) {
				// 5.2.1.0 (2010/10/01) 戻るリンク時に不要な改行が出力される件に対応
				msg = getResource().getLabel( displayMsg );
				if( msg != null && msg.length() > 0 ) { msg += BR; }
			}
		}
		else {
			if( "TITLE".equalsIgnoreCase( viewType ) ) {
				msg = makeTitle();
			}
			else if( "BODY".equalsIgnoreCase( viewType ) ) {
				msg = TaglibUtil.makeHTMLErrorTable( errMessage,getResource() );	// 3.5.5.2 (2004/04/02)
			}
			else {
				final String errMsg = "viewType属性に TITLE/BODY 以外の項目が指定されています。"
							+ "[" + viewType + "]" ;
				throw new HybsSystemException( errMsg );
			}
		}

		// 6.4.4.1 (2016/03/18) application オブジェクトに関連付ける 共通メッセージがあれば、表示します。
		final String cmnMsg = (String)getContextAttribute( CMN_MSG_ID );

		jspPrint( "<span class=\"errmsg\">" );	// 4.1.3.0 (2008/09/04)
		if( msg    != null && msg.length()    > 0 ) { jspPrint( msg ); }
		if( cmnMsg != null && cmnMsg.length() > 0 ) { jspPrint( cmnMsg ); }		// 6.4.4.1 (2016/03/18)
		jspPrint( "</span>" );

		return EVAL_PAGE ;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 2.0.1.0 (2002/10/10) デフォルト表示しないから、MSG0059=登録しました。に変更します。
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		command		= null;
		errMessage	= null;
		msgClear	= false;
		viewType	= "TITLE";		// TITLE/BODY
		displayMsg	= "MSG0059";	// 初期値は『登録しました。』
		warningMsg	= "ERR0020";	// データ登録時にワーニングが発生しました。
	}

	/**
	 * エラーメッセージをタグ情報の文字列に変換して返します。
	 *
	 * ここでは、正常なメッセージも異常なメッセージも作成します。
	 *
	 * @og.rev 3.6.0.1 (2004/09/29) ワーニング、エラー時のスタイルシートを適用
	 * @og.rev 3.6.0.7 (2004/11/06) target 属性を _new から _blank に変更
	 * @og.rev 4.0.0.0 (2007/10/18) メッセージリソース統合( getResource().getMessage ⇒ getResource().getLabel )
	 * @og.rev 5.1.7.0 (2010/06/01) エラー・ワーニングメッセージの後に改行を入れる(displayMsgと仕様を合わせる)
	 *
	 * @return	エラーメッセージのタグ情報文字列
	 */
	private String makeTitle() {
		final String href = getContextPath() + "/" + errMsgFile ;

		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
		final String title = warningMsg == null
								? "<span class=\"msg_error\">"
									+ errMessage.getTitle()
									 // 5.1.7.0 (2010/06/01) エラーメッセージの後に改行を入れる
									 + "</span>" + BR
								:"<span class=\"msg_warning\">"
									 + getResource().getLabel( warningMsg )
									 // 5.1.7.0 (2010/06/01) ワーニングメッセージの後に改行を入れる
									 + "</span>" + BR;

		final String key = "pageSize";
		final String val = String.valueOf( maxRowCount );
		final String urlEnc = XHTMLTag.urlEncode( key,val );

		// 6.1.1.0 (2015/01/17) Attributesの連結記述
		return XHTMLTag.link(
				new Attributes()
					.set( "href"	, href	 )
					.set( "target" , "_blank" )	// 3.6.0.7 (2004/11/06)
					.set( "body" , title )
				, urlEnc
			);
	}

	/**
	 * 【TAG】コマンド (NEW,RENEW,RESET,REVIEW)をセットします。
	 *
	 * @og.tag
	 * コマンドは,HTMLから(get/post)指定されますので,CMD_xxx で設定される
	 * フィールド定数値のいづれかを、指定できます。
	 *
	 * @param	cmd コマンド (public static final 宣言されている文字列)
	 * @see		<a href="../../../../constant-values.html#org.opengion.hayabusa.taglib.ErrorMessageTag.CMD_NEW">コマンド定数</a>
	 */
	public void setCommand( final String cmd ) {
		final String cmd2 = getRequestParameter( cmd );
		if( cmd2 != null && cmd2.length() > 0 ) { command = cmd2.toUpperCase(Locale.JAPAN); }
	}

	/**
	 * 【TAG】メッセージを初期化するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * メッセージは、一般には，エラーメッセージかワーニングです。
	 * 最終処理でメッセージが無ければ，標準でクリアします。
	 * また、command が NEW の場合も、メッセージは自動でクリアされます。
	 * 初期値は、クリアしない (true 以外)です。
	 *
	 * @param	flag  初期化 [true:クリアする/それ以外:しない]
	 */
	public void setClear( final String flag ) {
		msgClear = nval( getRequestParameter( flag ),msgClear );
	}

	/**
	 * 【TAG】表示形式『表題(TITLE)か、内容(BODY)』を指定します(初期値:TITLE)。
	 *
	 * @og.tag
	 * 一般には，表題(TITLE) を表示しておきます。
	 * 表題表示時には，リンクを張り、共通エラー表示画面をオープン
	 * 出来る様になっています。
	 *
	 * @param	type 表示形式 [TITLE:表題/BODY:内容]
	 */
	public void setViewType( final String type ) {
		viewType = nval( getRequestParameter( type ),viewType );
	}

	/**
	 * 【TAG】plsqlUpdate の結果を画面上に表示するメッセージIDを指定します(初期値:MSG0059『登録しました』)。
	 *
	 * @og.tag
	 * 指定したメッセージをリソースから取得して表示します。
	 * 表示させたくない場合は, displayMsg = "MSG0065" をセットしてください。
	 * 初期値は、MSG0059『登録しました。』を表示します。
	 *
	 * @og.rev 2.0.1.0 (2002/10/10) デフォルト表示しないから、MSG0059=登録しました。に変更します。
	 * @og.rev 3.2.0.0 (2003/05/22) 引数に何もセットされないときに、デフォルトの文字を表示するように変更。
	 *
	 * @param	id 処理結果表示メッセージID
	 */
	public void setDisplayMsg( final String id ) {
		displayMsg = nval( getRequestParameter( id ),displayMsg );
	//	String ids = getRequestParameter( id );
	//	if( ids != null ) { displayMsg = ids; }
	}

	/**
	 * 【TAG】登録処理実行後のワーニング結果を画面上に表示するメッセージIDを指定します(初期値:ERR0020)。
	 *
	 * @og.tag
	 * 指定したメッセージをリソースから取得して表示します。
	 * 表示させたくない場合は, warningMsg = "" をセットしてください。
	 * 初期値は、ERR0020『データ登録時にワーニングが発生しました。』を表示します。
	 *
	 * @og.rev 2.0.1.0 (2002/10/10) デフォルト表示しないから、ERR0020=データ登録時にワーニングが発生しました。に変更します。
	 *
	 * @param	id 警告時メッセージID
	 */
	public void setWarningMsg( final String id ) {
		final String ids = getRequestParameter( id );
		if( ids != null ) { warningMsg = ids; }
	}

	/**
	 * デバッグ時の文字列を返します。
	 *
	 * @return	このオブジェクトのデバッグ表現文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
	//			.println( "ERR_MSG_ID"	,ERR_MSG_ID	)
				.println( "errMsgFile"	,errMsgFile	)
				.println( "maxRowCount" ,maxRowCount)
				.println( "command"		,command	)
				.println( "msgClear"	,msgClear	)
				.println( "viewType"	,viewType	)
				.println( "displayMsg"	,displayMsg	)
				.println( "warningMsg"	,warningMsg	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
