/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

/**
 * カラム名配列や、値配列が設定される都度呼び出される リスナーインターフェースです。
 * 
 * これは、org.opengion.hayabusa.io.TableReader 等で、テーブルの読み取り処理を
 * 行う都度呼び出されるリスナーとして使用できます。
 *
 * 想定として、adjustColumns(読取元ファイルのデータ変換を行う) や、
 * checkColumns(読取元ファイルの整合性チェックを行う) 、nullCheck(NULL チェックすべきカラム列)
 * や、DirectTableInsertTag 等で、直接DB登録を行ったりするのを、外部から設定できます。
 *
 * @og.rev 6.2.1.0 (2015/03/13) ColumnActionListener 新規作成
 * @og.group データ処理
 *
 * @version  6.2
 * @author   Kazuhiko Hasegawa
 * @since    JDK8.0,
 */
public interface ColumnActionListener {

	/**
	 * 初期値を設定します。
	 *
	 * これは、通常インナークラスとして直接定義されるため、内部変数を使用して
	 * 処理することになりますが、final定義された変数しか使えません。
	 * そこで、コンストラクタ作成後、実行直前にパラメータを渡すことで、動的な
	 * 処理を行えるようにするための初期値設定です。
	 * Object配列(可変長引数)にしているのは、内部処理するに当たり、どのような
	 * 引数でも渡せるようにしているためで、あまりよろしくない手法です(^^)。
	 *
	 * @og.rev 6.2.1.0 (2015/03/13) ColumnActionListener 新規作成
	 *
	 * @param obj Object配列(可変長引数)
	 */
	default void init( Object... obj ) {
		// interface の初期メソッド
	}

	/**
	 * 一連の作業終了時に呼ばれます。
	 *
	 * これは、#init( Object... ) 設定したり、内部で作成したオブジェクトを
	 * 終了処理させるためのメソッドです。
	 * close 等の処理を行います。
	 * try ～ finally の finally で呼び出せば、必ず終了処理が実施されます。
	 *
	 * @og.rev 6.2.2.0 (2015/03/27) ColumnActionListener 対応。
	 * @og.rev 6.3.6.1 (2015/08/28) 引数は取らない様に変更します。
	 */
	default void end() {
		// interface の初期メソッド
	}

	/**
	 * カラム名の配列が設定された場合に、呼び出されます。
	 *
	 * @og.rev 6.2.1.0 (2015/03/13) ColumnActionListener 新規作成
	 *
	 * @param names カラム名配列
	 */
	void columnNames( final String[] names );

	/**
	 * １行分のデータが設定された場合に、呼び出されます。
	 *
	 * @og.rev 6.2.1.0 (2015/03/13) ColumnActionListener 新規作成
	 *
	 * @param   vals    文字列値の１行分の配列
	 * @param   rowNo   行番号(0～)
	 */
	void values( final String[] vals, final int rowNo );
}
