/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.business;

import org.opengion.fukurou.model.DataModel;					// 6.7.9.1 (2017/05/19) ArrayTableModel を、DataModel I/F に変更
import org.opengion.fukurou.db.Transaction;
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.HybsLoader;
import java.util.Map;											// 6.9.9.0 (2018/08/20)

/**
 * 業務ロジックを実行するためのヘルパークラスです。
 *
 * このクラスの役割は、外部パッケージ(タグクラスなど)からの各種の設定情報を
 * 業務ロジックの実体のクラスにセットすることです。
 *
 * 業務ロジックの実体となるクラス(AbstractBizLogic)では、各種設定情報のセッター
 * メソッドを、サブクラス(各業務ロジックの実装クラス)から隠蔽するため、パッケージ
 * プライベートで定義しています。
 *
 * このため、外部クラスから、直接設定情報をセットすることができないため、このヘルパー
 * クラスを介して、各種情報をセットしています。
 *
 * @og.rev 5.1.1.0 (2009/12/01) 新規作成
 * @og.group 業務ロジック
 *
 * @version 5.0
 * @author Hiroki Nakamura
 * @since JDK1.6,
 */
public class BizLogicHelper {
	private final AbstractBizLogic logic ;		// 5.1.9.0 (2010/08/01)

	/**
	 * コンストラクターを定義しています。
	 *
	 * @param name 業務ロジックのクラス名
	 * @param ldr クラスローダー
	 */
	public BizLogicHelper( final String name, final HybsLoader ldr ) {
		logic = (AbstractBizLogic)ldr.newInstance( name );
	}

	/**
	 * DBのトランザクションオブジェクトを指定します。
	 * 各実装クラスでは、コネクションのcommit,rollbackは行われません。
	 * (全てのDB処理は、1つのトランザクションとして処理されます。)
	 * このため、commit,rollbackは呼び出し元で行う必要があります。
	 * このメソッドは、1度しかセットすることができません。2回以上呼び出しするとエラーになります。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) 新規作成
	 *
	 * @param tran トランザクション
	 */
	public void setTransaction( final Transaction tran ) {
		logic.setTransaction( tran );
	}

	/**
	 * 接続先IDを指定します。
	 * このメソッドは、1度しかセットすることができません。2回以上呼び出しするとエラーになります。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) 新規作成
	 *
	 * @param id 接続先ID
	 */
	public void setDbid( final String id ) {
		logic.setDbid( id );
	}

	/**
	 * 業務ロジックのクラスをロードするためのクラスローダーをセットします。
	 * このメソッドは、1度しかセットすることができません。2回以上呼び出しするとエラーになります。
	 *
	 * @param ldr クラスローダー
	 */
	public void setLoader( final HybsLoader ldr ) {
		logic.setLoader( ldr );
	}

	/**
	 * 配列型テーブルモデルをセットします。
	 * このメソッドは、1度しかセットすることができません。2回以上呼び出しするとエラーになります。
	 *
	 * @og.rev 6.7.9.1 (2017/05/19) ArrayTableModel を、DataModel I/F に変更
	 *
	 * @param tbl 配列型テーブルモデル
	 */
	public void setTable( final DataModel<String> tbl ) {
		logic.setTable( tbl );
	}

	/**
	 * 固定値のキー配列を指定します。
	 * このメソッドは、1度しかセットすることができません。2回以上呼び出しするとエラーになります。
	 *
	 * @param ks キー配列(可変長引数)
	 */
	public void setKeys( final String... ks ) {
		logic.setKeys( ks );
	}

	/**
	 * 固定値の値配列を指定します。
	 * このメソッドは、1度しかセットすることができません。2回以上呼び出しするとエラーになります。
	 *
	 * @param vs 値配列(可変長引数)
	 */
	public void setVals( final String... vs ) {
		logic.setVals( vs );
	}

	/**
	 * この処理の実行ユーザーIDを指定します。
	 *
	 * @param id 実行ユーザーID
	 */
	public void setUserId( final String id ) {
		logic.setUserId( id );
	}

	/**
	 * 親(呼び出し)PGIDを指定します。
	 *
	 * @param id 親PGID
	 */
	public void setParentPgId( final String id ) {
		logic.setParentPgId( id );
	}

	/**
	 * デバッグモードにします。
	 */
	public void setDebug() {
		logic.setDebug();
	}

	/**
	 * デバッグメッセージを取得します。
	 *
	 * @return	デバッグメッセージ
	 */
	public String getDebugMsg() {
		return logic.getDebugMsg();
	}

	/**
	 * 処理を実行します。
	 * 処理の方法は、main()メソッドにより定義されます。
	 * 実装クラスで発生した全ての例外は、Throwableオブジェクトとしてスローされます。
	 * 呼び出し元では、例外を確実にcatchして、commit,rollbackを行ってください。
	 *
	 * @return	処理が成功したかどうか
	 * @throws Throwable 実行時の全エラーを上位に転送します。
	 */
	public boolean exec() throws Throwable {
		return logic.exec();
	}

	/**
	 * 結果ステータスを返します。
	 *
	 * @return	結果ステータス
	 */
	public int getKekka() {
		return logic.getKekka();
	}

	/**
	 * エラーメッセージオブジェクトを返します。
	 *
	 * @return	エラーメッセージ
	 */
	public ErrorMessage getErrMsg() {
		return logic.getErrMsg();
	}

	/**
	 * 業務ロジックの戻り値を返します。
	 *
	 * @return	戻り値
	 */
	public String getReturn() {
		return logic.getReturn();
	}

	/**
	 * 変数に関連付けた値を、返します。
	 * これは、BizLogicから、呼び出し元のJSPに、RETURN 変数以外の {&#064;XXXX} パラメータを返します。
	 * 既存のアトリビュートがあれば、上書きされます。
	 *
	 * @og.rev 6.9.9.0 (2018/08/20) 戻り値を返せるようにします。
	 *
	 * @return	変数に関連付けた値の内部マップオブジェクト
	 */
	public Map<String,String> getReturnMap() {
		return logic.getReturnMap();
	}

	/**
	 * このクラスは、テーブルモデルが外部から指定されている必要はありません。
	 *
	 * @see AbstractBizLogic#isRequireTable()
	 *
	 * @return	テーブルモデルが外部からセットされる必要があるかどうか
	 */
	public boolean isRequireTable() {
		return logic.isRequireTable();
	}
}
