/*
 * Copyright (c) 2017 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.fileexec;

import java.nio.file.Path;
import java.util.List ;											// 
import java.util.ArrayList ;									// 
import java.util.Arrays ;										// 

import static org.opengion.fukurou.fileexec.AppliExec.GE72.*;		// enum のショートカット

/**
 * RunExec_DBIN は、RunExec インターフェースの実装クラスで、ファイルをデータベースに登録します。
 *
 *<pre>
 *
 * GE72.RUNTYPEが、'1' の場合の処理を行います。
 *
 *</pre>
 *
 * @og.rev 7.0.0.0 (2017/07/07) 新規作成
 *
 * @version  7.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.8,
 */
public class RunExec_DBIN implements RunExec {
	private static final XLogger LOGGER= XLogger.getLogger( RunExec_DBIN.class.getName() );		// ログ出力

	private static final String DEF_ENCODE = "Windows-31J" ;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.9.7.0 (2018/05/14) PMD Each class should declare at least one constructor
	 */
	public RunExec_DBIN() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * 実際に処理を実行するプログラムのメソッド。
	 *
	 * @og.rev 6.8.1.5 (2017/09/08) LOGGER.debug 情報の追加
	 * @og.rev 6.9.7.0 (2018/05/14) PMD encode,clms72,skipCnt unreferenced before a possible exit point.
	 *
	 * @param	path 処理するファイルパス
	 * @param	ge72Data GE72 テーブルデータ
	 * @return	処理件数(正は成功、マイナスは異常時の行番号)
	 */
	public int exec( final Path path , final String[] ge72Data ) {
		LOGGER.debug( () -> "⑧ Path=" + path + " , GE72Data=" + Arrays.toString( ge72Data ) );

		// 6.9.7.0 (2018/05/14) PMD encode,clms72,skipCnt unreferenced before a possible exit point.
//		final String encode	= StringUtil.nval( ge72Data[FILE_ENC.NO] , DEF_ENCODE );	// UTF-8 , Windows-31J;
		final String table	= ge72Data[TABLE_NAME.NO];
//		final String clms72	= ge72Data[CLMS.NO];			// CLMS (#NAMEの設定)
//		final String params	= ge72Data[PARAMS.NO];			// 固定値の設定
//		final int    skipCnt= StringUtil.nval( ge72Data[SKIP_CNT.NO] , 0 );

		if( table == null || table.isEmpty() ) {
			// MSG2003 = DBINでは、tableは、必須です。
			throw MsgUtil.throwException( "MSG2003" );
		}

		final String encode	= StringUtil.nval( ge72Data[FILE_ENC.NO] , DEF_ENCODE );	// UTF-8 , Windows-31J;
		final String clms72	= ge72Data[CLMS.NO];			// CLMS (#NAMEの設定)

		// 一旦すべてのデータを読み取ります。よって、大きなファイルには向きません。
		final List<List<String>> dataList = new ArrayList<>();		// ファイルを読み取った行データごとの分割されたデータ
		final LineSplitter split = new LineSplitter( encode , clms72 );
		split.forEach( path , line -> dataList.add( line ) );		// １行ごとに、カラムを分割されたListオブジェクト

		final String[] clms = split.getColumns();					// ファイルの#NAME から、カラム列を取り出します。
		if( clms == null || clms.length == 0 ) {
			// MSG2004 = DBINでは、カラム列は、必須です。
			throw MsgUtil.throwException( "MSG2004" );
		}

		final String INS_QUERY = DBUtil.getInsertSQL( table,clms,null,null );

		final int skipCnt = StringUtil.nval( ge72Data[SKIP_CNT.NO] , 0 );
		final List<String[]> dbData = new ArrayList<>();
		if( !dataList.isEmpty() ) {
			for( int i=skipCnt; i<dataList.size(); i++ ) {			// skipCntの行から取り込む
				final List<String> line = dataList.get(i);
				dbData.add( line.toArray( new String[line.size()] ) );
			}
		}

		return DBUtil.execute( INS_QUERY , dbData );
	}
}
