/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.db.AbstractRenderer;
import org.opengion.hayabusa.db.CellRenderer;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.fukurou.util.StringUtil ;						// 6.2.2.0 (2015/03/27)

/**
 * SLABEL レンデラーは、桁数の長いデータをコンパクトに表示させる
 * LABEL レンデラーの類似クラスです。
 *
 * 全角２Byte / 半角および半角カタカナを １Byte で簡易計算し、指定の
 * 桁数でカットします。
 * 初期値は、20Byteで、桁数は、表示パラメータ(RENDERER_PARAM)で指定します。
 * 文字をカットした場合は、最後に『...』を追加し、カット前の文字を title 属性に
 * 設定することで、マウスをカット後の文字に載せると、カット前の値がチップ表示
 * されます。
 * &lt;span title="カット前の値"&gt;カット文字...&lt;/span&gt;
 * カットされなかった場合は、元の文字がそのまま表示されます。
 *
 *  カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、表示パラメータになにも指定しない(デフォルト)場合は、
 * すべて同一のオブジェクトを返します。それ以外は、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.rev 3.5.6.2 (2004/07/05) 新規作成
 * @og.group データ表示
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Renderer_SLABEL extends AbstractRenderer {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.2.2.3 (2015/04/10)" ;

	private static final CellRenderer DB_CELL = new Renderer_SLABEL() ;	// 20Byteでカット
	private final int cutSize;

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 */
	public Renderer_SLABEL() {
		super();		// 6.4.1.1 (2016/01/16) PMD refactoring. It is a good practice to call super() in a constructor
		cutSize = 20;
	}

	/**
	 * デフォルトコンストラクター。
	 *
	 * @param	clm	DBColumnオブジェクト
	 */
	private Renderer_SLABEL( final DBColumn clm ) {
		super();		// 6.4.1.1 (2016/01/16) PMD refactoring. It is a good practice to call super() in a constructor
		final String param = clm.getRendererParam();
		cutSize = Integer.parseInt( param );
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellRendererオブジェクト
	 * @og.rtnNotNull
	 */
	public CellRenderer newInstance( final DBColumn clm ) {
		final String param = clm.getRendererParam();

		// 6.4.1.1 (2016/01/16) PMD refactoring. A method should have only one exit point, and that should be the last statement in the method
		// 反転させたので注意
		return param == null || param.isEmpty() ? DB_CELL : new Renderer_SLABEL( clm );

	}

	/**
	 * データの表示用文字列を返します。
	 *
	 * 全角２Byte / 半角および半角カタカナを １Byte で簡易計算し、指定の
	 * 桁数でカットします。
	 * 初期値は、20Byteで、桁数は、表示パラメータ(RENDERER_PARAM)で指定します。
	 *
	 * @og.rev 6.2.2.0 (2015/03/27) BRと\nを相互に変換する処理を追加
	 * @og.rev 6.2.2.3 (2015/04/10) htmlフィルターに、BR→改行処理機能を追加。
	 *
	 * @param	value 入力値
	 *
	 * @return	データの表示用文字列
	 */
	@Override
	public String getValue( final String value ) {

		// 簡易的処理。すべてが全角であっても、制限以内である。
		final int len      = value.length();
		if( len*2 <= cutSize ) { return value; }

		int byteSize = 0;
		int adrs;
		for( adrs=0; adrs<len && byteSize<cutSize ; adrs++ ) {
			final char ch = value.charAt(adrs);
			if( ch <= 0x7f || ( ch >= 0xff61 && ch <= 0xff9f ) ) {
				byteSize ++;
			}
			else {
				byteSize +=2;
			}
		}

		// 正確にカウントした結果、制限以内であったため。
		if( adrs==len && byteSize<=cutSize ) {
			return value;
		}
		else if( byteSize>cutSize ) {		// オーバーした場合
			adrs-- ;
		}

		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
			.append( "<span title=\"" )
			.append( StringUtil.htmlFilter( value,true ) )
			.append( "\">" )
			.append( value.substring( 0,adrs ) )	// 切り出し
			.append( "...</span>" );

		return buf.toString();
	}

	/**
	 * データ出力用の文字列を作成します。
	 * ファイル等に出力する形式を想定しますので、HTMLタグを含まない
	 * データを返します。
	 * SLABEL ですが、出力時はカットしません。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) データ出力用のレンデラー
	 *
	 * @param   value 入力値
	 *
	 * @return  データ出力用の文字列
	 * @og.rtnNotNull
	 * @see		#getValue( String )
	 */
	@Override
	public String getWriteValue( final String value ) {
		return ( value == null ) ? "" : value;
	}
}
