/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.fukurou.util.StringUtil;
import static org.opengion.fukurou.system.HybsConst.BUFFER_MIDDLE;	// 6.1.0.0 (2014/12/26) refactoring

/**
 * ＤＢカラムの属性チェックに使用されるメソッドを集約した、クラスです。
 *
 * 全変数は、public static final 宣言されており、全メソッドは、public static synchronized 宣言されています。
 *
 * @og.group データ属性
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class DBTypeCheckUtil {
	/** String を Byte[]に変換するコード
	 * 例：) "MS932" , "JISAutoDetect" ,"JIS", "EUC_JP", "MS932", "SJIS" , "Windows-31J" , "Shift_JIS"
	 */
	private static final String CODE = HybsSystem.sys( "DB_ENCODE" );

	// 5.3.9.0 (2011/09/01) 文字数チェック方式の指定
	// 6.1.0.0 (2014/12/26) refactoring
	private static final boolean USE_TEXT_LEN = HybsSystem.sysBool( "DB_USE_TEXT_LENGTH" );

	/**
	 * オブジェクトを作らせない為の、private コンストラクタ
	 */
	private DBTypeCheckUtil() {}

	/**
	 * 文字列に使われている文字の範囲チェックを行います。
	 *
	 * 最小文字から最大文字、および、許可される文字を指定します。
	 * それ以外は、エラーと判定されます。
	 * ここで判定される以外に細かい制限をかけたい場合は、別のチェックと併用してください。
	 *
	 * @og.rev 5.6.0.3 (2012/01/24) 新規追加
	 *
	 * @param	value	元の文字列
	 * @param	minCh	許可される文字の最小値(含む)
	 * @param	maxCh	許可される文字の最大値(含む)
	 *
	 * @return	範囲チェックエラー文字列(正常時は、null)
	 */
	public static String rangeCheck( final String value ,final char minCh ,final char maxCh ) {
		final StringBuilder val = new StringBuilder( BUFFER_MIDDLE );
		boolean isError = false;
		for( int i=0; i<value.length(); i++ ) {
			final char ch = value.charAt( i );
			if( minCh <= ch && ch <= maxCh ) {
				val.append( ch );
			}
			else {
				val.append( "<span class=\"NG\">" ).append( ch ).append( "</span>" );
				isError = true;
			}
		}

		return isError ? val.toString() : null ;
	}

	/**
	 * 文字列の長さ(整数部)をチェックします。
	 *
	 * @param	value 元の文字列
	 * @param	sizeX	 整数部分の文字列の長さ
	 * @param	sizeY	 小数部分の文字列の長さ
	 *
	 * @return	エラー文字列長さ(正常時は、null)
	 */
	public static String sizeXCheck( final String value ,final int sizeX ,final int sizeY ) {
		int valuesizeX;
		final int pos = value.indexOf( '.' );
		if( pos >= 0 ) {
			valuesizeX = pos;
		}
		else {
			valuesizeX = value.length();
		}
		if( value.charAt(0) == '-' ) { valuesizeX--; }

		// 6.4.1.1 (2016/01/16) PMD refactoring. A method should have only one exit point, and that should be the last statement in the method
		return valuesizeX > sizeX
					? String.valueOf(valuesizeX)		// 整数部の長さが指定の長さよりも長いです。
					: null;

	}

	/**
	 * 文字列の長さ(小数部)をチェックします。
	 *
	 * @param	value 元の文字列
	 * @param	sizeX	 整数部分の文字列の長さ
	 * @param	sizeY	 小数部分の文字列の長さ
	 *
	 * @return	エラー文字列長さ(正常時は、null)
	 */
	public static String sizeYCheck( final String value ,final int sizeX ,final int sizeY ) {
		if( sizeY == 0 ) {
			return null;
		}
		int valuesizeY;
		final int pos = value.indexOf( '.' );
		if( pos >= 0 ) {
			valuesizeY = value.length() - pos - 1;
		}
		else {
			valuesizeY = 0;
		}

		if( valuesizeY > sizeY ) {
			// 小数部の長さが指定の長さよりも長いです。
			return String.valueOf(valuesizeY);
		} else {
			return null;
		}
	}

	/**
	 * 文字列の小数点の位置をチェックします。
	 * 小数点(.)が、２箇所以上存在する(存在する位置が異なる)場合エラー
	 *
	 * @param	value 元の文字列
	 *
	 * @return	エラー文字列(正常時は、null)
	 */
	public static String decimalPointCheck( final String value ) {
		String rtn = null;
		if( value.indexOf( '.' ) != value.lastIndexOf( '.' ) ) {
			rtn = changeErrorPath( value, '.' );
		}
		return rtn ;
	}

	/**
	 * 文字列の符号の位置をチェックします。
	 * マイナス(-)が、存在しないか、先頭以外の場合は、エラー
	 *
	 * @param	value 元の文字列
	 *
	 * @return	エラー文字列(正常時は、null)
	 */
	public static String decimalCodeCheck( final String value ) {
		String rtn = null;
		if( value.lastIndexOf( '-' ) > 0 ) {
			rtn = changeErrorPath( value, '-' );
		}
		return rtn ;
	}

	/**
	 * 文字列の整合性(整数)をチェックします。
	 * ０～９およびマイナス(-)を許可します。
	 *
	 * @param	value 元の文字列
	 *
	 * @return	エラー文字列(正常時は、null)
	 */
	public static String numberFormatCheck( final String value ) {
		boolean isError = false;
		int i = 0;
		char ch;
		while( i<value.length() ) {
			ch = value.charAt( i );
			if( ( '0'>ch || '9'<ch ) && '-'!=ch ) {
				isError = true;
				break;
			}
			i++;
		}
		if( isError ) {
			final StringBuilder val = new StringBuilder( BUFFER_MIDDLE );
			for( i=0; i<value.length(); i++ ) {
				ch = value.charAt( i );
				if( ( '0'>ch || '9'<ch ) &&  '-' != ch ) {
					val.append( "<span class=\"NG\">" ).append( ch ).append( "</span>" );
				}
				else {
					val.append( ch );
				}
			}
			return val.toString();
		} else {
			return null;
		}
	}

	/**
	 * 文字列の整合性(小数)をチェックします。
	 * ０～９、マイナス(-)および小数点(.)を許可します。
	 *
	 * og.rev 4.2.4.0 (2008/06/26) '.' or '-' のみはエラー
	 *
	 * @param	value 元の文字列
	 *
	 * @return	エラー文字列(正常時は、null)
	 */
	public static String decimalFormatCheck( final String value ) {
		boolean isError = false;
		int i = 0;
		char ch;
		while( i<value.length() ) {
			ch = value.charAt( i );
			if( ( '0'>ch || '9'<ch ) && '.'!=ch && '-'!=ch ) {
				isError = true;
				break;
			}
			i++;
		}

		// 4.2.4.0 (2008/06/26) '.' or '-' のみはエラー
		if( value.length() ==1 && ( value.charAt(0) == '.' || value.charAt(0) == '-' ) ) {
			isError = true;
		}

		if( isError ) {
			final StringBuilder val = new StringBuilder( BUFFER_MIDDLE );
			for( i=0; i<value.length(); i++ ) {
				ch = value.charAt( i );
				if( ( '0'>ch || '9'<ch ) && '.'!=ch && '-'!=ch ) {
					val.append( "<span class=\"NG\">" ).append( ch ).append( "</span>" );
				}
				else {
					val.append( ch );
				}
			}
			return val.toString();
		} else {
			return null;
		}
	}

	/**
	 * 日付文字列の整合性をチェックします。
	 *
	 * 整合性といっても、DBType_DATE のような厳密なチェックは、行いません。
	 * ここでは、yyyyMM(６桁)、yyyyMMdd(８桁)、yyyyMMddHHmmss(１４桁) の３種類のみ
	 * 対象にします。
	 * "0000XXXX" , "9999XXXX" は、常に許可されます。
	 * 月と日の関係も、ありません。（20130231 は OK）
	 * あくまで、月は、1～12 の範囲、日は、1～31の範囲チェックです。
	 *
	 * 厳密な日付チェックを行いたい場合は、DBType_DATE を使用してください。
	 * 
	 * @og.rev 5.6.0.3 (2012/01/24) 新規追加
	 *
	 * @param	value 元の文字列(nullは不可)
	 *
	 * @return	エラー文字列(正常時は、null)
	 */
	public static String ymdFormatCheck( final String value ) {
		if( value.startsWith( "0000" ) || value.startsWith( "9999" ) ) { return null; }		// 無条件 OK

		final int len = value.length() ;
		if( len >= 6 ) {	// 月のチェック
			final String val = ymdhmsCheck( value,4,6,1,12 );
			if( val != null ) { return val; }
		}

		if( len >= 8 ) {	// 日のチェック
			final String val = ymdhmsCheck( value,6,8,1,31 );
			if( val != null ) { return val; }
		}

		if( len >= 10 ) {	// 時のチェック
			final String val = ymdhmsCheck( value,8,10,0,24 );		// 240000 は許可します。
			if( val != null ) { return val; }
		}

		if( len >= 12 ) {	// 分のチェック
			final String val = ymdhmsCheck( value,10,12,0,60 );		// 60分は許可します。
			if( val != null ) { return val; }
		}

		if( len == 14 ) {	// 秒のチェック
			final String val = ymdhmsCheck( value,12,14,0,60 );		// うるう秒とは言いませんが、60秒は許可します。
			if( val != null ) { return val; }
		}

		return null;
	}

	/**
	 * 時刻文字列の整合性をチェックします。
	 *
	 * 整合性といっても、DBType_DATE のような厳密なチェックは、行いません。
	 * ここでは、HHmmss(６桁) のみ対象にします。
	 * 
	 * @og.rev 5.6.0.3 (2012/01/24) 新規追加
	 *
	 * @param	value 元の文字列(nullは不可)
	 *
	 * @return	エラー文字列(正常時は、null)
	 */
	public static String hmsFormatCheck( final String value ) {
		final int len = value.length() ;

		if( len >= 2 ) {	// 時のチェック
			final String val = ymdhmsCheck( value,0,2,0,24 );		// 240000 は許可します。
			if( val != null ) { return val; }
		}

		if( len >= 4 ) {	// 分のチェック
			final String val = ymdhmsCheck( value,2,4,0,60 );		// 60分は許可します。
			if( val != null ) { return val; }
		}

		if( len == 6 ) {	// 秒のチェック
			final String val = ymdhmsCheck( value,4,6,0,60 );		// うるう秒とは言いませんが、60秒は許可します。
			if( val != null ) { return val; }
		}

		return null;
	}

	/**
	 * 月、日、時、分、秒 のチェック用メソッド
	 *
	 * 同じようなパターンでチェックする為、共通メソッド化しておきます。
	 *
	 * @param	value	元の文字列
	 * @param	st		チェック開始桁数
	 * @param	ed		チェック終了桁数 (val.length() を超えない事)
	 * @param	minSu	許可範囲の最小値
	 * @param	maxSu	許可範囲の最大値
	 *
	 * @return	エラー文字列（正常な場合は、null）
	 */
	public static String ymdhmsCheck( final String value, final int st , final int ed , final int minSu , final int maxSu ) {
		String rtn = null;

		final int dt = Integer.parseInt( value.substring( st,ed ) );	
		if( dt < minSu || maxSu < dt ) {
			rtn = value.substring( 0,st ) + "<span class=\"NG\">" + value.substring( st,ed ) + "</span>" + value.substring( ed ) ;
		}
		return rtn;
	}

	/**
	 * 文字列のエラー文字列を返します。
	 *
	 * @param	val	元の文字列
	 * @param	inChar	エラー対象文字
	 *
	 * @return	エラー文字列
	 * @og.rtnNotNull
	 */
	private static String changeErrorPath( final String val, final char inChar ) {
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
		for( int i=0; i<val.length(); i++ ) {
			final char ch = val.charAt( i );
			if( inChar==ch ) {
				buf.append( "<span class=\"NG\">" ).append( ch ).append( "</span>" );
			} else {
				buf.append( ch );
			}
		}
		return buf.toString();
	}

	/**
	 * 文字列の長さをチェックします。
	 * バイト数に換算して比較チェックします。
	 *
	 * @og.rev 3.0.1.3 (2003/03/11) メソッド新規追加
	 * @og.rev 3.5.5.3 (2004/04/09) StringUtil の CODE を使用したメソッドを削除する。
	 * @og.rev 5.3.9.0 (2011/09/01) DB_USE_TEXT_LENGTH を考慮した、｢文字数｣、｢バイト数｣チェック
	 *
	 * @param	value 元の文字列
	 * @param	len	 文字列の長さ
	 *
	 * @return	エラー文字列(正常時は、null)
	 */
	public static String byteLengthCheck( final String value ,final int len ) {
		String rtn = null;

		// 5.3.9.0 (2011/09/01) ｢文字数｣、｢バイト数｣チェック
		final int valLen ;
		if( USE_TEXT_LEN ) {	// true:｢文字数｣チェック方式
			valLen = value.length();
		}
		else {					// false:｢バイト数｣チェック方式
			final byte[] byteValue = StringUtil.makeByte( value,CODE );	// 3.5.5.3 (2004/04/09)
			valLen = byteValue.length;
		}

		if( valLen > len ) {
			rtn = String.valueOf( valLen );
		}

		return rtn ;
	}

	/**
	 * 文字列の整合性を、dbType パラメータを利用してチェックします。
	 * regex が、null または、長さゼロの文字列の場合は、なにもしません。
	 *
	 * @og.rev 3.6.0.0 (2004/09/22) 新規作成
	 *
	 * @param	value 元の文字列
	 * @param	regex チェックする正規表現文字列
	 *
	 * @return	エラー文字列(正常時は、null)
	 */
	public static String matcheCheck( final String value,final String regex ) {
		// 6.4.1.1 (2016/01/16) PMD refactoring. A method should have only one exit point, and that should be the last statement in the method
		// 条件変更注意
		return regex == null || regex.isEmpty() || value.matches( regex )
					? null
					: "<span class=\"NG\">" + value + "</span> regex=" + regex ;

	}
}
