/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.taglet;

import com.sun.tools.doclets.Taglet;
import com.sun.javadoc.Tag;
import java.util.Map;

import org.opengion.fukurou.util.StringUtil;

/**
 * ソースコメントから、Javadoc を作成する場合のカスタムタグ情報を作成する
 * Taglet インターフェースの実装クラスを作成します。
 * og.formSample タグ(形式サンプル)を処理します。
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
@SuppressWarnings(value={"deprecation"})			// Ver7.0.0.0
public class TagletTag extends AbstractTaglet {

	private static final String NAME   = "og.tag";
	private static final String HEADER = "説明:";

	/**
	 * 実行時にドックレットがタグレットを読み込んで使用するには、
	 * そのタグレットが、次のシグニチャでマップ を引数として受け取る、
	 * レジスタ と呼ばれる static メソッドをもっている必要があります。
	 * このメソッドは、タグレット名をキーとして、カスタムタグレットの
	 * インスタンスをマップに追加します。 タグレットをオーバーライドする場合、
	 * 名前の競合を避けるため、新しいタグレットのインスタンスをマップに
	 * 追加する前に、オーバーライドされる側のタグレットをマップから
	 * 削除する必要があります。
	 *
	 * @param tagletMap タグレットマップ
	 */
	public static void register( final Map<String,Taglet> tagletMap ) {
	   final TagletTag tagTag = new TagletTag();
	   final Taglet tag = tagletMap.get(NAME);
	   if( tag != null ) {
		   tagletMap.remove(NAME);
	   }
	   tagletMap.put(NAME, tagTag);
	}

	/**
	 * このカスタムタグの名前を返します。
	 *
	 * @return カスタムタグの名前
	 * @og.rtnNotNull
	 */
	@Override
	public String getName() {
		return NAME;
	}

	/**
	 * このカスタムタグのタグ表現を受け取り、
	 * 文字列としての表現を返し、生成されたページに出力します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) インラインタグを処理できるように text(String) メソッド経由で出力する。
	 *
	 * @param tagTag このカスタムタグのタグ表現
	 *
	 * @return このタグの文字列としての表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString( final Tag tagTag ) {
		return "<span bgcolor=\"yellow\">"
			   + StringUtil.htmlFilter( text( tagTag ) )		// 6.0.2.0 (2014/09/19) インラインタグを処理
			   + "</span>";
	}

	/**
	 * このカスタムタグのタグ表現の配列を受け取り、
	 * 文字列としての表現を返し、生成されたページに出力します。
	 * このタグレットがインラインタグを表す場合、
	 * このメソッドは null を返します。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) table表記をpre に置き換えます。
	 * @og.rev 6.0.2.0 (2014/09/19) インラインタグを処理できるように text(String) メソッド経由で出力する。
	 *
	 * @param tagTags	このカスタムタグを表すタグの配列
	 *
	 * @return このタグの文字列としての表現
	 * @see		com.sun.tools.doclets.Taglet#toString( Tag[] )
	 */
	@Override
	public String toString( final Tag[] tagTags ) {
		if( tagTags.length == 0 ) {
			return null;			// インラインタグを表す
		}
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
			.append( "<dt><b>" ).append( HEADER ).append( "</b></dt>" )
			.append( "<dd><pre> " );
		for( int i=0; i<tagTags.length; i++ ) {
			buf.append( text(tagTags[i] ) );		// 6.0.2.0 (2014/09/19) インラインタグを処理
		}
		buf.append( "</pre></dd>\n" );
		return buf.toString();
	}
}
