/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.fukurou.system.DateSet;							// 6.4.2.0 (2016/01/29)
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.HybsDateUtil;
import org.opengion.hayabusa.db.DBTypeCheckUtil;

import java.util.Calendar;

/**
 * 文字列の日付属性(年/月/日 時:分:秒)の半角の日付を扱う為の、カラム属性を定義します。
 *
 * yyyyMMddHHmmss に対応している必要があります。
 * ただし、日付の整合性チェックは行いませんが、valueAdd( String value )での
 * 日付の加算時には、正式な日付データにて加算します。
 *
 * タイプチェックとして、以下の条件を判定します。
 * ・文字列長は、直接計算で文字数との比較
 * ・日付使用文字チェック「('0' &gt; c || '9' &lt; c)以外」エラー
 * ・文字パラメータの 正規表現チェック
 *
 * @og.group データ属性
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DBType_YMDH extends DBType_YMD {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.2.0 (2016/01/29)" ;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public DBType_YMDH() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * String引数の文字列を＋１した文字列を返します。
	 * ここでは、yyyyMMddHHmmss 形式のデータに、日を＋１します。
	 * 
	 * 引数が null の場合と、ゼロ文字列("")の場合は,物理的初期設定値(String getDefault())
	 * の値を返します。
	 * それ以外の場合は、日付計算を行い、日を加算します。よって、年月日フォーマットに
	 * 合わないデータを指定した場合は、エラーになりますので、ご注意ください。
	 *
	 * @og.rev 3.6.0.0 (2004/09/17) 日の加算ロジックの誤り修正。(roll ⇒ add)
	 * @og.rev 5.5.7.2 (2012/10/09) HybsDateUtil を利用するように修正します。
	 *
	 * @param	value	String引数の文字列
	 *
	 * @return	String引数の文字列を＋１した文字列
	 */
	@Override
	public String valueAdd( final String value ) {
		if( value == null || value.isEmpty() ) { return getDefault(); }

		final Calendar cal = HybsDateUtil.getCalendar( value );
		cal.add( Calendar.DATE, 1 );		// 日にちを１日加算します。

		return DateSet.getDate( cal.getTimeInMillis() , "yyyyMMddHHmmss" );
	}

	/**
	 * String引数の文字列に、第２引数に指定の文字列(数字、日付等)を加算して返します。
	 *
	 * ここでは、yyyyMMddHHmmss 形式のデータに、引数文の日を加算します。
	 *
	 * この YMDH は、引数に、日付単位を指定できます。単位は、yyyyMMddHHmmss 形式の
	 * １文字を指定します。大文字、小文字も識別します。value="5M" とすれば、５か月、value="5m"
	 * とすれば、５分 追加します。
	 * 指定しない場合は、日を加算します。
	 * 
	 * 引数が null の場合と、ゼロ文字列("")の場合は,物理的初期設定値(String getDefault())
	 * の値を返します。
	 * それ以外の場合は、日付計算を行い、日を加算します。よって、年月日フォーマットに
	 * 合わないデータを指定した場合は、エラーになりますので、ご注意ください。
	 *
	 * @og.rev 5.6.0.3 (2012/01/24) ADD に、引数の値を加算する機能を追加します。
	 * @og.rev 5.6.1.0 (2013/02/01) 加算する引数に、日付単位('y','M','d')を指定可能にします。
	 *
	 * @param   value  String引数
	 * @param   add    加算する日付文字列
	 *
	 * @return  引数の文字列に、日付を加算します
	 */
	@Override
	public String valueAdd( final String value,final String add ) {
		// 6.4.1.1 (2016/01/16) PMD refactoring. A method should have only one exit point, and that should be the last statement in the method
		return value == null || value.isEmpty() ? getDefault() : HybsDateUtil.getDatePlus( value,add,Calendar.DATE,"yyyyMMddHHmmss" );

	}

	/**
	 * エディターで編集されたデータを登録する場合に、データそのものを
	 * 変換して、実登録データを作成します。
	 * このメソッドでは、sizeX と無関係に １４ケタの YYYYMMDDHHMMS 形式の
	 * 日付データにフォーマットに統一します。
	 * 引数が null の場合と、ゼロ文字列("")の場合は,物理的初期設定値(String getDefault())
	 * の値を返します。
	 *
	 * @og.rev 5.5.7.2 (2012/10/09) HybsDateUtil を利用するように修正します。
	 * @og.rev 5.6.6.0 (2013/07/05) 日付の桁数範囲を、8-14 桁とします。
	 *
	 * @param	value	(一般に編集データとして登録されたデータ)
	 * @return  修正後の文字列(一般にデータベースに登録するデータ)
	 */
	@Override
	public String valueSet( final String value ) {
		// 6.4.1.1 (2016/01/16) PMD refactoring. A method should have only one exit point, and that should be the last statement in the method
		return value == null || value.isEmpty() ? getDefault() : HybsDateUtil.parseDate( value,8,14 );

	}

	/**
	 * データが登録可能かどうかをチェックします。
	 * データがエラーの場合は、そのエラー内容を返します。
	 *
	 * @og.rev 5.6.0.3 (2012/01/24) 日付チェックの整合性まで含めたチェックを入れる。
	 *
	 * @param   key		キー
	 * @param   value	値
	 * @param   sizeX	整数部分の文字列の長さ
	 * @param   sizeY	小数部分の文字列の長さ
	 * @param   typeParam	dbType パラメータ(文字パラメータ)
	 * @param   isStrict	厳密にチェックするかどうか[true:する/false:標準的]
	 *
	 * @return  エラー内容
	 * @og.rtnNotNull
	 */
	@Override
	public ErrorMessage valueCheck( final String key ,final String value ,
									final int sizeX ,final int sizeY ,final String typeParam ,final boolean isStrict) {

		final ErrorMessage msg = new ErrorMessage();
		if( value == null || value.isEmpty() ) { return msg; }

		final int len = (sizeY == 0) ? sizeX : sizeX + sizeY + 1;
		// 5.2.2.0 (2010/11/01) 厳密にチェック(isStrict=true)するフラグを追加
		if( isStrict ) {
			if( len != value.length() ) {
				// 文字列の長さが指定の長さと異なります。
				msg.addMessage( 0,ErrorMessage.NG,"ERR0011", key,value, 
										String.valueOf( value.length() ), String.valueOf( len ) );
			}
		}
		else {
			if( len < value.length() ) {
				// 文字列の長さが指定の長さよりも長いです。
				msg.addMessage( 0,ErrorMessage.NG,"ERR0006",key,value,
										String.valueOf( value.length() ),String.valueOf( len ) );
			}
		}

		// 5.6.0.3 (2012/01/24) 文字の範囲チェック
		String check = DBTypeCheckUtil.rangeCheck( value, '0', '9' );
		if( check != null ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,check );
		}

		// 5.6.0.3 (2012/01/24) 日付文字の整合性チェック
		// "0000","9999" で始まる日付は未条件許可。
		check = DBTypeCheckUtil.ymdFormatCheck( value );
		if( check != null ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,check );
		}

		// 3.6.0.0 (2004/09/22) dbType パラメータ(文字パラメータ)を使用したマッチチェック
		check = DBTypeCheckUtil.matcheCheck( value,typeParam );
		if( check != null ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,check );
		}

		return msg;
	}
}
