/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
import static org.opengion.fukurou.util.StringUtil.nval;

/**
 * テーブルレイアウトで column タグ等の存在しない空欄箇所に挿入して、
 * テーブルの行列の個数を合わせるタグです。
 *
 * 通常の空欄は、ラベルと値という２つの項目を別々のカラムに登録するように
 * &lt;td&gt;&lt;/td&gt;タグが、２つ続いた形式をしていますが、一般に、
 * 個数は表示上の都合で任意の個数の&lt;td&gt;タグを使用したいケースが
 * あります。
 * その場合、このSpaceTag を使っておけば,大量の作成済み画面を修正しなくても
 * このクラスを修正するだけで,すべての空欄のデザインがキープできます。
 *
 * @og.formSample
 * ●形式：&lt;og:space /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:space
 *       width              【TAG】全体の大きさ(width 属性)を指定します
 *       colspan            【TAG】td部分の colspan を指定します
 *       rowspan            【TAG】ラベル、フィールド共に rowspan を指定します
 *       labelWidth         【TAG】ラベル部分(前半部分)の大きさ(width 属性)を指定します
 *       valueWidth         【TAG】値部分(後半部分)の大きさ(width 属性)を指定します
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *      通常の、ラベル部、フィールド部が空の td タグを生成します。(tdタグを２つ作成します)
 *      &lt;og:space /&gt;
 *
 *      td タグの大きさを指定することができます。
 *      &lt;og:space
 *         labelWidth = &quot;ラベル部分(前半部分)の大きさ(width 属性)を指定します。&quot;
 *         valueWidth = &quot;値部分(後半部分)の大きさ(width 属性)を指定します。&quot;
 *      /&gt;
 *
 * @og.group 画面部品
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class SpaceTag extends CommonTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.2.0 (2016/01/29)" ;
	private static final long serialVersionUID = 642020160129L ;

	// 6.2.0.0 (2015/02/27) 初期値見直し
	private String labelWidth	= "";
	private String valueWidth	= "";
	private String width		= "";		// 5.1.9.0 (2010/08/01) 追加
	private String colspan		= "";		// 5.1.9.0 (2010/08/01) 追加
	private String rowspan		= "";		// 6.2.0.0 (2015/02/27) rowspan 属性を追加。

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public SpaceTag() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @return	後続処理の指示( SKIP_BODY )
	 */
	@Override
	public int doStartTag() {
		jspPrint( makeTag() );
		return SKIP_BODY ;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 5.1.9.0 (2010/08/01) width , colspan 属性追加
	 * @og.rev 6.2.0.0 (2015/02/27) rowspan 属性を追加。初期値見直し
	 */
	@Override
	protected void release2() {
		super.release2();
		labelWidth	= "";
		valueWidth	= "";
		width		= "";			// 5.1.9.0 (2010/08/01) 追加
		colspan		= "";			// 5.1.9.0 (2010/08/01) 追加
		rowspan		= "";			// 6.2.0.0 (2015/02/27) rowspan 属性を追加
	}

	/**
	 * 検索条件の入力用のテキストフィールドを作成します。
	 *
	 * テーブルタグによりフィールドは分割されます。
	 * 使用時は、テーブルタグを前後に使用して下さい。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) width , colspan 属性追加
	 * @og.rev 6.2.0.0 (2015/02/27) rowspan 属性を追加
	 *
	 * @return  入力用のテキストフィールドタグ
	 * @og.rtnNotNull
	 */
	private String makeTag() {
		final StringBuilder rtn = new StringBuilder( BUFFER_MIDDLE );

		// width が未指定の場合のみ、labelWidth、valueWidth を出力
		if( width.isEmpty() ) {
			rtn.append( "<td" )
				.append( labelWidth )
				.append( rowspan )
				.append( "></td><td" )
				.append( valueWidth )
				.append( colspan )
				.append( rowspan )
				.append( "></td>" );
		}
		// width 指定あり
		else {
			// width が指定されると、強制的に colspan を設定する。
			if( colspan.isEmpty() ) {
				colspan = " colspan=\"2\" ";
			}
			rtn.append( "<td" )				// width が指定された場合のみ、td タグは一つしか出力されない。
				.append( width )
				.append( colspan )
				.append( rowspan )
				.append( "></td>" );
		}
		return rtn.toString() ;
	}

	/**
	 * 【TAG】全体の大きさ(width 属性)を指定します。
	 *
	 * @og.tag
	 * 通常の space タグは、td ２個分のタグを出力しますが、width 属性を
	 * 指定した場合は、colspan="2" を自動セットして、width 属性を設定します。
	 * (colspan属性が未設定の場合。設定されていれば、その値が使用されます。)
	 * width 属性を設定した場合は、labelWidth、valueWidth は無視されます。
	 * 単位(px,% など)は、引数で付けて下さい。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) 新規追加
	 * @og.rev 6.2.0.0 (2015/02/27) 初期値設定方法変更
	 *
	 * @param   wd 幅
	 */
	public void setWidth( final String wd ) {
		width = nval( getRequestParameter( wd ),width );
		if( width.length() > 0 ) {
			width = " width=\"" + width + "\" ";
		}
	}

	/**
	 * 【TAG】フィールド部分の colspan を指定します。
	 *
	 * @og.tag
	 * 何も指定しない場合は、tdタグを２つ作成しますが、colspan を指定すると、
	 * 後ろの td に、ここで指定された、colspan 属性を付与します。
	 * ただし、width 属性を指定した場合だけ特殊で、td を一つだけ出力し、
	 * そこに、colspan 属性を付けます。(併用の場合のみ)
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) 新規追加
	 * @og.rev 6.2.0.0 (2015/02/27) 初期値設定方法変更
	 *
	 * @param   sp フィールド部分のcolspan
	 */
	public void setColspan( final String sp ) {
		// 
		colspan = nval( getRequestParameter( sp ),colspan );
		if( colspan != null && colspan.length() > 0 ) {
			colspan = " colspan=\"" + colspan + "\" ";
		}
	}

	/**
	 * 【TAG】ラベル、フィールド共に rowspan を指定します。
	 *
	 * @og.tag
	 * 通常は、ラベルとフィールドが、別のtdで囲われます。rowspan は、
	 * そのどちらの TDフィールドにも、設定します。
	 * 初期値は、rowspan を付けないです。
	 *
	 * &lt;og:space rowspan="3" /&gt;
	 *
	 * @og.rev 6.2.0.0 (2015/02/27) rowspan 属性を追加
	 *
	 * @param   sp ラベル、フィールド部分のrowspan
	 */
	public void setRowspan( final String sp ) {
		rowspan = nval( getRequestParameter( sp ),rowspan );
		if( rowspan.length() > 0 ) {
			rowspan = " rowspan=\"" + rowspan + "\" ";
		}
	}

	/**
	 * 【TAG】ラベル部分(前半部分)の大きさ(width 属性)を指定します。
	 *
	 * @og.tag
	 * 属性は td タグのラベル部分(前半部分)に、 width 属性で設定されます。
	 * colspan 属性は、このlabelWidth属性と同じ TD にはセットされません。
	 * ただし、通常のwidth 属性を使用した場合は、このlabelWidth属性は無視されます。
	 *
	 * @og.rev 6.2.0.0 (2015/02/27) 初期値設定方法変更
	 *
	 * @param   width 属性
	 */
	public void setLabelWidth( final String width ) {
		labelWidth = nval( getRequestParameter( width ),labelWidth );
		if( labelWidth.length() > 0 ) {
			labelWidth = " width=\"" + labelWidth + "\" ";
		}
	}

	/**
	 * 【TAG】値部分(後半部分)の大きさ(width 属性)を指定します。
	 *
	 * @og.tag
	 * 属性は td タグの width 属性に設定されます。
	 * colspan 属性を同時に指定した場合は、このvalueWidth属性と同じ TD にセットされます。
	 * ただし、通常のwidth 属性を使用した場合は、このvalueWidth属性は無視されます。
	 *
	 * @og.rev 6.2.0.0 (2015/02/27) 初期値設定方法変更
	 *
	 * @param   width 属性
	 */
	public void setValueWidth( final String width ) {
		valueWidth = nval( getRequestParameter( width ),valueWidth );
		if( valueWidth.length() > 0 ) {
			valueWidth = " width=\"" + valueWidth + "\" ";
		}
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION	)
				.println( "width"			,width		)
				.println( "colspan"			,colspan	)
				.println( "rowspan"			,rowspan	)
				.println( "labelWidth"		,labelWidth	)
				.println( "valueWidth"		,valueWidth	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
