/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.hayabusa.db.AbstractDBType;
import org.opengion.hayabusa.db.DBTypeCheckUtil;

/**
 * 半角英数字のみの文字列を扱う為の、カラム属性を定義します。
 *
 * 使用可能文字は、0-9,A-Z,a-z です。(,は含みません)
 * 小数点、カンマ、半角スペース などを含みません。
 *
 * タイプチェックとして、以下の条件を判定します。
 * ・文字列長は、直接計算で文字数との比較
 * ・半角文字チェック「0-9,A-Z,a-z」の範囲
 * ・文字パラメータの 正規表現チェック
 *
 * @og.rev 3.8.0.2 (2005/07/11) 新規作成
 * @og.group データ属性
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DBType_XLU9 extends AbstractDBType {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.2.2.0 (2010/11/01)" ;

	/**
	 * データが登録可能かどうかをチェックします。
	 * データがエラーの場合は、そのエラー内容を返します。
	 *
	 * @og.rev 5.2.2.0 (2010/11/01) 厳密にチェック(isStrict=true)するフラグを追加
	 *
	 * @param   key		キー
	 * @param   value	値
	 * @param   sizeX	整数部分の文字列の長さ
	 * @param   sizeY	少数部分の文字列の長さ
	 * @param   typeParam	dbType パラメータ
	 * @param   isStrict	厳密にチェックするかどうか[true:する/false:標準的]
	 *
	 * @return  エラー内容
	 */
//	public ErrorMessage valueCheck( final String key ,final String value ,
//									final int sizeX ,final int sizeY ,final String param ) {
	@Override
	public ErrorMessage valueCheck( final String key ,final String value ,
									final int sizeX ,final int sizeY ,final String typeParam ,final boolean isStrict) {

		ErrorMessage msg = new ErrorMessage();
		if( value == null || value.length() == 0 ) { return msg; }

		int len = (sizeY == 0) ? sizeX : sizeX + sizeY + 1;
		if( value.length() > len ) {
			// 文字列の長さが指定の長さよりも長いです。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0006", key,value, String.valueOf( value.length() ), String.valueOf( len ) );
		}

		StringBuilder val = new StringBuilder();
		boolean isError = false;
		for( int i=0; i<value.length(); i++ ) {
			char ch = value.charAt( i );
			boolean okFlag = (  (ch >= '0' && ch <= '9' ) || (ch >= 'A' && ch <= 'Z' ) ||
								(ch >= 'a' && ch <= 'z' ) );

			if( okFlag ) {
				val.append( ch );
			}
			else {
				val.append( "<span class=\"NG\">" ).append( ch ).append( "</span>" );
				isError = true;
			}
		}
		if( isError ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,val.toString() );
		}

		// 3.6.0.0 (2004/09/22) dbType パラメータを使用したマッチチェック
		String check = DBTypeCheckUtil.matcheCheck( value,typeParam );
		if( check != null ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,check );
		}

		return msg;
	}
}
