/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.report2;

import java.io.File;
import java.util.Map;
import java.util.HashMap;
import java.util.Locale;

import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;

import com.sun.star.beans.PropertyValue;
import com.sun.star.frame.XComponentLoader;
import com.sun.star.frame.XController;
import com.sun.star.frame.XDispatchHelper;
import com.sun.star.frame.XDispatchProvider;
import com.sun.star.frame.XModel;
import com.sun.star.frame.XStorable;
import com.sun.star.io.IOException;
import com.sun.star.lang.EventObject;
import com.sun.star.lang.IllegalArgumentException;
import com.sun.star.lang.XComponent;
import com.sun.star.uno.Exception;
import com.sun.star.uno.UnoRuntime;
import com.sun.star.util.CloseVetoException;
import com.sun.star.util.XCloseable;
import com.sun.star.view.PrintJobEvent;
import com.sun.star.view.PrintableState;
import com.sun.star.view.XPrintJobBroadcaster;
import com.sun.star.view.XPrintJobListener;
import com.sun.star.view.XPrintable;

/**
 * OpenOfficeを利用して様々な形式のファイルを読み込み、出力・印刷を行うための変換クラスです。
 *
 * 変換を行うことのできる入出力のフォーマット以下の通りです。
 *
 * [対応フォーマット]
 *  入力[Calc(ODS)   ,Excel(XLS)     ] ⇒ 出力[Calc(ODS)   ,Excel(XLS)     ,PDF]
 *  入力[Writer(ODT) ,Word(DOC)      ] ⇒ 出力[Writer(ODT) ,Word(DOC)      ,PDF]
 *  入力[Impress(ODP),PowerPoint(PPT)] ⇒ 出力[Impress(ODP),PowerPoint(PPT),PDF]
 *  入力[ * 上記の全て               ] ⇒ 印刷
 *
 * 変換を行うには、以下の2通りの方法があります。
 * (1)簡易的な変換メソッドを利用する場合
 *   {@link #convert(String, String)}を利用して、変換を行います。
 *   この場合、出力形式は、出力ファイルの拡張子に従って自動的に決定されます。
 *   このため、印刷処理などを行う場合は、(2)の方法で出力して下さい。
 * (2)段階的に書くメソッドを呼び出して変換する場合
 *   オブジェクトを生成した後、{@link #open()}、#(各種変換メソッド)、{@link #clone()}を
 *   順番に呼び出して変換を行います。
 *   この場合、出力形式は、それに対応するメソッドを呼び出ることで決定されます。
 *
 *   また、変換を行うための、各種メソッドは、例外としてThrowableを投げるように定義されています。
 *   このクラスを利用する場合は、このThrowableをcatchし、catch句で、必ず{@link #close( boolean )}に、
 *   "true"(エラー発生時のクローズ処理)を指定して、終了処理を行って下さい。
 *   (これを行わない場合、OpenOfficeの不要なプロセスが残ってしまう可能性があります)
 *
 * また、出力ファイルが既に存在する場合、出力ファイルは一旦削除された後、処理されます。
 * なお、入力ファイルと出力ファイルが同じ場合、何も処理されません。(例外も発行されません)
 *
 * 入力ファイルを、カンマ区切りで複数指定した場合、複数の入力ファイルをマージして出力します。
 * ※1 現状は、ファイルのマージは、入力ファイルがExcelまたはCalcの場合のみ対応しています。
 *
 * @og.group 帳票システム
 *
 * @version  4.0
 * @author   Hiroki.Nakamura
 * @since    JDK1.6
 */
public class DocConverter_OOO {

	final private boolean			isOnline;			// オンライン処理かどうか(オンライン処理の場合、プロセスはファクトリクラス経由で生成されます)
	final private String[]			mergeFile;

	private String					inputName;
	private String					origName;

//	private XComponent				doc;
	private XComponent				xComp;				// 5.1.8.0 (2010/07/01) メソッドと重なる変数名の変更
	private SOfficeProcess			soffice;

	// 入力、出力の拡張子とこれに対応するフィルター名
//	static final private HashMap<String,String> filterMap	= new HashMap<String,String>();
	static final private Map<String,String> filterMap	= new HashMap<String,String>();
	static {
		filterMap.put( "ods_ods", "calc8" );
		filterMap.put( "xls_ods", "calc8" );
		filterMap.put( "ods_xls", "MS Excel 97" );
		filterMap.put( "xls_xls", "MS Excel 97" );
		filterMap.put( "ods_pdf", "calc_pdf_Export" );
		filterMap.put( "xls_pdf", "calc_pdf_Export" );
		filterMap.put( "odt_odt", "writer8" );
		filterMap.put( "doc_odt", "writer8" );
		filterMap.put( "odt_doc", "MS Word 97" );
		filterMap.put( "doc_doc", "MS Word 97" );
		filterMap.put( "odt_pdf", "writer_pdf_Export" );
		filterMap.put( "doc_pdf", "writer_pdf_Export" );
		filterMap.put( "odp_odp", "impress8" );
		filterMap.put( "ppt_odp", "impress8" );
		filterMap.put( "odp_ppt", "MS PowerPoint 97" );
		filterMap.put( "ppt_ppt", "MS PowerPoint 97" );
		filterMap.put( "odp_pdf", "impress_pdf_Export" );
		filterMap.put( "ppt_pdf", "impress_pdf_Export" );
	};

	/**
	 * コンストラクタです。
	 *
	 * #DocConverter(input, true)と同じです。
	 *
	 * @param input ファイル一覧(カンマ区切り)
	 * @see #DocConverter_OOO(String[])
	 */
	public DocConverter_OOO( final String input ) {
		this( StringUtil.csv2Array( input ), true );
	}

	/**
	 * コンストラクタです。
	 *
	 * #DocConverter(input, true)と同じです。
	 *
	 * @param input ファイル一覧(配列)
	 * @see #DocConverter_OOO(String[], boolean)
	 */
	public DocConverter_OOO( final String input[] ) {
		this( input, true );
	}

	/**
	 * コンストラクタです。
	 *
	 * isOnline(isOl)がtrueに指定された場合、soffice.binのプロセスをファクトリークラス経由で生成し、
	 * キャッシュします。
	 * 但し、システムリソースが読み込まれないような、バッチファイルから起動した場合は、この方法は
	 * 利用できないため、isOnlineをfalseに指定する必要があります。
	 *
	 * @param input ファイル一覧(配列)
	 * @param isOl オンライン(Web環境での使用)かどうか
	 */
	public DocConverter_OOO( final String input[], final boolean isOl ) {
		if( input == null || input.length == 0 || input[0].length() == 0 ) {
			throw new HybsSystemException( "入力ファイルが指定されていません。" );
		}
		File inFile = new File( input[0] );
		if( !inFile.exists() ) {
			throw new HybsSystemException( "入力ファイルが存在しません。[file=" + input[0] + "]" );
		}
		isOnline = isOl;
		inputName = input[0];
		origName = input[0];

		if( input.length == 1 ) {
			mergeFile = null;
		}
		else {
			if( !"xls".equals( getSuffix( input[0] ) ) && !"ods".equals( getSuffix( input[0] ) ) ) {
				throw new HybsSystemException( "ファイルのマージを行う場合、入力ファイルは、ExcelまたはCacl形式である必要があります。" );
			}

			mergeFile = new String[input.length-1];
			for( int i=0; i<mergeFile.length; i++ ) {
				if( input[i+1].length() == 0 || !( new File( input[i+1] ) ).exists() ) {
					throw new HybsSystemException( "マージファイルが指定されていないか、または存在しません。[file=" + input[i+1] + "]" );
				}
				if( inputName.equals( input[i] + 1 ) ) {
					throw new HybsSystemException( "マージファイルに入力ファイルと同じファイルが指定されてます。[file=" + input[i+1] + "]" );
				}
				if( !"xls".equals( getSuffix( input[i+1] ) ) && !"ods".equals( getSuffix( input[i+1] ) ) ) {
					throw new HybsSystemException( "ファイルのマージを行う場合、マージファイルは、ExcelまたはCacl形式である必要があります。" );
				}
				mergeFile[i] = input[i+1];
			}
		}
	}

	/**
	 * SOficeのコンポーネントを起動します。
	 *
	 * 正常に処理が終了した場合は、#close()を発行し、終了処理を行って下さい。
	 * また、例外が発生した場合は、#close(true)を発行し、終了処理を行って下さい。
	 *
	 * @og.rev 5.1.7.0 (2010/06/01) マージ処理対応
	 *
	 * @throws Throwable 何らかのエラーが発生した場合。
	 * @see #close()
	 * @see #close(boolean)
	 */
	public void open() throws Throwable {
		// プロセスの取得
		if( soffice == null ) {
			if( isOnline ) {
				soffice = ProcessFactory.newInstance();
			}
			else {
				soffice = new SOfficeProcess( "docconverter.class" );
				soffice.bootstrap();
			}

			// マージする場合は、マージ対象のファイルをテンポラリにコピーする(readOnly回避のため)
			// テンプレート(無題)として上げると、シートコピー先として特定できなくなるため
			if( mergeFile != null ) {
				File origFile = new File( origName );
				inputName = soffice.getTempPath() + System.currentTimeMillis() + "_" + origFile.getName();
				FileUtil.copy( origFile, new File( inputName ) );
			}
		}

//		PropertyValue[] calcProps = new PropertyValue[1];
//		calcProps[0] = new PropertyValue();
//		calcProps[0].Name = "Hidden";
//		calcProps[0].Value = true;
//
//		String url = "file:///" + inputName.replace( '\\', '/' );
//
//		@SuppressWarnings("cast")	// OpenOffice 3.2 での冗長なキャスト警告の抑止。キャストをはずすと、旧3.1 では、エラーになる。
//		XComponentLoader cloader = (XComponentLoader) UnoRuntime.queryInterface( XComponentLoader.class, soffice.getDesktop() );
//		try {
//			doc = cloader.loadComponentFromURL( url, "_default", 0, calcProps );
//		}
//		catch( IOException ex ) {
//			throw new HybsSystemException( "OpenOfficeの立ち上げ時にエラーが発生しました(入出力エラー)。", ex );
//		}
//		catch( IllegalArgumentException ex ) {
//			throw new HybsSystemException( "OpenOfficeの立ち上げ時にエラーが発生しました(パラメーター不正)。", ex );
//		}

		// 5.1.7.0 (2010/06/01) マージ処理対応
		xComp = getComponent( inputName, ( mergeFile == null ? true : false ), false );

		if( mergeFile != null ) {
			for( int i=0; i<mergeFile.length; i++ ) {
				merge( mergeFile[i] );
			}
		}
	}

	/**
	 * ドキュメントコンポーネントを取得します。
	 *
	 * @param	input			ファイル名
	 * @param	isHidden		隠し属性[true/false]
	 * @param	isAsTemplate	OpenOffice上のTemplate属性[true/false]
	 *
	 * @return	ドキュメントコンポーネント
	 */
	@SuppressWarnings("cast")	// OpenOffice 3.2 での冗長なキャスト警告の抑止。キャストをはずすと、旧3.1 では、エラーになる。
	private XComponent getComponent( final String input, final boolean isHidden, final boolean isAsTemplate ) {
		PropertyValue[] calcProps = new PropertyValue[2];
		calcProps[0] = new PropertyValue();
		calcProps[0].Name = "Hidden";
		calcProps[0].Value = isHidden;
		calcProps[1] = new PropertyValue();
		calcProps[1].Name = "AsTemplate";
		calcProps[1].Value = isAsTemplate;

		String url = "file:///" + input.replace( '\\', '/' );

		XComponent rtnDoc;
		XComponentLoader cloader = (XComponentLoader) UnoRuntime.queryInterface( XComponentLoader.class, soffice.getDesktop() );
		try {
			rtnDoc = cloader.loadComponentFromURL( url, "_blank", 0, calcProps );
		}
		catch( IOException ex ) {
			throw new HybsSystemException( "OpenOfficeの立ち上げ時にエラーが発生しました(入出力エラー)。", ex );
		}
		catch( IllegalArgumentException ex ) {
			throw new HybsSystemException( "OpenOfficeの立ち上げ時にエラーが発生しました(パラメーター不正)。", ex );
		}
		return rtnDoc;
	}

	/**
	 * ドキュメント(xls,ods)のマージを行います。
	 *
	 * @param mergeInputName マージ対象のファイル名
	 */
	@SuppressWarnings("cast")	// OpenOffice 3.2 での冗長なキャスト警告の抑止。キャストをはずすと、旧3.1 では、エラーになる。
	private void merge( final String mergeInputName ) {
		// マージする副ファイルは、テンプレート(無題)として上げる(readOnly回避のため)
		XComponent subDoc = getComponent( mergeInputName, false, true );

		XDispatchProvider dispatchProvider
			= (XDispatchProvider)UnoRuntime.queryInterface( XDispatchProvider.class
				,((XController)UnoRuntime.queryInterface( XController.class
					,((XModel)UnoRuntime.queryInterface( XModel.class
						,subDoc
					)).getCurrentController()
				)).getFrame()
			);

		XDispatchHelper xDispatchHelper = soffice.getDispatcher();
		xDispatchHelper.executeDispatch(dispatchProvider, ".uno:TableSelectAll", "", 0, new PropertyValue[0]);

		String title = ( new File( inputName ).getName() );
		title = ( title ).substring( 0, title.indexOf( '.' ) );

		PropertyValue[] moveProps = new PropertyValue[3];
		moveProps[0] = new PropertyValue();
		moveProps[0].Name = "DocName";
		moveProps[0].Value = title;
		moveProps[1] = new PropertyValue();
		moveProps[1].Name = "Index";
		moveProps[1].Value = 32767;
		moveProps[2] = new PropertyValue();
		moveProps[2].Name = "Copy";
		moveProps[2].Value = true;
		xDispatchHelper.executeDispatch(dispatchProvider, ".uno:Move", "", 0, moveProps);

		// シートリンク方式の場合は、CalcをHiddenの状態で実行できるが、画像などのオブジェクトがリンクされないため、
		// 採用見送り。。
//		XSpreadsheets mainSp = ( (XSpreadsheetDocument)UnoRuntime.queryInterface( XSpreadsheetDocument.class, xComp ) ).getSheets();
//		String[] mainSheetNames = mainSp.getElementNames();
//
//		XSpreadsheets subSp = ( (XSpreadsheetDocument)UnoRuntime.queryInterface( XSpreadsheetDocument.class, subDoc ) ).getSheets();
//		String[] subSheetNames = subSp.getElementNames();
//		for( int i=0; i<1; i++ ) {
//			XSpreadsheet newSp = insertSheet( mainSp, subSheetNames[i], 0, (short)(mainSheetNames.length + i) );
//			XSheetLinkable mainLink = (XSheetLinkable) UnoRuntime.queryInterface( XSheetLinkable.class, newSp );
//			mainLink.link( "file:///" + mergeInputName.replace( '\\', '/' ), subSheetNames[i], "", "", com.sun.star.sheet.SheetLinkMode.NORMAL );
//			mainLink.setLinkMode( com.sun.star.sheet.SheetLinkMode.NONE );
//		}

		closeComponent( subDoc );
	}

//	/**
//	 * スプレッドシートオブジェクトに対して、シートを新規に追加します。
//	 * 追加するシート名が既に存在する場合、シート名のサフィックスとして1,2...の連番を付加します。
//	 *
//	 * @param sheets
//	 * @param sheetName
//	 * @param suffix
//	 * @param index
//	 * @return 追加シートのシートオブジェクト
//	 */
//	private XSpreadsheet insertSheet( final XSpreadsheets sheets, final String sheetName, final int suffix, final short index ) {
//		String sn = sheetName + ( suffix == 0 ? "" : String.valueOf( suffix ) );
//		try {
//			sheets.insertNewByName( sn, index );
//		}
//		catch ( com.sun.star.uno.RuntimeException ex ) {
//			if( suffix < 256 ) {
//				return insertSheet( sheets, sheetName, suffix+1, index );
//			}
//			else {
//				throw new HybsSystemException( "シートの追加に失敗しました", ex );
//			}
//		}
//
//		XSpreadsheet insSheet = null;
//		try {
//			insSheet = (XSpreadsheet)UnoRuntime.queryInterface( XSpreadsheet.class, sheets.getByName( sn ) );
//		}
//		catch( NoSuchElementException ex ) {
//			throw new HybsSystemException( "追加したシートにアクセスできません。", ex );
//		}
//		catch( WrappedTargetException ex ) {
//			throw new HybsSystemException( "追加したシートにアクセスできません。", ex );
//		}
//		return insSheet;
//	}

	/**
	 * Calcコンポーネントをクローズします。
	 *
	 * このクローズ処理は、処理が正常終了した場合に呼び出しする必要があります。
	 * 例外が発生した場合は、#close(true)を発行し、終了処理を行って下さい。
	 *
	 * このメソッドは#close(false)と同じです。
	 *
	 * @throws Throwable 何らかのエラーが発生した場合。
	 * @see #close(boolean)
	 */
	public void close() throws Throwable  {
		close( false );
	}

	/**
	 * Calcコンポーネントをクローズします。
	 *
	 * 引数のisErrがtrueの場合、この変換オブジェクトで生成されたプロセスは強制的に破棄されます。
	 * falseの場合は、プロセスは、ファクトリクラスを経由して、キャッシュに戻されます。
	 * (バッチ処理の場合は、いずれの場合も、プロセスは強制的に破棄されます)
	 *
	 * 起動から変換、クローズまでの書く処理で例外が発生した場合は、#close(true)を発行し、終了処理を行って下さい。
	 *
	 * #close(false)は#close()と同じであるため、通常利用することはありません。
	 *
	 * @og.rev 4.2.4.1 (2008/07/07 ) 終了処理を60回で終わるように修正
	 * @og.rev 4.3.0.0 (2008/07/15 ) ↑は6秒しか待っていなかったので、60秒待つように修正
	 *
	 * @param	isErr	trueの場合、この変換オブジェクトで生成されたプロセスは強制的に破棄されます。
	 */
	@SuppressWarnings("cast")	// OpenOffice 3.2 での冗長なキャスト警告の抑止。キャストをはずすと、旧3.1 では、エラーになる。
	public void close( final boolean isErr ) {
		if( xComp != null ) {
			closeComponent( xComp );
		}

		if( soffice != null ) {
			if( isOnline ) {
				if( isErr ) {
					ProcessFactory.remove( soffice );
				}
				else {
					ProcessFactory.release( soffice );
				}
			}
			else {
				soffice.close();
			}
		}

		// マージした場合は、テンポラリにコピーしたファイルを削除
		if( mergeFile != null ) {
			if( ! ( new File( inputName ) ).delete() ) {
				System.err.println( "テンポラリにコピーしたファイルを削除できませんでした。[" + inputName + "]" );
			}
		}
	}

	/**
	 * ドキュメントコンポーネントをクローズします。
	 *
	 * @param comp ドキュメントコンポーネント
	 */
	@SuppressWarnings("cast")	// OpenOffice 3.2 での冗長なキャスト警告の抑止。キャストをはずすと、旧3.1 では、エラーになる。
	private void closeComponent( final XComponent comp ) {
		XCloseable closeable = null;
		for( int i = 0;; ++i ) {
			try {
				closeable = (XCloseable) UnoRuntime.queryInterface( XCloseable.class, comp );
				closeable.close( true );
				break;
			}
			catch( CloseVetoException ex ) {
				// 4.2.4.1 (2008/07/07 )
				// 4.3.4.4 (2009/01/01)
				if( i == 600 ) { throw new HybsSystemException( "sofficeプロセスに接続できません。", ex ); }
				try {
					Thread.sleep( 100 );
				}
				catch( InterruptedException ex2 ) {
//					throw new HybsSystemException( ex2 );
				}
			}
		}
	}

	/**
	 * 印刷を行います。
	 *
	 * 正常に処理が終了した場合は、#close()を発行し、終了処理を行って下さい。
	 * また、例外が発生した場合は、#close(true)を発行し、終了処理を行って下さい。
	 *
	 * @og.rev 4.3.0.0 (2008/07/16) スプールが終わるまでwaitし、さらにプリンタ発行の状況を監視し、正常終了かどうかを判断
	 * @og.rev 4.3.7.3 (2009/06/22) 存在しないプリンターを指定した場合のエラーハンドリングを追加
	 * @og.rev 5.1.2.0 (2010/01/01) CentOS等は、OS_INFOがLinux UNKNOWNとなるため、判定条件を変更
	 *
	 * @param	printer	プリンター名
	 * @throws Throwable 何らかのエラーが発生した場合。
	 */
	public void print( final String printer ) throws Throwable {
		if( xComp == null ) { throw new HybsSystemException( "初めに、#open()を実行して下さい" ); }

		if( printer == null || printer.length() == 0 ) {
			throw new HybsSystemException( "プリンターが指定されていません。" );
		}

		@SuppressWarnings("cast")	// OpenOffice 3.2 での冗長なキャスト警告の抑止。キャストをはずすと、旧3.1 では、エラーになる。
		XPrintable xprintable = (XPrintable) UnoRuntime.queryInterface( XPrintable.class, xComp );
		@SuppressWarnings("cast")	// OpenOffice 3.2 での冗長なキャスト警告の抑止。キャストをはずすと、旧3.1 では、エラーになる。
		XPrintJobBroadcaster selection = (XPrintJobBroadcaster) UnoRuntime.queryInterface(XPrintJobBroadcaster.class, xprintable);
		MyPrintJobListener listener = new MyPrintJobListener ();
		selection.addPrintJobListener( listener );

		PropertyValue[] tmpProps = new PropertyValue[1];
		tmpProps[0] = new PropertyValue();
		tmpProps[0].Name = "Name";
		// 5.1.2.0 (2010/01/01) CentOS等は、OS_INFOがLinux UNKNOWNとなるため、判定条件を変更
		// OSがLinuxの場合は、プリンタ名称の前後に"<",">"を付加
//		tmpProps[0].Value = "Linux".equals( HybsSystem.sys( "OS_INFO" ) ) ? "<" + printer + ">" : printer;
		tmpProps[0].Value = "LINUX".indexOf( HybsSystem.sys( "OS_INFO" ).toUpperCase( Locale.JAPAN ) ) >= 0 ? "<" + printer + ">" : printer;

		// 4.3.4.4 (2009/01/01)
		try {
			xprintable.setPrinter( tmpProps );
		}
		catch ( IllegalArgumentException ex ) {
			throw new HybsSystemException( "印刷時にエラーが発生しました。", ex );
		}

		// 4.3.7.3 (2009/06/22) 存在しないプリンタを指定した場合は、PropertyValueに
		// デフォルトプリンターが入るため、引数の値と合致しているかで正しく設定されたかを確認
		String curPrinter = null;
		PropertyValue[] chkProps = xprintable.getPrinter();
		for( int i=0; i<chkProps.length; i++ ) {
			if( "Name".equals( chkProps[i].Name) ) {
				curPrinter = (String)chkProps[i].Value;
				break;
			}
		}
		if( !(printer.equalsIgnoreCase( curPrinter ) ) ) {
			String errMsg = "プリンター[" + printer + "]を発行先に指定できませんでした。" + HybsSystem.CR
							+ "存在しないプリンタ名が指定されている可能性があります。";
			throw new HybsSystemException( errMsg );
		}

		// 4.3.0.0 (2008/07/16)
		PropertyValue[] printProps = new PropertyValue[1];
		printProps[0] = new PropertyValue();
		printProps[0].Name = "Wait";
		printProps[0].Value = true;

		// 4.3.4.4 (2009/01/01)
		try {
			xprintable.print( printProps );
		}
		catch ( IllegalArgumentException ex ) {
			throw new HybsSystemException( "印刷時にエラーが発生しました。", ex );
		}

		// 4.3.0.0 (2008/07/16)
		if( listener.getStatus() == null
				|| ( listener.getStatus() != PrintableState.JOB_COMPLETED && listener.getStatus() != PrintableState.JOB_SPOOLED ) ){
			throw new HybsSystemException ( "Error Occured while spooling print job. Check Spooler-Service!!!");
		}
	}

	/**
	 * プリンタジョブの状況を監視するリスナーです。
	 *
	 * @author Hiroki.Nakamura
	 */
	private static class MyPrintJobListener implements XPrintJobListener {
		private PrintableState status = null;

		@Override
		public void printJobEvent( final PrintJobEvent event ) {
			status = event.State;
		}

		@Override
		public void disposing( final EventObject event ) {
			// 何もありません。(PMD エラー回避)
		}

		public PrintableState getStatus() {
			return status;
		}
	}

	/**
	 * PDF出力を行います。
	 *
	 * 入力形式で未対応の場合(形式は入力ファイルの拡張子で判別)、例外が発行されます。
	 *
	 * 正常に処理が終了した場合は、#close()を発行し、終了処理を行って下さい。
	 * また、例外が発生した場合は、#close(true)を発行し、終了処理を行って下さい。
	 *
	 * @param	outputName	出力ファイル名
	 * @param	pdfPasswd	PDFパスワード
	 * @throws Throwable 何らかのエラーが発生した場合。
	 */
	public void pdf( final String outputName, final String pdfPasswd ) throws Throwable  {
		savePdf( outputName, getFilterName( getSuffix( inputName ), "pdf" ), pdfPasswd );
	}

	/**
	 * Calc(ods)出力を行います。
	 *
	 * 入力形式で未対応の場合(形式は入力ファイルの拡張子で判別)、例外が発行されます。
	 *
	 * 正常に処理が終了した場合は、#close()を発行し、終了処理を行って下さい。
	 * また、例外が発生した場合は、#close(true)を発行し、終了処理を行って下さい。
	 *
	 * @param	outputName	出力ファイル名
	 * @throws Throwable 何らかのエラーが発生した場合。
	 */
	public void ods( final String outputName ) throws Throwable  {
		saveDoc( outputName, getFilterName( getSuffix( inputName ), "ods" ) );
	}

	/**
	 * Excel(xls)出力を行います。
	 *
	 * 入力形式で未対応の場合(形式は入力ファイルの拡張子で判別)、例外が発行されます。
	 *
	 * 正常に処理が終了した場合は、#close()を発行し、終了処理を行って下さい。
	 * また、例外が発生した場合は、#close(true)を発行し、終了処理を行って下さい。
	 *
	 * @param	outputName	出力ファイル名
	 * @throws Throwable 何らかのエラーが発生した場合。
	 */
	public void xls( final String outputName ) throws Throwable {
		saveDoc( outputName, getFilterName( getSuffix( inputName ), "xls" ) );
	}

	/**
	 * Writer(ods)出力を行います。
	 *
	 * 入力形式で未対応の場合(形式は入力ファイルの拡張子で判別)、例外が発行されます。
	 *
	 * 正常に処理が終了した場合は、#close()を発行し、終了処理を行って下さい。
	 * また、例外が発生した場合は、#close(true)を発行し、終了処理を行って下さい。
	 *
	 * @param	outputName	出力ファイル名
	 * @throws Throwable 何らかのエラーが発生した場合。
	 */
	public void odt( final String outputName ) throws Throwable {
		saveDoc( outputName, getFilterName( getSuffix( inputName ), "odt" ) );
	}

	/**
	 * Word(doc)出力を行います。
	 *
	 * 入力形式で未対応の場合(形式は入力ファイルの拡張子で判別)、例外が発行されます。
	 *
	 * 正常に処理が終了した場合は、#close()を発行し、終了処理を行って下さい。
	 * また、例外が発生した場合は、#close(true)を発行し、終了処理を行って下さい。
	 *
	 * @param	outputName	出力ファイル名
	 * @throws Throwable 何らかのエラーが発生した場合。
	 */
	public void doc( final String outputName ) throws Throwable {
		saveDoc( outputName, getFilterName( getSuffix( inputName ), "doc" ) );
	}

	/**
	 * Impress(odp)出力を行います。
	 *
	 * 入力形式で未対応の場合(形式は入力ファイルの拡張子で判別)、例外が発行されます。
	 *
	 * 正常に処理が終了した場合は、#close()を発行し、終了処理を行って下さい。
	 * また、例外が発生した場合は、#close(true)を発行し、終了処理を行って下さい。
	 *
	 * @param	outputName	出力ファイル名
	 * @throws Throwable 何らかのエラーが発生した場合。
	 */
	public void odp( final String outputName ) throws Throwable {
		saveDoc( outputName, getFilterName( getSuffix( inputName ), "odp" ) );
	}

	/**
	 * PowerPoint(ppt)出力を行います。
	 *
	 * 入力形式で未対応の場合(形式は入力ファイルの拡張子で判別)、例外が発行されます。
	 *
	 * 正常に処理が終了した場合は、#close()を発行し、終了処理を行って下さい。
	 * また、例外が発生した場合は、#close(true)を発行し、終了処理を行って下さい。
	 *
	 * @param	outputName	出力ファイル名
	 * @throws Throwable 何らかのエラーが発生した場合。
	 */
	public void ppt( final String outputName ) throws Throwable {
		saveDoc( outputName, getFilterName( getSuffix( inputName ), "ppt" ) );
	}

	/**
	 * 出力ファイルから出力形式を自動判別し、変換を行います。
	 *
	 * 入出力形式で未対応の場合(形式は入出力ファイルの拡張子で判別)、例外が発行されます。
	 *
	 * 正常に処理が終了した場合は、#close()を発行し、終了処理を行って下さい。
	 * また、例外が発生した場合は、#close(true)を発行し、終了処理を行って下さい。
	 *
	 * @param	outputName	出力ファイル名
	 * @throws Throwable 何らかのエラーが発生した場合。
	 */
	public void auto( final String outputName ) throws Throwable {
		String outSuffix = getSuffix( outputName );
		if( "pdf".equalsIgnoreCase( outSuffix ) ) {
			savePdf( outputName, getFilterName( getSuffix( inputName ), outSuffix ), null );
		}
		else {
			saveDoc( outputName, getFilterName( getSuffix( inputName ), outSuffix ) );
		}
	}

	/**
	 * フィルター名を指定して、各種ファイル形式に出力を行います。
	 *
	 * @param	outputName	出力ファイル名
	 * @param	filter		フィルター名
	 */
	private void saveDoc(  final String outputName, final String filter ) {
		if( xComp == null ) { throw new HybsSystemException( "初めに、#open()を実行して下さい" ); }
		if( !checkOutput( outputName ) ){ return; }

		PropertyValue[] storeProps = new PropertyValue[1];
		storeProps[0] = new PropertyValue();
		storeProps[0].Name = "FilterName";
		storeProps[0].Value = filter;

		String url = "file:///" + outputName.replace( '\\', '/' );
		@SuppressWarnings("cast")	// OpenOffice 3.2 での冗長なキャスト警告の抑止。キャストをはずすと、旧3.1 では、エラーになる。
		XStorable xstorable = (XStorable) UnoRuntime.queryInterface( XStorable.class, xComp );
		try {
			xstorable.storeAsURL( url, storeProps );
		}
		catch ( Throwable th ) {
			throw new HybsSystemException( "ファイルへの変換時にエラーが発生しました。[filter=" + filter + "]", th );
		}
	}

	/**
	 * フィルターを指定してPDF出力を行います。
	 *
	 * @param	outputName	出力ファイル名
	 * @param	filter		フィルター名
	 * @param	pdfPasswd	PDFパスワード
	 */
	private void savePdf( final String outputName, final String filter, final String pdfPasswd ) {
		if( xComp == null ) { throw new HybsSystemException( "初めに、#open()を実行して下さい" ); }
		if( !checkOutput( outputName ) ){ return; }

		PropertyValue[] storeProps;
		if( pdfPasswd == null || pdfPasswd.length() == 0 ) {
			storeProps = new PropertyValue[1];
			storeProps[0] = new PropertyValue();
			storeProps[0].Name = "FilterName";
			storeProps[0].Value = filter;
		}
		// 帳票要求テーブルでPDFパスワードが設定されている場合
		else {
			PropertyValue[] filterProps = new PropertyValue[2];
			filterProps[0] = new PropertyValue();
			filterProps[0].Name = "EncryptFile";
			filterProps[0].Value = true;
			filterProps[1] = new PropertyValue();
			filterProps[1].Name = "DocumentOpenPassword";
			filterProps[1].Value = pdfPasswd;

			storeProps = new PropertyValue[2];
			storeProps[0] = new PropertyValue();
			storeProps[0].Name = "FilterName";
			storeProps[0].Value = "calc_pdf_Export";
			storeProps[1] = new PropertyValue();
			storeProps[1].Name = "FilterData";
			storeProps[1].Value = filterProps;
		}

		String url = "file:///" + outputName.replace( '\\', '/' );
		@SuppressWarnings("cast")	// OpenOffice 3.2 での冗長なキャスト警告の抑止。キャストをはずすと、旧3.1 では、エラーになる。
		XStorable xstorable = (XStorable) UnoRuntime.queryInterface( XStorable.class, xComp );
		try {
			xstorable.storeToURL( url, storeProps );
		}
		catch ( Throwable th ) {
			throw new HybsSystemException( "PDFファイルへの変換時にエラーが発生しました。[filter=" + filter + "]", th );
		}
	}

	/**
	 * 出力ファイルのチェックを行います。
	 *
	 * @param	outputName	出力ファイル名
	 *
	 * @return	処理対象かどうか(入力ファイルと出力ファイルが同じ場合は、falseが返ります)
	 */
	private boolean checkOutput( final String outputName ) {
		if( outputName == null || outputName.length() == 0 ) {
			throw new HybsSystemException( "出力ファイルが指定されていません。" );
		}

		File inFile = new File( inputName );
		File outFile = new File( outputName );

		if( outFile.exists() ) {
			if( inFile.getAbsoluteFile().equals( outFile.getAbsoluteFile() ) ) {
				// 入力と出力が同じファイルの場合な何もしない
				return false;
			}
			else if( !outFile.delete() ) {
				throw new HybsSystemException( "出力先の既存ファイルが削除できません。[file=" + outputName + "]" );
			}
		}
		return true;
	}

	/**
	 * 入出力の形式(拡張子)からフィルター名を取得します。
	 *
	 * @param	inSuffix	入力拡張子
	 * @param	outSuffix	出力拡張子
	 *
	 * @return	フィルター名
	 */
	private static String getFilterName( final String inSuffix, final String outSuffix ) {
		String filterName = filterMap.get( inSuffix + "_" + outSuffix );
		if( filterName == null ) {
			String errMsg = "入力形式、出力形式は、以下の対応表に基づき、設定して下さい。" + HybsSystem.CR
							+ "入力[Calc(ods)   ,Excel(xls)     ] ⇒ 出力[Calc(ods)   ,Excel(xls)     ,PDF]" + HybsSystem.CR
							+ "入力[Writer(odt) ,Word(doc)      ] ⇒ 出力[Writer(odt) ,Word(doc)      ,PDF]" + HybsSystem.CR
							+ "入力[Impress(odp),PowerPoint(ppt)] ⇒ 出力[Impress(odp),PowerPoint(ppt),PDF]" + HybsSystem.CR;
			throw new HybsSystemException( errMsg );
		}
		return filterName;
	}

	/**
	 * ファイル名から拡張子(小文字)を求めます。
	 *
	 * @param	fileName	ファイル名
	 *
	 * @return	拡張子(小文字)
	 */
	private static String getSuffix( final String fileName ) {
		String suffix = null;
		if( fileName != null ) {
//			int sufIdx = fileName.lastIndexOf( "." );
			int sufIdx = fileName.lastIndexOf( '.' );
			if( sufIdx >= 0 ) {
				suffix = fileName.substring( sufIdx + 1 ).toLowerCase( Locale.JAPAN );
			}
		}
		return suffix;
	}

	/**
	 * ドキュメントの変換を行うための簡易メソッドです。
	 *
	 * 変換方法は、出力ファイルの拡張子により自動的に決定されます。
	 *
	 * @param	inputFile	入力ファイル名
	 * @param	outputFile	出力ファイル名
	 * @see #convert(String[], String, boolean)
	 */
	public static final void convert( final String inputFile, final String outputFile ) {
		convert( StringUtil.csv2Array( inputFile ), outputFile );
	}

	/**
	 * ドキュメントの変換を行うための簡易メソッドです。
	 *
	 * 変換方法は、出力ファイルの拡張子により自動的に決定されます。
	 *
	 * @param	inputFile	入力ファイル名配列
	 * @param	outputFile	出力ファイル名
	 * @see #convert(String[], String, boolean)
	 */
	public static final void convert( final String[] inputFile, final String outputFile ) {
		convert( inputFile, outputFile, true );
	}

	/**
	 * ドキュメントの変換を行うための簡易メソッドです。
	 *
	 * 変換方法は、出力ファイルの拡張子により自動的に決定されます。
	 *
	 * isOnlineがtrueに指定された場合、soffice.binのプロセスをファクトリークラス経由で生成し、
	 * キャッシュします。
	 * 但し、システムリソースが読み込まれないような、バッチファイルから起動した場合は、この方法は
	 * 利用できないため、isOnlineをfalseに指定する必要があります。
	 *
	 * @param	inputFile	入力ファイル名配列
	 * @param	outputFile	出力ファイル名
	 * @param	isOnline	オンライン(Web環境での使用)かどうか
	 */
	public static final void convert( final String inputFile[], final String outputFile, final boolean isOnline ) {
		DocConverter_OOO dc = new DocConverter_OOO( inputFile, isOnline );
		try {
			dc.open();
			dc.auto( outputFile );
			dc.close();
		}
		catch ( Throwable th ) {
			dc.close( true );
			throw new HybsSystemException( th );
		}
	}

	/**
	 * ドキュメントの変換を行います。
	 *
	 * 変換方法は、出力ファイルの拡張子により自動的に決定されます。
	 *
	 * @og.rev 4.3.1.1 (2008/08/23) mkdirs の戻り値判定
	 *
	 * @param	args	コマンド引数配列
	 * @throws Exception 何らかのエラーが発生したとき。
	 */
	public static void main( final String[] args ) throws Exception {
		if( args.length < 2 ) {
			System.out.println( "usage : OdsConverter [inputFile or inputDir] [outputDir]" );
			return;
		}

		File input = new File( args[0] );
		File output = new File( args[1] );

		// 4.3.1.1 (2008/08/23) mkdirs の戻り値判定
		if( output.mkdirs() ) {
			System.err.println( args[1] + " の ディレクトリ作成に失敗しました。" );
		}

		if( input.isDirectory() ) {
			File[] inputFiles = input.listFiles();
			for( int i = 0; i<inputFiles.length; i++ ) {
				String inputFile = inputFiles[i].getAbsolutePath();
				String outputFile = output.getAbsolutePath() + File.separator + inputFiles[i].getName().replace( ".xls", ".ods" );
				convert( StringUtil.csv2Array( inputFile ), outputFile, false );
			}
		}
		else {
			String inputFile = input.getAbsolutePath();
			String outputFile = output.getAbsolutePath() + File.separator + input.getName().replace( ".xls", ".ods" );
			convert( StringUtil.csv2Array( inputFile ), outputFile, false );
		}
	}
}
