/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.io;


import java.util.List;

import org.jfree.chart.plot.Plot;
import org.jfree.chart.plot.XYPlot;
import org.jfree.chart.plot.DatasetRenderingOrder;
import org.jfree.chart.plot.ValueMarker;
import org.jfree.chart.axis.DateAxis;
import org.jfree.chart.axis.ValueAxis;
import org.jfree.chart.renderer.xy.XYItemRenderer;
import org.jfree.chart.labels.StandardXYToolTipGenerator;
import org.jfree.data.xy.XYDataset;
import org.jfree.ui.Layer;

/**
 * ChartPlot_XY は、ChartPlot インターフェースを継承した実体クラスです。
 * JFreeChart では、XYPlot 関係の プロットを構築して、レンデラーや、データセットを
 * 設定していきます。
 * ここでは、複数のデータセットはサポートしていません。
 *
 * @og.rev 5.6.1.0 (2013/02/01) 新規作成
 *
 * @version  0.9.0	2007/06/21
 * @author	 Kazuhiko Hasegawa
 * @since	 JDK1.1,
 */
public class ChartPlot_XYTime implements ChartPlot {

	/**
	 * Plot オブジェクトを取得します。
	 *
	 * Plot オブジェクト には、その種類の応じた、データセットやレンデラーを
	 * 設定する必要があります。
	 * また、複数のデータセットや、それに関係する属性情報も、設定する必要が
	 * あります。
	 * Plot は、JFreeChart オブジェクトにつき、一つ用意しなければなりません。
	 * チャート合成時でも、Plot は一つです。
	 * 
	 * @og.rev 5.9.17.1 (2017/02/10) 色設定の反映
	 *
	 * @param	create	ChartCreateオブジェクト
	 *
	 * @return	Plotオブジェクト
	 */
	public Plot getPlot( final ChartCreate create ) {
		XYPlot xyPlot = new XYPlot();

		XYItemRenderer rend ;
		ValueAxis      yaxis ;

		DateAxis daxis = create.getDateAxis();
		xyPlot.setDomainAxis( daxis );

		xyPlot.setOrientation( create.getPlotOrientation() );
		xyPlot.setDatasetRenderingOrder(DatasetRenderingOrder.FORWARD);

		// クリッカブル・マップ
		HybsURLGenerator urlGen = create.getURLGenerator();
//		boolean useToolTip = create.getUseToolTip();	// 4.3.1.0 (2008/08/09) ツールチップス利用フラグ
		boolean useToolTip = create.isUseToolTip();		// 4.9.9.9 (2009/08/07) メソッド名変更

		List<ChartDataset> datasetList = create.getDatasetList();
		for(int idx=0; idx<datasetList.size(); idx++) {
			ChartDataset chDataset = datasetList.get(idx);


			// 4.1.1.0 (2008/02/04) XYItemRenderer を直接取得します。
			String chartType = chDataset.getChartType();
			TypeRenderer typeRen = ChartFactory.getTypeRenderer( chartType );
			rend = (XYItemRenderer)typeRen.getRenderer();

			xyPlot.setRenderer(idx,rend,false);
			// クリッカブル・マップ
			if( urlGen != null ) {
				rend.setURLGenerator( urlGen );
			}
			if( useToolTip ){	// 4.3.1.0 (2008/08/09) ツールチップスの利用
				rend.setBaseToolTipGenerator( new StandardXYToolTipGenerator() );
			}

			// 色設定  5.9.17.1 (2017/02/10)
			java.awt.Color[] clrs = chDataset.getSeriesColors();
			if( clrs != null && clrs.length>0){
				for( int i=0;i<clrs.length; i++){
					rend.setSeriesPaint( i, clrs[i] );
				}
			}
			
			
			yaxis = chDataset.makeNumberAxis();
			xyPlot.setRangeAxis( idx,yaxis,false );

			// 縦軸マーカーの設定(横ライン)
			ValueMarker[] marker = chDataset.getValueMarkers();
			for( int i=0; i<marker.length; i++ ) {
				xyPlot.addRangeMarker(idx,marker[i],Layer.FOREGROUND);
			}

			xyPlot.setDataset( idx,(XYDataset)chDataset.getDataset() );
		}

		return xyPlot;
	}
}
