/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.io;

import java.util.List;

import org.jfree.data.general.Dataset;
import org.jfree.data.general.PieDataset;
// import org.jfree.data.general.ValueDataset;			// 5.7.8.0 (2014/07/04)
import org.jfree.data.category.CategoryDataset;
import org.jfree.data.xy.XYDataset;
import org.jfree.chart.plot.Plot;
import org.jfree.chart.plot.MultiplePiePlot;
import org.jfree.chart.plot.PiePlot;
import org.jfree.chart.plot.PiePlot3D;
import org.jfree.chart.plot.RingPlot;
import org.jfree.chart.plot.SpiderWebPlot;
import org.jfree.chart.plot.PolarPlot;
// import org.jfree.chart.plot.MeterPlot;				// 5.7.8.0 (2014/07/04)
// import org.jfree.chart.plot.ThermometerPlot;			// 5.7.8.0 (2014/07/04)
// import org.jfree.chart.plot.CompassPlot;				// 5.7.8.0 (2014/07/04)
import org.jfree.chart.labels.StandardCategoryToolTipGenerator;
import org.jfree.chart.labels.StandardPieToolTipGenerator;

/**
 * ChartPlot_Pie は、ChartPlot インターフェースを継承した実体クラスです。
 * JFreeChart では、各種オブジェクトの組み合わせで、色々なグラフを作成できます。
 * チャートタイプが、複数種類存在するため、ここでは、特殊な方法として、各タイプ毎に
 * オブジェクトを構築しています。(ファクトリメソッド的な処理)
 *
 * @version  0.9.0	2007/06/21
 * @author	 Kazuhiko Hasegawa
 * @since	 JDK1.1,
 */
public class ChartPlot_Pie implements ChartPlot {

	/**
	 * Plot オブジェクトを取得します。
	 *
	 * Plot オブジェクト には、その種類の応じた、データセットやレンデラーを
	 * 設定する必要があります。
	 * また、複数のデータセットや、それに関係する属性情報も、設定する必要が
	 * あります。
	 * Plot は、JFreeChart オブジェクトにつき、一つ用意しなければなりません。
	 * チャート合成時でも、Plot は一つです。
	 *
	 * @og.rev 5.3.0.0 (2010/12/01) 特殊プロットの追加
	 * @og.rev 5.7.8.0 (2014/07/04) MeterPlot 、Compass 、Thermometer の機能追加
	 * @og.rev 5.9.10.4 (2016/07/19) Pieに色指定の反映
	 * @og.rev 5.9.17.1 (2017/02/10) Pie3D,Ringにも色指定反映
	 *
	 * @param	create	ChartCreateオブジェクト
	 *
	 * @return	Plotオブジェクト
	 */
	public Plot getPlot( final ChartCreate create ) {

		List<ChartDataset> datasetList = create.getDatasetList();
		ChartDataset chDataset = datasetList.get(0);

		Dataset dtset = chDataset.getDataset();

		// クリッカブル・マップ
		HybsURLGenerator urlGen = create.getURLGenerator();
		boolean useToolTip = create.isUseToolTip();		// 4.9.9.9 (2009/08/07) メソッド名変更

		Plot plot = null;
		String type = chDataset.getChartType();
		if( "MultiplePie".equalsIgnoreCase( type ) ) {
			plot = new MultiplePiePlot();
			((MultiplePiePlot)plot).setDataset( (CategoryDataset)dtset );
		}
		else if( "Pie".equalsIgnoreCase( type ) ) {
			plot = new PiePlot();
			((PiePlot)plot).setDataset( (PieDataset)dtset );
			if( urlGen != null ) {
				((PiePlot)plot).setURLGenerator( urlGen );
			}
			if( useToolTip ){	// 4.3.1.0 (2008/08/09) ツールチップスの利用
				((PiePlot)plot).setToolTipGenerator( new StandardPieToolTipGenerator() );
			}
			
			// 5.9.10.4 (2016/07/19) 色指定の反映
			java.awt.Color[] clrs = chDataset.getSeriesColors();
			if( clrs != null && clrs.length>0){
				for( int i=0;i<clrs.length; i++){
					((PiePlot)plot).setSectionPaint(i,clrs[i]); // 非推奨だがとりあえず暫定的にindexを使っておく
				}
			}
			
		}
		else if( "Pie3D".equalsIgnoreCase( type ) ) {
			plot = new PiePlot3D();
			((PiePlot3D)plot).setDataset( (PieDataset)dtset );
			if( urlGen != null ) {
				((PiePlot)plot).setURLGenerator( urlGen );
			}
			if( useToolTip ){	// 4.3.1.0 (2008/08/09) ツールチップスの利用
				((PiePlot)plot).setToolTipGenerator( new StandardPieToolTipGenerator() );
			}
			
			// 5.9.17.1 (2017/02/10)
			java.awt.Color[] clrs = chDataset.getSeriesColors();
			if( clrs != null && clrs.length>0){
				for( int i=0;i<clrs.length; i++){
					((PiePlot3D)plot).setSectionPaint(i,clrs[i]); // 非推奨だがとりあえず暫定的にindexを使っておく
				}
			}
		}
		else if( "Ring".equalsIgnoreCase( type ) ) {
			plot = new RingPlot();
			((RingPlot)plot).setDataset( (PieDataset)dtset );
			if( urlGen != null ) {
				((PiePlot)plot).setURLGenerator( urlGen );
			}
			if( useToolTip ){	// 4.3.1.0 (2008/08/09) ツールチップスの利用
				((RingPlot)plot).setToolTipGenerator( new StandardPieToolTipGenerator() );
			}
			
			// 5.9.17.1 (2017/02/10)
			java.awt.Color[] clrs = chDataset.getSeriesColors();
			if( clrs != null && clrs.length>0){
				for( int i=0;i<clrs.length; i++){
					((RingPlot)plot).setSectionPaint(i,clrs[i]); // 非推奨だがとりあえず暫定的にindexを使っておく
				}
			}
		}
		else if( "SpiderWeb".equalsIgnoreCase( type ) ) {
			plot = new SpiderWebPlot();
			((SpiderWebPlot)plot).setDataset( (CategoryDataset)dtset );
			if( urlGen != null ) {
				((SpiderWebPlot)plot).setURLGenerator( urlGen );
			}
			if( useToolTip ){	// 4.3.1.0 (2008/08/09) ツールチップスの利用
				((SpiderWebPlot)plot).setToolTipGenerator( new StandardCategoryToolTipGenerator() );
			}
		}
	 	// 5.3.0.0 (2010/12/01) 特殊プロットの追加
		else if( "Polar".equalsIgnoreCase( type ) ) {
			plot = new PolarPlot();
			((PolarPlot)plot).setDataset( (XYDataset)dtset );
		}
		else if( "Meter".equalsIgnoreCase( type ) ) {
			// 5.7.8.0 (2014/07/04) 機能追加
			plot = chDataset.makeMeterPlot();
//			plot = new MeterPlot();
//			((MeterPlot)plot).setDataset( (ValueDataset)dtset );
		}
		else if( "Thermometer".equalsIgnoreCase( type ) ) {
			// 5.7.8.0 (2014/07/04) 機能追加
			plot = chDataset.makeThermometerPlot();
//			plot = new ThermometerPlot();
//			((ThermometerPlot)plot).setDataset( (ValueDataset)dtset );
		}
		else if( "Compass".equalsIgnoreCase( type ) ) {
			// 5.7.8.0 (2014/07/04) 機能追加
			plot = chDataset.makeCompassPlot();
//			plot = new CompassPlot();
//			((CompassPlot)plot).addDataset( (ValueDataset)dtset );
		}

		return plot;
	}
}
